/*\t*******************************************************************/
/*    Last Edited .........  28-MAR-88 01:02:40                       */
/*    Filename  ...........  liuStrg.c                                */
/*    Project .............  n/a                                      */
/*    Author  .............  S. Balsarotti / B. Resnik / C. Strebel / */
/*                           S. Weidner / M. Ratcliff                 */
/*    Drawing Number  .....                                           */
/*    Revision  ...........                                           */
/*    Language  ...........  C                                        */
/*    Operating System  ...  n/a                                      */
/*    Processor  ..........  n/a                                      */
/*    Function:                                                       */
/*      Tons of handy dandy string processing utilities.              */
/*                                                                    */
/*\t*******************************************************************/


/*\r*******************************************************************/
/*                         Revision History                           */
/**********************************************************************/
/*

   Date    By           Change Description
dd-mmm-yy  nnn          text
---------  ----         -----------------------------------------------
22-DEC-88  SCB          Standards Update
29-MAR-89  BPR          More standards
21-JUL-89  SPT          Moved LIUmatchesTemplate to the File Utils Mod.
21-AUG-89  MJE          Fixed LIUreplaceChar to return error if input
                        string is NULL.
13-OCT-89  CES          Updates hierarchy information and charts
30-OCT-89  MJE          Updated module to reflect changes from code review.
29-JUN-90  BPR          Added functions LIUstrstri, LIUstrncmpi, and
                        LIUstrchri.
                        Replaced "LIL_PROC_NAME" with LIL_PROC_NAME.
03-sep-91  bpr          Replaced INT32 *pasNum with INT32 pasNum in
                        LIUstrncmpi
10-NOV-92  MJWR         Added LIUstripTrailingBlanks and
                        LIUstripLeadingBlanks functions.
15-Feb-93  WV           Added conditional compilation for include files.
25-JUN-93  MJWR         Ported to C++/Windows environment.
17-Jun-94  MJWR         Make ERAxxx compile time controlled
**\r*/

/*\i*******************************************************************/
/*                       Module Include Files                         */
/**********************************************************************/

/*                       System Include Files                         */
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <errno.h>

/*              Constant Include Files                  */

#include "liidSys.h"
#include "erapdefs.h"

/*              External Variable Include Files         */

/***************** External Procedure Include Files *******************/

#include "erapRpt.h"
#include "liuStrg.h"

/*\i*/

/*\m*******************************************************************/
/*                       Module Declarations                          */
/**********************************************************************/


/*                         Module Constants                           */


/*                         Module Variables                           */

/*                     Module Procedure Declarations                  */

/*\m*/


/*\p*******************************************************************/
/*                                                                    */
/*  NAME: LIUmakeUppercase                                            */
/*                                                                    */
/*  PURPOSE: Eliminate all lowercase characters from a string.        */
/*                                                                    */
/*                                                                    */
/*  SAMPLE CALL:  lilErr = LIUmakeUppercase (inputStr, outputStr);    */
/*                                                                    */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasInstringPtr     PAS    Input string                           */
/*                                                                    */
/*   refOutstringPtr    REF    Capitalized string                     */
/*                                                                    */
/*   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    */
/*                                                                    */
/*\p*******************************************************************/

STAT_TYPE LIUmakeUppercase
              /* Copy string using all uppercase characters */
(
const CHAR     *pasInstringPtr,
CHAR           *refOutstringPtr
)

{ /* LIUmakeUppercase */

/**********************************************************************/
/*                       LOCAL CONSTANT DECLARATIONS                  */
/**********************************************************************/

#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUmakeUppercase";
#endif
/**********************************************************************/
/*                        LOCAL VARIABLE DECLARATIONS                 */
/**********************************************************************/

STAT_TYPE       lilErr;
INT32           lilLen;
INT32           lili;
CHAR            lilch;

/**********************************************************************/
/*                            Procedure Body                          */
/**********************************************************************/

lilErr = SUCCEEDED_;
if (pasInstringPtr == NULL || refOutstringPtr == NULL)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  }  /* end if */

lilLen = strlen(pasInstringPtr);

for (lili=0; lili < lilLen; lili++)
  {
  lilch = pasInstringPtr[(int)lili];
  if (isalpha(lilch))
    {
    lilch = toupper(lilch);
    }
  refOutstringPtr[(int)lili] = lilch;
} /* End for loop */

refOutstringPtr[(int)lilLen] = '\0';

return( lilErr);
} /* LIUmakeUppercase */


/*\p*******************************************************************/
/*                                                                    */
/*  NAME: LIUreplaceChar                                              */
/*                                                                    */
/*  PURPOSE: Replace all instances of a character in a string with    */
/*  another char.                                                     */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*  = LIUreplaceChar (textPtr, '\n', '@');                            */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   refText            REF    Pointer to the text string to be       */
/*                             scanned for characters to be replaced  */
/*   pasSrc             PAS    character that will be replaced        */
/*   pasTar             PAS    character to replace it                */
/*   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    */
/*\p*******************************************************************/

STAT_TYPE LIUreplaceChar
/* Translate specified character in string */
(CHAR   *refText,
 CHAR    pasSrc,
 CHAR    pasTar)

{ /* LIUreplaceChar */
/*******************************************************************/
/*                      Local Constant Declarations                */
/*******************************************************************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUreplaceChar";
#endif


/*******************************************************************/
/*                      Local Variable Declarations                */
/*******************************************************************/

STAT_TYPE     lilErr;
INT32         lilDone;
INT32         lilTemp;
CHAR          lilTempStr[320];
CHAR          lilSrc[2];

/*******************************************************************/
/*                         Procedure Body                          */
/*******************************************************************/

lilErr = SUCCEEDED_;
lilSrc[0] = pasSrc;
lilSrc[1] = NULL;
if (refText == NULL)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog(LIL_PROC_NAME, "Input string is not valid.");
#endif
  }
else
  {
  if (pasSrc == pasTar)
    { 
/*** Do Nothing *****/
    }
  else
    {
    strcpy(lilTempStr, refText);
    lilDone = FALSE_;
    while(!lilDone)
      {
      lilTemp = strcspn(lilTempStr, lilSrc);
      if ((lilTemp > -1) && (lilTemp < (INT32)strlen(lilTempStr)))
        {
        lilTempStr[(int)lilTemp] = pasTar;
        }
      else
        {
        lilDone = TRUE_;
        } /* end if-else */
      } /* end while */
    strcpy(refText, lilTempStr);
    }  /* end else */
  } /* end else - if */
return(lilErr);

} /* end LIUreplaceChar */


/*\p*******************************************************************/
/*                                                                    */
/*  NAME: LIUintToString                                              */
/*                                                                    */
/*  PURPOSE: This function will convert an integer value to its       */
/*  string representation and also will place a decimal point         */
/*  anyWhere within the string.                                       */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*  = LIUintToString(Integer, Shift, StrLen, StrPtr);                 */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasInt             PAS    The base integer to convert to string. */
/*   pasShift           PAS    The position of the decimal point.     */
/*   pasStrLen          PAS    The length of string to return.        */
/*   refStr             REF    returning string.                      */
/*   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    */
/*                                                                    */
/*\p*******************************************************************/

STAT_TYPE LIUintToString
   /* Get ASCII string for integer & add decimal point */
(
INT32 pasInt,
INT32 pasShift,
INT32 pasStrLen,
CHAR *refStr
)

{ /* LIUintToString */
/*******************************************************************/
/*                      Local Constant Declarations                */
/*******************************************************************/
#define        LIL_STR_LEN      255
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUintToString";
#endif

/*******************************************************************/
/*                      Local Variable Declarations                */
/*******************************************************************/

  STAT_TYPE     lilErr;
  CHAR          lilStr1[LIL_STR_LEN];
  CHAR          lilStr2[LIL_STR_LEN];
  INT32         lilLen;
  INT32         lili;
  INT32         lilT;
  INT32         lilj;
  INT32         lilPos;
  INT32         lilShf;

/*******************************************************************/
/*                         Procedure Body                          */
/*******************************************************************/

lilErr = SUCCEEDED_;
if (pasInt < 0)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog ( LIL_PROC_NAME, "Cannot process negative numbers");
#endif
  } /* end if */

if (refStr == NULL)
  {
    lilErr = FAILED_;
#ifdef LIU_USE_ERA
    ERAerrorLog ( LIL_PROC_NAME, "String pointer is NULL");
#endif
  }
else
  {
  sprintf(lilStr1, "%d\0", pasInt);
  lilLen = strlen(lilStr1);
  if ( (lilLen - pasShift + 2 > pasStrLen) || (pasShift + 2 > pasStrLen)
                          || (pasStrLen > LIL_STR_LEN))
    {
    lilErr = FAILED_;
#ifdef LIU_USE_ERA
    ERAerrorLog ( LIL_PROC_NAME, "Insufficient string memory");
#endif
    } /* end if */
  else
    {
    if ((pasShift >= 0 ) && (lilLen <= pasShift))
      {
      /* pad the left side with zeros. */
  
      lilT = (pasShift - lilLen) + 1;
      for(lili = 0; lili < lilT; lili++)
        {
        lilStr2[(int)lili] = '0';
        } /* end for */
      for (lili = lilT; lili < (lilT + lilLen); lili++)
        {
        lilStr2[(int)lili] = lilStr1[(int)(lili - lilT)];
        }
      lilStr2[(int)lili] = '\0';
      lilPos = 1;
      } /* end if */
    else
      if (pasShift < 0)
      {
    
      /* pad the rigth side with zeros. */

      if (pasShift < 0)
        {
        lilShf = -pasShift;
        }
      else
        {
        lilShf = pasShift;
        }
      lilT = lilShf + 1;
      for (lili = 0; lili < lilLen; lili++)
        {
        lilStr2[(int)lili] = lilStr1[(int)lili];
        }
      for (lili = lilLen; lili < (lilLen + lilT); lili++)
        {
        lilStr2[(int)lili] = '0';
        }
      lilStr2[(int)lili] = '\0';
      lilPos = lilT + lilLen - 1;
      } 
    else
      {
      strcpy(lilStr2, lilStr1);
      lilPos = strlen(lilStr2) - pasShift;
      } /* end if-else-if */ 
  
  /*************************************************************/
  /* If the decimal point shift is not equal to zero then place*/
  /* the decimal point at the appropriate place in the string. */
  /* Otherwise leave the decimal point off.                    */
  /*************************************************************/

    if (pasShift != 0)
      { 
      lilj = 0;  
      for (lili = 0; lili < (INT32)(strlen(lilStr2) + 1); lili++) 
        {
        if (lili == lilPos)
          {
          refStr[(int)lili] = '.';
          } 
        else
          {
          refStr[(int)lili] = lilStr2[(int)lilj];
          lilj++;
          } /* end if-else */
        } /* end for */
      } 
    else
      {
      lili = strlen(lilStr2) + 1;
      strcpy(refStr, lilStr2);
    
      } /* end if - else */
    refStr[(int)lili] = '\0';

  /****************************************************************/
  /* If the string length is ok then return the string else pad   */
  /* the begining of the string with enough blank spaces to make  */
  /* up the requested final string length.                        */
  /****************************************************************/

    if ((INT32)strlen(refStr) == pasStrLen)
      lilErr = SUCCEEDED_;
    else
      {
      lilT = pasStrLen - strlen(refStr);
      strcpy(lilStr2, refStr);
      for (lili = 0; lili < lilT; lili++)
        refStr[(int)lili] = ' ';
      for (lili = lilT; lili < (lilT + strlen(lilStr2)); lili++)
        refStr[(int)lili] = lilStr2[(int)(lili - lilT)];
      refStr[(int)lili] = '\0';
      lilErr = SUCCEEDED_;
      } /* end if - else */
    } /* end else */
  } /* end else */
return(lilErr);
} /* end LIUintToString */

/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrcmp                                                  */
/*   and   LIUstrcmpi                                                 */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Case-sensitive followed by ..                              */
/*         Case-insensitive comparison of two strings; works          */
/*    same as standard strcmp otherwise.                              */
/*                                                                    */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*           lilErr = LIUstrcmpi    (strPtr1, strPtr2);               */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr1            PAS    String to be compared.                 */
/*   pasStr2            PAS    String to be compared.                 */
/*   lilStatus          FNC    (INT32) = 0, pasStr1 = pasStr2         */
/*                                     < 0, pasStr1 < pasStr2         */
/*                                     > 0, pasStr1 > pasStr2         */
/*                                                                    */
/*\p*******************************************************************/

INT32 LIUstrcmp // For completeness only
                   /* Compare strings, case-sensitive */
(
const CHAR *pasStr1,
const CHAR *pasStr2
)
{
return ((INT32)strcmp( (const char *)pasStr1, (const char *)pasStr2 ));
}

INT32 LIUstrcmpi
                   /* Compare strings, case-insensitive */
(
const CHAR *pasStr1,
const CHAR *pasStr2
)

{

/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/

INT32 lilIndex, lilMaxIndex;
INT32 lilStatus;

/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

if ((pasStr1 == NULL) || (pasStr2 == NULL))
  {
  lilStatus = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrcmpi", "Input strings are equal to NULL");
#endif
  }
else
  {
  lilIndex = 0;
  lilStatus = 0;
  lilMaxIndex = (INT32)strlen (pasStr1);
  if ((INT32)strlen (pasStr2) < lilMaxIndex)
    lilMaxIndex = strlen (pasStr2);
  while ( (lilIndex < lilMaxIndex) &&
          (tolower (pasStr1 [(int)lilIndex]) == tolower (pasStr2 [(int)lilIndex])) )
    lilIndex++;
  
  if  (lilIndex == lilMaxIndex)
    {
    if (strlen(pasStr1) != strlen (pasStr2))
      {
      if ((INT32)strlen(pasStr1) == lilMaxIndex)
        --lilStatus;
      else
        ++lilStatus;
      }
    }
  else
    {
    if (tolower (pasStr1 [(int)lilIndex]) < tolower (pasStr2 [(int)lilIndex]) )
      --lilStatus;
    else
      ++lilStatus;
    }
  }
return (lilStatus);

} /* end of function LIUstrcmpi */

/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrstr                                                  */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Returns pointer to a substring of pasStr1 that matches     */
/*    pasStr2; otherwise returns NULL                                 */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*        subStrPtr = LIUstrstr    (strPtr1, strPtr2);                */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr1            PAS    String to be searched.                 */
/*   pasStr2            PAS    String to be matched.                  */
/*   lilPtr             FNC    (INT8 *) pointer to substring match.   */
/*                             NULL if no match.                      */
/*                                                                    */
/*\p*******************************************************************/

CHAR *LIUstrstr  /* Find substring within string */
(
const CHAR *pasStr1,
const CHAR *pasStr2
)

{

/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/

CHAR *lilPtr;
INT32 lilStr2Len;

/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

if ((pasStr1 == NULL) || (pasStr2 == NULL))
{
  lilPtr = NULL;
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrstr", "Input strings are equal to NULL.");
#endif
}
else
{ 
  lilStr2Len = strlen (pasStr2);
  lilPtr = (CHAR *)pasStr1;
  while ((lilPtr != NULL) && 
            (strncmp (lilPtr, pasStr2, (int)lilStr2Len) != 0))
    lilPtr = strchr (++lilPtr, pasStr2 [0] );
} /* end if - else */

return (lilPtr);

} /* end of function LIUstrstr */

/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUfindSubfield                                            */
/*                                                                    */
/*  PURPOSE:  to find out if the template that is passed in matches   */
/*            any of the subfield in the field that was passed in,    */
/*            and if so, which subfield.                              */
/*                                                                    */
/*  SAMPLE CALL:  lilErr = LIUfindSubfield( lilField,                 */
/*                                          "MY_TEMPLATE", &lilPos);  */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasListPtr         PAS    (CHAR *) points to a string containing */
/*                             substrings separated by a single NULL  */
/*                             character.  pasListPtr  is double NULL */
/*                             terminated. (two consecutive NULL's)   */
/*   pasItemPtr         PAS    ptr to template string to search for   */
/*   refPosition        REF    subfield number (starting at 0)        */
/*                             that matches the template.             */
/*   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    */
/*                                                                    */
/*\p*******************************************************************/

STAT_TYPE LIUfindSubfield
              /* Get number of subfield matching template */
(
const CHAR        *pasListPtr,
const CHAR        *pasItemPtr,
INT32             *refPosition
)

{
/**********************************************************************/
/*                   Local Constant Declarations                      */
/**********************************************************************/
#ifdef LIU_USE_ERA
CHAR      *LIL_PROC_NAME  =  "LIUfindSubfield";
#endif
/**********************************************************************/
/*                   Local Variable Declarations                      */
/**********************************************************************/
STAT_TYPE  lilErr            ;/* error return flag                    */
CHAR      *lilSubfieldPtr    ;/* current position in the passed field */
INT32      lilSubfieldLen    ;/* length of current subfield           */
INT32      lilI              ;/* loop counter (='s current subfield)  */
INT32      lilFound          ;/* (T/F) has record been found?         */

/**********************************************************************/
/*                          Procedure Body                            */
/**********************************************************************/

lilErr = SUCCEEDED_;

if ((pasListPtr == NULL) || (pasItemPtr == NULL) || (refPosition == NULL))
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  }
else
  {
  lilSubfieldPtr = (CHAR *)pasListPtr ;
  lilFound = FALSE_;
  lilI = 0;
  while (lilSubfieldPtr[0] != '\0') 
    {
  /* find length of current subfield (including the NULL) */
    lilSubfieldLen = (INT32)strlen( lilSubfieldPtr) + 1;

  /* compare subfield to the template */
    if (strncmp(pasItemPtr     ,lilSubfieldPtr, (int)lilSubfieldLen) == 0)
      { /* then */

      /* template has been found, note position, stop looping */
      lilFound     = TRUE_;
      *refPosition = lilI;
      lilSubfieldPtr = "\0";
  
      } 
    else
      {
      /* go on looking in the remaining subfields */
      lilI++;
      lilSubfieldPtr += (int)lilSubfieldLen;
      } /* endif */
  
    } /* endwhile */
  
  /* report status of search */
  if (lilFound == TRUE_) 
    { /* then */
    lilErr = SUCCEEDED_;
    }
  else
    {
    lilErr = FAILED_;
    } /* end else */
  } /* end if - else */  
return( lilErr);
} /* end LIUfindSubfield */
/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrncmpi                                                */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Case-insensitive comparison of two strings; works          */
/*    same as standard strncmp otherwise.  The maximum numbers of     */
/*    characters to compare is specified.                             */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*           lilErr = LIUstrncmpi    (strPtr1, strPtr2, 30);          */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr1            PAS    String to be compared.                 */
/*   pasStr2            PAS    String to be compared.                 */
/*   pasNum             PAS    Number of characters to compare.       */
/*   lilStatus          FNC    (INT32) = 0, pasStr1 = pasStr2         */
/*                                     < 0, pasStr1 < pasStr2         */
/*                                     > 0, pasStr1 > pasStr2         */
/*                                                                    */
/*\p*******************************************************************/

INT32 LIUstrncmpi
                   /* Compare strings, case-insensitive */
(
const CHAR *pasStr1,
const CHAR *pasStr2,
INT32       pasNum
)

{

/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/

INT32 lilIndex, lilMaxIndex;
INT32 lilStatus;
INT32 lilStrlen1;
INT32 lilStrlen2;

/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

if ((pasStr1 == NULL) || (pasStr2 == NULL))
  {
  lilStatus = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrncmpi", "Input strings are equal to NULL");
#endif
  }
else
  {
  lilIndex = 0;
  lilStatus = 0;

  lilStrlen1 = strlen (pasStr1);
  lilStrlen2 = strlen (pasStr2);

  if (lilStrlen1 > lilStrlen2)
    lilMaxIndex = lilStrlen2;
  else
    lilMaxIndex = lilStrlen1;

  if (lilMaxIndex > pasNum)
    lilMaxIndex = pasNum;

  while ( (lilIndex < lilMaxIndex) &&
          (tolower (pasStr1 [(int)lilIndex]) ==
           tolower (pasStr2 [(int)lilIndex])) )
    lilIndex++;
  
  if  (lilIndex == lilMaxIndex)
    {
    if  (pasNum > lilMaxIndex)
      {
      if (lilStrlen1 != lilStrlen2)
        {
        if (lilStrlen1 < lilStrlen2)
          {
          --lilStatus;
          }
        else
          {
          ++lilStatus;
          }
        }
      }
    }
  else
    {
    if (tolower (pasStr1 [(int)lilIndex]) < tolower (pasStr2 [(int)lilIndex]) )
      --lilStatus;
    else
      ++lilStatus;
    }
  }
return (lilStatus);

} /* end of function LIUstrncmpi */

/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrstri                                                 */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Case insensitive LIUstrstr.                                */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*        subStrPtr = LIUstrstri    (strPtr1, strPtr2);               */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr1            PAS    String to be searched.                 */
/*   pasStr2            PAS    String to be matched.                  */
/*   lilPtr             FNC    (INT8 *) pointer to substring match.   */
/*                             NULL if no match.                      */
/*                                                                    */
/*\p*******************************************************************/

CHAR *LIUstrstri /* Find substring within string */
(
const CHAR *pasStr1,
const CHAR *pasStr2
)

{

/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/

CHAR *lilPtr;
INT32 lilStr2Len;

/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

if ((pasStr1 == NULL) || (pasStr2 == NULL))
{
  lilPtr = NULL;
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrstri", "Input strings are equal to NULL.");
#endif
}
else
{ 
  lilStr2Len = strlen (pasStr2);
  lilPtr = (CHAR *)pasStr1;
  while ((lilPtr != NULL) && 
            (LIUstrncmpi (lilPtr, pasStr2, lilStr2Len) != 0))
    lilPtr = LIUstrchri (++lilPtr, pasStr2[0] );
} /* end if - else */

return (lilPtr);

} /* end of function LIUstrstri */


/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrchri                                                 */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Case-insensitive strchr.                                   */
/*                                                                    */
/*                                                                    */
/*  SAMPLE CALL:                                                      */
/*                                                                    */
/*           lilStr = LIUstrchri    (strPtr1, strPtr2);               */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr             PAS    String to be searched.                 */
/*   pasChar            PAS    Character to search for.               */
/*   lilStr             FNC    (CHAR *) Location in search string of  */
/*                             first occurence of pasChar or NULL.    */
/*                                                                    */
/*\p*******************************************************************/

CHAR *LIUstrchri
                   /* case-insensitive strchr */
(const CHAR *pasStr,
 CHAR        pasChar)

{
/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/
CHAR  *lilStrPtr;
CHAR   lilChar;
/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

lilStrPtr = NULL;

if (pasStr == NULL)
  {
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrchri", "Input string is equal to NULL");
#endif
  }
else
  {
  lilChar = tolower (pasChar);
  lilStrPtr = (CHAR *)pasStr;
  while ((*lilStrPtr != '\0') && (tolower(*lilStrPtr) != lilChar))
    lilStrPtr++;
  if (!(*lilStrPtr)) lilStrPtr = NULL;
  }

return (lilStrPtr);
} /* end of function LIUstrchri */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripLeadingBlanks

    PURPOSE:  to remove any leading blanks from the referenced
      string.  This includes all blanks, new lines, carriage
      returns, and tab characters.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripLeadingBlanks
( CHAR *refStr )

{ /* LIUstripLeadingBlanks procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUstripLeadingBlanks";
#endif

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               lilErr;
CHAR                   *lilPtr;

/************************* Procedure Body *****************************/

lilErr = SUCCEEDED_;

if (refStr == NULL)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog( LIL_PROC_NAME, "Bad string pointer");
#endif
  }
else
  {
  lilErr = SUCCEEDED_;
  lilPtr = refStr;
  /**************************************************
  * Skip over all leading blanks to find start of
  * source string.
  */
  while (LIUisWhiteSpace( *lilPtr ) ) lilPtr++;
  /**************************************************
  * If pointer to end of blanks is not equal to
  * referenced string, then there are some
  * blanks to skip.  Copy from the skip pointer
  * back to the referenced string up to and including
  * the string terminator.
  */
  if (lilPtr != refStr) /* Then leading blanks to nuke */
    {
    do
      {
      *refStr = *lilPtr;
      refStr++;
      lilPtr++;
      }
    while (*lilPtr != '\0');
    *refStr = '\0';
    }
  }

return(lilErr);
} /* LIUstripLeadingBlanks end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripTrailingBlanks

    PURPOSE:  to strip all trailing blanks, if any, from the
      referenced string.  Simply convert all trailing blanks,
      tabs, newlines, and/or carriage returns to null terminators
      until some other character is encountered or the entire
      string has been processed.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripTrailingBlanks
( CHAR *refStr )

{ /* LIUstripTrailingBlanks procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUstripTrailingBlanks";
#endif

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               lilErr;
CHAR                   *lilPtr;

/************************* Procedure Body *****************************/

lilErr = SUCCEEDED_;

if (refStr == NULL)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog( LIL_PROC_NAME, "Null string pointer");
#endif
  }
else
  {
  lilPtr = refStr;
  lilPtr += strlen(refStr);
  if (lilPtr != refStr) /* then nonzero length */
    {
    lilPtr--;
    do
      {
      if (LIUisWhiteSpace( *lilPtr) )
        {
        *lilPtr = '\0';
        lilPtr--;
        }
      }
    while ((lilPtr >= refStr) && (LIUisWhiteSpace( *lilPtr )));
    }
  }

return(lilErr);
} /* LIUstripTrailingBlanks end */


/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripLTblanks

    PURPOSE:  to strip leading and trailing blanks from a string.
**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripLTblanks
( CHAR *refStr )

{ /* LIUstripLTblanks procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
/* CHARPTR LIL_PROC_NAME = "LIUstripLTblanks"; */
#endif

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE  lilErr;

/************************* Procedure Body *****************************/
lilErr = LIUstripTrailingBlanks( refStr );
if (lilErr == SUCCEEDED_) 
  {
  if (LIUisWhiteSpace( *refStr ))
    {
    lilErr = LIUstripLeadingBlanks( refStr );
    }
  }

return(lilErr);
} /* LIUstripLTblanks end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUisWhiteSpace

    PURPOSE:  to check if a character is "white space" such as
      a blank, tab, new line, or carriage return.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilWht             FNC    TRUE_ if blank space, else FALSE_      **
**\p*******************************************************************/

INT32 LIUisWhiteSpace
( INT32 pasCh )


{ /* LIUisWhiteSpace procedure */

/******************* Local Constant Declarations **********************/

/******************* Local Variable Declarations **********************/
int  lilWht;

/************************* Procedure Body *****************************/

lilWht = FALSE_;

if ((pasCh == ' ') ||
    (pasCh == '\t')||
    (pasCh == '\n')||
    (pasCh == '\r'))
  {
  lilWht = TRUE_;
  }

return(lilWht);
} /* LIUisWhiteSpace end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripULblanks

    PURPOSE:  to strip leading and trailing blanks from a string.
      Then convert any embedded blanks to underscore
      characters.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripULblanks
( CHAR *refStr )

{ /* LIUstripULblanks procedure */

/******************* Local Constant Declarations **********************/

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE  lilErr;
CHAR      *lilCptr;

/************************* Procedure Body *****************************/
lilErr = LIUstripLTblanks( refStr );

if (lilErr == SUCCEEDED_)
  {
  lilCptr = refStr;
  while (*lilCptr != '\0')
    {
    if (LIUisWhiteSpace( *lilCptr ) )
      {
      *lilCptr = '_';
      }
    lilCptr++;
    }
  }

return(lilErr);
} /* LIUstripULblanks end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstrncpy

    PURPOSE:  to copy no more than 'pasDestBufSize-1' characters
    from 'pasSrc' to 'refDest'.  Then make certain that 'refDest'
    is null terminated.  Works same as 'strncpy' but GUARANTEES
    the destination string is null terminated - something strncpy
    fails to do.  (If you pass a string that is NOT properly
    terminated to other string functions, you may violate your
    memory and crash the computer!)

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstrncpy
( CHAR              *refDest,
  const CHAR        *pasSrc,
  INT32             pasDestBufSize )

{ /* LIUstrncpy procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUstrncpy";
#endif

#define LIL_MIN_BUF_SIZE 2

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE  lilErr;

/************************* Procedure Body *****************************/


if ((refDest == NULL) || (pasSrc == NULL))
  {
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  lilErr = FAILED_;
  }
else if (pasDestBufSize < LIL_MIN_BUF_SIZE)
  {
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  lilErr = FAILED_;
  }
else
  {
  lilErr = SUCCEEDED_;
  strncpy( refDest, pasSrc, (size_t)pasDestBufSize );
  refDest[(int)pasDestBufSize-1] = '\0';
  }

return(lilErr);
} /* LIUstrncpy end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripLeadingChars

    PURPOSE:  to remove any leading characters from the referenced
      string.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripLeadingChars
( int   pasCharToStrip,
  CHAR *refStr )

{ /* LIUstripLeadingChars procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUstripLeadingChars";
#endif

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               lilErr;
CHAR                   *lilPtr;

/************************* Procedure Body *****************************/

lilErr = SUCCEEDED_;

if (refStr == NULL)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog( LIL_PROC_NAME, "Bad string pointer");
#endif
  }
else
  {
  lilErr = SUCCEEDED_;
  lilPtr = refStr;
  /**************************************************
  * Skip over all leading chars to find start of
  * source string.
  */
  while ( *lilPtr == pasCharToStrip ) lilPtr++;
  /**************************************************
  * If pointer to end of chars is not equal to
  * referenced string, then there are some
  * chars to skip.  Copy from the skip pointer
  * back to the referenced string up to and including
  * the string terminator.
  */
  if (lilPtr != refStr) /* Then leading chars to nuke */
    {
    do
      {
      *refStr = *lilPtr;
      refStr++;
      lilPtr++;
      }
    while (*lilPtr != '\0');
    *refStr = '\0';
    }
  }

return(lilErr);
} /* LIUstripLeadingChars end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripTrailingChars

    PURPOSE:  to strip all trailing chars, if any, from the
      referenced string.  Simply convert all trailing chars,
      tabs, newlines, and/or carriage returns to null terminators
      until some other character is encountered or the entire
      string has been processed.

**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripTrailingChars
( int   pasCharToStrip,
  CHAR *refStr )

{ /* LIUstripTrailingChars procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUstripTrailingChars";
#endif
/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE               lilErr;
CHAR                   *lilPtr;

/************************* Procedure Body *****************************/

lilErr = SUCCEEDED_;

if (refStr == NULL)
  {
  lilErr = FAILED_;
#ifdef LIU_USE_ERA
  ERAerrorLog( LIL_PROC_NAME, "Null string pointer");
#endif
  }
else
  {
  lilPtr = refStr;
  lilPtr += strlen(refStr);
  if (lilPtr != refStr) /* then nonzero length */
    {
    lilPtr--;
    do
      {
      if ( *lilPtr == pasCharToStrip )
        {
        *lilPtr = '\0';
        lilPtr--;
        }
      }
    while ((lilPtr > refStr) && (LIUisWhiteSpace( *lilPtr )));
    }
  }

return(lilErr);
} /* LIUstripTrailingChars end */


/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUstripLTchars

    PURPOSE:  to strip leading and trailing chars from a string.
**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUstripLTchars
( int   pasCharToStrip,
  CHAR *refStr )

{ /* LIUstripLTchars procedure */

/******************* Local Constant Declarations **********************/

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE  lilErr;

/************************* Procedure Body *****************************/
lilErr = LIUstripTrailingChars( pasCharToStrip, refStr );

if (lilErr == SUCCEEDED_) 
  {
  if ( *refStr == pasCharToStrip )
    {
    lilErr = LIUstripLeadingChars( pasCharToStrip, refStr );
    }
  }

return(lilErr);
} /* LIUstripLTchars end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUfortranToCstr

    PURPOSE:  convert a fortran string to a C string.
      The fortran string is EXACTLY 'pasFortranStringLen'
      bytes long.  It is padded with trailing blanks.
      It has NO null terminator.  The 'refCstring' is
      a C caharacter array AT LEAST 'pasFortranStringLen+1'
      bytes long.  After extracting and terminating the
      C string from the Fortran string, trailing blanks
      are stripped from the C string.  Leading
      blanks are NOT stripped however, as they MAY BE
      significant.
**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUfortranToCstr
(INT32          pasFortranStringLen,
 CHAR          *pasFortranString,
 CHAR          *refCstring )

{ /* LIUfortranToCstr procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUfortranToCstr";
#endif

/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE  lilErr;

/************************* Procedure Body *****************************/

lilErr = FAILED_;

if (pasFortranStringLen <= 0)
  {
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  }
else if (pasFortranString == NULL || refCstring == NULL)
  {
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  }
else
  {
  memcpy((char *)refCstring,
         (char *)pasFortranString,
         (size_t)pasFortranStringLen);
  refCstring[(int)pasFortranStringLen] = '\0';
  LIUstripTrailingBlanks(refCstring);
  lilErr = SUCCEEDED_;
  }

return(lilErr);
} /* LIUfortranToCstr end */

/*\p********************************************************************
**                                                                    **
**                                                                    **
    NAME:  LIUcToFortranStr

    PURPOSE:  convert a NULL terminated C string to a
      FORTRAN format string.

      The fortran string is EXACTLY 'pasFortranStringLen'
      bytes long.  Up to, but no more than, 'pasFortranStringLen'
      characters are copied from the C string to the fortran
      string.  The FORTRAN string is then padded, if necessary,
      with trailing blanks - no null terminator is used.
**                                                                    **
**                                                                    **
**  INTERFACE DEFINITION:                                             **
**     variable         def.          expected/description            **
**   ------------       -----  -------------------------------------  **
**   lilErr             FNC    (SUCCEEDED_ / FAILED_) error return    **
**\p*******************************************************************/

STAT_TYPE LIUcToFortranStr
(INT32          pasFortranStringLen,
 CHAR          *pasCstring,
 CHAR          *refFortranString )

{ /* LIUcToFortranStr procedure */

/******************* Local Constant Declarations **********************/
#ifdef LIU_USE_ERA
CHARPTR LIL_PROC_NAME = "LIUcToFortranStr";
#endif
/******************* Local Variable Declarations **********************/
/* error return flag       */
STAT_TYPE  lilErr;
int        lilLen;

/************************* Procedure Body *****************************/

lilErr = FAILED_;

if (pasFortranStringLen <= 0)
  {
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  }
else if (refFortranString == NULL || pasCstring == NULL)
  {
#ifdef LIU_USE_ERA
  ERAparameterError( LIL_PROC_NAME );
#endif
  }
else
  {
  strncpy((char *)refFortranString,
          (char *)pasCstring,
          (size_t)pasFortranStringLen);
  lilLen = (int)strlen(pasCstring);
  while (lilLen < pasFortranStringLen)
    {
    refFortranString[lilLen] = ' ';
    lilLen++;
    }
  lilErr = SUCCEEDED_;
  }

return(lilErr);
} /* LIUcToFortranStr end */

/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrchrLast                                              */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Find the last occurrence of the specified                  */
/*      character.                                                    */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr             PAS    String to be searched.                 */
/*   pasChar            PAS    Character to search for.               */
/*   lilStr             FNC    (CHAR *) Location in search string of  */
/*                             last occurence of pasChar or NULL.     */
/*                                                                    */
/*\p*******************************************************************/

CHAR *LIUstrchrLast
(const CHAR *pasStr,
 CHAR pasChar)

{
/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/
CHAR  *lilStrPtr;
/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

lilStrPtr = NULL;

if (pasStr == NULL)
  {
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrchrLast", "Input string is equal to NULL");
#endif
  }
else
  {
  lilStrPtr = (CHAR *)pasStr;
  lilStrPtr += (strlen(pasStr)-1);
  while ((lilStrPtr != pasStr) && (tolower(*lilStrPtr) != pasChar))
    lilStrPtr--;
  if (*lilStrPtr != pasChar) lilStrPtr = NULL;
  }
return (lilStrPtr);
} /* end of function LIUstrchrLast */

/*\p*******************************************************************/
/*                                                                    */
/*  NAME:  LIUstrchriLast                                             */
/*                                                                    */
/*  PURPOSE:                                                          */
/*         Find the last occurrence of the specified                  */
/*      character.                                                    */
/*                                                                    */
/*  INTERFACE DEFINITION:                                             */
/*     variable         def.          expected/description            */
/*   ------------       -----  -------------------------------------  */
/*   pasStr             PAS    String to be searched.                 */
/*   pasChar            PAS    Character to search for.               */
/*   lilStr             FNC    (CHAR *) Location in search string of  */
/*                             last occurence of pasChar or NULL.     */
/*                                                                    */
/*\p*******************************************************************/

CHAR *LIUstrchriLast
                   /* case-insensitive LIUstrchrLast */
(CHAR *pasStr,
 CHAR pasChar)

{
/*******************************************************************/
/*                        Local Variable Declarations              */
/*******************************************************************/
CHAR  *lilStrPtr;
CHAR   lilChar;
/*******************************************************************/
/*                            Procedure Body                       */
/*******************************************************************/

lilStrPtr = NULL;

if (pasStr == NULL)
  {
#ifdef LIU_USE_ERA
  ERAerrorLog("LIUstrchriLast", "Input string is equal to NULL");
#endif
  }
else
  {
  lilStrPtr = pasStr;
  lilStrPtr += (strlen(pasStr)-1);
  lilChar = tolower (pasChar);
  while ((lilStrPtr != pasStr) && (tolower(*lilStrPtr) != lilChar))
    lilStrPtr--;
  if (tolower(*lilStrPtr) != lilChar) lilStrPtr = NULL;
  }

return (lilStrPtr);
} /* end of function LIUstrchriLast */

/*\p********************************************************************
**                                                                    **
    NAME:  LIUisAllOneChar

    PURPOSE:  to see if the entire string conintains a repeating
      list of characters all equal to 'pasCharToCheck'. If so,
      return TRUE_, else FALSE_.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   lilOne             BOOLEAN, TRUE_ or FALSE_                      **
**\p*******************************************************************/

BOOLEAN LIUisAllOneChar( int   pasCharToCheck,
                         const CHAR *pasStr )
{
/*********** Local Constant & Variable Declarations *******************/
BOOLEAN lilOne;
char   *lilCptr;

/************************* Procedure Body *****************************/
lilOne = FALSE_;
if (pasStr)
  {
  if (*pasStr)
    {
    lilOne  = TRUE_;
    lilCptr = (CHAR *)pasStr;
    while (*lilCptr && lilOne == TRUE_)
      {
      if (*lilCptr != pasCharToCheck)
        {
        lilOne = FALSE_;
        }
      else
        {
        lilCptr++;
        }
      }
    }
  }
return(lilOne);
} /* LIUisAllOneChar end */

/*\p********************************************************************
**                                                                    **
    NAME:  LIUgetPlural
    NAME:  LIUgetAblankStr

    PURPOSE:  to return "s" if integer is != 1, else return ""
      Return " ", pointer to a single blank string. Useful
      when print formatting fields.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   lilPptr            const char *, plural string pointer           **
**\p*******************************************************************/
LOCAL char *limPlural   = "s";
LOCAL char *limSingular = "";
LOCAL char *limAblank   = " ";

const char *LIUgetPlural( INT32 pasCount )
{
/*********** Local Constant & Variable Declarations *******************/
const char   *lilPptr;

/************************* Procedure Body *****************************/
lilPptr = (const char *)limPlural;
if (pasCount == 1)
  {
  lilPptr = (const char *)limSingular;
  }
return(lilPptr);
} /* LIUgetPlural end */

const char *LIUgetAblankStr( )
{
return( (const char *)limAblank );
} /* LIUgetAblankStr end */

/*\p********************************************************************
**                                                                    **
    NAME:  LIUgetAcharStr
    NAME:  LIUdeleteAcharStr

    PURPOSE:  Get a string filled with the character specified,
      and null terminated - provide a complimentary delete
      function, which is equivalent to LIMDELA(charPtr);

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   lilCptr            char *, filled string pointer                 **
**\p*******************************************************************/


char *LIUgetAcharStr( int pasCharToGet,
                      int pasLenOfStr )
{

/*********** Local Constant & Variable Declarations *******************/
char   *lilCptr;

/************************* Procedure Body *****************************/

if (pasLenOfStr < 2)
  pasLenOfStr = 2;

lilCptr = new char[pasLenOfStr+1];

if (lilCptr)
  {
  memset( lilCptr, pasCharToGet, pasLenOfStr);
  lilCptr[pasLenOfStr] = '\0';
  }

return(lilCptr);
}



STAT_TYPE LIUdeleteAcharStr( char **pasAcharStr )
{

if (pasAcharStr)
  {
  if (*pasAcharStr)
    {
    delete [] (*pasAcharStr);
    *pasAcharStr = NULL;
    }
  }
return(SUCCEEDED_);
}

