/*--------------------------------------------------------------------*/
/*                                                                    */
/*  HALLOCTR.H                                                        */
/*                                                                    */
/*--------------------------------------------------------------------*/

#if !defined( HALLOCTR_H )
#define HALLOCTR_H

#if defined(__DPMI32__) || defined(__WIN32__)
#error Can not use huge allocation in a 32 bit model
#endif

#if !defined( __STDLIB_H )
#include <stdlib.h>
#endif  // __STDLIB_H

#if !defined( __DOS_H )
#include <dos.h>
#endif  // __DOS_H

#if !defined( __ALLOC_H )
#include <alloc.h>
#endif  // __ALLOC_H

#if !defined( __EXCEPT_H )
#include <except.h>
#endif  // __EXCEPT_H

#if !defined( __CLASSLIB_DEFS_H )
#include "classlib\defs.h"
#endif  // __CLASSLIB_DEFS_H

#if defined( BI_CLASSLIB_NO_po )
#pragma option -po-
#endif

class _BIDSCLASS THugeXAllocation
{
public:
    THugeXAllocation(unsigned long sz) : size(sz)
      {
      }

    unsigned long size;
};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  class THugeStandardAllocator                                          */
/*                                                                        */
/*  Provides class-specific farmalloc and farfree that                    */
/*  simply call the global farmalloc and farfree functions.   That        */
/*  is, THugeStandardAllocator does not provide any specialized behavior. */
/*  It is used in the non-managed versions of the parametrized            */
/*  containers.                                                           */
/*                                                                        */
/*------------------------------------------------------------------------*/

class _BIDSCLASS THugeStandardAllocator
{
public:
    void __huge * farmalloc(unsigned long sz)
        {
        void __huge * temp = (void __huge *) ::farmalloc( sz );
        if (!temp)
          {
          throw(THugeXAllocation(sz));
          }
        return (temp);
        }
    void farfree( void __huge *ptr )
        { ::farfree( (void __far *)ptr ); }
};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  class THugeSharedAllocator                                            */
/*                                                                        */
/*  Provides class-specific farmalloc and farfree that                    */
/*  allocate from shared memory.                                          */
/*                                                                        */
/*------------------------------------------------------------------------*/

class _BIDSCLASS THugeSharedAllocator
{
public:
    void __huge * farmalloc(unsigned long sz);
    void farfree( void __huge *ptr );
};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T, class Alloc> class THugeManaged_T                  */
/*                                                                        */
/*  Provides a parametrized wrapper for type T which uses a               */
/*  class-specific farmalloc and farfree functions supplied               */
/*  by Alloc.                                                             */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class THugeManaged_T : private Alloc
{

public:

    THugeManaged_T() {}
    THugeManaged_T( const T& t ) : data(t)
        {
        }

    const THugeManaged_T& operator = ( const THugeManaged_T& t )
        {
        data = t.data;
        return *this;
        }

    operator T&()
        {
        return data;
        }

    Alloc::farmalloc;
    Alloc::farfree;

private:

    T data;

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  THugeSharedAllocator::farmalloc                                       */
/*  THugeSharedAllocator::farfree                                         */
/*                                                                        */
/*  When compiling for WINDOWS, allocates memory as GMEM_DDESHARE.        */
/*  When compiling for DOS, uses the global farmalloc and                 */
/*  farfree functions.                                                    */
/*                                                                        */
/*------------------------------------------------------------------------*/

#if defined( _Windows )

#include <windows.h>

inline void __huge * THugeSharedAllocator::farmalloc(unsigned long sz)

  {

  HGLOBAL hMem = GlobalAlloc( GPTR | GMEM_DDESHARE, DWORD(sz) );

  if (hMem == NULL)
    {
    throw(THugeXAllocation(sz));
    }
  return ((void __huge *)GlobalLock( hMem ));
  }

inline void THugeSharedAllocator::farfree( void __huge *ptr )

  {

  HGLOBAL hMem = (HGLOBAL)GlobalHandle(FP_SEG((void __far *)ptr));

  if (GlobalUnlock(hMem))
      GlobalFree(hMem);
  }

#else

inline void __huge * THugeSharedAllocator::farmalloc(unsigned long sz)

  {

  void __huge * temp = (void __huge *) ::farmalloc( sz );

  if (!temp)
    {
    throw(THugeXAllocation(sz));
    }
  return (temp);
  }

inline void THugeSharedAllocator::farfree( void __huge *ptr )

  {
  ::farfree( (void __far *)ptr );
  }

#endif  // __WINDOWS__

#if defined( BI_CLASSLIB_NO_po )
#pragma option -po.
#endif

#endif  // HALLOCTR_H
