/*------------------------------------------------------------------------*/
/*                                                                        */
/*  HARRAYS.H                                                             */
/*                                                                        */
/*------------------------------------------------------------------------*/

#if !defined( HARRAYS_H )
#define HARRAYS_H

#if defined(__DPMI32__) || defined(__WIN32__)
#error Can not use huge arrays in a 32 bit model
#endif

//#define TEMPLATES

#if !defined( __MEM_H )
#include <mem.h>
#endif  // __MEM_H

#if !defined( __CLASSLIB_DEFS_H )
#include "classlib\defs.h"
#endif  // __CLASSLIB_DEFS_H

#if !defined( __CLASSLIB_SHDDEL_H )
#include "classlib\shddel.h"
#endif  // __CLASSLIB_SHDDEL_H

#if !defined( HALLOCTR_H )
#include "halloctr.h"
#endif  // HALLOCTR_H

#if !defined( HVECTIMP_H )
#include "hvectimp.h"
#endif  // HVECTIMP_H

#pragma option -Vo-
#if defined( BI_CLASSLIB_NO_po )
#pragma option -po-
#endif

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  [INTERNAL USE ONLY]                                                   */
/*                                                                        */
/*  template <class Vect, class T> class THugeArrayAsVectorImp            */
/*                                                                        */
/*  Implements the type-independent array operations, using a vector      */
/*  as the underlying implementation.  The type Vect specifies the        */
/*  form of the vector, either a TCHugeVectorImp<T0>, a                   */
/*  TSHugeVectorImp<T0>, a TICHugeVectorImp<T0>, or a                     */
/*  TISHugeVectorImp<T0>.  The type T specifies the                       */
/*  type of the objects to be put in the array.  When using               */
/*  TCHugeVectorImp<T0> or a TSHugeVectorImp<T0> T should be the same as  */
/*  T0. When using TICHugeVectorImp<T0> or TISHugeVectorImp<T0> T should  */
/*  be of type pointer to T0.  See THugeArrayAsVector and                 */
/*  TIHugeArrayAsVector for examples.                                     */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class Vect, class T> class THugeArrayAsVectorImp
{

public:

    THugeArrayAsVectorImp( long upper, long lower, long delta ) :
        Data( upper-lower+1,delta ),
        Lowerbound( lower )
        {
        }

    long LowerBound() const
        {
        return Lowerbound;
        }

    long UpperBound() const
        {
        return BoundBase( Data.Limit() )-1;
        }

    unsigned long ArraySize() const
        {
        return Data.Limit();
        }

    int IsFull() const
        {
        return Data.GetDelta() == 0 && Data.Count() >= Data.Limit();
        }

    int IsEmpty() const
        {
        return Data.Count() == 0;
        }

    unsigned long GetItemsInContainer() const
        {
        return Data.Count();
        }

    void Reallocate( unsigned long sz, unsigned long offset = 0 )
        {
        Data.Resize( sz, offset );
        }


    void SetData( long loc, const T & t )
        {
        PRECONDITION( loc >= Lowerbound && loc <= UpperBound() );
        Data[ ZeroBase(loc) ] = t;
        }

    void RemoveEntry( long loc )
        {
        SqueezeEntry( ZeroBase(loc) );
        }

    void SqueezeEntry( unsigned long loc )
        {
        PRECONDITION( loc < Data.Count() );
        Data.Detach( loc );
        }

    unsigned long ZeroBase( long loc ) const
        {
        return loc - Lowerbound;
        }

    long BoundBase( unsigned long loc ) const
        {
        return loc == ULONG_MAX ? LONG_MAX : loc + Lowerbound;
        }

    void Grow( long loc )
        {
        if( loc < LowerBound() )
            Reallocate( ArraySize() + (loc - Lowerbound) );
        else if( loc >= BoundBase( Data.Limit()) )
            Reallocate( ZeroBase(loc) );
        }

    long Lowerbound;

    Vect Data;

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  [INTERNAL USE ONLY]                                                   */
/*                                                                        */
/*  template <class Vect, class T> class TDHugeArrayAsVectorImp           */
/*                                                                        */
/*  Implements the fundamental array operations for direct arrays, using  */
/*  a vector as the underlying implementation.                            */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class Vect, class T> class TDHugeArrayAsVectorImp :
    public THugeArrayAsVectorImp<Vect,T>
{

public:

    typedef void (*IterFunc)(T __huge &, void *);
    typedef int  (*CondFunc)(const T __huge &, void *);

    TDHugeArrayAsVectorImp( long upper, long lower, long delta ) :
        THugeArrayAsVectorImp<Vect,T>( upper, lower, delta )
        {
        }

    int Add( const T & t )
        {
        return Data.Add(t);
        }

    int Detach( const T & t )
        {
        return Data.Detach(t);
        }

    int Detach( long loc )
        {
        return Data.Detach( ZeroBase(loc) );
        }

    int Destroy( const T & t )
        {
        return Detach(t);
        }

    int Destroy( long loc )
        {
        return Detach(loc);
        }

    int HasMember( const T & t ) const
        {
        return Data.Find(t) != ULONG_MAX;
        }

    long Find( const T & t ) const
        {
        return BoundBase( Data.Find( t ) );
        }

    T __huge & operator []( long loc )
        {
        Grow( loc+1 );
        return Data[ZeroBase(loc)];
        }

    T __huge & operator []( long loc ) const
        {
        PRECONDITION( loc >= Lowerbound && ZeroBase(loc) < Data.Count() );
        return Data[ZeroBase(loc)];
        }

    void ForEach( IterFunc iter, void *args )
        {
        if( !IsEmpty() )
            Data.ForEach( iter, args );
        }

    T __huge *FirstThat( CondFunc cond, void *args ) const
        {
        if( IsEmpty() )
            return 0;
        return Data.FirstThat( cond, args );
        }

    T __huge *LastThat( CondFunc cond, void *args ) const
        {
        if( IsEmpty() )
            return 0;
        return Data.LastThat( cond, args );
        }

    void Flush()
        {
        Data.Flush();
        }

protected:

    const T __huge & ItemAt( long i ) const
        {
        return Data[ ZeroBase(i) ];
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  [INTERNAL USE ONLY]                                                   */
/*                                                                        */
/*  template <class Vect, class T> class TIHugeArrayAsVectorImp           */
/*                                                                        */
/*  Implements the fundamental array operations for indirect arrays,      */
/*  using a vector as the underlying implementation.                      */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class Vect, class T> class TIHugeArrayAsVectorImp :
    public THugeArrayAsVectorImp<Vect,T *>, public TShouldDelete
{

public:

    typedef void (*IterFunc)(T &, void *);
    typedef int  (*CondFunc)(const T &, void *);

    TIHugeArrayAsVectorImp( long upper, long lower, long delta ) :
        THugeArrayAsVectorImp<Vect,T *>( upper, lower, delta )
        {
        }

    ~TIHugeArrayAsVectorImp()
        {
        Flush();
        }

    int Add( T *t )
        {
        return Data.Add(t);
        }

    int Detach( T *t,
                TShouldDelete::DeleteType dt = TShouldDelete::NoDelete )
        {
        return Data.Detach(t,DelObj(dt));
        }

    int Detach( long loc,
                TShouldDelete::DeleteType dt = TShouldDelete::NoDelete )
        {
        return Data.Detach( ZeroBase(loc), DelObj(dt) );
        }

    int Destroy( T *t )
        {
        return Detach(t,TShouldDelete::Delete);
        }

    int Destroy( long loc )
        {
        return Detach(loc,TShouldDelete::Delete);
        }

    int HasMember( const T *t ) const
        {
        return Data.Find(t) != ULONG_MAX;
        }

    long Find( const T *t ) const
        {
        return BoundBase( Data.Find( t ) );
        }

    T * __huge & operator []( long loc )
        {
        Grow( loc+1 );
        return Data[ZeroBase(loc)];
        }

    T * __huge & operator []( long loc ) const
        {
        PRECONDITION( loc >= Lowerbound && ZeroBase(loc) < Data.Count() );
        return Data[ZeroBase(loc)];
        }

    void ForEach( IterFunc iter, void *args )
        {
        if( !IsEmpty() )
            Data.ForEach( iter, args );
        }

    T *FirstThat( CondFunc cond, void *args ) const
        {
        if( IsEmpty() )
            return 0;
        return Data.FirstThat( cond, args );
        }

    T *LastThat( CondFunc cond, void *args ) const
        {
        if( IsEmpty() )
            return 0;
        return Data.LastThat( cond, args );
        }

    void Flush( DeleteType dt = DefDelete )
        {
        Data.Flush(DelObj(dt));
        }

protected:

    T * __huge & ItemAt( long i ) const
        {
        return Data[ ZeroBase(i) ];
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMHugeArrayAsVector              */
/*  template <class T,class Alloc> class TMHugeArrayAsVectorIterator      */
/*                                                                        */
/*  Implements a managed array of objects of type T, using a vector as    */
/*  the underlying implementation.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMHugeArrayAsVectorIterator;

template <class T, class Alloc> class TMHugeArrayAsVector :
    public TDHugeArrayAsVectorImp<TMCHugeVectorImp<T,Alloc>,T>
{

    friend TMHugeArrayAsVectorIterator<T,Alloc>;

public:

    TMHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TDHugeArrayAsVectorImp<TMCHugeVectorImp<T,Alloc>,T>( upper, lower, delta )
        {
        }

    int AddAt( const T & t, long loc )
        {
        return Data.AddAt( t, ZeroBase(loc) );
        }

};

template <class T, class Alloc> class TMHugeArrayAsVectorIterator :
    public TMCHugeVectorIteratorImp<T,Alloc>
{

public:

    TMHugeArrayAsVectorIterator( const TMHugeArrayAsVector<T,Alloc>& a ) :
        TMCHugeVectorIteratorImp<T,Alloc>( a.Data ) {}

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class THugeArrayAsVector                           */
/*  template <class T> class THugeArrayAsVectorIterator                   */
/*                                                                        */
/*  Implements an array of objects of type T, using a vector as the       */
/*  underlying implementation and THugeStandardAllocator as its memory    */
/*  manager.                                                              */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class THugeArrayAsVector :
    public TMHugeArrayAsVector<T,THugeStandardAllocator>
{

public:

    THugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TMHugeArrayAsVector<T,THugeStandardAllocator>( upper, lower, delta )
        {
        }

};

template <class T> class THugeArrayAsVectorIterator :
    public TMHugeArrayAsVectorIterator<T,THugeStandardAllocator>
{

public:

    THugeArrayAsVectorIterator( const THugeArrayAsVector<T>& a ) :
        TMHugeArrayAsVectorIterator<T,THugeStandardAllocator>(a)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMSHugeArrayAsVector             */
/*  template <class T,class Alloc> class TMSHugeArrayAsVectorIterator     */
/*                                                                        */
/*  Implements a managed, sorted array of objects of type T, using a      */
/*  vector as the underlying implementation.                              */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMSHugeArrayAsVectorIterator;

template <class T, class Alloc> class TMSHugeArrayAsVector :
    public TDHugeArrayAsVectorImp<TMSHugeVectorImp<T,Alloc>,T>
{

    friend TMSHugeArrayAsVectorIterator<T,Alloc>;

public:

    TMSHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TDHugeArrayAsVectorImp<TMSHugeVectorImp<T,Alloc>,T>( upper, lower, delta )
        {
        }

};

template <class T, class Alloc> class TMSHugeArrayAsVectorIterator :
    public TMSHugeVectorIteratorImp<T,Alloc>
{

public:

    TMSHugeArrayAsVectorIterator( const TMSHugeArrayAsVector<T,Alloc>& a ) :
        TMSHugeVectorIteratorImp<T,Alloc>( a.Data ) {}

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TSHugeArrayAsVector                          */
/*  template <class T> class TSHugeArrayAsVectorIterator                  */
/*                                                                        */
/*  Implements a sorted array of objects of type T, using a vector as     */
/*  the underlying implementation and THugeStandardAllocator as its       */
/*  memory manager.                                                       */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TSHugeArrayAsVector :
    public TMSHugeArrayAsVector<T,THugeStandardAllocator>
{

public:

    TSHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TMSHugeArrayAsVector<T,THugeStandardAllocator>( upper, lower, delta )
        {
        }

};

template <class T> class TSHugeArrayAsVectorIterator :
    public TMSHugeArrayAsVectorIterator<T,THugeStandardAllocator>
{

public:

    TSHugeArrayAsVectorIterator( const TSHugeArrayAsVector<T>& a ) :
        TMSHugeArrayAsVectorIterator<T,THugeStandardAllocator>( a ) {}

}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMIHugeArrayAsVector             */
/*  template <class T,class Alloc> class TMIHugeArrayAsVectorIterator     */
/*                                                                        */
/*  Implements a managed indirect array of objects of type T, using a     */
/*  vector as the underlying implementation.                              */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMIHugeArrayAsVectorIterator;

template <class T, class Alloc> class TMIHugeArrayAsVector :
    public TIHugeArrayAsVectorImp<TMICHugeVectorImp<T,Alloc>,T>
{

    friend TMIHugeArrayAsVectorIterator<T,Alloc>;

public:

    TMIHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TIHugeArrayAsVectorImp<TMICHugeVectorImp<T,Alloc>,T>( upper, lower, delta )
        {
        }

    int AddAt( T *t, long loc )
        {
        return Data.AddAt( t, ZeroBase(loc) );
        }

};

template <class T, class Alloc> class TMIHugeArrayAsVectorIterator :
    public TMICHugeVectorIteratorImp<T,Alloc>
{

public:

    TMIHugeArrayAsVectorIterator( const TMIHugeArrayAsVector<T,Alloc>& a ) :
        TMICHugeVectorIteratorImp<T,Alloc>( a.Data ) {}

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TIHugeArrayAsVector                          */
/*  template <class T> class TIHugeArrayAsVectorIterator                  */
/*                                                                        */
/*  Implements an indirect array of objects of type T, using a vector as  */
/*  the underlying implementation and THugeStandardAllocator as its       */
/*  memory manager.                                                       */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TIHugeArrayAsVector :
    public TMIHugeArrayAsVector<T,THugeStandardAllocator>
{

public:

    TIHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TMIHugeArrayAsVector<T,THugeStandardAllocator>( upper, lower, delta )
        {
        }

};

template <class T> class TIHugeArrayAsVectorIterator :
    public TMIHugeArrayAsVectorIterator<T,THugeStandardAllocator>
{

public:

    TIHugeArrayAsVectorIterator( const TIHugeArrayAsVector<T>& a ) :
        TMIHugeArrayAsVectorIterator<T,THugeStandardAllocator>(a)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMISHugeArrayAsVector            */
/*  template <class T,class Alloc> class TMISHugeArrayAsVectorIterator    */
/*                                                                        */
/*  Implements a managed, indirect sorted array of objects of type T,     */
/*  using a vector as the underlying implementation.                      */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMISHugeArrayAsVectorIterator;

template <class T, class Alloc> class TMISHugeArrayAsVector :
    public TIHugeArrayAsVectorImp<TMISHugeVectorImp<T,Alloc>,T>
{

    friend TMISHugeArrayAsVectorIterator<T,Alloc>;

public:

    TMISHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TIHugeArrayAsVectorImp<TMISHugeVectorImp<T,Alloc>,T>( upper, lower, delta )
        {
        }

};

template <class T, class Alloc> class TMISHugeArrayAsVectorIterator :
    public TMISHugeVectorIteratorImp<T,Alloc>
{

public:

    TMISHugeArrayAsVectorIterator( const TMISHugeArrayAsVector<T,Alloc>& a ) :
        TMISHugeVectorIteratorImp<T,Alloc>( a.Data ) {}

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TISHugeArrayAsVector                         */
/*  template <class T> class TISHugeArrayAsVectorIterator                 */
/*                                                                        */
/*  Implements an indirect sorted array of objects of type T, using a     */
/*  vector as the underlying implementation and THugeStandardAllocator as */
/*  its memory manager.                                                   */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TISHugeArrayAsVector :
    public TMISHugeArrayAsVector<T,THugeStandardAllocator>
{

public:

    TISHugeArrayAsVector( long upper, long lower = 0, long delta = 0 ) :
        TMISHugeArrayAsVector<T,THugeStandardAllocator>( upper, lower, delta )
        {
        }

};

template <class T> class TISHugeArrayAsVectorIterator :
    public TMISHugeArrayAsVectorIterator<T,THugeStandardAllocator>
{

public:

    TISHugeArrayAsVectorIterator( const TISHugeArrayAsVector<T>& a ) :
        TMISHugeArrayAsVectorIterator<T,THugeStandardAllocator>(a)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class THugeArray                                   */
/*  template <class T> class THugeArrayIterator                           */
/*                                                                        */
/*  Easy names for THugeArrayAsVector and THugeArrayAsVectorIterator      */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class THugeArray :
    public THugeArrayAsVector<T>
{

public:

    THugeArray( long upper, long lower = 0, long delta = 0 ) :
        THugeArrayAsVector<T>( upper, lower, delta )
        {
        }

};

template <class T> class THugeArrayIterator :
    public THugeArrayAsVectorIterator<T>
{

public:


    THugeArrayIterator( const THugeArray<T>& a ) :
        THugeArrayAsVectorIterator<T>(a)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TSHugeArray                                  */
/*  template <class T> class TSHugeArrayIterator                          */
/*                                                                        */
/*  Easy names for TSHugeArrayAsVector and TSHugeArrayAsVectorIterator    */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TSHugeArray :
    public TSHugeArrayAsVector<T>
{

public:

    TSHugeArray( long upper, long lower = 0, long delta = 0 ) :
        TSHugeArrayAsVector<T>( upper, lower, delta )
        {
        }

};

template <class T> class TSHugeArrayIterator :
    public TSHugeArrayAsVectorIterator<T>
{

public:


    TSHugeArrayIterator( const TSHugeArray<T>& a ) :
        TSHugeArrayAsVectorIterator<T>(a)
        {
        }

};

#if defined( BI_CLASSLIB_NO_po )
#pragma option -po.
#endif

#pragma option -Vo.

#endif  // HARRAYS_H
