/*------------------------------------------------------------------------*/
/*                                                                        */
/*  HVECTIMP.H                                                            */
/*                                                                        */
/*------------------------------------------------------------------------*/

#if !defined( HVECTIMP_H )
#define HVECTIMP_H

#if defined(__DPMI32__) || defined(__WIN32__)
#error Can not use huge vectors in a 32 bit model
#endif

#include <iostream.h>

#if !defined( __LIMITS_H )
#include <limits.h>
#endif  // __LIMITS_H

#if !defined( __CHECKS_H )
#include <checks.h>
#endif  // __CHECKS_H

#if !defined( __CLASSLIB_DEFS_H )
#include "classlib\defs.h"
#endif  // __CLASSLIB_DEFS_H

#if !defined( __CLASSLIB_STDTEMPL_H )
#include "classlib\stdtempl.h"
#endif  // __CLASSLIB_STDTEMPL_H

#if !defined( HALLOCTR_H )
#include "halloctr.h"
#endif  // HALLOCTR_H

#if !defined( __CLASSLIB_MEMMGR_H )
#include "classlib\memmgr.h"
#endif  // __CLASSLIB_MEMMGR_H

#if !defined( __CLASSLIB_VOIDP_H )
#include "classlib\voidp.h"
#endif  // __CLASSLIB_VOIDP_H

#pragma option -Vo-
#if defined( BI_CLASSLIB_NO_po )
#pragma option -po-
#endif

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T, class Alloc> class THugeVectorImpBase              */
/*                                                                        */
/*  Implements the base functionality for a managed vector of objects of  */
/*  type T.  Assumes that T has meaningful copy semantics and a default   */
/*  constructor.                                                          */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class THugeVectorImpBase :
    public Alloc
{

public:

    THugeVectorImpBase() : Data(0), Lim(0) {}

    THugeVectorImpBase( unsigned long sz, unsigned long d ) :
        Data( (T __huge *)(farmalloc(sizeof(T) * sz)) ),
        Lim(sz)
        {
        }

    THugeVectorImpBase( const THugeVectorImpBase<T,Alloc>& );

    const THugeVectorImpBase<T,Alloc>& operator = (
                const THugeVectorImpBase<T,Alloc>& );

    ~THugeVectorImpBase()
        {
        farfree (Data);
        }

    unsigned long Limit() const
        {
        return Lim;
        }

    virtual unsigned long Top() const
        {
        return Lim;
        }

    virtual unsigned long Count() const
        {
        return Lim;
        }

    int Resize( unsigned long, unsigned long = 0 );

    virtual unsigned long GetDelta() const
        {
        return 0;
        }

protected:

    T __huge * Data;
    unsigned long Lim;

    virtual void Zero( unsigned long, unsigned long)
        {
        }

};

template <class T, class Alloc>
THugeVectorImpBase<T,Alloc>::THugeVectorImpBase(
        const THugeVectorImpBase<T,Alloc>& v ) :
    Data( (T __huge *)(farmalloc(sizeof(T) * v.Lim)) ),
    Lim(v.Lim)
{
    PRECONDITION( Lim == 0 || (Data != 0 && v.Data != 0) );
    for( unsigned long i = 0; i < Lim; i++ )
        Data[i] = v.Data[i];
}

template <class T, class Alloc>
const THugeVectorImpBase<T,Alloc>& THugeVectorImpBase<T,Alloc>::operator = ( const THugeVectorImpBase<T,Alloc>& v )
{
    if( Data != v.Data )
        {
        farfree (Data);
        Data = (T __huge *)(farmalloc(sizeof(T) * v.Lim));
        CHECK( Data != 0 );
        Lim = v.Lim;
        for( unsigned long i = 0; i < Lim; i++ )
            Data[i] = v.Data[i];
        }
    return *this;
}

inline unsigned long NextDelta( unsigned long sz, unsigned long delta )
{
    return (sz%delta) ? ((sz+delta)/delta)*delta : sz;
}

template <class T, class Alloc>
int THugeVectorImpBase<T,Alloc>::Resize( unsigned long newSz, unsigned long offset )
{
    if( newSz <= Lim || GetDelta() == 0 )
        return 0;
    unsigned long sz = Lim + NextDelta( newSz - Lim, GetDelta() );
    T __huge * temp = (T __huge *)(farmalloc(sizeof(T) * sz));
    unsigned long last = min( sz-offset, Lim );
    for( unsigned long i = 0; i < last; i++ )
        temp[i+offset] = Data[i];
    farfree ( Data );
    Data = temp;
    Lim = sz;
    Zero( last+offset, sz );
    return 1;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T, class Alloc> class TMHugeVectorImp                 */
/*                                                                        */
/*  Implements a managed vector of objects of type T.  Assumes that       */
/*  T has meaningful copy semantics and a default constructor.            */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T,class Alloc> class TMHugeVectorIteratorImp;

template <class T,class Alloc> class TMHugeVectorImp :
    public THugeVectorImpBase<T,Alloc>
{

public:

    typedef void (*IterFunc)(T __huge &, void *);
    typedef int  (*CondFunc)(const T __huge &, void *);

    friend TMHugeVectorIteratorImp<T,Alloc>;

    TMHugeVectorImp() : THugeVectorImpBase<T,Alloc>() {}
    TMHugeVectorImp( unsigned long sz, unsigned long d = 0 ) : THugeVectorImpBase<T,Alloc>(sz,d) {}

    T __huge & operator [] ( unsigned long index )
        {
        PRECONDITION( Lim == 0 || Data != 0 && index < Lim );
        return Data[index];
        }

   T __huge & operator [] ( unsigned long index ) const
        {
        PRECONDITION( Lim > 0 && Data != 0 && index < Lim );
        return Data[index];
        }

    void Flush( unsigned long = ULONG_MAX, unsigned long = 0 ) {}

    void ForEach( IterFunc iter, void *args )
        {
        ForEach( iter, args, 0, Count() );
        }

    void ForEach( IterFunc iter, void *args,
                  unsigned long start, unsigned long stop );

    T __huge *FirstThat( CondFunc cond, void *args,
                           unsigned long start, unsigned long stop) const;

    T __huge *FirstThat( CondFunc cond, void *args ) const
        {
        return FirstThat( cond, args, 0, Count() );
        }

    T __huge *LastThat( CondFunc cond, void *args,
                          unsigned long start, unsigned long stop ) const;

    T __huge *LastThat( CondFunc cond, void *args ) const
        {
        return LastThat( cond, args, 0, Count() );
        }

};

template <class T, class Alloc>
void TMHugeVectorImp<T,Alloc>::ForEach( IterFunc iter, void *args,
                                    unsigned long start, unsigned long stop )
{
    for( unsigned long cur = start; cur < stop; cur++ )
        iter( Data[cur], args );
}

template <class T, class Alloc>
T __huge *TMHugeVectorImp<T,Alloc>::FirstThat( CondFunc cond, void *args,
                                    unsigned long start, unsigned long stop ) const
{
    for( unsigned long cur = start; cur < stop; cur++ )
        if( cond( Data[cur], args ) != 0 )
            return &(T __huge &)Data[cur];
    return 0;
}

template <class T, class Alloc>
T __huge *TMHugeVectorImp<T,Alloc>::LastThat( CondFunc cond, void *args,
                                   unsigned long start, unsigned long stop ) const
{
    T __huge *res = 0;
    for( unsigned long cur = start; cur < stop; cur++ )
        if( cond( Data[cur], args ) != 0 )
            res = &(T __huge &)Data[cur];
    return res;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T, class Alloc> class TMHugeVectorIteratorImp         */
/*                                                                        */
/*  Implements a vector iterator.  This iterator works with any direct,   */
/*  managed vector.  For indirect vectors, see TMIHugeVectorIteratorImp.  */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMHugeVectorIteratorImp
{

public:

    TMHugeVectorIteratorImp( const TMHugeVectorImp<T,Alloc>&v )
        {
        Vect = &v;
        Restart(0,v.Top());
        }

    TMHugeVectorIteratorImp( const TMHugeVectorImp<T,Alloc>&v,
                         unsigned long start,
                         unsigned long stop
                         )
        {
        Vect = &v;
        Restart( start, stop );
        }

    operator int() const
        {
        return Cur < Upper;
        }

    const T __huge & Current() const
        {
        PRECONDITION( Cur < Upper );
        return (*Vect)[Cur];
        }

    const T __huge & operator ++ ( int )
        {
        const T __huge & temp = Current();
        Cur++;
        return temp;
        }

    const T __huge & operator ++ ()
        {
        PRECONDITION( Cur < Upper );
        Cur++;
        return Current();
        }

    void Restart()
        {
        Restart(Lower,Upper);
        }

    void Restart( unsigned long start, unsigned long stop )
        {
        Cur = Lower = start;
        Upper = stop;
        }

private:

    const TMHugeVectorImp<T,Alloc> *Vect;
    unsigned long Cur;
    unsigned long Lower, Upper;

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class THugeVectorImp                               */
/*  template <class T> class THugeVectorIteratorImp                       */
/*                                                                        */
/*  Implements a vector of objects of type T using THugeStandardAllocator */
/*  as its memory manager. Assumes that T has meaningful copy semantics   */
/*  and a default constructor.                                            */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class THugeVectorImp :
    public TMHugeVectorImp<T,THugeStandardAllocator>
{

public:

    THugeVectorImp()
        {
        }

    THugeVectorImp( unsigned long sz, unsigned long = 0 ) :
        TMHugeVectorImp<T,THugeStandardAllocator>( sz )
        {
        }

    THugeVectorImp( const THugeVectorImp<T>& v ) :
        TMHugeVectorImp<T,THugeStandardAllocator>( v )
        {
        }

};

template <class T> class THugeVectorIteratorImp :
    public TMHugeVectorIteratorImp<T,THugeStandardAllocator>
{

public:

    THugeVectorIteratorImp( const THugeVectorImp<T>& v ) :
        TMHugeVectorIteratorImp<T,THugeStandardAllocator>(v)
        {
        }

    THugeVectorIteratorImp( const THugeVectorImp<T>& v,
                        unsigned long start,
                        unsigned long stop
                        ) :
        TMHugeVectorIteratorImp<T,THugeStandardAllocator>(v,start,stop)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T, class Alloc> class TMCHugeVectorImp                */
/*  template <class T, class Alloc> class TMCHugeVectorIteratorImp        */
/*                                                                        */
/*  Implements a managed, counted vector of objects of type T.  Assumes   */
/*  that T has meaningful copy semantics and a default constructor.       */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMCHugeVectorImp :
    public TMHugeVectorImp<T,Alloc>
{

public:

    TMCHugeVectorImp() :
        Count_(0),
        Delta(0)
        {
        }

    TMCHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMHugeVectorImp<T,Alloc>( sz ),
        Count_(0),
        Delta(d)
        {
        }

    int Add( const T& );
    int AddAt( const T&, unsigned long);
    int Detach( const T& t )
        {
        return Detach( Find(t) );
        }

    int Detach( unsigned long loc );

    int IsEmpty() const
        {
        return Count_ == 0;
        }

    void Flush( unsigned long stop = ULONG_MAX,
                unsigned long start = 0 )
        {
        TMHugeVectorImp<T,Alloc>::Flush( stop, start );
        Count_ = 0;
        }

    virtual unsigned long Find( const T& ) const;

    virtual unsigned long Top() const
        {
        return Count_;
        }

    virtual unsigned long Count() const
        {
        return Count_;
        }

    virtual unsigned long GetDelta() const
        {
        return Delta;
        }

protected:

    unsigned long Count_;
    unsigned long Delta;

};

template <class T,class Alloc> class TMCHugeVectorIteratorImp :
    public TMHugeVectorIteratorImp<T,Alloc>
{

public:

    TMCHugeVectorIteratorImp( const TMCHugeVectorImp<T,Alloc>& v ) :
        TMHugeVectorIteratorImp<T,Alloc>(v)
        {
        }

    TMCHugeVectorIteratorImp( const TMCHugeVectorImp<T,Alloc>& v,
                          unsigned long start,
                          unsigned long stop
                        ) :
        TMHugeVectorIteratorImp<T,Alloc>(v,start,stop)
        {
        }

};

template <class T, class Alloc> int TMCHugeVectorImp<T,Alloc>::Add( const T& t )
{
    if( Count_ >= Lim && !Resize( Count_+1 ) )
        return 0;
    Data[Count_++] = t;
    return 1;
}

template <class T, class Alloc> 
int TMCHugeVectorImp<T,Alloc>::AddAt( const T& t, unsigned long loc )
{
    if( loc >= Lim && !Resize(loc+1) )
        return 0;
    if( Count_ == Lim && !Resize(Lim+1) )
        return 0;
    if( loc > Count_ )
        Count_ = loc;
    for( unsigned long cur = Count_; cur > loc; cur-- )
        Data[cur] = Data[cur-1];
    Data[loc] = t;
    Count_++;
    return 1;
}

template <class T, class Alloc>
int TMCHugeVectorImp<T,Alloc>::Detach( unsigned long loc )
{
    if( loc >= Lim )
        return 0;
    if( loc >= Count_ )
        {
        Zero( loc, loc+1 ); // removing an element that's not
        return 1;           // in the counted portion
        }
    Count_--;
    for( unsigned long cur = loc; cur < Count_; cur++ )
        Data[cur] = Data[cur+1];
    return 1;
}

template <class T, class Alloc>
unsigned long TMCHugeVectorImp<T,Alloc>::Find( const T& t ) const
{
    for( unsigned long loc = 0; loc < Count_; loc++ )
        if( Data[loc] == t )
            return loc;
    return ULONG_MAX;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TCHugeVectorImp                              */
/*  template <class T> class TCHugeVectorIteratorImp                      */
/*                                                                        */
/*  Implements a counted vector of objects of type T using                */
/*  THugeStandardAllocator as its memory manager.  Assumes                */
/*  that T has meaningful copy semantics and a default constructor.       */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TCHugeVectorImp :
    public TMCHugeVectorImp<T,THugeStandardAllocator>
{

public:

    TCHugeVectorImp()
        {
        }

    TCHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMCHugeVectorImp<T,THugeStandardAllocator>( sz, d )
        {
        }

};

template <class T> class TCHugeVectorIteratorImp :
    public TMCHugeVectorIteratorImp<T,THugeStandardAllocator>
{

public:

    TCHugeVectorIteratorImp( const TCHugeVectorImp<T>& v ) :
        TMCHugeVectorIteratorImp<T,THugeStandardAllocator>(v)
        {
        }

    TCHugeVectorIteratorImp( const TCHugeVectorImp<T>& v,
                         unsigned long start,
                         unsigned long stop
                        ) :
        TMCHugeVectorIteratorImp<T,THugeStandardAllocator>(v,start,stop)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T, class Alloc> class TMSHugeVectorImp                */
/*  template <class T, class Alloc> class TMSHugeVectorIteratorImp        */
/*                                                                        */
/*  Implements a managed, sorted vector of objects of type T.  Assumes    */
/*  that T has meaningful copy semantics, a meaningful < operator,        */
/*  and a default constructor.                                            */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMSHugeVectorImp :
    public TMCHugeVectorImp<T,Alloc>
{

public:

    TMSHugeVectorImp()
        {
        }

    TMSHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMCHugeVectorImp<T,Alloc>( sz, d )
        {
        }

    int Add( const T& );

    virtual unsigned long Find( const T& ) const;

};

template <class T,class Alloc> class TMSHugeVectorIteratorImp :
    public TMCHugeVectorIteratorImp<T,Alloc>
{

public:

    TMSHugeVectorIteratorImp( const TMSHugeVectorImp<T,Alloc>& v ) :
        TMCHugeVectorIteratorImp<T,Alloc>(v)
        {
        }

    TMSHugeVectorIteratorImp( const TMSHugeVectorImp<T,Alloc>& v,
                          unsigned long start,
                          unsigned long stop
                        ) :
        TMCHugeVectorIteratorImp<T,Alloc>(v,start,stop)
        {
        }

};

template <class T, class Alloc> int TMSHugeVectorImp<T,Alloc>::Add( const T& t )
{
    unsigned long loc = Count_++;
    if( Count_ > Lim )
        if( !Resize( Count_ ) )
            {
            --Count_;
            return 0;
            }
    while( loc > 0 && t < Data[loc-1] )
        {
        Data[loc] = Data[loc-1];
        loc--;
        }
    Data[loc] = t;
    return 1;
}

template <class T, class Alloc>
unsigned long TMSHugeVectorImp<T,Alloc>::Find( const T& t ) const
{
    if( Count_ == 0 )
        return ULONG_MAX;

    unsigned long lower = 0;
    unsigned long upper = Count_-1;

    while( lower < upper && upper != ULONG_MAX )
        {
        unsigned long middle = (lower+upper)/2;
        if( (T __huge &)Data[middle] == (T&)t )
            return middle;
        if( (T __huge &)Data[middle] < (T&)t )
            lower = middle+1;
        else
            upper = middle-1;
        }

    if( lower == upper && (T __huge &)Data[lower] == (T&)t )
        return lower;
    else
        return ULONG_MAX;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TSHugeVectorImp                              */
/*  template <class T> class TSHugeVectorIteratorImp                      */
/*                                                                        */
/*  Implements a sorted vector of objects of type T using                 */
/*  THugeStandardAllocator as its memory manager.  Assumes                */
/*  that T has meaningful copy semantics, a meaningful < operator,        */
/*  and a default constructor.                                            */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TSHugeVectorImp :
    public TMSHugeVectorImp<T,THugeStandardAllocator>
{

public:

    TSHugeVectorImp()
        {
        }

    TSHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMSHugeVectorImp<T,THugeStandardAllocator>( sz, d )
        {
        }

};

template <class T> class TSHugeVectorIteratorImp :
    public TMSHugeVectorIteratorImp<T,THugeStandardAllocator>
{

public:

    TSHugeVectorIteratorImp( const TSHugeVectorImp<T>& v ) :
        TMSHugeVectorIteratorImp<T,THugeStandardAllocator>(v)
        {
        }

    TSHugeVectorIteratorImp( const TSHugeVectorImp<T>& v,
                         unsigned long start,
                         unsigned long stop
                        ) :
        TMSHugeVectorIteratorImp<T,THugeStandardAllocator>(v,start,stop)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMIHugeVectorImp                 */
/*                                                                        */
/*  Implements a managed vector of pointers to objects of type T.         */
/*  Since pointers always have meaningful copy semantics, this class      */
/*  can handle any type of object.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T,class Alloc> class TMIHugeVectorImp :
    public THugeVectorImpBase<TVoidPointer,Alloc>
{

public:

    typedef void (*IterFunc)(T&, void *);
    typedef int  (*CondFunc)(const T&, void *);

    TMIHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        THugeVectorImpBase<TVoidPointer,Alloc>(sz,d) {}

    T * __huge & operator [] ( unsigned long index )
        {
        PRECONDITION( Lim == 0 || Data != 0 && index < Lim );
        return *STATIC_CAST(T * __huge *,STATIC_CAST(void __huge *,&Data[index]));
        }

    T * __huge & operator [] ( unsigned long index ) const
        {
        PRECONDITION( Lim > 0 && Data != 0 && index < Lim );
        return *STATIC_CAST(T * __huge *,STATIC_CAST(void __huge *,&Data[index]));
        }

    void Flush( unsigned del = 0,
                unsigned long stop = ULONG_MAX,
                unsigned long start = 0 );

    void ForEach( IterFunc iter, void *args )
        {
        ForEach( iter, args, 0, Count() );
        }

    void ForEach( IterFunc iter, void *args,
                  unsigned long start, unsigned long stop );

    T *FirstThat( CondFunc cond, void *args ) const
        {
        return FirstThat( cond, args, 0, Count() );
        }

    T *FirstThat( CondFunc cond, void *args,
                           unsigned long start, unsigned long stop ) const;

    T *LastThat( CondFunc cond, void *args ) const
        {
        return LastThat( cond, args, 0, Count() );
        }

    T *LastThat( CondFunc cond, void *args,
                          unsigned long start, unsigned long stop ) const;

protected:

    void Zero( unsigned long, unsigned long);

private:

    static void DelObj(T &, void * );

};

template <class T, class Alloc>
void TMIHugeVectorImp<T,Alloc>::DelObj(T & tRef, void * )
{
    delete &tRef ;
}

template <class T, class Alloc>
void TMIHugeVectorImp<T,Alloc>::Flush( unsigned del,
                                   unsigned long upr,
                                   unsigned long lwr )
{
    upr = min( upr, Limit() );
    if( del )
        ForEach( DelObj, 0, lwr, upr );
    Zero( lwr, upr );
}

template <class T, class Alloc>
void TMIHugeVectorImp<T,Alloc>::ForEach( IterFunc iter, void *args,
                                    unsigned long start, unsigned long stop )
{
    for( unsigned long cur = start; cur < stop; cur++ )
        if( Data[cur] != 0 )
            iter( *STATIC_CAST(T *,STATIC_CAST(void *,Data[cur])), args );
}

template <class T, class Alloc>
T *TMIHugeVectorImp<T,Alloc>::FirstThat( CondFunc cond,
                                     void *args,
                                     unsigned long start,
                                     unsigned long stop ) const
{
    for( unsigned long cur = start; cur < stop; cur++ )
        if( Data[cur] != 0 &&
            cond( *STATIC_CAST(T *,STATIC_CAST(void *,Data[cur])), args ) != 0 )
            return STATIC_CAST(T *,STATIC_CAST(void *,Data[cur]));
    return 0;
}

template <class T, class Alloc>
T *TMIHugeVectorImp<T,Alloc>::LastThat( CondFunc cond,
                                    void *args,
                                    unsigned long start,
                                    unsigned long stop ) const
{
    T *res = 0;
    for( unsigned long cur = start; cur < stop; cur++ )
        if( Data[cur] != 0 &&
            cond( *STATIC_CAST(T *,STATIC_CAST(void *,Data[cur])), args ) != 0 )
            res = STATIC_CAST(T *,STATIC_CAST(void *,Data[cur]));
    return res;
}

template <class T, class Alloc>
void TMIHugeVectorImp<T,Alloc>::Zero( unsigned long lwr, unsigned long upr )
{
    for( unsigned long i = lwr; i < min( Limit(), upr ); i++ )
        Data[i] = 0;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMIHugeVectorIteratorImp         */
/*                                                                        */
/*  Implements an iterator for a managed indirect vector of pointers to   */
/*  objects of type T.                                                    */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T, class Alloc> class TMIHugeVectorIteratorImp
{

public:

    TMIHugeVectorIteratorImp( const TMIHugeVectorImp<T,Alloc>& v )
        {
        Vect = &v;
        Restart(0,v.Top());
        }

    TMIHugeVectorIteratorImp( const TMIHugeVectorImp<T,Alloc>& v,
                          unsigned long start,
                          unsigned long stop )
        {
        Vect = &v;
        Restart( start, stop );
        }

    operator int() const
        {
        return Cur < Upper;
        }

    T *Current() const
        {
        PRECONDITION( Cur < Upper );
        return STATIC_CAST(T *,STATIC_CAST(void *,(*Vect)[Cur]));
        }

    T *operator ++ ( int )
        {
        T *temp = Current();
        Cur++;
        return temp;
        }

    T *operator ++ ()
        {
        PRECONDITION( Cur < Upper );
        Cur++;
        return Current();
        }

    void Restart()
        {
        Restart(Lower,Upper);
        }

    void Restart( unsigned long start, unsigned long stop )
        {
        Cur = Lower = start;
        Upper = stop;
        }

private:

    const TMIHugeVectorImp<T,Alloc> *Vect;
    unsigned long Cur;
    unsigned long Lower, Upper;

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TIHugeVectorImp                              */
/*  template <class T> class TIHugeVectorIteratorImp                      */
/*                                                                        */
/*  Implements a vector of pointers to objects of type T using            */
/*  THugeStandardAllocator as its memory manager.                         */
/*  Since pointers always have meaningful copy semantics, this class      */
/*  can handle any type of object.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TIHugeVectorImp :
    public TMIHugeVectorImp<T,THugeStandardAllocator>
{

public:

    TIHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMIHugeVectorImp<T,THugeStandardAllocator>(sz,d)
        {
        }

};

template <class T> class TIHugeVectorIteratorImp :
    public TMIHugeVectorIteratorImp<T,THugeStandardAllocator>
{

public:

    TIHugeVectorIteratorImp( const TIHugeVectorImp<T>& v ) :
        TMIHugeVectorIteratorImp<T,THugeStandardAllocator>(v)
        {
        }

    TIHugeVectorIteratorImp( const TIHugeVectorImp<T>& v,
                         unsigned long l, unsigned long u ) :
        TMIHugeVectorIteratorImp<T,THugeStandardAllocator>(v,l,u)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMICHugeVectorImp                */
/*  template <class T,class Alloc> class TMICHugeVectorIteratorImp        */
/*                                                                        */
/*  Implements a managed, counted vector of pointers to objects of type T.*/
/*  Since pointers always have meaningful copy semantics, this class      */
/*  can handle any type of object.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T,class Alloc> class TMICHugeVectorImp :
    public TMIHugeVectorImp<T,Alloc>
{

public:

    TMICHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMIHugeVectorImp<T,Alloc>(sz), Delta(d), Count_(0) {}

    int Add( T *t );
    int AddAt( T *, unsigned long);
    int Detach( const T *t, int del = 0 )
        {
        return Detach( Find(t), del );
        }

    int Detach( unsigned long loc, int del = 0 );

    int IsEmpty() const
        {
        return Count_ == 0;
        }

    void Flush( int del = 0,
                unsigned long stop = ULONG_MAX,
                unsigned long start = 0 )
        {
        TMIHugeVectorImp<T,Alloc>::Flush( del, stop, start );
        Count_ = 0;
        }

    unsigned long Find( const T *t ) const;

    virtual unsigned long Top() const
        {
        return Count_;
        }

    virtual unsigned long Count() const
        {
        return Count_;
        }

    virtual unsigned long GetDelta() const
        {
        return Delta;
        }

protected:

    unsigned long Count_;
    unsigned long Delta;

};

template <class T,class Alloc> class TMICHugeVectorIteratorImp :
    public TMIHugeVectorIteratorImp<T,Alloc>
{

public:

    TMICHugeVectorIteratorImp( const TMICHugeVectorImp<T,Alloc>& v ) :
        TMIHugeVectorIteratorImp<T,Alloc>(v)
        {
        }

    TMICHugeVectorIteratorImp( const TMICHugeVectorImp<T,Alloc>& v,
                           unsigned long start,
                           unsigned long stop ) :
        TMIHugeVectorIteratorImp<T,Alloc>(v,start,stop)
        {
        }

};

template <class T, class Alloc> 
int TMICHugeVectorImp<T,Alloc>::AddAt( T *t, unsigned long loc )
{
    if( loc >= Lim && !Resize(loc+1) )
        return 0;
    if( Count_ == Lim && !Resize(Lim+1) )
        return 0;
    if( loc > Count_ )
        Count_ = loc;
    for( unsigned long cur = Count_; cur > loc; cur-- )
        Data[cur] = Data[cur-1];
    Data[loc] = t;
    Count_++;
    return 1;
}

template <class T, class Alloc>
int TMICHugeVectorImp<T,Alloc>::Detach( unsigned long loc, int del )
{
    if( loc >= Lim )
        return 0;
    if( del )
        delete STATIC_CAST(T *,STATIC_CAST(void *,Data[loc]));
    if( loc >= Count_ )
        {
        Zero( loc, loc+1 ); // removing an element that's not
        return 1;           // in the counted portion
        }
    Count_--;
    for( unsigned long cur = loc; cur < Count_; cur++ )
        Data[cur] = Data[cur+1];
    Zero( Count_, Count_+1 );
    return 1;
}

template <class T,class Alloc>
unsigned long TMICHugeVectorImp<T,Alloc>::Find( const T *t ) const
{
    if( Top() != 0 )
        {
        for( unsigned long loc = 0; loc < Top(); loc++ )
            if( Data[loc] &&
              *STATIC_CAST(T *,STATIC_CAST(void *,Data[loc])) == *t )
                return loc;
        }
    return ULONG_MAX;
}

template <class T,class Alloc> int TMICHugeVectorImp<T,Alloc>::Add( T *t )
{
    while( Count_ < Limit() && (*this)[Count_] != 0 )
        Count_++;
    if( Count_ >= Lim && !Resize( Count_+1 ) )
        return 0;
    Data[Count_++] = t;
    return 1;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TICHugeVectorImp                             */
/*  template <class T> class TICHugeVectorIteratorImp                     */
/*                                                                        */
/*  Implements a counted vector of pointers to objects of type T using    */
/*  THugeStandardAllocator as its memory manager.                         */
/*  Since pointers always have meaningful copy semantics, this class      */
/*  can handle any type of object.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TICHugeVectorImp :
    public TMICHugeVectorImp<T,THugeStandardAllocator>
{

public:

    TICHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMICHugeVectorImp<T,THugeStandardAllocator>( sz, d )
        {
        }

};

template <class T> class TICHugeVectorIteratorImp :
    public TMICHugeVectorIteratorImp<T,THugeStandardAllocator>
{

public:

    TICHugeVectorIteratorImp( const TICHugeVectorImp<T>& v ) :
        TMICHugeVectorIteratorImp<T,THugeStandardAllocator>(v)
        {
        }

    TICHugeVectorIteratorImp( const TICHugeVectorImp<T>& v,
                          unsigned long l, unsigned long u ) :
        TMICHugeVectorIteratorImp<T,THugeStandardAllocator>(v,l,u)
        {
        }

};

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T,class Alloc> class TMISHugeVectorImp                */
/*  template <class T,class Alloc> class TMISHugeVectorIteratorImp        */
/*                                                                        */
/*  Implements a managed, sorted vector of pointers to objects of type T. */
/*  This is implemented through the template TInternalIHugeVectorImp.     */
/*  Since pointers always have meaningful copy semantics, this class      */
/*  can handle any type of object.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T,class Alloc> class TMISHugeVectorImp :
    public TMICHugeVectorImp<T,Alloc>
{

public:

    TMISHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMICHugeVectorImp<T,Alloc>(sz)
        {
        Delta = d;
        }

    unsigned long Find( const T *t ) const;
    int Add( T *t );

};

template <class T,class Alloc> class TMISHugeVectorIteratorImp :
    public TMICHugeVectorIteratorImp<T,Alloc>
{

public:

    TMISHugeVectorIteratorImp( const TMISHugeVectorImp<T,Alloc>& v ) :
        TMICHugeVectorIteratorImp<T,Alloc>(v)
        {
        }

    TMISHugeVectorIteratorImp( const TMISHugeVectorImp<T,Alloc>& v,
                           unsigned long start,
                           unsigned long stop ) :
        TMICHugeVectorIteratorImp<T,Alloc>(v,start,stop)
        {
        }

};

template <class T,class Alloc>
unsigned long TMISHugeVectorImp<T,Alloc>::Find( const T *t ) const
{
    if( Count_ == 0 )
        return ULONG_MAX;

    unsigned long lower = 0;
    unsigned long upper = Count_-1;

    while( lower < upper && upper != ULONG_MAX )
        {
        unsigned long middle = (lower+upper)/2;
        if( *STATIC_CAST(T *,STATIC_CAST(void *,Data[middle])) == *t )
            return middle;
        if( *STATIC_CAST(T *,STATIC_CAST(void *,Data[middle])) < *t )
            lower = middle+1;
        else
            upper = middle-1;
        }

    if( lower == upper &&
        *STATIC_CAST(T *,STATIC_CAST(void *,Data[lower])) == *t )
        return lower;
    else
        return ULONG_MAX;
}

template <class T,class Alloc> int TMISHugeVectorImp<T,Alloc>::Add( T *t )
{
    unsigned long loc = Count_++;
    if( Count_ > Lim )
        if( !Resize( Count_ ) )
            {
            --Count_;
            return 0;
            }
    while( loc > 0 &&
           *t < *STATIC_CAST(T *,STATIC_CAST(void *,(*this)[loc-1])) )
        {
        Data[loc] = Data[loc-1];
        loc--;
        }
    Data[loc] = t;
    return 1;
}

/*------------------------------------------------------------------------*/
/*                                                                        */
/*  template <class T> class TISHugeVectorImp                             */
/*  template <class T> class TISHugeVectorIteratorImp                     */
/*                                                                        */
/*  Implements a sorted vector of pointers to objects of type T using     */
/*  THugeStandardAllocator as its memory manager.                         */
/*  This is implemented through the template TInternalIHugeVectorImp.     */
/*  Since pointers always have meaningful copy semantics, this class      */
/*  can handle any type of object.                                        */
/*                                                                        */
/*------------------------------------------------------------------------*/

template <class T> class TISHugeVectorImp :
    public TMISHugeVectorImp<T,THugeStandardAllocator>
{

public:

    TISHugeVectorImp( unsigned long sz, unsigned long d = 0 ) :
        TMISHugeVectorImp<T,THugeStandardAllocator>( sz, d )
        {
        }

};

template <class T> class TISHugeVectorIteratorImp :
    public TMISHugeVectorIteratorImp<T,THugeStandardAllocator>
{

public:

    TISHugeVectorIteratorImp( const TISHugeVectorImp<T>& v ) :
        TMISHugeVectorIteratorImp<T,THugeStandardAllocator>(v)
        {
        }

    TISHugeVectorIteratorImp( const TISHugeVectorImp<T>& v,
                          unsigned long l, unsigned long u
                         ) :
        TMISHugeVectorIteratorImp<T,THugeStandardAllocator>(v,l,u)
        {
        }

};

#if defined( BI_CLASSLIB_NO_po )
#pragma option -po.
#endif

#pragma option -Vo.

#endif  // HVECTIMP_H
