//	Copyright (C) 1996, 1997 Meta Four Software.  All rights reserved.
//
//	Auxiliary view declarations
//
//! rev="$Id: k4viewx.h,v 1.17 1997/05/27 12:37:20 jcw Rel $"

#ifndef __K4VIEW_H__
	#error This file is included by "k4view.h", it cannot be used standalone
#endif

/////////////////////////////////////////////////////////////////////////////
// Declarations in this file

	class c4_Sequence;					// a collection of rows

	class c4_Reference; 				// refers to the actual data values
		class c4_IntRef;
		class c4_FloatRef;
		class c4_DoubleRef;
		class c4_StringRef;
		class c4_BytesRef;
		class c4_ViewRef;

	class c4_Dependencies;				// not defined here
	class c4_Handler;					// not defined here
	class c4_Notifier;					// not defined here

/////////////////////////////////////////////////////////////////////////////
//: A sequence is an abstract base class for views on ranges of records.
//
//	Sequences represent arrays of rows (or indexed collections / tables).
//	Insertion and removal of entries is allowed, but could take linear time.
//	A reference count is maintained to decide when the object should go away.

class c4_Sequence
{
	int _refCount;
	    /* Reference count. */
	c4_Dependencies* _dependencies;
	    /* Pointer to depency list, or null if nothing depends on this. */
	int _lastPropIndex;
	    /* Optimization: cached property index. */
protected:
	int _lastPropId; // see c4_HandlerSeq::Reset()
	    /* Optimization: cached property id. */

public: 
/* General */
	c4_Sequence ();
	    //: Abstract constructor.
	
	virtual int Compare(int, c4_Cursor) const;
	    //: Compares the specified row with another one.
	void SetAt(int, c4_Cursor);
	    //: Replaces the contents of a specified row.
	virtual int RemapIndex(int, const c4_Sequence*) const;
	    //: Remaps the index to an underlying view.
	
	c4_String Describe() const;
	    //: Returns a descriptions of the current data structure.

/* Reference counting */
	void IncRef();
	    //: Increments the reference count of this sequence.
	void DecRef();
	    //: Decrements the reference count, delete objects when last.
	int NumRefs() const;
	    //: Returns the current number of references to this sequence.

/* Adding / removing rows */
	virtual int Size() const = 0;
	    //: Returns the current number of rows.
	void Resize(int, int =-1);
	    //: Changes number of rows, either by inserting or removing them.
	
	virtual void InsertAt(int, c4_Cursor, int =1);
	    //: Inserts one or more rows into this sequence.
	virtual void RemoveAt(int, int =1);
	    //: Removes one or more rows from this sequence.
	virtual void Move(int, int);
	    //: Move a row to another position.

/* Properties */
	int NthProperty(int) const;
	    //: Returns the id of the N-th property.
	int PropIndex(int, bool =false);
	    //: Finds the index of a property, or creates a new entry.
	
	virtual int NumHandlers() const = 0;
	    //: Returns the number of data handlers in this sequence.
	virtual c4_Handler& NthHandler(int) const = 0;
	    //: Returns a reference to the N-th handler in this sequence.
	virtual const c4_Sequence* HandlerContext(int) const = 0;
	    //: Returns the context of the N-th handler in this sequence.
	virtual int AddHandler(c4_Property&, c4_Handler*) = 0;
	    //: Adds the specified data handler to this sequence.

/* Element access */
	virtual bool Get(int, int, c4_Bytes&);
	    //: Retrieves one data item from this sequence.
	virtual void Set(int, int, const c4_Bytes&);
	    //: Stores a data item into this sequence.
	
/* Dependency notification */
	void Attach(c4_Sequence* child_);
	    //: Registers a sequence to receive change notifications.
	void Detach(c4_Sequence* child_);
	    //: Unregisters a sequence which received change notifications.
	c4_Dependencies* GetDependencies() const;
	    //: Returns a pointer to the dependencies, or null.

	virtual c4_Notifier* PreChange(c4_Notifier& nf_);
	    //: Called just before a change is made to the sequence.
	virtual void PostChange(c4_Notifier& nf_);
	    //: Called after changes have been made to the sequence.
	
protected:
	virtual ~c4_Sequence ();

public: //! for c4_Table::Sequence setup
	virtual void SetSize(int size_) = 0;

private:
	c4_Sequence (const c4_Sequence&);	// not implemented
	void operator= (const c4_Sequence&); // not implemented
};

/////////////////////////////////////////////////////////////////////////////
//: A reference is used to get or set typed data, using derived classes.
//
//	Objects of this class are only intended to be used as a temporary handle
//	while getting and setting properties in a row.	They are normally only
//	constructed as result of function overload operators: "property (row)".
//
//	See: c4_IntRef, c4_FloatRef, c4_DoubleRef, 
//	c4_StringRef, c4_BytesRef, c4_ViewRef

class c4_Reference
{
	c4_Cursor _cursor;
	    /* The cursor which points to the data. */
	int _propId;
	    /* The property id associated to this reference. */

public:
	c4_Reference (c4_RowRef, int);
	    //: Constructor.

	c4_Reference& operator= (const c4_Reference&);
	    //: Assignment of one data item.

	bool GetData(c4_Bytes&) const;
	    //: Retrieves the value of the referenced data item.
	void SetData(const c4_Bytes&) const;
	    //: Stores a value into the referenced data item.
	
	friend bool operator== (const c4_Reference&, const c4_Reference&);
	    //: Returns true if the contents of both references is equal.
	friend bool operator!= (const c4_Reference&, const c4_Reference&);
	    //: Returns true if the contents of both references is not equal.

private:
	void operator& () const;			// not implemented
};

/////////////////////////////////////////////////////////////////////////////

	//: Used to get or set integer values.
class c4_IntRef : public c4_Reference
{
public:
	c4_IntRef (const c4_Reference&);
	    //: Constructor.
	operator long () const;
	    //: Gets the value as long integer.
	c4_IntRef& operator= (long);
	    //: Sets the value to the specified long integer.
};

#if !q4_TINY

	//: Used to get or set floating point values.
class c4_FloatRef : public c4_Reference
{
public:
	c4_FloatRef (const c4_Reference&);
	    //: Constructor.
	operator double () const;
	    //: Gets the value as floating point.
	c4_FloatRef& operator= (double);
	    //: Sets the value to the specified floating point.
};

	//: Used to get or set double precision values.
class c4_DoubleRef : public c4_Reference
{
public:
	c4_DoubleRef (const c4_Reference&);
	    //: Constructor.
	operator double () const;
	    //: Gets the value as floating point.
	c4_DoubleRef& operator= (double);
	    //: Sets the value to the specified floating point.
};

#endif // !q4_TINY

	//: Used to get or set string values.
class c4_StringRef : public c4_Reference
{
public:
	c4_StringRef (const c4_Reference&);
	    //: Constructor.
	operator c4_String () const;
	    //: Gets the value as string.
	c4_StringRef& operator= (const char*);
	    //: Sets the value to the specified string.
};

	//: Used to get or set binary object values.
class c4_BytesRef : public c4_Reference
{
public:
	c4_BytesRef (const c4_Reference&);
	    //: Constructor.
	operator c4_Bytes () const;
	    //: Gets the value as string.
	c4_BytesRef& operator= (const c4_Bytes&);
	    //: Sets the value to the specified string.
};

	//: Used to get or set view values.
class c4_ViewRef : public c4_Reference
{
public:
	c4_ViewRef (const c4_Reference&);
	    //: Constructor.
	operator c4_View () const;
	    //: Gets the value as view.
	c4_ViewRef& operator= (const c4_View&);
	    //: Sets the value to the specified view.
};

/////////////////////////////////////////////////////////////////////////////
