//----------------------------------------------------------------------------
// Mr.Clean v. 1.00 - (C) Copyright 1995 by Kent Reisdorph
//
// Novemeber 11, 1995
//
// dialogs.cpp - Source for dialogs used in Mr.Clean, an application
// that cleans misc files out of Borland C++ project directories
//
//----------------------------------------------------------------------------
#include "mrcmain.h"
#pragma hdrstop

/// code for ChooseDirDialog /////////////////////////////////////

// response table. We had to create our own OK button and use it
// because the commmon dialog OK button validates the filename field
// which in our case will always be empty.
DEFINE_RESPONSE_TABLE1(ChooseDirDialog, TFileOpenDialog)
	EV_COMMAND(IDC_OK, CmOkClicked),
END_RESPONSE_TABLE;

ChooseDirDialog::ChooseDirDialog(TWindow*        parent,
										TData&          data,
										TResId          templateId,
										const char far* title,
										TModule*        module)
	: TFileOpenDialog(parent, data, templateId, title, module)
{
	// create a pointer to the static control that the common dialog
	// uses to report the directory. We are going to take the text from
	// this control later.
	dir = new TStatic(this, 1088);
}

void
ChooseDirDialog::SetupWindow()
{
	TFileOpenDialog::SetupWindow();
	// Gotta do this to get KB handling in the listbox. Evidently
	// the common dialogs use a default ID of 1 to achieve this effect.
	// This id, of course, conflicts with the BWCC OK button. For some
	// strange reason the dialog will not come up if you remove the
	// BWCC OK button, so I had to disable and make it !visible but it
	// still had to be on the dialog. Strange!

	// Addendum: I decided not to use a BWCC dialog box for the
	// ChooseDirDialog - too many problems.
	SetDefaultId(1);
}

// CmOkClicked(). Handler for our OK button. Just gets the text from the
// static control and copies it to TOpenSave::TData.FileName. We could
// do this other ways, but copying the directory to FileName follows
// the OWL common dialog conventions.
void
ChooseDirDialog::CmOkClicked()
{
	char str[256];
	dir->GetText(str, sizeof(str));
	strcpy(Data.FileName, str);
	TDialog::CmOk();
}

/// code for TCenteredDialog /////////////////////////////////////

// SetupWindow(). To determine the x position we get the width of the
// screen, divide by 2 to find the center, and subtract half the width
// of the dialog. Likewise for y pos. That's the whole class.
void TCenteredDialog::SetupWindow()
{
	TDialog::SetupWindow();
	TRect rect = GetWindowRect();
	int x = (GetSystemMetrics(SM_CXSCREEN)/2) - (rect.Width()/2);
	int y = (GetSystemMetrics(SM_CYSCREEN)/2) - (rect.Height()/2);
	SetWindowPos(0, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
}

/// code for TimedDialog /////////////////////////////////////

// This class just pops up a dialog for the number of seconds passed
// in the constructor's timeout parameter. Using a timeout of 0 will
// leave the dialog on the screen. Clicking the left mouse button
// closes the dialog
DEFINE_RESPONSE_TABLE1(TimedDlg, TDialog)
	EV_WM_TIMER,
	EV_WM_LBUTTONDOWN,
END_RESPONSE_TABLE;

TimedDlg::TimedDlg(TWindow* parent, int resId, const char far* text,
	int timeout, TModule* module)
		: TCenteredDialog(parent, resId, module)
{
	timeOut = timeout * 1000;
	(const char far*)dlgText = text;
  textWin = new TStatic(this, IDC_TEXT);
}

void
TimedDlg::SetupWindow()
{
	TCenteredDialog::SetupWindow();
	textWin->SetText(dlgText);
	if (timeOut) {
		SetTimer(1, timeOut);
	}
}

void
TimedDlg::EvTimer(UINT)
{
	KillTimer(1);
	CloseWindow();
}

void
TimedDlg::EvLButtonDown(uint, TPoint&)
{
	CloseWindow();
}

/// code for ConfigDialog /////////////////////////////////////
DEFINE_RESPONSE_TABLE1(ConfigDlg, TDialog)
	EV_BN_CLICKED(IDC_ADD, AddItem),
	EV_BN_CLICKED(IDC_REMOVE, RemoveItem),
	EV_BN_CLICKED(IDOK, CmOk),
	EV_EN_CHANGE(IDC_EDIT, EditChanged),
	EV_BN_CLICKED(IDC_DUMMY, DummyClicked),
	EV_LBN_SELCHANGE(IDC_LISTBOX, SelChanged),
END_RESPONSE_TABLE;

ConfigDlg::ConfigDlg(TWindow* parent)
	: TCenteredDialog(parent, IDD_CONFIG),
	fData(OFN_PATHMUSTEXIST, "", 0, "", "*")
{
	// set up the controls
	listbox = new TListBox(this, IDC_LISTBOX);
	edit = new TEdit(this, IDC_EDIT, 80);
	dir = new TStatic(this, IDC_PATH);
	addBtn = new TButton(this, IDC_ADD);
	removeBtn = new TButton(this, IDC_REMOVE);
	globalBtn = new TRadioButton(this, IDC_GLOBAL);
	projectBtn = new TRadioButton(this, IDC_PROJECT);
}

void
ConfigDlg::SetupWindow()
{
	TCenteredDialog::SetupWindow();
	// initially disable the add and remove buttons. We will enable
	// them as conditions dictate
	addBtn->EnableWindow(false);
	removeBtn->EnableWindow(false);
	// set the default for global configuration, not project specific
	globalBtn->SetCheck(true);
	projectBtn->SetCheck(false);
	// set the default ID to a non-existant button. The response table
	// sends the button press for the non-button to the DummyClicked
	// function. This prevents the dialog from closing when the enter
	// key is pressed.
	SetDefaultId(IDC_DUMMY);
	// read the mrclean.cfg file
	ReadFile();
	MrCleanWindow* w = dynamic_cast<MrCleanWindow*>(Parent);
	dir->SetText(w->path);
}

// AddItem() function. Takes the text from the edit control and adds
// it to the listbox after doing some simple validation.
void
ConfigDlg::AddItem()
{
	char str[80];
	// get the text from the edit control
	edit->GetText(str, sizeof(str));
	// if there is no text then return
	if (strlen(str) == 0) return;
	// if the user entered *.* then warn them
	if (!strcmp(str, "*.*"))
		if (MessageBox("WARNING!! This filespec will delete all files "
			"in your project directory!", "Mr.Clean Warning",
			MB_OKCANCEL | MB_ICONEXCLAMATION ) == IDCANCEL) { edit->Clear(); return; }
	// I like the string class for manipulating strings
	string s(str);
	// if the more than 3 characters follow the '.' then the filespec
	// is invalid. Beep and return (don't add the string to the listbox).
	if (s.find_first_of(".") < s.length()- 4) { MessageBeep(-1); return; }
	// convert the string to upper case - no reason, really
	strupr(str);
	// add the string to the listbox
	listbox->AddString(str);
	// now that we have strings in the listbox we can enable the "Remove" button.
	removeBtn->EnableWindow(true);
	// clear the edit control
	edit->Clear();
	// set focus back to the edit control
	edit->SetFocus();
}

// RemoveItem(). Message handler function for the "Remove" button.
// removes an entry from the listbox
void
ConfigDlg::RemoveItem()
{
	// remove the string at the currently selected location
	listbox->DeleteString(listbox->GetSelIndex());
	// if no more entries remain in the list box then disable the
	// "Remove" button.
	if (listbox->GetCount() == 0) removeBtn->EnableWindow(false);
}

// CmOk(). Called when the user clicks OK. We need to write the
// config file if OK was clicked.
void
ConfigDlg::CmOk()
{
	char temp[256];
	// get a pointer to the parent window
	MrCleanWindow* w = dynamic_cast<MrCleanWindow*>(Parent);
	// if the global button was not checked then the Project Specific
	// button was checked.
	if (!globalBtn->GetCheck()) {
		// if a path was not passed in the command line then we need to
		// pop up the FileSave common dialog so the user can select the
		// project directory for the config file.
		if (!strcmp(w->path, "")) {
			//strcpy(fData.FileName, "mrclean.cfg");
			if ((new ChooseDirDialog(
				this, fData, IDD_CHOOSEDIR)->Execute()) == IDCANCEL) return;
			wsprintf(temp, "%s\\", fData.FileName);
			// change the path member of MrCleanWindow to reflect the new path
			strcpy(w->path, temp);
			dir->SetText(temp);
			strcat(temp, "mrclean.cfg");
			return;
		}
		// if a path was passed in the command line then use it
		else wsprintf(temp, "%smrclean.cfg", w->path);
	}
	// if the Global button is checked then use the Mr.Clean directory
	// The parent's cfgFile member contains the path and filename
	else strcpy(temp, w->cfgFile);
	// open a file for output
	ofstream outfile(temp);
	// something went wrong
	if (!outfile) {
		MessageBox("Error Opening Config File", "Mr. Clean Error",
			MB_OK | MB_ICONEXCLAMATION);
		Destroy();
	}
	// write the check states of the 20 buttons. The control IDs
	// are sequential to make this easy.
	for (int i=101;i<124;i++) {
		int check = (int)SendDlgItemMsg(i, BM_GETCHECK, 0, 0);
		outfile << check << endl;
	}
	char str[80];
	// any items in the list box?
	int count = listbox->GetCount();
	// write the number of items to the file
	outfile << count << endl;
	// get each string out of the listbox and write it to the file
	for (i=0;i<count;i++) {
		listbox->GetString(str, i);
		outfile << str << endl;
	}
	// close the file
	outfile.close();
	// all done - close the dialog
	TDialog::CmOk();
}

// ReadFile()
void
ConfigDlg::ReadFile()
{
	// get a pointer to the parent window
	MrCleanWindow* w = dynamic_cast<MrCleanWindow*>(Parent);
	char temp[256];
	// need this for the OpenFile function
	OFSTRUCT ofStruct;
	// first check to see if a mrclean.cfg file exists in the path
	// passed in the command line
	wsprintf(temp, "%smrclean.cfg", w->path);
	// use OpenFile with the OF_EXIST flag. This opens the file and then
	// closes it again. If the file does not exist then -1 is returned.
	if (OpenFile(temp, &ofStruct, OF_EXIST) == -1) {
		// no file? Ok, then use the default mrclean.cfg and set the
		// radio buttons to Global
		strcpy(temp, w->cfgFile);
		globalBtn->SetCheck(true);
		projectBtn->SetCheck(false);
	}
	// file does exist in the path provided? Use it and set the radio
	// buttons to Project Specific
	else
		if (strcmp(w->path, "")) {
			projectBtn->SetCheck(true);
			globalBtn->SetCheck(false);
		}
	// open a file for output
	ifstream infile(temp);
	// we don't really need a config file so if it doesn't
	// exist we will just create a new one.
	if (!infile) return;
	// read the values for each checkbox control and set the check state
	// of the button with that value.
	for (int i=101;i<124;i++) {
		int val;
		infile >> val;
		SendDlgItemMsg(i, BM_SETCHECK, val, 0);
	}
	int count;
	char str[80];
	// get the number of strings to put into the listbox
	infile >> count;
	// if count is non-zero read that many string and stick them
	// into the listbox.
	if (count) {
		for(i=0;i<count;i++) {
			infile >> str;
			listbox->AddString(str);
		}
	}
	// all done - close the file
	infile.close();
}

// EditChanged() Called in response to a EN_CHANGE message. Used
// to enable or disable the Add button.
void
ConfigDlg::EditChanged()
{
	char str[80];
	// get the text from the edit control
	edit->GetText(str, sizeof(str));
	// if there is text then enable the Add button
	if (strlen(str) > 0) addBtn->EnableWindow(true);
	// if there is no text then disable the Add button. The user could
	// enter text and then backspace until no text remains.
	else addBtn->EnableWindow(false);
}

