//----------------------------------------------------------------------------
// Mr.Clean v. 1.00 - (C) Copyright 1995 by Kent Reisdorph
//
// Novemeber 11, 1995
//
// mrclean.cpp - Source for Mr.Clean, an application that cleans
// misc files out of Borland C++ project directories
//
//----------------------------------------------------------------------------
#include "mrcmain.h"
#pragma hdrstop

/// code for MrCleanWindow /////////////////////////////////////
DEFINE_RESPONSE_TABLE1(MrCleanWindow, TWindow)
	EV_COMMAND(CM_CHOOSEDIR, CmChooseDir),
	EV_COMMAND(CM_XIT, CmExit),
	EV_COMMAND(CM_CLEAN, CmClean),
	EV_COMMAND(CM_CONFIG, CmConfig),
	EV_COMMAND(CM_ABOUT, CmAbout),
	EV_WM_TIMER,
END_RESPONSE_TABLE;

MrCleanWindow::MrCleanWindow(const char far* _path, const char far* cmd)
	: TWindow(0, 0, 0),
	fData(OFN_PATHMUSTEXIST, "", 0, "", "*")
{
	bitmap = 0;
	pal = 0;
	// no mode passed
	mode = -1;
	// set the mode according to the string passed in the mode param
	if(!strcmpi(cmd, "CONFIG")) mode = CONFIG;
	if (!strcmpi(cmd, "CLEAN")) {
		mode = CLEAN;
		// if mode is CLEAN then make the window invisible
		GetApplication()->nCmdShow = SW_HIDE;
	}
	strcpy(path, "");
	strcpy(target, _path);
	// the following code trims the filename off the command line
	// path. This condition will occur if the program is set up to run
	// from the Tool menu in the IDE.
	if (strlen(_path) > 0) {
		string s(_path);
		// if a filename was passed with the path then find the last "\"
		// and chop the string off there.
		if (s.find_first_of(".") != NPOS)
			s.remove(s.find_last_of("\\"));
		// if a filename was not passed then check to be sure the user
		// didn't put a "\" at the end of the path. If so, chop it off.
		else if (s.find_last_of("\\") == s.length() - 1) s.remove(s.length() - 1);
		// copy the string to our path variable
		strcpy(path, s.c_str());
		char str[256];
		// get the current working directory and save it
		getcwd(str, sizeof(str));
		// try to change the directory to the one passed in the command line.
		// If an error occurs then the path is not valid. We will inform the
		// user later in EvTimer. Copy "error" to the path since it isn't any
		// good to us now.
		if (chdir(path) == -1) strcpy(path, "error");
		// if the chdir succeeded then add a backslash onto the end of the
		// path. We'll need it later.
		else strcat(path, "\\");
		// change the directory back to the current working directory
		chdir(str);
	}
	// Store the path and filename of the default config file.
	// First, get the module filename. This will be the path and filename
	// of the application.
	GetApplication()->GetModuleFileName(cfgFile, sizeof(cfgFile));
	// copy the string ".CFG" where ".EXE" is: Find the end of the string
	// and back up 4 characters.
	memcpy(&cfgFile[strlen(cfgFile)-4], ".CFG", 4);
}

MrCleanWindow::~MrCleanWindow()
{
	delete bitmap;
	delete pal;
}

void
MrCleanWindow::SetupWindow()
{
	TWindow::SetupWindow();
	if (mode != CLEAN) {
		int w, h;
		int sw = GetSystemMetrics(SM_CXSCREEN);
		int sh = GetSystemMetrics(SM_CYSCREEN);
		// the bitmap is available to registered users. It's way cool!
		// read the bitmap, get a palette
		// if the bitmap is not available a TXOwl exception will be
		// thrown which we will catch...
		try {
			TDib* dib = new TDib("mrclean.bmp");
			pal = new TPalette(*dib);
			bitmap = new TBitmap(*dib, pal);
			delete dib;
			int x = GetSystemMetrics(SM_CYFRAME) * 2;
			w = bitmap->Width() + x;
			h = bitmap->Height() + x + GetSystemMetrics(SM_CYMENU) * 2;
		}
		// catch the exception. Zero the bitmap and palette pointers
		// and set the window size to 75% of the screen.
		catch (TXOwl) {
			bitmap = 0;
			pal = 0;
			w = sw * .75;
			h = sh * .75;
		}
		// move the window
		Parent->MoveWindow(sw/2 - (w/2), sh/2 - (h/2), w, h);
		// if a path was passed in the command line then change the
		// caption to reflect the path
		if (strlen(path) > 0) {
			char str[256];
			wsprintf(str, "Mr.Clean - %s", path);
			str[strlen(str) - 1] = 0;
			Parent->SetCaption(str);
		}
	}
	// set a timer so that automatic processing can begin if necessary
	SetTimer(1,1);
}

void
MrCleanWindow::Paint(TDC& dc, bool, TRect&)
{
	// if no bitmap then don't do anything
	if (!bitmap) return;
	// otherwise display it
	TMemoryDC memdc(dc);
	memdc.SelectObject(*pal);
	memdc.RealizePalette();
	memdc.SelectObject(*bitmap);
	dc.BitBlt(0, 0, bitmap->Width(),  bitmap->Height(), memdc, 0, 0, SRCCOPY);
	memdc.RestoreObjects();
}

// EvTimer. Here's where we do automatic processing
void
MrCleanWindow::EvTimer(uint)
{
	// kill the timer
	KillTimer(1);
	// if the path was invalid then tell the user and then close the window
	if (!strcmpi(path, "error")) {
		MessageBox("The path entered in the command line is not a valid path.",
			"Mr.Clean Error");
		Destroy();
		return;
	}
	// if the mode is clean then DoClean and close the window
	// if the mode is config then call the config function
	switch (mode) {
		case CLEAN :
			DoClean();
			Destroy();
			break;
		case CONFIG :
			CmConfig();
			break;
	}
}

// CmChooseDir() Allows the user to change the directory to be cleaned.
void
MrCleanWindow::CmChooseDir()
{
	if (ChooseDirDialog(this, fData, IDD_CHOOSEDIR).Execute() == IDCANCEL) return;
	// change the application's caption to reflect the new path
	strcpy(path, fData.FileName);
	strcat(path, "\\");
	char str[256];
	wsprintf(str, "Mr.Clean - %s", path);
	str[strlen(str) - 1] = 0;
	Parent->SetCaption(str);
}

// close the window unconditionally
void
MrCleanWindow::CmExit()
{
	Destroy();
}

// handler for the Clean Now menu item
void
MrCleanWindow::CmClean()
{
	DoClean();
}

// handler for the Configure menu item. Just executes the config
// dialog box.
void
MrCleanWindow::CmConfig()
{
	new ConfigDlg(this)->Execute();
	// if a path exists then change the applications title to show
	// the path.
	if (strlen(path) > 0) {
		char str[256];
		wsprintf(str, "Mr.Clean - %s", path);
		str[strlen(str) - 1] = 0;
		Parent->SetCaption(str);
	}
}

// handler for the About menu item. Executes the about dialog box.
void
MrCleanWindow::CmAbout()
{
	new TCenteredDialog(this, IDD_ABOUT)->Execute();
}

// DoClean(). Performs the deletion of the files.
void
MrCleanWindow::DoClean()
{
	char temp[256];
	// needed for the OpenFile function
	OFSTRUCT ofStruct;
	// make a path and filename with the passed path
	wsprintf(temp, "%smrclean.cfg", path);
	// check and see if a mrclean.cfg file exists there. OpenFile with
	// the OF_EXIST path will check for existance of a file. If the file
	// does not exist then use the default config file.
	if (OpenFile(temp, &ofStruct, OF_EXIST) == -1) strcpy(temp, cfgFile);
	// open the file
	ifstream infile(temp);
	// something went wrong
	if (!infile) {
		MessageBox("Error Opening Config File", "Mr. Clean Error",
			MB_OK | MB_ICONEXCLAMATION);
		return;
	}
	// read the values from the file for the standard file format
	// check boxes. Our IDs are numbered from 101 to 122.
	int val;
	for (int i=101;i<123;i++) {
		infile >> val;
		// if val is nonzero then we need to delete the files associated
		// with that checkbox. We do that by loading a string resource that
		// has the same ID as the checkbox. The strings contain the filespec
		// associated with the checkbox. String #101 = "*.BAK", etc.
		if (val) {
			char str[6];
			// load the string associated with i
			::Module->LoadString(i, str, sizeof(str));
			// make a complete path with the path and filespec
			wsprintf(temp, "%s%s", path, str);
			// do the deletion
			DeleteFiles(temp);
		}
	}
	// get the "Target .EXE Only" value
	infile >> val;
	// if you are using CodeGuard then it will break on this function if
	// the TARGET is not in the project directory. IOW, we might be trying
	// to delete a file that doesn't exist. It's no problem, but CG will
	// report it as a function failure.
	if (val) remove(target);
	int count;
	char str[80];
	// get the number of strings of user defined filespecs
	infile >> count;
	if (count) {
		for(i=0;i<count;i++) {
			// read a string
			infile >> str;
			// make a complete path
			wsprintf(temp, "%s%s", path, str);
			// do the deletions
			DeleteFiles(temp);
		}
	}
	// close the file
	infile.close();
	// tell the user that the process is done
	new TimedDlg(this, IDD_TIMED, "Directory Clean Done!", 1)->Execute();
}

// DeleteFiles() Does the actual deletion of the files
void
MrCleanWindow::DeleteFiles(const char far* fspec)
{
	// use the fflbk structure with findfirst and findnext to find each
	// file. The DOS function remove() does not allow wildcards so we
	// need to delete each file individually.
	struct ffblk ffblk;
	// need a flag to indicate when we're done
	int done;
	char str[256];
	// pass the filespec to findfirst
	done = findfirst(fspec, &ffblk, 0);
	//  The first file to meet the filespec condition is now contained in ffblk
	while (!done)
	{
		// make a complete path with the filename contained in the ff_name
		// member of the ffblk structure
		wsprintf(str, "%s%s", path, ffblk.ff_name);
		// delete the file. If an access violation occured report it.
		// Why? I don't really know.<g>
		if (remove(str) == -1 && errno == 5) {
			char temp[300];
			Parent->Show(SW_NORMAL);
			wsprintf(temp, "Cannot Delete\n\n%s\n\nAccess Denied", str);
			MessageBox(temp, "Mr. Clean Message", MB_OK);
		}
		// find the next file and loop
		done = findnext(&ffblk);
	}
}

// The application class. Pretty basic. Just a few mods to allow for
// the command line arguments.
class TMrCApp : public TApplication {
	public:
		char command[10];
		char path[256];
		TMrCApp(const char far* _path, const char far* cmd)
			: TApplication("Mr.Clean")
		{
			// copy the values passed to us from OwlMain to member variables
			strcpy(path, _path);
			strcpy(command, cmd);
		}
		void InitMainWindow()
		{
			// Enable BWCC and Ctl3d
			EnableBWCC();
      EnableCtl3d();
			// setup the main window. Pass the command line args to the
			// MrCleanWindow constructor.
			SetMainWindow(new TFrameWindow(0,
				"Mr.Clean - BC++ Project Directory Cleaner",
				new MrCleanWindow(path, command)));
			// assign our menu to the frame
			MainWindow->AssignMenu(MENU_1);
			// assign our icon to the frame
			MainWindow->SetIcon(this, ICON_1);
		}
};

int OwlMain(int argc, char* argv[])
{
	char path[256] = "";
	char cmd[10] = "";
	// if args are passed to us then pass them on in the TMrCApp constructor
	if (argc > 1) strcpy(path, argv[1]);
	if (argc > 2)	strcpy(cmd, argv[2]);
	TMrCApp app(path, cmd);
	return app.Run();
}
