




/*
 *
 *          Copyright (C) 1994, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1994. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



#include "ui/colormap.h"
#include "ui/applic.h"
#include "ui/cntroler.h"
#include "ui/dsplsurf.h"
#include "ui/color.h"

#if defined (__MS_WINDOWS__)
#include <windows.h>

#elif defined(__OS2__)
class UI_ColorMapEntry {
    ulong h;
} ; // Temporary
#elif defined (__X_MOTIF__)
#include <X11/Intrinsic.h>
extern "C" { short sprintf (char *, ...); };
#endif

UI_ColorMapType UI_ColorMap::_defaultcmap;


UI_ColorMap::UI_ColorMap (UI_DisplaySurface& d): _dsurface (d)
{
    _numcolors = 0;
    _colors    = NULL;

#if defined (__MS_WINDOWS__)
    LOGPALETTE lp;

    lp.palVersion    = 0x300;
    lp.palNumEntries = 1;
    _cmap   = CreatePalette (&lp);
    if (!_cmap) {
        CL_Error::Warning ("UI_ColorMap::UI_ColorMap: %s",
                           "Cant Allocate the colormap");
        return;
    }

#elif defined (__X_MOTIF__)
    XSetWindowAttributes     xattributes;
    Display *dpy   = XtDisplay (_dsurface.Client ());
    short screen_num = DefaultScreen (dpy);
    
    _defaultcmap = DefaultColormap (dpy, screen_num);
    if (!(_cmap = XCreateColormap (dpy, RootWindow (dpy, screen_num),
                                   DefaultVisual(dpy,screen_num),AllocNone))) {
        CL_Error::Warning ("UI_ColorMap::UI_ColorMap: %s",
                           "Cant Allocate the colormap");
        return;
    }
    
    xattributes.colormap = _cmap;
    ulong valuemask = 0;
    valuemask |= CWColormap;
    XChangeWindowAttributes (dpy, XtWindow (_dsurface.Client ()),
                             valuemask, &xattributes);
    
#endif
}



UI_ColorMap::~UI_ColorMap ()
{
#if defined (__MS_WINDOWS__)
    DeleteObject (_cmap);

#elif defined (__X_MOTIF__)
    Display *dpy = XtDisplay (_dsurface.Client ());
    if (_colors)
        XFreeColors   (dpy, _cmap, _colors, _numcolors, 0);
    if (_cmap != _defaultcmap)    
        XFreeColormap (dpy, _cmap);
    
#endif

    if (_colors)
        delete [] _colors;
}




bool UI_ColorMap::Load (const UI_ColorMapValues v [])
{
    UI_Color c;
    
    for (short i = 0; v [i].red != -1; ++i) {
        c.Red   (v [i].red);
        c.Green (v [i].green);
        c.Blue  (v [i].blue);
        Add (c);
    }
    return TRUE;
}



UI_ColorMapHandle UI_ColorMap::Add (const UI_Color &c)
{
    if (_numcolors >= UI_MAXCOLORS)
        return 0;

#if defined (__MS_WINDOWS__)
    if (_colors) 
        delete [] _colors;
  
    _numcolors ++;
    _colors = new UI_ColorMapEntry [_numcolors];
    if (!_colors) {
        CL_Error::Warning ("ColorMap::Add: memory allocation failed");
        return 0;
    }
    ResizePalette (_cmap, _numcolors); 
    if (_numcolors != 1)
        GetPaletteEntries (_cmap, 0, _numcolors - 1, _colors);
    
    short index = _numcolors - 1;

    _colors [index].peRed   = LOBYTE (c.Red   ());
    _colors [index].peGreen = LOBYTE (c.Green ());
    _colors [index].peBlue  = LOBYTE (c.Blue  ());
    _colors [index].peFlags = NULL;

    if (!SetPaletteEntries (_cmap, index, 1, &(_colors [index]))) 
        CL_Error::Warning ("UI_ColorMap::Add: %s",
                           "Cant allocate Read/Write cells");
    return index;

#elif defined (__X_MOTIF__)
    Display *dpy   = XtDisplay (_dsurface.Client ());

    XColor &xcolor  = c.NativeForm ();
    
    ulong plane_mask [1];

    if (_colors) {
        XFreeColors (dpy, _cmap, _colors, _numcolors, 0);
        delete [] _colors;
        _colors = 0;
    }
    _numcolors ++;
    _colors = new UI_ColorMapEntry [_numcolors];
    if (!_colors) {
        CL_Error::Warning ("ColorMap::Add: memory allocation failed");
        return 0;
    }
    while (1) {
        if (XAllocColorCells (dpy, _cmap, False,
                              plane_mask, 0, _colors, _numcolors))
            break;
        _numcolors --;
        if (_numcolors == 0) {
            CL_Error::Warning ("UI_ColorMap::Add:  %s",
                               "Can't allocate read/write cells");
            return 0;
        }
    
    }
    
    xcolor.flags = DoRed | DoGreen | DoBlue;
    if (_cmap != _defaultcmap)
        _colors [_numcolors-1] = _numcolors - 1;
    xcolor.pixel = _colors [_numcolors-1];
    XStoreColor (dpy, _cmap, &xcolor);
    
    return _colors [_numcolors-1];
    
#endif
}



bool UI_ColorMap::Remove (UI_ColorMapHandle h)
{
    bool retval = TRUE;

    ushort j = 0;
    UI_ColorMapEntry *newcolors = new UI_ColorMapEntry [_numcolors-1];

    for (ushort i = 0; i < _numcolors; ++i) {
        if (i == h) {
            for (j = i; j < _numcolors-1; ++j) 
                newcolors [j] = _colors [j+1];
            break;
        }
        newcolors [i] = _colors [i];
    }

    delete [] _colors;
    _colors = newcolors;
    _numcolors --;

#if defined (__MS_WINDOWS__)
    if (!SetPaletteEntries (_cmap, 0, _numcolors, _colors)) retval = FALSE;

#elif defined (__X_MOTIF__)
    Display *dpy = XtDisplay (_dsurface.Client ());
    
    if (!XFreeColors (dpy, _cmap, &h, 1, 0)) retval = FALSE;
     
#endif

    return retval;
}



UI_ColorMapHandle UI_ColorMap::Match (const UI_Color &c)
{
#if defined (__MS_WINDOWS__)
    short index = GetNearestPaletteIndex (_cmap, c.NativeForm ());
    return index;

#elif defined (__X_MOTIF__)
    Display *dpy  = XtDisplay (_dsurface.Client ());
    XColor xcolor = c.NativeForm ();
    int scr       = DefaultScreen (dpy);

    if (XAllocColor (dpy, _cmap, &xcolor)) {
        XFreeColors (dpy, _cmap, &xcolor.pixel, 1, 0);
        return xcolor.pixel;
    }
    else
        return BlackPixel (dpy, scr);

#endif
}



UI_ColorMapHandle UI_ColorMap::ExactMatch (const UI_Color &c)
{
#if defined (__MS_WINDOWS__)
    for (short i = 0; i < _numcolors; ++i) {
        if (_colors [i].peRed   == LOBYTE (c.Red   ()) &&
            _colors [i].peGreen == LOBYTE (c.Green ()) && 
            _colors [i].peBlue  == LOBYTE (c.Blue  ())) 
            return i;
    }
    return 0;
            
#elif defined (__X_MOTIF__)
    Display *dpy  = XtDisplay (_dsurface.Client ());
    int scr       = DefaultScreen (dpy);
    XColor xcolor = c.NativeForm ();
    char colorrep [30];

    sprintf (colorrep, "RGB:%4x/%4x/%4x", xcolor.red,
             xcolor.green, xcolor.blue);
    XParseColor (dpy, _cmap, colorrep, &xcolor);

    return xcolor.pixel;

#endif 
}



UI_Color UI_ColorMap::operator[] (UI_ColorMapHandle& h) const
{
#if defined (__MS_WINDOWS__)
    return UI_Color (_colors [h].peRed, _colors [h].peGreen,
                     _colors [h].peBlue);

#elif defined (__X_MOTIF__)

    XColor xcolor;
    Display *dpy = XtDisplay (_dsurface.Client ());
    
    xcolor.pixel = h;
    XQueryColor (dpy, _cmap, &xcolor);

    UI_Color *c = new UI_Color (xcolor.red, xcolor.green, xcolor.blue);

    return *c;
#endif 
}



bool UI_ColorMap::Install ()
{
    bool retval = TRUE;
    
#if defined (__MS_WINDOWS__)
    if (!SelectPalette (_dsurface.Handle (), _cmap, FALSE)) retval = FALSE;
    if (!RealizePalette (_cmap)) retval = FALSE;

#elif defined (__X_MOTIF__)
    extern UI_Application* _TheApplication;

    UI_Application *app        = _TheApplication;
    Widget w                   = _dsurface.Client ();
    Widget toplevel            = app->Controller().ShellWidget();
    Widget colormapwidgets [2] = {w, toplevel};
    Display *dpy               = XtDisplay (_dsurface.Client());

    XSetWindowColormap (dpy, XtWindow (w), _cmap);
    XSetWindowColormap (dpy, XtWindow (toplevel), _cmap);
    XtSetWMColormapWindows (toplevel, colormapwidgets, 2);

#endif 

    return retval;
}



bool UI_ColorMap::UseDefault()
{
#if defined (__MS_WINDOWS__)
    ushort numentries = GetDeviceCaps (_dsurface.Handle(), SIZEPALETTE);
    LOGPALETTE FAR* lp = new LOGPALETTE;
       
    if (_colors)
        delete [] _colors;
    _colors = new UI_ColorMapEntry [numentries];
    if (!_colors) {
        CL_Error::Warning ("ColorMap::Add: memory allocation failed");
        return FALSE;
    }
    GetSystemPaletteEntries (_dsurface.Handle(), 0, numentries, _colors);
    DeleteObject (_cmap);

    _defaultcmap = CreatePalette (lp);
    _cmap        = _defaultcmap;

    SetPaletteEntries (_defaultcmap, 0, numentries, _colors);
    SelectPalette     (_dsurface.Handle(), _defaultcmap, FALSE);
    _numcolors   = numentries;

#elif defined (__X_MOTIF__)
    Display *dpy = XtDisplay (_dsurface.Client ());
    if (_cmap)
        XFreeColormap (dpy, _cmap);
    _cmap = 0;
    _cmap = _defaultcmap;
    XSetWindowColormap (dpy, XtWindow (_dsurface.Client ()), _cmap);
    
#endif 

    return TRUE;
}



bool UI_ColorMap::Clear()
{
    bool retval = TRUE;

#if defined (__MS_WINDOWS__)
    if(!ResizePalette (_cmap, 0)) retval = FALSE;

#elif defined (__X_MOTIF__)
    Display *dpy = XtDisplay (_dsurface.Client ());
    
    if (!XFreeColors (dpy, _cmap, _colors, _numcolors, 0)) retval = FALSE;

#endif

    _numcolors = 0;
    if (_colors)
        delete [] _colors;
    return retval;
}




bool UI_ColorMap::Replace (const UI_ColorMapHandle h, UI_Color &c)
{ 
    bool retval = TRUE;

#if defined (__MS_WINDOWS__)
    _colors [h].peRed   = LOBYTE (c.Red   ());
    _colors [h].peGreen = LOBYTE (c.Green ());
    _colors [h].peBlue  = LOBYTE (c.Blue  ());
    _colors [h].peFlags = PC_RESERVED;

    if (!SetPaletteEntries (_cmap, h, 1, &(_colors [h]))) retval = FALSE;
    
#elif defined (__X_MOTIF__)
    Display *dpy  = XtDisplay (_dsurface.Client());
    XColor xcolor = c.NativeForm ();

    xcolor.pixel  = h;
    if (!XStoreColor (dpy, _cmap, &xcolor)) retval = FALSE;

#endif 
    
    return retval;
}



bool UI_ColorMap::operator== (const UI_ColorMap &cmap) const
{
    if (_numcolors != cmap._numcolors) return FALSE;

    for (ulong i = 0; i < _numcolors; ++i) {
        if ((*this) [(UI_ColorMapHandle &) i] 
            == cmap [(UI_ColorMapHandle &) i]) continue;
        else return FALSE;
    }

    return TRUE;
}



void UI_ColorMap::operator= (const UI_ColorMap &cmap)
{
    Clear ();
    for (ulong i = 0; i < _numcolors; ++i) {
        UI_Color *c = &(cmap [(ulong &) i]);
        Add (*c);
        delete c;    
    }
}
