

#include "ui/lineseg.h"
#include "ui/dwgsurf.h"

UI_LineSegment::UI_LineSegment (const UI_Point& p1, const UI_Point& p2)
: _p1 (p1), _p2(p2)
{
}


UI_LineSegment::UI_LineSegment ()
{
}


bool UI_LineSegment::Intersects (const UI_LineSegment& l) const
{
    // This code is a formulation of the algorithm on page 889 of Cormen,
    // Leiserson and Rivest's book.
    UI_Rectangle r1 = BoundingRectangle ();
    UI_Rectangle r2 = l.BoundingRectangle ();
    if (!r1.IsContainedIn(r2) || !r2.IsContainedIn (r1))
        return FALSE;
    long v1 = (l._p1 - _p1) * (_p2 - _p1);
    long v2 = (l._p2 - _p1) * (_p2 - _p1);
    if (v1 < 0 && v2 > 0 || v1 >0 && v2 < 0)
        return TRUE;
    if (v1 == 0)
        return TRUE;
    return FALSE;
}


bool UI_LineSegment::DrawOn (UI_DrawingSurface& sfc, const UI_Point& p)
    const
{
    sfc.DrawLine  (p + _p1, p + _p2);
    return TRUE;
}



bool UI_LineSegment::ReshapeTo (const UI_Point& p1, const UI_Point& p2)
{
    if (!PrepareToChange())
        return FALSE;
    _p1 = p1;
    _p2 = p2;
    Notify ();
    return TRUE;
}


void UI_LineSegment::operator= (const CL_Object& o)
{
    if (o.IsA (*this))
        *this = (const UI_LineSegment&) o;
}


UI_LineSegment& UI_LineSegment::operator= (const UI_LineSegment& o)
{
    if (PrepareToChange()) {
        _p1 = o._p1;
        _p2 = o._p2;
        Notify();
    }
    return *this;
}


bool UI_LineSegment::OnBoundary (const UI_Point& p) const
{
    UI_LineSegment l (p, p);
    return Intersects (l);
}


UI_Rectangle UI_LineSegment::BoundingRectangle() const
{
    long x1 = _p1.XCoord();
    long y1 = _p1.YCoord();
    long x2 = _p2.XCoord();
    long y2 = _p2.YCoord();
    if (x2 > x1) {
        if (y2 > y1)
            return UI_Rectangle (_p1, x2-x1+1, y2-y1+1); // Quadrant 1
        // --------------------------------^^ ------^^
        // The rectangle's width and height include its extremities.
        else
            return UI_Rectangle (UI_Point (x1, y2), x2-x1+1, y2-y1+1); 
                                                 // Quadrant 4
    }
    else {
        if (y2 > y1)
            return UI_Rectangle (UI_Point (x2, y1), x2-x1+1, y2-y1+1);
        else
            return UI_Rectangle (_p2, x2-x1+1, y2-y1+1);
    }
}




bool UI_LineSegment::IntersectsBoundary (const UI_Rectangle& r) const
{
    return (r.Includes (_p1) ^ r.Includes (_p2));
}


bool UI_LineSegment::IsContainedIn (const UI_Rectangle& r) const
{
    return (r.Includes (_p1) && r.Includes (_p2));
}


UI_LineSegment::EndPoints UI_LineSegment::Extremities () const
{
    UI_LineSegment::EndPoints s;
    s.p1 = _p1;
    s.p2 = _p2;
    return s;
}



UI_Point UI_LineSegment::Center () const
{
    return UI_Point ((_p1.XCoord() + _p2.XCoord())/2,
                     (_p1.YCoord() + _p2.YCoord())/2);
}
