-- clock.e
-- Displays time in ANALOG and/or DIGITAL format (12 or 24 hour)
-- Also displays time in words
-- (May be useful for helping teach children how to read the time)

-- Written by Neil Kube, and placed in the PUBLIC DOMAIN

-- neilkube@ozemail.com.au

without warning

---------------------------------------------------------------------------
--      The following keys are active in this program:
--          KEY         ACTION
--          ---         ------
--          Esc         Exits program
--          t           Puts clock into real-time mode (displays current time)
--          m           increments minute
--          h           increments hour
--          1           sets 12 hour format in digital display
--          2           sets 24 hour format
---------------------------------------------------------------------------

include graphics.e
include NK_funcs.e

constant PI = 3.14159265
constant cx = 220, cy = 240 -- Centre of clock face
constant HOUR = 4, MIN = 5

sequence    HourNumbers, 
	    MinuteNumbers,
	    longhands, 
	    shorthands,
	    Words, Words_1, Words_2, sBlankLine, xx, this_time,
	    sTime

integer iKey

atom aHourHandColor, aMinHandColor, aHourNumColor, aMinNumColor, 
aMinMarkColor, aBackgroundColor, aPrevLong, aPrevShort, aTextColor, aShowTime,
aDigHourColor, aDigMinColor, gmode, format
    
aHourHandColor  = RED
aMinHandColor   = BLUE
aHourNumColor   = RED
aMinNumColor    = BLUE
aMinMarkColor   = BLUE
aDigHourColor   = RED --YELLOW
aDigMinColor    = BLUE -- YELLOW
aBackgroundColor = GREEN
aTextColor      = RED

aPrevLong = 1
aPrevShort = 1

format = 12     -- Default to 12 hour format

Words = {}
Words_1 = { "twelve",
	    "one", "two", "three", "four", "five", "six", 
	    "seven", "eight", "nine", "ten", "eleven", "twelve",
	    "one", "two", "three", "four", "five", "six", 
	    "seven", "eight", "nine", "ten", "eleven", "twelve" }
Words_2 = {
	"+0 o'clock",
	"five minutes past +0",
	"ten minutes past +0",
	"quarter past +0",
	"twenty minutes past +0",
	"twenty-five minutes past +0",
	"half past +0",
	"twenty-five minutes to +1",
	"twenty minutes to +1",
	"quarter to +1",
	"ten minutes to +1",
	"five minutes to +1",
	"+1 o'clock"
	    }

sBlankLine = repeat(32, 60) -- 60 spaces

procedure DrawMinuteNumber(atom n)
    -- Note that as the numbers are not changing, the second color may be
    -- specified as 0 - transperant
    if n < 10 then
	ShowDigital(n, {aMinNumColor, 0}, MinuteNumbers[1+(n/5)], 1, 1)
    else
	ShowDigital(n, {aMinNumColor, 0}, MinuteNumbers[1+(n/5)], 1, 2)
    end if
end procedure

procedure DrawHourNumber(atom n)
    if n < 10 then
	ShowDigital(n, {aHourNumColor, 0}, HourNumbers[n], 1.5, 1)
    else
	ShowDigital(n, {aHourNumColor, 0}, HourNumbers[n], 1.5, 2)
	if n = 10 then
	    DrawMinuteNumber(50)
	end if
    end if
end procedure

procedure ClockFace()
atom    x, y
atom    x1short, x1long, x2, x3, x4, 
	y1short, y1long, y2, y3, y4

if graphics_mode(18) then      --18, 258 or 260
    abort(1)
end if

bk_color(aBackgroundColor)
text_color(aTextColor)
clear_screen()

MinuteNumbers = {}
for n = 0 to 55 by 5 do 
    -- Calculate position of Minute numbers
    x = cx + 190 * cos((n/5-3) * PI / 6)
    y = cy + 190 * sin((n/5-3) * PI / 6) - 12
    -- Offset position of number to the left to centralise it
    if n < 10 then
	x = x-7
    else    -- 10, 11 or 12 - 2 digit numbers require greater offset to left
	x = x-17
    end if
    MinuteNumbers = append(MinuteNumbers, {x,y})
    DrawMinuteNumber(n)

    this_time = Words_2[(n/5)+1]
    Words = append( Words, this_time )
    
    for m = 1 to 4 do
	-- Calculate position of intermediate marks
	x = cos(((n+m)/5-3) * PI / 6)
	y = sin(((n+m)/5-3) * PI / 6)
	-- Draw intermediate minute marks
	-- Start & end posn. detemined by multiplication factors to {x,y} values
	draw_line(aMinMarkColor,{{cx,cy}+180*{x,y}, {cx,cy}+190*{x,y}})
	-- Create written description of the time
	if m < 3 then   -- ie 1 or 2 minutes after a 5 minute mark
	    xx = "just after " & this_time
	else    
	    this_time = Words_2[(n/5)+2]
	    xx = "almost " & this_time
	end if
	Words = append( Words, xx )
    end for

end for

-- Create 12-item sequence, each item containing a {x,y} coordinate
-- defining the position of an hour number.
HourNumbers = {}
for n = 1 to 12 do      -- Calc position of Hour numbers
    x = cx + 150 * cos((n-3) * PI / 6)
    y = cy + 150 * sin((n-3) * PI / 6) - 18
    -- Offset position of number to the left to centralise it
    if n < 10 then
	x = x-10
    else    -- 10, 11 or 12 - 2 digit numbers require greater offset to left
	x = x-29
    end if
    HourNumbers = append(HourNumbers, {x, y})
    DrawHourNumber(n)
end for

-- Create 60-item sequences, each item containing set of {x,y} coordinates
-- defining the shape of the clock hands at a particular position.
shorthands = {}
longhands = {}
for n = 1 to 60 do
    x1short = cx + 105 * cos((n/5-3) * PI / 6)
    x1long = cx + 150 * cos((n/5-3) * PI / 6)
    x2 = cx +  12 * cos(((n-15)/5-3) * PI / 6)
    x3 = cx +  12 * cos(((n-30)/5-3) * PI / 6)
    x4 = cx +  12 * cos(((n+15)/5-3) * PI / 6)
    y1short = cy + 105 * sin((n/5-3) * PI / 6)
    y1long = cy + 150 * sin((n/5-3) * PI / 6)
    y2 = cy +  12 * sin(((n-15)/5-3) * PI / 6)
    y3 = cy +  12 * sin(((n-30)/5-3) * PI / 6)
    y4 = cy +  12 * sin(((n+15)/5-3) * PI / 6)
    shorthands = append(shorthands, {{x1short,y1short}, {x2,y2}, {x3,y3}, {x4,y4}})
    longhands = append(longhands, {{x1long,y1long}, {x2,y2}, {x3,y3}, {x4,y4}})
end for
end procedure -- ClockFace

procedure ShowTime(sequence Time)   --{year, month, day, hour, minute, second}
    atom Minute, MinutePos, Hour, HourPos, x, location
    
    Minute = floor(Time[MIN])
    if Minute < 1 then
	MinutePos = Minute + 60
    elsif Minute > 60 then
	MinutePos = Minute - 60
    else    
	MinutePos = Minute
    end if
    
    Hour = floor(Time[HOUR])
  
    if format = 12 then
	if Hour > 12 then
	    Hour = Hour - 12
	end if
	if Hour = 0 then
	    Hour = 12
	end if
    else    -- 24 hour
	if Hour = 24 then
	    Hour = 0
	end if
    end if
    
    
    HourPos = Hour * 5 + floor(Minute / 12)
    if HourPos < 1 then
	HourPos = HourPos + 60
    elsif HourPos > 60 then
	HourPos = HourPos - 60
    end if

    -- Remove hands from old position
    if aPrevLong = MinutePos then   -- Minute hand hasn't moved
				    -- no need to redraw it
    else
	polygon(aBackgroundColor, 1, longhands[aPrevLong])  -- Erase old hand

	x = floor(aPrevLong/5 + .5)                         -- Calc which hour
	if x = 0 then                                       -- number to redraw
	    x = 12
	end if
	DrawHourNumber( x )                                 -- Redraw it

	polygon(aMinHandColor,1,longhands[MinutePos])       -- Draw new hand
	
	aPrevLong = MinutePos                               -- Remember
	polygon(aHourHandColor, 1, shorthands[aPrevShort])  -- Redraw old hour
							    -- hand - it may
							    -- have been erased
    end if
    
    if aPrevShort = HourPos then    -- Hour hand hasn't moved
				    -- no need to redraw
    else
	polygon(aBackgroundColor, 1, shorthands[aPrevShort])    -- Erase old
	polygon(aMinHandColor, 1, longhands[aPrevLong])     -- Redraw old min
							    -- hand - it may
							    -- have been erased
	polygon(aHourHandColor,1,shorthands[HourPos])           -- Draw new
	aPrevShort = HourPos                                    -- Remember
    end if
    
    -- Display time digitally in HH MM format
    ShowDigital(Hour, {aDigHourColor, aBackgroundColor}, {410,20}, 3, 2)
    ShowDigital(Minute, {aDigMinColor, aBackgroundColor}, {530,20}, 3, 2)
    
    -- Display time in words
    position(30,2)
    puts(1, sBlankLine)
    position(30,2)
    if MinutePos = 60 then
	this_time = Words[1]
    else        
	this_time = Words[MinutePos+1]
    end if
    -- Replace "+0" with current hour name
    -- or replace "+1" with next hour name
    location = match("+", this_time)
    if this_time[location + 1] = 48 then -- "0" - this hour
	this_time = this_time[1..location-1]
	& Words_1[Hour+1] & this_time[location+2..length(this_time)]
    else    
	this_time = this_time[1..location-1]
	& Words_1[Hour+2] & this_time[location+2..length(this_time)]
    end if
    printf(1, "%s", {this_time})
    
end procedure

ClockFace()
aShowTime = TRUE
sTime = date()
ShowTime( sTime )

while TRUE do
    iKey = wait(60) --get_key()
    if iKey = 27 then   -- Esc pressed - Quit
	gmode = graphics_mode(-1)
	exit
    elsif iKey = 104 then   -- h - advance hour
	sTime = sTime + {0,0,0,1,0,0,0,0}
	if sTime[HOUR] = 24 then
	    sTime = sTime - {0,0,0,24,0,0,0,0}
	end if
	aShowTime = FALSE
    elsif iKey = 109 then   -- m - advance minute
	sTime = sTime + {0,0,0,0,1,0,0,0}
	if sTime[MIN] > 59 then
	    sTime = sTime - {0,0,0,-1,60,0,0,0}
	    if sTime[HOUR] > 23 then
		sTime = sTime - {0,0,0,24,0,0,0,0}
	    end if
	end if
	aShowTime = FALSE
    elsif iKey = 116 then   -- t - show system time
	aShowTime = TRUE
    elsif iKey = 49 then   -- 1 - 12 hour format
	format = 12
    elsif iKey = 50 then   -- 2 - 24 hour format
	format = 24
    end if
    if aShowTime = TRUE then
	sTime = date()
    end if
    ShowTime( sTime )

end while


