MODULE Convert;

	(********************************************************)
	(*							*)
	(*	Converts "Accounts" data files from		*)
	(*	version 1 to version 2.00 format.		*)
	(*							*)
	(*  Programmer:		P. Moylan			*)
	(*  Last edited:	22 April 1993			*)
	(*  Status:		Complete but not tested		*)
	(*							*)
	(********************************************************)

(************************************************************************)
(*									*)
(*  Remark: this module defines a number of types which logically	*)
(*  should be imported from other modules.  The reason is that we are	*)
(*  necessarily concerned with the fine structure of how disk records	*)
(*  are implemented, and therefore cannot reasonably take the usual	*)
(*  attitude of deciding that implementation details will be looked	*)
(*  after in lower-level modules.					*)
(*									*)
(************************************************************************)

FROM SYSTEM IMPORT
    (* type *)	BYTE, ADR;

FROM IOErrorCodes IMPORT
    (* const*)	OK;

FROM FileSys IMPORT
    (* type *)	File,
    (* proc *)	OpenFile, CloseFile, ReadRecord, WriteRecord;

FROM Lib IMPORT
    (* proc *)	ParamCount, ParamStr;

FROM Windows IMPORT
    (* type *)	Window, Colour, FrameType, DividerType,
    (* proc *)	OpenWindow, CloseWindow, WriteString, WriteLn,
		PressAnyKey;

FROM Menus IMPORT
    (* type *)	ItemText;

(************************************************************************)
(*	Types common to both old and new file formats:			*)
(************************************************************************)

CONST NoteFieldSize = 38;

TYPE
    fieldtype = (yearfield, monthfield, dayfield);
    Date = ARRAY fieldtype OF BYTE;
    AbbreviationType = ARRAY [0..2] OF CHAR;
    Money = LONGINT;
    NoteFieldSubscript = [0..NoteFieldSize-1];
    NoteType = ARRAY NoteFieldSubscript OF CHAR;

    FileRecordNumber = LONGCARD;
    TitleType = ItemText;

(************************************************************************)
(*	Types specific to the old file format:				*)
(************************************************************************)

TYPE
    OldTransactionKind = (debit, credit, oldseparator, oldnote);
    OldTransactionRecord =
			RECORD
			    datefield:	Date;
			    kind:	OldTransactionKind;
			    code:	AbbreviationType;
			    amountfield:Money;
			    notefield:	NoteType;
			END (*RECORD*);

    OldFileHeader = RECORD
			NoOfEntries: FileRecordNumber;
			OpeningBalance, ClosingBalance: Money;
			LongTitle: TitleType;
			abbreviation: AbbreviationType;
		    END (*RECORD*);

(************************************************************************)
(*	Types specific to the new file format:				*)
(************************************************************************)

CONST LongNoteFieldSize = NoteFieldSize + SIZE(AbbreviationType)
						+ SIZE(Money);

TYPE
    NewTransactionKind = (normal, separator, note);
    LongNoteFieldSubscript = [0..LongNoteFieldSize-1];
    LongNoteType = ARRAY LongNoteFieldSubscript OF CHAR;
    NewTransactionRecord =
			RECORD
			    datefield:	Date;
			    CASE kind:	NewTransactionKind OF
				normal:
				    code:	AbbreviationType;
				    amountfield:Money;
				    notefield:	NoteType;
				|
				separator, note:
				    longnotefield: LongNoteType;
			    END (*CASE*);
			END (*RECORD*);

    NewFileHeader = RECORD
			minorversion, majorversion: SHORTCARD;
			NoOfEntries: FileRecordNumber;
			OpeningBalance, ClosingBalance: Money;
			LongTitle: TitleType;
			abbreviation: AbbreviationType;
		    END (*RECORD*);

(************************************************************************)
(*									*)
(*			    GLOBAL VARIABLES				*)
(*									*)
(*  (Normally I frown on the use of global variables, but here their	*)
(*   use seems to be fairly innocuous).					*)
(*									*)
(************************************************************************)

VAR
    OldStream, NewStream: File;

(************************************************************************)
(*		DEALING WITH THE DATA FILE HEADERS			*)
(************************************************************************)

PROCEDURE ConvertHeader (VAR (*OUT*) NumberOfDataRecords: FileRecordNumber;
				VAR (*OUT*) failure: BOOLEAN);

    (* Reads the header of the input file, writes the header of the	*)
    (* output.  Returns the number of data records (not counting the	*)
    (* header) of the input file.					*)

    VAR amount: CARDINAL;
	OldHeader: OldFileHeader;
	NewHeader: NewFileHeader;

    BEGIN
	IF (ReadRecord (OldStream, ADR(OldHeader), SIZE(OldHeader), amount)
				= OK) AND (amount = SIZE(OldHeader)) THEN
	    NewHeader.minorversion := 0;
	    NewHeader.majorversion := 2;
	    NewHeader.NoOfEntries := OldHeader.NoOfEntries;
	    NewHeader.OpeningBalance := OldHeader.OpeningBalance;
	    NewHeader.ClosingBalance := OldHeader.ClosingBalance;
	    NewHeader.LongTitle := OldHeader.LongTitle;
	    NewHeader.abbreviation := OldHeader.abbreviation;
	    failure := WriteRecord (NewStream,ADR(NewHeader),
					SIZE(NewHeader)) <> OK;
	ELSE
	    failure := TRUE;
	END (*IF*);
	NumberOfDataRecords := NewHeader.NoOfEntries;
    END ConvertHeader;

(************************************************************************)
(*		THE DATA CONVERSION PROCEDURES				*)
(************************************************************************)

PROCEDURE PadNote1 (VAR (*IN*) source: NoteType;
			VAR (*OUT*) dest: LongNoteType);

    (* Copies source to dest, padding with leading minus signs.	*)

    CONST gap = LongNoteFieldSize - NoteFieldSize;

    VAR j: NoteFieldSubscript;

    BEGIN
	FOR j := 0 TO gap-1 DO
	    dest[j] := "-";
	END (*FOR*);
	FOR j := 0 TO NoteFieldSize-1 DO
	    dest[j+gap] := source[j];
	END (*FOR*);
    END PadNote1;

(************************************************************************)

PROCEDURE PadNote2 (VAR (*IN*) source: NoteType;
			VAR (*OUT*) dest: LongNoteType);

    (* Copies source to dest, padding with trailing spaces.	*)

    VAR j: LongNoteFieldSubscript;

    BEGIN
	FOR j := 0 TO NoteFieldSize-1 DO
	    dest[j] := source[j];
	END (*FOR*);
	FOR j := NoteFieldSize TO LongNoteFieldSize-1 DO
	    dest[j] := " ";
	END (*FOR*);
    END PadNote2;

(************************************************************************)

PROCEDURE ConvertKind (original: OldTransactionRecord;
			VAR (*OUT*) result: NewTransactionRecord): BOOLEAN;

    (* Converts a transaction kind from the old to the new format.	*)
    (* Returns FALSE if unknown kind is encountered.			*)

    BEGIN
	CASE original.kind OF
	    debit:	result.kind := normal;
			result.code := original.code;
			result.amountfield := -original.amountfield;
			result.notefield := original.notefield;
	  |
	    credit:	result.kind := normal;
			result.code := original.code;
			result.amountfield := original.amountfield;
			result.notefield := original.notefield;
	  |
	    oldseparator: result.kind := separator;
			PadNote1 (original.notefield, result.longnotefield);
	  |
	    oldnote:	result.kind := note;
			result.amountfield := 0;
			PadNote2 (original.notefield, result.longnotefield);
	  |
	  ELSE RETURN FALSE;
	END (*CASE*);
	RETURN TRUE;
    END ConvertKind;

(************************************************************************)

PROCEDURE ConvertOneRecord (VAR (*OUT*) error: BOOLEAN);

    (* Performs the conversion for one record.	*)

    VAR InRecord: OldTransactionRecord;  OutRecord: NewTransactionRecord;
	amount: CARDINAL;

    BEGIN
	error := (ReadRecord (OldStream, ADR(InRecord),
				SIZE(InRecord), amount) <> OK)
			OR (amount <> SIZE(InRecord));
	IF NOT error THEN
	    OutRecord.datefield := InRecord.datefield;
	    error := (NOT ConvertKind (InRecord, OutRecord))
			OR (WriteRecord (NewStream, ADR(OutRecord),
					SIZE(OutRecord)) <> OK);
	END (*IF*);
    END ConvertOneRecord;

(************************************************************************)
(*			OPENING AND CLOSING FILES			*)
(************************************************************************)

PROCEDURE OpenTheFiles (VAR (*OUT*) OpenFailure: BOOLEAN);

    (* Opens both the input and output file. *)

    VAR OldFileName, NewFileName: ARRAY [0..64] OF CHAR;

    BEGIN
	OpenFailure := ParamCount() <> 2;
	IF NOT OpenFailure THEN
	    ParamStr (OldFileName, 1);
	    ParamStr (NewFileName, 2);
	    OpenFailure :=
			(OpenFile (OldStream, OldFileName, FALSE) <> OK)
			OR (OpenFile (NewStream, NewFileName, TRUE) <> OK);
	END (*IF*);
    END OpenTheFiles;

(************************************************************************)

PROCEDURE CloseTheFiles;

    (* Closes both the input and output file. *)

    BEGIN
	CloseFile (OldStream);
	CloseFile (NewStream);
    END CloseTheFiles;

(************************************************************************)

PROCEDURE DoTheConversion;

    (* The master procedure, which processes the old data file and	*)
    (* creates the corresponding new file.				*)

    VAR NumberOfRecordsRemaining: FileRecordNumber;
	IOfailure: BOOLEAN;
	status: Window;

    BEGIN
	OpenWindow (status, white, black, 0, 24, 0, 79, noframe, nodivider);
	OpenTheFiles (IOfailure);
	IF IOfailure THEN
	    WriteString (status, "Cannot proceed - failed to open one ");
	    WriteString (status, "or both of the data files.");
	ELSE
	    WriteString (status, "Starting the conversion.  Please wait ...");
	    WriteLn (status);
	    ConvertHeader (NumberOfRecordsRemaining, IOfailure);
	    WHILE (NumberOfRecordsRemaining > 0) AND NOT IOfailure DO
		ConvertOneRecord (IOfailure);
		DEC (NumberOfRecordsRemaining);
	    END (*WHILE*);
	    CloseTheFiles;
	    IF IOfailure THEN
		WriteString (status, "Conversion failed because of I/O error.");
	    ELSE
		WriteString (status, "Conversion successfully completed.");
	    END (*IF*);
	    WriteLn (status);
	END (*IF*);
	PressAnyKey (status);  CloseWindow (status);
    END DoTheConversion;

(************************************************************************)
(*				MAIN PROGRAM				*)
(************************************************************************)

BEGIN
    DoTheConversion;
END Convert.

