IMPLEMENTATION MODULE Menus;

	(****************************************************************)
	(*								*)
	(*	Displays menus on screen, allows terminal user to	*)
	(*			select from them.			*)
	(*								*)
	(*  Programmer:		P. Moylan				*)
	(*  Last edited:	25 February 1994			*)
	(*  Status:		OK					*)
	(*								*)
	(****************************************************************)

FROM Storage IMPORT
    (* proc *)	ALLOCATE, DEALLOCATE;

FROM TaskControl IMPORT
    (* type *)	Lock,
    (* proc *)	CreateLock, Obtain, Release;

FROM Keyboard IMPORT
    (* proc *)	InKey, PutBack, StuffKeyboardBuffer;

FROM Windows IMPORT
    (* const*)	MaxColumnNumber,
    (* type *)	Window, FrameType, DividerType,
    (* proc *)	OpenWindow, CloseWindow, ChangeScrollingRegion,
		WriteChar, WriteString, ColourSwap, Blink,
		SetCursor, SaveCursor, ScrollUp, ScrollDown, EraseLine,
		NewScrollingRegion, ResetScrollingRegion;

FROM Mouse IMPORT
    (* type *)	Buttons, ButtonSet,
    (* proc *)	MouseAvailable, HideMouseCursor, ShowMouseCursor;

FROM UserInterface IMPORT
    (* type *)	UIWindow, Capability, CapabilitySet,
    (* proc *)	AllowMouseControl, AddActiveRegion;

(************************************************************************)

CONST
    gap = 1;			(* space between menu columns *)
    ClickIndicator = CHR(1);	(* special code to indicate mouse click	*)
    LeftOnly = ButtonSet{LeftButton};
    Esc = CHR(01BH);		(* keyboard Escape character *)

TYPE
    ItemNo = CARDINAL;

    (* The following declaration uses subscript bounds [1..1] because	*)
    (* Modula-2 provides no way of declaring a variable-length array	*)
    (* (except as a procedure parameter).  The correct array size will	*)
    (* be established on a call to ALLOCATE.				*)

    TextPointer = POINTER TO ARRAY [1..1] OF ItemText;
    Menu = POINTER TO MenuDetails;

    (********************************************************************)
    (*									*)
    (* The fields in a MenuDetails record have the following meaning:	*)
    (*									*)
    (*	wname		The window used to display this menu		*)
    (*			 on the screen					*)
    (*	foreground,	The colours to use when displaying the menu	*)
    (*	 background							*)
    (*	ScreenPosition	The row and column numbers which will be	*)
    (*			 occupied by the window frame on the screen.	*)
    (*			 After allowing for the frame and header, the	*)
    (*			 actual text starts at row (firstrow+3) and	*)
    (*			 column (firstcol+1).  To fit the text exactly	*)
    (*			 in the window frame, we would need		*)
    (*			    NoOfItems = (lastrow-firstrow-4)*NoOfColumns*)
    (*			 If NoOfItems is smaller than this, the		*)
    (*			 window will include some blank areas.  If it	*)
    (*			 is larger, the menu scrolls.			*)
    (*	heading		The text to display in the menu header		*)
    (*	NoOfItems	Number of menu items				*)
    (*	ItemsPerColumn  Number of items allocated to each column	*)
    (*	NoOfColumns	Number of columns of menu items			*)
    (*	hstep		The increment in item number resulting from	*)
    (*			 one "cursor right" operation.			*)
    (*	vstep		The increment in item number resulting from	*)
    (*			 one "cursor down" operation.  Note that hstep	*)
    (*			 and vstep depend on whether the items are	*)
    (*			 numbered horizontally or vertically, and that	*)
    (*			 in turn depends on whether we have a short fat	*)
    (*			 display or a tall thin one.			*)
    (*	LocationInWindow: The location of the menu itself, relative to	*)
    (*			 the window in which it is displayed, and not	*)
    (*			 counting the space taken by borders or a	*)
    (*			 header.  The top left character is at location	*)
    (*			 (firstrow,firstcol), the menu takes 'height'	*)
    (*			 screen lines and is 'width' characters wide.	*)
    (*	ColumnWidth	The number of characters of TextPtr^[j] which	*)
    (*			 will be written on the screen.  Approximately	*)
    (*			 equal to (lastcol-firstcol-1)/NoOfColumns.	*)
    (*  CurrentItemNo	The item number currently selected		*)
    (*  row		The current menu row selected.  Note that row=1	*)
    (*			 means the top row of items displayed; this is	*)
    (*			 not necessarily the top row of the entire menu	*)
    (*			 since some items may have scrolled off the top.*)
    (*  column		The current menu column selected		*)
    (*	ExtraAtTop	The number of rows which have disappeared off	*)
    (*				the top of the display			*)
    (*	ExtraAtBottom	The number of rows which have disappeared off	*)
    (*				the bottom of the display		*)
    (*	SpecialMode	Says whether special mode is active		*)
    (*	RanOffEdge	Set if a cursor movement ran us off the edge	*)
    (*				of the menu				*)
    (*	TextPtr^[j]	The text to display for menu item j.		*)
    (*									*)
    (********************************************************************)

    MenuDetails =

	   RECORD
		(* The screen window - recorded in both modes.		*)

		wname: Window;

		(* Details about the screen window - normal mode only.	*)

		foreground, background: Colour;
		ScreenPosition:
		    RECORD
			firstrow, lastrow: RowRange;
			firstcol, lastcol: ColumnRange;
		    END (*RECORD*);
		heading: ItemText;

		(* The layout of the menu within its window, determined	*)
		(* at the time that the menu is created.		*)

		NoOfItems: ItemNo;
		ItemsPerColumn: ItemNo;
		NoOfColumns: MenuColumn;

		(* Further layout details, determined at the time we	*)
		(* discover how much space the menu will be given on	*)
		(* the screen.						*)

		hstep, vstep: CARDINAL;
		LocationInWindow:
		    RECORD
			firstrow, firstcol: CARDINAL;
			height, width: CARDINAL;
		    END (*RECORD*);
		ColumnWidth: ColumnRange;

		(* Information about the current state of the menu.	*)

		CurrentItemNo: ItemNo;
		row: RowRange;
		column: MenuColumn;
		ExtraAtTop, ExtraAtBottom: CARDINAL;
		SpecialMode, RanOffEdge: BOOLEAN;

		(* Pointer to the text of the menu items.	*)

		TextPtr: TextPointer;

	    END (*RECORD*);

(************************************************************************)

VAR
    (* The following record contains the details of the last mouse	*)
    (* click.  Because users can't move a mouse very fast, we don't	*)
    (* bother to keep a queue of clicks, we just record the last seen.	*)
    (* In the event that a click arrives before the last has been	*)
    (* consumed, the earlier click is lost.  I haven't yet seen an	*)
    (* application where that has been a major problem.			*)

    LastMouseClick: RECORD
			access: Lock;
			win: Window;
			X: ColumnRange;  Y: RowRange;
			valid: BOOLEAN;
		    END (*RECORD*);

(************************************************************************)
(*			MISCELLANEOUS UTILITIES				*)
(************************************************************************)

PROCEDURE resize (VAR (*INOUT*) item: ItemText;  size: ColumnRange);

    (* Makes item equal to the given size, by space filling on the	*)
    (* right and inserting a Nul to terminate the text.			*)

    VAR j: ColumnRange;

    BEGIN
	j := 0;
	WHILE (ORD(item[j]) <> 0) AND (j < size) DO INC(j) END (*WHILE*);
	WHILE j < size DO item[j] := " "; INC(j) END (*WHILE*);
	IF j <= MaxColumnNumber THEN item[j] := CHR(0) END (*IF*);
    END resize;

(************************************************************************)
(*			   CREATING A MENU				*)
(************************************************************************)

PROCEDURE CreateMenu (VAR (*OUT*) M: Menu; columns: MenuColumn;
			VAR (*IN*) Messages: ARRAY OF ItemText;
			NumberOfItems: CARDINAL);

    (* Introduces a menu into the system, but does not display it yet.	*)

    VAR j, count: CARDINAL;

    BEGIN
	NEW (M);
	WITH M^ DO
	    NoOfColumns := columns;

	    (* Store the header text, with space fill.	*)

	    heading := Messages[0];
	    resize (heading, MaxColumnNumber);

	    (* Work out how many menu items there are.	*)

	    count := NumberOfItems;
	    IF (count = 0) OR (count > HIGH (Messages)) THEN
		count := HIGH (Messages);
	    END (*IF*);
	    NoOfItems := count;
	    ItemsPerColumn := (count + NoOfColumns - 1) DIV NoOfColumns;
	    CurrentItemNo := 1;

	    (* Store the item text.	*)

	    ALLOCATE (TextPtr, NoOfItems*SIZE(ItemText));
	    FOR j := 1 TO NoOfItems DO
		(*# save, check(index=>off) *)
		TextPtr^[j] := Messages[j];
		(*# restore *)
	    END (*FOR*);

	END (*WITH*);

	(* Give the menu a default initial position, size, and colour.	*)

	PositionMenu (M, white, black, 0, 10, 0, MaxColumnNumber);

    END CreateMenu;

(************************************************************************)
(*			  POSITIONING A MENU				*)
(************************************************************************)

PROCEDURE SetRelativeLocation (M: Menu;  row1, col1, rows, columns: CARDINAL);

    (* Gives initial values to M^.LocationInWindow and M^.ColumnWidth,	*)
    (* and resizes the item text to the space available.  Also sets	*)
    (* M^.hstep and M^.vstep, based on the following criterion: if the	*)
    (* display will be wider than it is tall then we use row major	*)
    (* ordering (hstep = 1), whereas for tall narrow menus we use	*)
    (* column major ordering (vstep = 1).  This distinction is actually	*)
    (* irrelevant to the caller, but it affects the appearance of the	*)
    (* menu, and the decision taken here seems to give a result which	*)
    (* someone reading the screen would consider intuitively logical.	*)

    VAR j: ItemNo;

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		firstrow := row1;  firstcol := col1;
		height := rows;  width := columns;
		IF ItemsPerColumn <= height THEN
		    height := ItemsPerColumn;
		END (*IF*);
		IF NoOfColumns > height THEN
		    hstep := 1;  vstep := NoOfColumns;
		ELSE
		    hstep := ItemsPerColumn;  vstep := 1;
		END (*IF*);
	    END (*WITH*);

	    resize (heading, columns);
	    ColumnWidth := (columns - (NoOfColumns-1)*gap) DIV NoOfColumns;
	    FOR j := 1 TO NoOfItems DO
		(*# save, check(index=>off) *)
		resize (TextPtr^[j], ColumnWidth);
		(*# restore *)
	    END (*FOR*);

	END (*WITH*);

    END SetRelativeLocation;

(************************************************************************)

PROCEDURE PositionMenu (M: Menu; ForegroundColour, BackgroundColour: Colour;
			startline, endline: RowRange;
			leftcol, rightcol: ColumnRange);

    (* Sets the screen location and colours of the window which will	*)
    (* hold the menu.  Calling this procedure automatically enables	*)
    (* Normal Mode.							*)

    VAR j: CARDINAL;

    BEGIN
	WITH M^ DO
	    SpecialMode := FALSE;
	    foreground := ForegroundColour;
	    background := BackgroundColour;

	    (* Work out the space available on the screen.	*)

	    WITH ScreenPosition DO
		firstrow := startline;  lastrow := endline;
		firstcol := leftcol;  lastcol := rightcol;
	    END (*WITH*);

	    SetRelativeLocation (M, 3, 1, endline-startline-3,
						rightcol-leftcol-1);

	END (*WITH*);
    END PositionMenu;

(************************************************************************)
(*			    CLOSING A MENU				*)
(************************************************************************)

PROCEDURE DestroyMenu (M: Menu);

    (* Removes a menu from the system, freeing up the space it used.	*)

    BEGIN
	DEALLOCATE (M^.TextPtr, M^.NoOfItems*SIZE(ItemText));
	DISPOSE (M);
    END DestroyMenu;

(************************************************************************)
(*			    SCREEN DISPLAY				*)
(************************************************************************)

PROCEDURE RefreshRow (M: Menu);

    (* Refreshes the current menu row.	*)

    VAR screenrow: RowRange;  j: ColumnRange;  item: ItemNo;
	count: CARDINAL;

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		screenrow := firstrow + row - 1;
		j := LocationInWindow.firstcol;
	    END (*WITH*);
	    SetCursor (wname, screenrow, j);  EraseLine (wname, 1);
	    count := NoOfColumns;
	    item := 1 + vstep*(row + ExtraAtTop - 1);
	    LOOP
		(*# save, check(index=>off) *)
		WriteString (wname, TextPtr^[item]);
		(*# restore *)
		DEC (count);
		IF (count = 0) OR (item+hstep > NoOfItems) THEN
		    EXIT (*LOOP*);
		END (*IF*);
		INC (item, hstep);  INC (j, ColumnWidth + gap);
		SetCursor (wname, screenrow, j);
	    END (*LOOP*);
	END (*WITH*);
    END RefreshRow;

(************************************************************************)

PROCEDURE DisplayMOREatTop (M: Menu);

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		SetCursor (wname, firstrow-1, firstcol+width-6);
	    END (*WITH*);
	    WriteString (wname, "*MORE*");
	END (*WITH*);
    END DisplayMOREatTop;

(************************************************************************)

PROCEDURE RemoveMOREatTop (M: Menu);

    CONST DoubleBar = '';

    VAR j: [1..6];

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		SetCursor (wname, firstrow-1, firstcol+width-6);
	    END (*WITH*);
	    FOR j := 1 TO 6 DO
		WriteChar (wname, DoubleBar);
	    END (*FOR*);
	END (*WITH*);
    END RemoveMOREatTop;

(************************************************************************)

PROCEDURE DisplayMOREatBottom (M: Menu);

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		SetCursor (wname, firstrow+height, firstcol+width-6);
	    END (*WITH*);
	    WriteString (wname, "*MORE*");
	END (*WITH*);
    END DisplayMOREatBottom;

(************************************************************************)

PROCEDURE RemoveMOREatBottom (M: Menu);

    CONST HorizontalBar = '';

    VAR j: [1..6];

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		SetCursor (wname, firstrow+height, firstcol+width-6);
	    END (*WITH*);
	    FOR j := 1 TO 6 DO
		WriteChar (wname, HorizontalBar);
	    END (*FOR*);
	END (*WITH*);
    END RemoveMOREatBottom;

(************************************************************************)

PROCEDURE Highlight (M: Menu);

    (* Toggles the highlighted state of the current menu item.	*)

    BEGIN
	IF MouseAvailable() THEN HideMouseCursor END (*IF*);
	WITH M^ DO
	    ColourSwap (wname, LocationInWindow.firstrow+row-1,
		(column-1)*(ColumnWidth+gap) + LocationInWindow.firstcol,
			ColumnWidth);
	END (*WITH*);
	IF MouseAvailable() THEN ShowMouseCursor END (*IF*);
    END Highlight;

(************************************************************************)

PROCEDURE BlinkCurrent (M: Menu);

    (* Toggles the blinking state of the current menu item.	*)

    BEGIN
	WITH M^ DO
	    Blink (wname, LocationInWindow.firstrow+row-1,
		(column-1)*(ColumnWidth+gap) + LocationInWindow.firstcol,
			ColumnWidth);
	END (*WITH*);
    END BlinkCurrent;

(************************************************************************)
(*			    CURSOR MOVEMENTS				*)
(************************************************************************)

PROCEDURE UpARow (M: Menu);

    (* Moves to the next item up, if present.  Scrolls if necessary.	*)

    BEGIN
	WITH M^ DO
	    IF row > 1 THEN
		DEC (row);  DEC (CurrentItemNo, vstep);
	    ELSIF ExtraAtTop > 0 THEN
		DEC (CurrentItemNo, vstep);
		ScrollDown (wname);
		DEC (ExtraAtTop);  INC (ExtraAtBottom);
		RefreshRow (M);
		IF NOT SpecialMode THEN
		    IF MouseAvailable() THEN HideMouseCursor END (*IF*);
		    IF ExtraAtTop = 0 THEN RemoveMOREatTop(M) END (*IF*);
		    IF ExtraAtBottom = 1 THEN DisplayMOREatBottom(M) END(*IF*);
		    IF MouseAvailable() THEN ShowMouseCursor END (*IF*);
		END (*IF*);
	    ELSIF SpecialMode THEN
		RanOffEdge := TRUE;
	    END (*IF*);
	END (*WITH*);
    END UpARow;

(************************************************************************)

PROCEDURE DownARow (M: Menu);

    (* Moves to the next item down, if present.  Scrolls if necessary.	*)

    BEGIN
	WITH M^ DO
	    IF row < LocationInWindow.height THEN
		INC (row);  INC (CurrentItemNo, vstep);
	    ELSIF ExtraAtBottom > 0 THEN
		INC (CurrentItemNo, vstep);  ScrollUp (wname);
		INC (ExtraAtTop);  DEC (ExtraAtBottom);
		RefreshRow (M);
		IF NOT SpecialMode THEN
		    IF MouseAvailable() THEN HideMouseCursor END (*IF*);
		    IF ExtraAtTop = 1 THEN DisplayMOREatTop(M) END (*IF*);
		    IF ExtraAtBottom = 0 THEN RemoveMOREatBottom(M) END (*IF*);
		    IF MouseAvailable() THEN ShowMouseCursor END (*IF*);
		END (*IF*);
	    ELSIF SpecialMode THEN
		RanOffEdge := TRUE;
	    END (*IF*);
	END (*WITH*);
    END DownARow;

(************************************************************************)

PROCEDURE MoveRight (M: Menu);

    (* Moves to the next item right, if present.	*)

    BEGIN
	WITH M^ DO
	    IF column < NoOfColumns THEN
		INC (column);  INC (CurrentItemNo, hstep);
	    ELSIF SpecialMode THEN
		RanOffEdge := TRUE;
	    END (*IF*);
	END (*WITH*);
    END MoveRight;

(************************************************************************)

PROCEDURE MoveLeft (M: Menu);

    (* Moves to the next item left, if present.	*)

    BEGIN
	WITH M^ DO
	    IF column > 1 THEN
		DEC (column);  DEC (CurrentItemNo, hstep);
	    ELSIF SpecialMode THEN
		RanOffEdge := TRUE;
	    END (*IF*);
	END (*WITH*);
    END MoveLeft;

(************************************************************************)

PROCEDURE GotoItem (M: Menu;  newitem: ItemNo);

    (* Moves to the menu item whose number is specified.  We move a row	*)
    (* at a time, rather than taking one big leap, since this is less	*)
    (* disconcerting to the user.					*)

    BEGIN
	WITH M^ DO
	    IF newitem <> CurrentItemNo THEN
		IF vstep = 1 THEN	(* we are using column major order *)
		    column := ((newitem-1) DIV hstep) + 1;
		ELSE			(* we are using row major order *)
		    column := ((newitem-1) MOD vstep) + 1;
		END (*IF*);
		CurrentItemNo := (column-1)*hstep
					+ (ExtraAtTop+row-1)*vstep + 1;
		WHILE CurrentItemNo > newitem DO UpARow(M) END (*WHILE*);
		WHILE CurrentItemNo < newitem DO DownARow(M) END (*WHILE*);
	    END (*IF*);
	END (*WITH*);
    END GotoItem;

(************************************************************************)

PROCEDURE RepositionTo (M: Menu; leading: CHAR);

    (* Finds the next menu item whose first character matches	*)
    (* "leading", and adjusts the display appropriately.	*)

    VAR newitem: ItemNo;

    BEGIN
	WITH M^ DO
	    newitem := CurrentItemNo;
	    REPEAT
		IF newitem >= NoOfItems THEN newitem := 1
		ELSE INC (newitem)
		END (*IF*);
		(*# save, check(index=>off) *)
	    UNTIL (CAP(TextPtr^[newitem][0]) = leading)
				OR (newitem = CurrentItemNo);
		(*# restore *)
	END (*WITH*);
	GotoItem (M, newitem);
    END RepositionTo;

(************************************************************************)

PROCEDURE HandleFunctionKey (M: Menu;  VAR (*INOUT*) option: CHAR);

    (* Deals with the case where the user typed a function key - i.e.	*)
    (* any key which produces a two-code sequence where the first code	*)
    (* is CHR(0).  On entry, the CHR(0) has already been read.		*)

    VAR count: CARDINAL;

    BEGIN
	option := InKey();
	WITH M^ DO
	    IF option = "H" THEN UpARow(M)		(* cursor up *)
	    ELSIF option = "P" THEN DownARow(M)		(* cursor down *)
	    ELSIF option = "M" THEN MoveRight(M)	(* cursor right *)
	    ELSIF option = "K" THEN MoveLeft(M)		(* cursor left *)
	    ELSIF option = "G" THEN			(* home *)
		GotoItem (M, 1);
	    ELSIF option = "O" THEN			(* end *)
		GotoItem (M, NoOfColumns*ItemsPerColumn);
		GotoItem (M, NoOfItems);
	    ELSIF option = "I" THEN			(* page up *)
		IF row = 1 THEN
		    IF ExtraAtTop > 0 THEN
			count := LocationInWindow.height;
			REPEAT
			    UpARow(M);  DEC (count);
			UNTIL (count=0) OR (ExtraAtTop=0);
		    END (*IF*)
		ELSE
		    WHILE row > 1 DO UpARow(M) END (*WHILE*)
		END (*IF*)
	    ELSIF option = "Q" THEN			(* page down *)
		IF row = LocationInWindow.height THEN
		    IF ExtraAtBottom > 0 THEN
			count := LocationInWindow.height;
			REPEAT
			    DownARow(M);  DEC (count);
			UNTIL (count=0) OR (ExtraAtBottom=0);
		    END (*IF*)
		ELSE
		    WHILE row < LocationInWindow.height DO
			DownARow(M);
		    END (*WHILE*)
		END (*IF*)
	    ELSIF SpecialMode THEN			(* unknown option *)
		PutBack (option);   PutBack (CHR(0));
	    END (*IF*);
	END (*WITH*);
    END HandleFunctionKey;

(************************************************************************)
(*			DEALING WITH MOUSE CLICKS			*)
(************************************************************************)

PROCEDURE SelectItemAt (M: Menu;  r: RowRange;  c: ColumnRange);

    VAR NewItemNo: ItemNo;  OnAnItem: BOOLEAN;

    BEGIN
	WITH M^ DO
	    WITH LocationInWindow DO
		DEC (r, firstrow);  DEC (c, firstcol);
	    END (*WITH*);
	    OnAnItem := c MOD (ColumnWidth+gap) < ColumnWidth;
	    c := c DIV (ColumnWidth+gap);
	    IF c >= NoOfColumns THEN OnAnItem := FALSE END(*IF*);

	    IF OnAnItem THEN
		(* We have now reduced (r,c) to be the coordinates of	*)
		(* an item in the visible part of the array, with (0,0)	*)
		(* corresponding to the top left position.		*)

		NewItemNo := c*hstep + (ExtraAtTop+r)*vstep + 1;

		(* The first click on an item simply means that we	*)
		(* should go to that item; a second click on the same	*)
		(* item means that we should accept it as the result.	*)

		IF NewItemNo = CurrentItemNo THEN
		    PutBack (" ");
		ELSE
		    HideMouseCursor;
		    GotoItem (M, NewItemNo);
		    ShowMouseCursor;
		END (*IF*);
	    END (*IF*);

	END (*WITH*);

    END SelectItemAt;

(************************************************************************)

PROCEDURE InterpretMouseClick (M: Menu);

    (* This procedure is called when we know that a mouse click has	*)
    (* been detected and its details stored in LastMouseClick.  This	*)
    (* procedure checks whether the click is relevant to menu M, and	*)
    (* takes the appropriate action if so.				*)

    VAR OK: BOOLEAN;  row: RowRange;  column: ColumnRange;

    BEGIN
	WITH LastMouseClick DO
	    Obtain (access);
	    OK := valid AND (win = M^.wname) AND NOT M^.SpecialMode;
	    IF OK THEN
		column := X;  row := Y;
	    END (*IF*);
	    valid := FALSE;
	    Release (access);
	END (*WITH*);
	IF OK THEN
	    SelectItemAt (M, row, column);
	END (*IF*);
    END InterpretMouseClick;

(************************************************************************)

PROCEDURE RecordClick (w: Window;  row: RowRange;  col: ColumnRange);

    (* This procedure is called asynchronously as the result of a mouse	*)
    (* click.  The parameters tell us which window was clicked on, and	*)
    (* where in that window the click occurred, but they don't tell us	*)
    (* which menu is involved.  Rather than work that out here, we	*)
    (* stuff a special character into the keyboard.  Procedure		*)
    (* MakeTheSelection will pick up that special character and from	*)
    (* that deduce that it needs to look at the LastMouseClick data.	*)

    BEGIN
	WITH LastMouseClick DO
	    Obtain (access);
	    win := w;
	    X := col;  Y := row;
	    valid := TRUE;
	    Release (access);
	    StuffKeyboardBuffer (ClickIndicator);
	END (*WITH*);
    END RecordClick;

(************************************************************************)

PROCEDURE ClickOnTop (w: Window;  row: RowRange;  col: ColumnRange);

    (* This procedure is triggered by a mouse click on the window	*)
    (* divider.  We turn this into a "cursor up" command.		*)

    BEGIN
	StuffKeyboardBuffer (CHR(0));
	StuffKeyboardBuffer ("H");
    END ClickOnTop;

(************************************************************************)

PROCEDURE ClickOnTopMORE (w: Window;  row: RowRange;  col: ColumnRange);

    (* This procedure is triggered by a mouse click on the top *MORE*	*)
    (* indicator.  We turn this into a "page up" command.		*)

    BEGIN
	StuffKeyboardBuffer (CHR(0));
	StuffKeyboardBuffer ("I");
    END ClickOnTopMORE;

(************************************************************************)

PROCEDURE ClickOnBottom (w: Window;  row: RowRange;  col: ColumnRange);

    (* This procedure is triggered by a mouse click on the bottom	*)
    (* of the window frame.  We turn this into a "cursor down" command.	*)

    BEGIN
	StuffKeyboardBuffer (CHR(0));
	StuffKeyboardBuffer ("P");
    END ClickOnBottom;

(************************************************************************)

PROCEDURE ClickOnBottomMORE (w: Window;  row: RowRange;  col: ColumnRange);

    (* This procedure is triggered by a mouse click on the bottom	*)
    (* *MORE* indicator.  We turn this into a "page down" command.	*)

    BEGIN
	StuffKeyboardBuffer (CHR(0));
	StuffKeyboardBuffer ("Q");
    END ClickOnBottomMORE;

(************************************************************************)
(*			    SCREEN DISPLAY				*)
(************************************************************************)

PROCEDURE OpeningDisplay (M: Menu;  initialvalue: CARDINAL);

    (* Sets up the initial state of the display of M.  It is assumed	*)
    (* that window M^.wname has already been opened and that the	*)
    (* position of the menu has already been set.			*)

    VAR j: RowRange;

    BEGIN
	IF initialvalue = 0 THEN
	    initialvalue := 1;
	END (*IF*);
	WITH M^ DO
	    RanOffEdge := FALSE;
	    CurrentItemNo := 1;  column := 1;  ExtraAtTop := 0;
	    ExtraAtBottom := ItemsPerColumn - LocationInWindow.height;
	    FOR j := 1 TO LocationInWindow.height DO
		row := j;  RefreshRow (M);
	    END (*FOR*);
	    row := 1;
	    IF (ExtraAtBottom > 0) AND NOT SpecialMode THEN
		IF MouseAvailable() THEN HideMouseCursor END (*IF*);
		DisplayMOREatBottom (M);
		IF MouseAvailable() THEN ShowMouseCursor END (*IF*);
	    END (*IF*);
	END (*WITH*);
	GotoItem (M, initialvalue);
    END OpeningDisplay;

(************************************************************************)

PROCEDURE DisplayMenu (w: Window;  M: Menu;
				rows, columns, initialvalue: CARDINAL);

    (* Displays menu M at the current cursor position in window w,	*)
    (* with initialvalue specifying a field to highlight.		*)
    (* Calling this procedure automatically enables special mode.	*)

    VAR row1, col1: CARDINAL;

    BEGIN
	SaveCursor (w, row1, col1);
	SetRelativeLocation (M, row1, col1, rows, columns);
	NewScrollingRegion (w, row1, row1+rows-1, col1, col1+columns-1);
	WITH M^ DO
	    wname := w;  SpecialMode := TRUE;
	END (*WITH*);
	OpeningDisplay (M, initialvalue);
	Highlight (M);  ResetScrollingRegion (w);
    END DisplayMenu;

(************************************************************************)
(*		     MAKING A SELECTION FROM A MENU			*)
(************************************************************************)

PROCEDURE MakeTheSelection (M: Menu);

    (* Allows the keyboard user to alter the state of menu M by use of	*)
    (* the cursor control keys, or by typing the initial letter of a	*)
    (* menu item.  Returns when Space or Enter is typed, also returns	*)
    (* with M^.CurrentItem=0 if Esc is typed.  In Special Mode only,	*)
    (* also returns if a cursor movement key would run us off the edge	*)
    (* of the menu.  (In Normal Mode, any attempt to run off the edge	*)
    (* is ignored.)  In addition, in Special Mode the final key typed	*)
    (* remains available (e.g. by InKey()) to the caller.		*)

    TYPE CHARSET = SET OF CHAR;

    CONST CR = CHR(0DH);

    VAR option: CHAR;

    BEGIN
	WITH M^ DO
	    RanOffEdge := FALSE;
	    LOOP
		Highlight (M);
		IF SpecialMode THEN BlinkCurrent (M) END (*IF*);
		option := InKey ();
		IF SpecialMode THEN BlinkCurrent (M) END(*IF*);
		Highlight (M);
		IF option = ClickIndicator THEN
		    InterpretMouseClick (M);
		ELSIF option = " " THEN EXIT(*LOOP*)
		ELSIF option = CR THEN EXIT (*LOOP*)
		ELSIF option = Esc THEN
		    CurrentItemNo := 0;  EXIT (*LOOP*)
		ELSIF option IN CHARSET{"0".."9", "A".."Z", "a".."z"} THEN
		    RepositionTo (M, CAP(option));
		ELSIF option = CHR(0) THEN
		    HandleFunctionKey (M, option);
		    IF SpecialMode AND RanOffEdge THEN
			PutBack (option);  option := CHR(0);
			EXIT (*LOOP*)
		    END (*IF*);
		END (*IF*);
	    END (*LOOP*);
	    IF SpecialMode THEN
		PutBack (option);
	    END (*IF*);
	    IF CurrentItemNo > NoOfItems THEN
		CurrentItemNo := 0;
	    END (*IF*);
	END (*WITH*);
    END MakeTheSelection;

(************************************************************************)

PROCEDURE SelectFromMenu (M: Menu): CARDINAL;

    (* Displays menu M on the screen, allows terminal user to use	*)
    (* cursor keys to move about the menu and the ENTER key to select	*)
    (* an item.  (The space bar is also accepted, as an alternative to	*)
    (* the ENTER key, to select an item).  An item may also be selected	*)
    (* by typing its initial letter, followed by space or ENTER.	*)
    (* Returns the number of the item which was selected.		*)
    (* (Item numbers start from 1).  An answer of 0 indicates that the	*)
    (* user typed the ESC key to return without selecting anything.	*)

    (* Remark: it is possible with the cursor keys to "select" a	*)
    (* nonexistent item at the bottom of the last column.  The result	*)
    (* will be 0 in this case.  Although this might appear to be a bug,	*)
    (* it is deliberate.  I found by experiment that the more "logical"	*)
    (* approach of stopping the user from moving the cursor into a	*)
    (* blank region was a nuisance for the user.			*)

    VAR UIW: UIWindow;

    BEGIN
	WITH M^ DO
	    IF SpecialMode THEN
		WITH LocationInWindow DO
		    NewScrollingRegion (wname, firstrow, firstrow+height-1,
						firstcol, firstcol+width-1);
		END (*WITH*);
		Highlight (M);
	    ELSE
		IF MouseAvailable() THEN HideMouseCursor END(*IF*);
		WITH ScreenPosition DO
		    OpenWindow (wname, foreground, background,
				firstrow, lastrow, firstcol, lastcol,
				simpleframe, doubledivider);
		    WriteString (wname, heading);
		    ChangeScrollingRegion (wname, 3, lastrow-firstrow-1);
		END (*WITH*);
		IF MouseAvailable() THEN
		    UIW := AllowMouseControl (wname, heading,
				CapabilitySet {wshow, wmove, wescape});
		    WITH LocationInWindow DO
			AddActiveRegion (UIW, firstrow, firstrow+height-1,
				firstcol, firstcol+width-1,
				LeftOnly, RecordClick);
			AddActiveRegion (UIW, firstrow-1, firstrow-1,
				firstcol, firstcol+width-7,
				LeftOnly, ClickOnTop);
			AddActiveRegion (UIW, firstrow-1, firstrow-1,
				firstcol+width-6, firstcol+width-1,
				LeftOnly, ClickOnTopMORE);
			AddActiveRegion (UIW, firstrow+height, firstrow+height,
				firstcol, firstcol+width-7,
				LeftOnly, ClickOnBottom);
			AddActiveRegion (UIW, firstrow+height, firstrow+height,
				firstcol+width-6, firstcol+width-1,
				LeftOnly, ClickOnBottomMORE);
		    END (*WITH*);
		    ShowMouseCursor;
		END (*IF*);
		OpeningDisplay (M, 1);
	    END (*IF*);
	    MakeTheSelection (M);
	    IF SpecialMode THEN
		Highlight (M);  ResetScrollingRegion (wname);
	    ELSE
		IF MouseAvailable() THEN HideMouseCursor END(*IF*);
		CloseWindow (wname);
		IF MouseAvailable() THEN ShowMouseCursor END(*IF*);
	    END (*IF*);
	    RETURN CurrentItemNo;
	END (*WITH*);
    END SelectFromMenu;

(************************************************************************)

BEGIN
    WITH LastMouseClick DO
        CreateLock (access);
        valid := FALSE;
    END (*WITH*);
END Menus.

