-- NAME: sfx2.e
-- OBJECT: playing wave file on sound blaster card.
-- BY Jacques Deschenes, Baie-Comeau, P.Q., Canada, e-mail:desja@quebectel.com
--    I borrowed some code from Greg Harris for sound card detection.
-- creation date: April 7th, 1997
-- version 2.0
-- revision hitory
--   may 12th, 1997
--     SB_INT was not initialise in DetectCard()
--   may 13th, 1997
--     corrected a bug in TstInt() function.
--     improper masks were used to disable interrupt.
--   may 14th, 1997
--     correction to DetectDma16() forgotten to divided block size by 2 
--     reworked TstInt()  was disrupting network
--     ** problem doesn't seem to be solved yet.
--   June 14th, 1997
--     corrected a bug in GetBlasterInfo() reported by Robert Evans

-- global constants
--      ERROR_MESSAGES   sequence containing error messages indexed by
--                       iSbCardError
--
--      constant indexing ERROR_MESSAGES
--      ERROR_RESET_DSP = 1,         -- dsp couldn't be reset
--      ERROR_WRITE_DSP = 2,         -- dsp write failed 
--      ERROR_READ_DSP = 3,          -- dsp read failed
--      ERROR_NO_BLASTER = 4,        -- environment variable BLASTER not found
--      ERROR_ENV_BAD =5,            -- environment variable BLASTER bad.
--      ERROR_MEM_ALLOC = 6,         -- error allocating memory 
--      ERROR_NOT_INITIALISED = 7,   -- CloseSfx() as been called
--      ERROR_CARD_NOT_FOUND = 8,    -- Sound card not detected.  
--      ERROR_CARD_NOT_SUPPORTED = 9 -- Sound card version prior to 2.0 not
--                                      supported
--  constants for SetVolumes
--      VOICE_VOLUME = 4             -- parameter for SetVolume()
--      MASTER_VOLUME = #22          -- parameter for SetVolume() 
--      VOL_OFF = 0                  -- parameter for SetVolume() 
--      VOL_MAX = 7                  -- parameter for SetVolume()
--
--  index to access Wave info sequence
--  siFMT_ID = 1,    -- format identifier
--  siCHANNELS = 2,  -- number of channels
--  siSPS = 3,       -- samples per seconds 
--  siBITS_PS = 4,   --  bits per sample 8 or 16
--  siDATA_LEN = 5,  -- data length
--  siDATA_OFS = 6,  -- data offset in file
--  siDATA_BUFF = 7  -- data buffer address



-- global variables
--    integer iSbCardError  -- should be zero if there is no error
--    atom DSPVer           -- DSP version number 

-- global functions
--     SoundDone(integer index)
--     PlaySound(sequence SoundInfo, integer loop)
--     LoadWaveFile(sequence FileName)

-- global procedures
--     SetVolume(integer WhichOne, byte volume)
--     FreeSoundBuffer(sequence SoundInfo)
--     PlayWaveFile(sequence FileName, integer volume)
--     StopSound(integer index)
--     CloseSfx()


include ports.e
--include delay.e
include dma.e
include doswrap.e

-- with trace
without warning
without type_check

--***************** DEBUG CODE *****************************************
integer detect
detect = 0   -- 0 mean do not auto detect sound card
procedure WriteLogMsg(sequence FileName, sequence msg)
integer fh
   fh = open(FileName,"a")
   if fh = -1 then
     return
   end if
   puts(fh,msg&'\n')
   close(fh)
end procedure

integer debug
debug = 0
constant debug_file = "SFX2.LOG"
integer fh 
if debug then
    fh = open(debug_file,"w")
    if fh = -1 then
	debug = 0
    else
	puts(fh,"This file is produce by SFX2 debug version.\n\n")
	close(fh)
    end if
end if
--***************** END OF DEBUG CODE **********************************

constant TRUE = 1, FALSE = 0

word BaseIo  -- sound card base io address
byte IRQ      -- Hardware interrupt number used by the sound card
byte SB_INT -- sound card software interrupt
byte DMA8      -- 8 bit DMA channel number
byte DMA16     -- 16 bit DMA channel number
integer DmaChannel  -- DmaChannel currently used by sound card

constant cBLOCK_SIZE = 512,  -- dsp data block size
	 cBUFFER_SIZE = 2*cBLOCK_SIZE  -- dma buffer size

atom DmaBuffer,  -- dma buffer address
     MixerBuffer -- sound mixer buffer address

sequence OrigVector -- original interrupt vector

DMA16 = 0
DMA8 = 0

word DSP_RESET,        -- dsp reset i/o port
     DSP_READ_DATA,    -- dsp read data i/o port
     DSP_WRITE_DATA ,  -- dsp write data i/o port
     DSP_WRITE_STATUS, -- dsp write status i/o port
     DSP_DATA_AVAIL    -- dsp read status i/o port

constant MONO = 1, STEREO = 2
     
constant DSP_DATA_READY = #AA -- byte returned by dsp when data is avaible
    
global integer iSbCardError --ResetDSP(), WriteDSP and ReadDSP()
	-- set this variable in case of error due to sound card.
	-- this error code is an index for the following ERROR_MESSAGES

iSbCardError = 0  -- no error
	 
global constant ERROR_MESSAGES={
    "Failed to reset DSP.\n",
    "Failed to write to DSP.\n",
    "Failed to read from DSP.\n",
    "\"BLASTER\" environment variable not found.\n",
    "Bad syntax in \"BLASTER\" environment variable.\n",
    "Not enough memory.\n",
    "sfx2.e not initialised.\n",
    "Sound card not detected.\n",
    "Sound card not supported.\n"
    } -- ERROR_MESSAGES

global constant 
    ERROR_RESET_DSP = 1,         -- dsp couldn't be reset
    ERROR_WRITE_DSP = 2,         -- dsp write failed 
    ERROR_READ_DSP = 3,          -- dsp read failed
    ERROR_NO_BLASTER = 4,        -- environment variable BLASTER not found
    ERROR_ENV_BAD =5,            -- environment variable BLASTER bad.
    ERROR_MEM_ALLOC = 6,         -- error allocating memory 
    ERROR_NOT_INITIALISED = 7,   -- CloseSfx() as been called
    ERROR_CARD_NOT_FOUND = 8,    -- Sound card not detected.  
    ERROR_CARD_NOT_SUPPORTED = 9 -- Sound card version prior to 2.0 not support

global atom DSPVer  -- DSP version number 


-- some DSP commands
constant 
     SET_BLOCK_SIZE = #48,  -- set data block size
     PAUSE_DMA = #D0,       -- 8 bits dma pause
     SET_TIME_CONSTANT = #40, -- set dsp sample rate
     SPEAKER_ON = #D1,        -- turn on dsp speaker
     SPEAKER_OFF = #D3,       -- turn off dsp speaker
     DMA_CONT = #D4,          -- continu dma transfert
     EXIT_AUTO_INIT = #DA,    -- exit auto 8 bits auto init mode.
     GET_DSP_VERSION = #E1,   
     SET_SINGLE_CYCLE = #14,  -- initialise 8 bits single cycle mode.
     SET_AUTO_INIT = #1C,     -- initialise 8 bits auto init mode.
     INVOKE_INT = #F2,        -- make dsp to trigger an interrupt( not doc.)
     HSPEED_SINGLE = #91,     -- initialise an 8 bits high speed single cycle
     HSPEED_AUTO_INIT = #90   -- initialise an 8 bits high speed auto init.

constant TIME_OUT = .05 --time out for DSP ACCESS (abort after .05 seconds)

-----------------------------------------------------------------------
-- code borrowed from hardint.ex demo
atom segment, code_segment

segment = allocate(4)
lock_memory(segment, 4)

sequence save_segment_code
save_segment_code = {
    #53,   -- push ebx
    #0E,   -- push cs   or #1E push ds -- only a 16-bit value
    #5B,   -- pop ebx  
    #89, #1D} & int_to_bytes(segment) & -- mov segment, ebx
    {#5B,   -- pop ebx
    #C3    -- ret
}       
atom save_segment
save_segment = allocate(length(save_segment_code))
poke(save_segment, save_segment_code)
call(save_segment) -- save code segment

code_segment = bytes_to_int(peek({segment,4})) 

poke(save_segment+1, #1E) 
call(save_segment) -- save data segment

----------------------
-- constant to access fields in sound record
constant  iDATA_ADDR = 0, iDATA_LEN = 4, iPOS_IDX = 8, 
	  iSND_LOOP = #C, iSND_PLAY = #E, REC_LEN = 16

constant MAX_SOUNDS = 4  -- maximum number of sounds that can be mixed together
	  
-- interrupt service routine

constant DATA_LEN = 112 -- length of allocated data segment used by interrupt.
atom DSP_ISR,       -- address of interrupt service routine
     DATA_SEG,      -- address of data segment
     IRQ_NO,        -- sound card irq number
     BASE_IO,       -- sound card base i/o address
     HALF_FF,       -- switch to indicate lower/upper half of dma buffer.
     DMA_BUFF_ADDR, -- address of dma buffer 
     BLOCK_SIZE,    -- size of data block used by dsp 
     MX_BUFF_ADDR,  -- sound mixer buffer address
     NB_SOUNDS,     -- number of sounds currently playing.
     TMP_NB_SNDS,   -- temporary variable used by isr.
     TMP_COUNT,     -- temporary variable used by isr.
     MAX_COUNT,     -- maximum bytes added in mixer buffer.
     UPPER_LIMIT,   -- upper limit of current data buffer. 
     INT_COUNT,     -- incremented at each interrupt.
     SOUNDS         -- sounds record array. 

DATA_SEG = allocate(DATA_LEN)
lock_memory(DATA_SEG,DATA_LEN)

IRQ_NO = DATA_SEG + 0
BASE_IO = DATA_SEG + 1
HALF_FF = DATA_SEG + 3
DMA_BUFF_ADDR = DATA_SEG + 4
BLOCK_SIZE = DATA_SEG + 8
MX_BUFF_ADDR = DATA_SEG + 12
NB_SOUNDS = DATA_SEG + 16
TMP_NB_SNDS = DATA_SEG + 17
TMP_COUNT = DATA_SEG + 18
MAX_COUNT = DATA_SEG + 22
UPPER_LIMIT = DATA_SEG + 26
INT_COUNT = DATA_SEG + 30
SOUNDS = DATA_SEG + 34

constant sDSP_ISR = {
  --PROC DSP_INTERRUPT
  #1E,#06,                              -- PUSH DS ES            
  #60,                                  -- PUSHAD                
  #BB}&peek({segment,4})&{              -- MOV  EBX, DATA_SEG    
  #53,                                  -- PUSH EBX              
  #1F,                                  -- POP  DS               
  #53,                                  -- PUSH EBX              
  #07,                                  -- POP  ES               
  #66,#8B,#15}&int_to_bytes(BASE_IO)&{  -- MOV  DX, [BASE_IO]
  #66,#83,#C2,#0E,                      -- ADD  DX, 0E           
  #EC,                                  -- IN AL, DX             
  #66,#BA,#20,#00,                      -- MOV  DX, 20           
  #B0,#20,                              -- MOV  AL, 20           
  #EE,                                  -- OUT  DX, AL           
  #80,#3D}&int_to_bytes(IRQ_NO)&{#07,   -- CMP  [IRQ_NO], 7
  #76,#09,#90,#90,#90,#90,              -- JBE  @MIX_SND
  #66,#BA,#A0,#00,                      -- MOV  DX, 0A0          
  #EE,                                  -- OUT  DX, AL           
				      -- @MIX_SND:
  #E8,#0A,#00,#00,#00,                  -- CALL MIX_SOUNDS
				      -- @EXIT:
  #FF,#05}&int_to_bytes(INT_COUNT)&{    -- INC  [INT_COUNT]
  #61,                                  -- POPAD
  #07,#1F,                              -- POP  ES DS
  #CF,                                  -- IRETD
  --PROC MIX_SOUNDS
  #FC,                                  -- CLD
  #E8,#45,#00,#00,#00,                  -- CALL INIT_MIXER
  #C7,#05}&int_to_bytes(MAX_COUNT)&{#00,#00,#00,#00,-- MOV [MAX_COUNT], 0
  #C6,#05}&int_to_bytes(TMP_NB_SNDS)&{#00,-- MOV [TMP_NB_SNDS], 0
  #BB,#FF,#FF,#FF,#FF,                  -- MOV EBX, -1
				      -- @MIX_NEXT:
  #43,                                  -- INC EBX
  #83,#FB,#04,                          -- CMP EBX, MAX_SOUNDS
  #74,#23,#90,#90,#90,#90,              -- JE @DONE
  #8B,#D3,                              -- MOV EDX, EBX
  #C1,#E2,#04,                          -- SHL EDX, 4             
  #81,#C2}&int_to_bytes(SOUNDS)&{       -- ADD EDX, OFFSET SOUNDS
  #66,#83,#7A,#0E,#01,                  -- CMP [EDX + SOUND.SND_PLAY], 1
  #75,#E4,                              -- JNE @MIX_NEXT
  #FE,#05}&int_to_bytes(TMP_NB_SNDS)&{  -- INC [TMP_NB_SNDS]
  #E8,#1C,#00,#00,#00,                  -- CALL ADD_SOUND
  #EB,#D7,                              -- JMP @MIX_NEXT
				      -- @DONE:
  #E8,#C9,#00,#00,#00,                  -- CALL COPY_MIXER
  #C3,                                  -- RET
  --PROC INIT_MIXER
  #8B,#3D}&int_to_bytes(MX_BUFF_ADDR)&{ -- MOV EDI, [MX_BUFF_ADDR]
  #8B,#0D}&int_to_bytes(BLOCK_SIZE)&{   -- MOV ECX, [BLOCK_SIZE]
  #66,#B8,#00,#00,                      -- MOV AX, 0
  #F3,#66,#AB,                          -- REP STOSW
  #C3,                                  -- RET
  --PROC ADD_SOUND
  #8B,#3D}&int_to_bytes(MX_BUFF_ADDR)&{ -- MOV EDI, [MX_BUFF_ADDR]
  #8B,#72,#08,                          -- MOV ESI, [EDX+SOUND.POS_IDX]
  #8B,#0A,                              -- MOV ECX, [EDX+SOUND.DATA_ADDR]
  #03,#4A,#04,                          -- ADD ECX, [EDX+SOUND.DATA_LEN]
  #89,#0D}&int_to_bytes(UPPER_LIMIT)&{  -- MOV [UPPER_LIMIT], ECX
  #2B,#CE,                              -- SUB ECX, ESI
  #3B,#0D}&int_to_bytes(BLOCK_SIZE)&{   -- CMP ECX, [BLOCK_SIZE]
  #76,#0A,#90,#90,#90,#90,              -- JBE @ADD_1
  #8B,#0D}&int_to_bytes(BLOCK_SIZE)&{   -- MOV ECX, [BLOCK_SIZE]
				      -- @ADD_1:
  #89,#0D}&int_to_bytes(TMP_COUNT)&{    -- MOV [TMP_COUNT], ECX
  #0B,#C9,                              -- OR ECX, ECX
  #74,#11,#90,#90,#90,#90,              -- JE @ADD_3      
				      -- @ADD_2:
  #AC,                                  -- LODSB
  #66,#98,                              -- CBW
  #66,#03,#07,                          -- ADD AX, [EDI]
  #66,#AB,                              -- STOSW
  #E2,#F6,                              -- LOOP @ADD_2
  #89,#72,#08,                          -- MOV [EDX+SOUND.POS_IDX], ESI
				      -- @ADD_3:
  #A1}&int_to_bytes(TMP_COUNT)&{        -- MOV EAX, [TMP_COUNT]
  #3B,#35}&int_to_bytes(UPPER_LIMIT)&{  -- CMP ESI, [UPPER_LIMIT]      
  #72,#52,#90,#90,#90,#90,              -- JB @ADD_5
  #66,#83,#7A,#0C,#01,                  -- CMP [EDX+SOUND.SND_LOOP],1  
  #74,#15,#90,#90,#90,#90,              -- JE  @RST_POS                
  #66,#C7,#42,#0E,#00,#00,              -- MOV [EDX+SOUND.SND_PLAY],0  
  #FE,#0D}&int_to_bytes(NB_SOUNDS)&{    -- DEC [NB_SOUNDS]             
  #EB,#35,#90,#90,#90,                  -- JMP @ADD_5
				      -- @RST_POS:
  #8B,#02,                              -- MOV EAX, [EDX+SOUND.DATA_ADDR]
  #8B,#F0,                              -- MOV ESI, EAX
  #89,#42,#08,                          -- MOV [EDX+SOUND.POS_IDX], EAX   
  #A1}&int_to_bytes(TMP_COUNT)&{        -- MOV EAX, [TMP_COUNT]
  #3B,#05}&int_to_bytes(BLOCK_SIZE)&{   -- CMP EAX, [BLOCK_SIZE]          
  #74,#1E,#90,#90,#90,#90,              -- JE @ADD_5
  #8B,#0D}&int_to_bytes(BLOCK_SIZE)&{   -- MOV ECX, [BLOCK_SIZE]          
  #2B,#C8,                              -- SUB ECX, EAX
				      -- @ADD_4:
  #AC,                                  -- LODSB
  #66,#98,                              -- CBW
  #66,#03,#07,                          -- ADD AX, [EDI]
  #66,#AB,                              -- STOSW
  #E2,#F6,                              -- LOOP @ADD_4
  #89,#72,#08,                          -- MOV [EDX+SOUND.POS_IDX], ESI
  #A1}&int_to_bytes(BLOCK_SIZE)&{       -- MOV EAX, [BLOCK_SIZE]
				      -- @ADD_5:
  #3B,#05}&int_to_bytes(MAX_COUNT)&{    -- CMP EAX, [MAX_COUNT]           
  #76,#09,#90,#90,#90,#90,              -- JBE @EXIT_ADD
  #A3}&int_to_bytes(MAX_COUNT)&{        -- MOV [MAX_COUNT], EAX           
				      -- @EXIT_ADD:
  #C3,                                  -- RET
  --PROC COPY_MIXER
  #8B,#35}&int_to_bytes(MX_BUFF_ADDR)&{ -- MOV ESI, [MX_BUFF_ADDR]
  #8B,#3D}&int_to_bytes(DMA_BUFF_ADDR)&{-- MOV EDI, [DMA_BUFF_ADDR]
  #B0,#01,                              -- MOV AL, 1
  #2A,#05}&int_to_bytes(HALF_FF)&{      -- SUB AL, [HALF_FF]
  #A2}&int_to_bytes(HALF_FF)&{          -- MOV [HALF_FF], AL
  #75,#0A,#90,#90,#90,#90,              -- JNE @LOWER
  #03,#3D}&int_to_bytes(BLOCK_SIZE)&{   -- ADD EDI, [BLOCK_SIZE]
				      -- @LOWER:
  #8A,#3D}&int_to_bytes(TMP_NB_SNDS)&{  -- MOV BH, [TMP_NB_SNDS]
  #0A,#FF,                              -- OR BH, BH
  #8B,#0D}&int_to_bytes(BLOCK_SIZE)&{   -- MOV ECX, [BLOCK_SIZE]
  #74,#46,#90,#90,#90,#90,              -- JE @SILENCE_FILL
  #8B,#0D}&int_to_bytes(MAX_COUNT)&{    -- MOV ECX, [MAX_COUNT]
				      -- @COPY_LOOP:
  #66,#AD,                              -- LODSW
  #66,#3D,#7F,#00,                      -- CMP AX, 07F
  #7E,#0B,#90,#90,#90,#90,              -- JLE @COPY_2
  #B0,#7F,                              -- MOV AL, 07F
  #EB,#0F,#90,#90,#90,                  -- JMP @COPY_3
				      -- @COPY_2:
  #66,#3D,#80,#FF,                      -- CMP AX, -080
  #7D,#06,#90,#90,#90,#90,              -- JGE @COPY_3
  #B0,#80,                              -- MOV AL, -080
				      -- @COPY_3:
  #04,#80,                              -- ADD AL, 80
  #AA,                                  -- STOSB
  #E2,#DC,                              -- LOOP @COPY_LOOP
  #8B,#0D}&int_to_bytes(BLOCK_SIZE)&{   -- MOV ECX, [BLOCK_SIZE]
  #3B,#0D}&int_to_bytes(MAX_COUNT)&{    -- CMP ECX, [MAX_COUNT]
  #74,#0E,#90,#90,#90,#90,              -- JE @EXIT_COPY
  #2B,#0D}&int_to_bytes(MAX_COUNT)&{    -- SUB ECX, [MAX_COUNT]
				      -- @SILENCE_FILL:
  #B0,#80,                              -- MOV AL, 080    
  #F3,#AA,                              -- REP STOSB
				      -- @EXIT_COPY:
  #C3,                                  -- RET
  0
}
DSP_ISR = allocate(length(sDSP_ISR))
lock_memory(DSP_ISR,length(sDSP_ISR))
poke(DSP_ISR,sDSP_ISR)

-----------------------------------------------------------------------

global procedure delay(atom duration)
atom start
    start =time()
    while time() - start < duration do
    end while
end procedure

procedure EnableIRQ(integer IrqNumber)
  if IrqNumber < 8 then 
    Output(and_bits(Input(#21),not_bits(power(2,IrqNumber))),#21)
  else       
    Output(and_bits(Input(#A1),not_bits(power(2,IrqNumber-8))),#A1)
  end if
end procedure --EnableIRQ()

procedure DisableIRQ(integer IrqNumber)
  if IrqNumber < 8 then 
    Output(or_bits(Input(#21),power(2,IrqNumber)),#21)
  else       
    Output(or_bits(Input(#A1),power(2,IrqNumber-8)),#A1)
  end if
end procedure --DisableIRQ()

procedure ResetDSP()
-- set iSbCardError if unable to reset dsp.
atom start, TimeOut  
  Output(1,DSP_RESET) 
  delay(.001) -- problably not needed.  DSP chip ask for a 3 micro second delay
  Output(0,DSP_RESET) 
  start = time()
  TimeOut = 0
  while not TimeOut and floor(Input(DSP_DATA_AVAIL)/#80)=0 do 
    TimeOut = time()-start >= TIME_OUT
  end while 
  if TimeOut then
    iSbCardError = ERROR_RESET_DSP
    if debug then
	WriteLogMsg(debug_file,ERROR_MESSAGES[iSbCardError])
	abort(1)
    end if
  end if
  if Input(DSP_READ_DATA) = DSP_DATA_READY then
	 iSbCardError = 0 -- reset success.
     else
	 iSbCardError = ERROR_RESET_DSP -- fail to reset dsp.
	 if debug then
	  WriteLogMsg(debug_file,ERROR_MESSAGES[iSbCardError])
	  abort(1)
	 end if
  end if
end procedure --ResetDSP()

procedure WriteDSP(byte value) 
-- write a byte to dsp write register
atom start integer TimeOut
  if iSbCardError then
     return
  end if
  start = time()
  TimeOut = 0
  while not TimeOut and floor(Input(DSP_WRITE_STATUS)/#80) = 1 do 
    TimeOut = time() - start > TIME_OUT
  end while
  if TimeOut then
    iSbCardError = ERROR_WRITE_DSP -- failed to write to dsp.
    if debug then
	WriteLogMsg(debug_file,ERROR_MESSAGES[iSbCardError])
	abort(1)
    end if
    return
  end if
  Output(value,DSP_WRITE_DATA)
end  procedure --WriteDSP()

function ReadDSP() 
-- read a byte from dsp read register
atom start 
integer TimeOut
  if iSbCardError then
     return -1
  end if
  TimeOut = 0
  start = time()
  while not TimeOut and floor(Input(DSP_DATA_AVAIL)/#80) = 0 do 
    TimeOut = time() - start > TIME_OUT
  end while
  if TimeOut then
    iSbCardError = ERROR_READ_DSP -- failed to read dsp data.
    if debug then
	WriteLogMsg(debug_file,ERROR_MESSAGES[iSbCardError])
	abort(1)
    end if
    return -1
  end if
  return Input(DSP_READ_DATA)
end function -- ReadDSP()

function GetDSPVersion()
-- return the Digital sound processor version number.
integer major, minor
  WriteDSP(GET_DSP_VERSION)
  major = ReadDSP()
  minor = ReadDSP()
  return major+ minor/100
end function -- GetDSPVersion

-- MIXER REGISTERS
global constant VOICE_VOLUME = 4  -- volume levels(0..7)
global constant MASTER_VOLUME = #22  --(0..7)
constant MIXER_ADDR = #4  -- write to that port to select internal reg.
constant MIXER_DATA = #5   -- read or  write here data to (from) registers.
constant FILTER_STEREO = #E  -- outpout filter and mono/stereo register
constant MIXER_RESET = 0  -- write any byte to that register to reset mixer.
constant FILTER_SOURCE = #C -- program low pass input filter and input source.

-- volume min and max value
global constant VOL_OFF = 0,
		VOL_MAX = 7

global procedure SetVolume(integer WhichOne, byte volume)
-- Set the 6 different volume controls of CT1345 mixer, part of sbpro card.
-- volume is limited to 8 levels 0..7  ( 4 levels for MIC_VOLUME 0..3)
-- for WhichOne use one of the _VOLUME constant above 
-- Except for MIC_VOLUME, there is 2 volume left and right
-- for right volume number is multiplied by 2 and for left by 32
-- so byte to write to mixer = volume * 2 + volume * 32
byte  r,l
    volume = remainder(volume,8) -- insure it's inside bound
    r = volume*2  -- right volume
    l = volume * 32 -- left volume
    Output(WhichOne,BaseIo + MIXER_ADDR) -- select specific volume register
    Output(r+l,BaseIo + MIXER_DATA)    -- write volume level to register.
end procedure -- SetVolume()

function HexToI(sequence HStr)
-- convert an hexadecimal string to an integer
-- always return a number if HStr is not valid return 0
-- Only upper case letters are recognise
integer i
    i = 0
    for  n = 1 to length(HStr) do
	if HStr[n] >= '0' and HStr[n] <= '9' then
	    i = i * 16 + HStr[n] - '0'
	elsif HStr[n] >= 'A' and HStr[n] <= 'Z' then
	    i = i * 16 + HStr[n] - 'A' + 10
	else exit
	end if
    end for
    return i
end function -- HexToI()

function AToI(sequence NbStr)
-- convert a decimal string to it's integer value
integer i
    i = 0
    for n = 1 to length(NbStr) do
	if NbStr[n] >= '0' and NbStr[n] <= '9' then
	    i = i * 10 + NbStr[n] - '0'
	else
	    exit
	end if
    end for
    return i
end function -- AToI()

function upper(sequence Str)
-- convert to upper case
sequence UpStr
    UpStr = ""
    for i = 1 to length(Str) do
	if Str[i] >= 'a' and Str[i] <='z' then
	    UpStr = UpStr & 'A' + (Str[i] - 'a')
	else
	    UpStr = UpStr & Str[i]
	end if
    end for
    return UpStr
end function -- upper()

-- ****************** sound card detection code *****************

constant TST_DATA_LEN = 16
atom INT_CODE, TST_DATA_SEG,
     dBASE_IO,
     dDMA_COUNT,
     dIRQ_NO,
     dINT_COUNT

TST_DATA_SEG = allocate(TST_DATA_LEN)
lock_memory(TST_DATA_SEG,TST_DATA_LEN)

dBASE_IO = TST_DATA_SEG + 0
dDMA_COUNT = TST_DATA_SEG + 2
dIRQ_NO = TST_DATA_SEG + 4
dINT_COUNT = TST_DATA_SEG + 5

constant sINT_CODE = {
  #1E,                                  -- PUSH DS               
  #60,                                  -- PUSHAD                
  #BB}&peek({segment,4})&{              -- MOV  EBX, DATA_SEG    
  #53,                                  -- PUSH EBX              
  #1F,                                  -- POP  DS               
  #66,#BA,#0C,#00,                      -- MOV  DX,  0C          
  #32,#C0,                              -- XOR  AL, AL
  #EE,                                  -- OUT  DX, AL           
  #66,#8B,#15}&int_to_bytes(dDMA_COUNT)&{-- MOV  DX, [dma_count]     
  #EC,                                  -- IN AL, DX             
  #86,#E0,                              -- XCHG AH,AL            
  #EC,                                  -- IN AL, DX             
  #66,#3D,#FF,#FF,                      -- CMP   AX, 0FFFF       
  #75,#30,#90,#90,#90,#90,              -- JNE  @exit
  #66,#8B,#15}&int_to_bytes(dBASE_IO)&{  -- MOV  DX, [BASE_IO]
  #66,#83,#C2,#0E,                      -- ADD  DX, 0E           
  #EC,                                  -- IN AL, DX             
  #66,#BA,#20,#00,                      -- MOV  DX, 20           
  #B0,#20,                              -- MOV  AL, 20           
  #EE,                                  -- OUT  DX, AL           
  #80,#3D}&int_to_bytes(dIRQ_NO)&{#07,   -- CMP  [IRQ_NO], 7
  #76,#10,#90,#90,#90,#90,              -- JBE  @exit
  #66,#BA,#A0,#00,                      -- MOV  DX, 0A0          
  #EE,                                  -- OUT  DX, AL           
  #66,#FF,#05}&int_to_bytes(dINT_COUNT)&{-- INC  [INT_COUNT]
				      -- @exit:
  #61,                                  -- POPAD
  #1F,                                  -- POP  DS
  #EA,#00,#00,#00,#00,#00,#00           -- JMP ORIG_HANDLER
}

INT_CODE = allocate(length(sINT_CODE))
lock_memory(INT_CODE,length(sINT_CODE))
poke(INT_CODE,sINT_CODE)

constant  DMA_COUNT_REG={1,3,5,7}

function TstInt(integer IrqNo)
sequence OrigVector
integer IntNo, Master, Slave
atom start, DmaBuff
    poke(dIRQ_NO,IrqNo)
    poke(dDMA_COUNT,{DMA_COUNT_REG[DMA8+1],0})
    poke(dINT_COUNT,{0,0,0,0})
    if IrqNo < 8 then
	IntNo = IrqNo + 8
    else
	IntNo = IrqNo - 8 + #70
    end if
    DmaBuff = AllocateDMABuffer(32)
    mem_set(DmaBuff,128,32)
    WriteDSP(#40)  -- set dsp time constant
    WriteDSP(211)  -- time constant
    Master = Input(#21)  -- get master PIC interrupt mask
    Slave = Input(#A1)   -- get slave PIC interrupt mask
    OrigVector = get_vector(IntNo)
    poke(INT_CODE + length(sINT_CODE) - 6,int_to_bytes(OrigVector[2])&
	 remainder(OrigVector[1],256)&floor(OrigVector[1]/256))
    set_vector(IntNo,{code_segment,INT_CODE})
    EnableIRQ(IrqNo)
    WriteLogMsg(debug_file,sprintf("Testing irq No. %d",{IrqNo}))
    SetDma(DmaBuff,32,DMA8,DMA_OUT+DMA_SINGLE)
    WriteDSP(#14)  -- set dsp mode for single cycle dma transfert
    WriteDSP(remainder(32-1,256)) -- write block length low byte
    WriteDSP(floor((32-1)/256))   -- high byte
    start = time()
    while time() - start < .05 do
    end while
    ResetDSP()
    free_low(DmaBuff)
    ResetDma(DMA8)
    set_vector(IntNo,OrigVector)
    Output(Master,#21)
    Output(Slave,#A1)
    if bytes_to_int(peek({dINT_COUNT,4})) > 0 then
	return 1
    else
       return 0
    end if
end function  -- TstInt()

constant SB_INT_NO= {2,3,5,7,10}
function DetectIRQ()
integer DspIrq
  DspIrq = -1
  poke(dBASE_IO,{remainder(BaseIo,256),floor(BaseIo/256)})
  for i = 1 to length(SB_INT_NO)  do
    if TstInt(SB_INT_NO[i]) then
	DspIrq = SB_INT_NO[i]
	if debug then
	  WriteLogMsg(debug_file,sprintf("Found IRQ to be %d",{DspIrq}))
	end if
	exit
    end if
  end for
  return DspIrq
end function

constant DMA_8_CHANNELS = {0,1,3}
constant DMA_16_CHANNELS = {5,6,7}

function DetectDma8()
atom DmaBuff, DmaCount, start
integer Dma8

    DmaBuff = AllocateDMABuffer(32)
    mem_set(DmaBuff,128,32)
    WriteDSP(#40)
    WriteDSP(211)
    for i = 1 to length(DMA_8_CHANNELS) do
      SetDma(DmaBuff,32,DMA_8_CHANNELS[i],DMA_OUT+DMA_SINGLE)
      WriteDSP(#14)
      WriteDSP(remainder(32-1,256))
      WriteDSP(floor((32-1)/256))
      start = time()
      while time() - start < .05 do
      end while
      DmaCount = ReadCurrentCount(DMA_8_CHANNELS[i])
      if DmaCount = 65535 then
	 free_low(DmaBuff)
	 Dma8 = DMA_8_CHANNELS[i]
	 if debug then
	   WriteLogMsg(debug_file,sprintf("Found DMA8 to be %d",{Dma8}))
	 end if
	 return Dma8
      else
	 ResetDSP()
	 ResetDma(DMA_8_CHANNELS[i])
      end if
    end for
    free_low(DmaBuff)
    return -1
end function

function DetectDma16()
atom DmaBuff, DmaCount, start
integer Dma16
    
    DmaBuff = AllocateDMABuffer(32)
    mem_set(DmaBuff,0,32)
    WriteDSP(#41) -- set sampling rate
    WriteDSP(floor(22050/256))
    WriteDSP(remainder(22050,256))
    for i = 1 to length(DMA_16_CHANNELS) do
      SetDma(DmaBuff,32,DMA_16_CHANNELS[i],DMA_OUT+DMA_SINGLE)
      WriteDSP(#B0) -- set 16 bits output
      WriteDSP(#10) -- set 16 bits mono signed mode
      WriteDSP(remainder(16-1,256))
      WriteDSP(floor((16-1)/256))
      start = time()
      while time() - start < .05 do
      end while
      DmaCount = ReadCurrentCount(DMA_16_CHANNELS[i])
      if DmaCount = 65535 then
	 free_low(DmaBuff)
	 Dma16 = DMA_16_CHANNELS[i]
	 if debug then
	    WriteLogMsg(debug_file,sprintf("Found DMA16 to be %d",{Dma16}))
	 end if
	 return Dma16
      else
	 ResetDSP()
	 ResetDma(DMA_16_CHANNELS[i])
      end if
    end for
    free_low(DmaBuff)
    return -1
end function
-----------------------------------------------------------------

-------------Write Sound Card Register----------------
procedure WriteSndReg(integer reg, integer val) -- by Greg Harris
    --Write to a sound card register
    integer temp
	Output(reg, #388)
	for tmp = 1 to 6 do      --wait 6 miliseconds
	    temp = Input(#388)
	end for
	Output(val, #389)
	for tmp = 1 to 35 do    --wait 35 miliseconds
	    temp = Input(#388)
	end for
end procedure

----------------Detect Sound Card--------------------

function Detect_Sound_Card() -- By Greg Harris
    --Detects a Adlib Compatible Card
    --Return 1 if true 0 if not found
    integer a, b, c, success
	success = 0             
	WriteSndReg(#4, #60)
	WriteSndReg(#4, #80)
	b=Input(#388)
	WriteSndReg(#2, #FF)
	WriteSndReg(#4, #21)
	for tmp = 1 to 130 do
	    a = Input(#388)
	end for
	c = Input(#388)
	WriteSndReg(#4, #60)
	WriteSndReg(#4, #80)
	if and_bits(b,#E0) = 0 then
	    if and_bits(c,#E0) = #C0 then
		success = 1
		for tmp = 1 to #F5 do     --reset Sound Card
		    WriteSndReg(tmp, 0)
		end for
	    end if
	end if
    return success
end function

function Get_Base_Address() -- by Greg Harris
integer baseport, resetport, readport, data
data = 0
baseport = #210
resetport = #216
readport = #21A
    while (data != DSP_DATA_READY) and (baseport<#270) do
       Output(1, resetport)
       for x = 1 to 3 do
       end for
       Output(0, resetport)
       data = 0

       for x = 0 to 99 do
       data = Input(readport)
       end for

       if data = DSP_DATA_READY then
	  exit
       else
	    baseport = baseport + #10
	    resetport = resetport + #10
	    readport = readport + #10
       end if
    end while

    if baseport = #270 then
	return 0
    else
	return baseport
    end if
end function

function DetectCard()
    integer temp
    if not detect then 
      return 0
    end if
    if debug then
	WriteLogMsg(debug_file,"Entering DetectCard()")
    end if
    if Detect_Sound_Card() then
	temp = Get_Base_Address()
	BaseIo = temp
	if debug then
	  WriteLogMsg(debug_file,sprintf("Found base address to be %d",{BaseIo}))
	end if
	DSP_RESET = BaseIo +  6        -- dsp reset i/o port
	DSP_READ_DATA = BaseIo + #A    -- dsp read data i/o port
	DSP_WRITE_DATA = BaseIo + #C   -- dsp write data i/o port
	DSP_WRITE_STATUS = BaseIo +#C  -- dsp write status i/o port
	DSP_DATA_AVAIL = BaseIo + #E   -- dsp read status i/o port
	if debug then
	    WriteLogMsg(debug_file,"Calling DetectDma8()")
	end if
	DMA8 = DetectDma8()
	if DMA8 = -1 then
	    return 0
	end if
	if debug then
	    WriteLogMsg(debug_file,"Calling DetectIRQ()")
	end if
	IRQ = DetectIRQ()
	if IRQ = -1 then
	    return 0
	end if
	if IRQ < 8 then
	    SB_INT = IRQ + 8
	else
	    SB_INT = IRQ -8 + #70 
	end if
	DSPVer = GetDSPVersion()
	if DSPVer >= 4.0 then
	    if debug then
	      WriteLogMsg(debug_file,"Calling DetectDma16()")
	    end if
	    DMA16 = DetectDma16()
	end if
	return 1
    else
	return 0
    end if
end function


-----------------------------------------------------------------

function GetBlasterInfo()
-- Get sound blaster card info from environment variable BLASTER
object blaster  integer i
integer lb
  
  blaster =  getenv("BLASTER")
  if atom(blaster) then
    return ERROR_NO_BLASTER
  end if
    blaster = upper(blaster)
    lb = length(blaster)
    i = match("A",blaster)
    if not i then
	return ERROR_ENV_BAD
    end if
    if lb >= i+3 then
      BaseIo = HexToI(blaster[i+1..i+3])
    else
      return ERROR_ENV_BAD
    end if
    if BaseIo < #210 then
	return ERROR_ENV_BAD
    end if
    i = match("I",blaster)
    if not i then
	return ERROR_ENV_BAD
    end if
    if lb >= i+2 then
      IRQ = AToI(blaster[i+1..i+2])
    elsif lb >= i+1 then
      IRQ = blaster[i+1] - '0'
    else
      return ERROR_ENV_BAD
    end if
    if not find(IRQ,{2,5,7,10}) then
	return ERROR_ENV_BAD
    end if
    if IRQ < 8 then
       SB_INT = IRQ + 8
    else
       SB_INT = IRQ - 8 + #70
    end if
    i = match("D",blaster)
    if not i then
	return ERROR_ENV_BAD
    end if
    if lb < i+1 then
	return ERROR_ENV_BAD
    end if
    DMA8 = blaster[i+1] - '0'
    if not find(DMA8,{0,1,3}) then  -- validate 8 bit dma channel
	return ERROR_ENV_BAD
    end if
    i = match("H",blaster)
    if i then
      if lb >= i+1 then
	DMA16 = blaster[i+1] - '0' 
      else
	return ERROR_ENV_BAD
      end if    
      if not find(DMA16,{5,6,7}) then  -- validate dma channel
	return ERROR_ENV_BAD
      end if
    end if
    DSP_RESET = BaseIo +  6        -- dsp reset i/o port
    DSP_READ_DATA = BaseIo + #A    -- dsp read data i/o port
    DSP_WRITE_DATA = BaseIo + #C   -- dsp write data i/o port
    DSP_WRITE_STATUS = BaseIo +#C  -- dsp write status i/o port
    DSP_DATA_AVAIL = BaseIo + #E   -- dsp read status i/o port
    DSPVer = GetDSPVersion()
    return 0
end function -- GetBlasterInfo()

procedure InitDSP()
-- initialise DSP and some sound card parameters
  if debug then
    WriteLogMsg(debug_file,"Entering InitDSP()")
  end if
  iSbCardError = GetBlasterInfo()
  if iSbCardError then
    if not DetectCard() then
      iSbCardError = ERROR_CARD_NOT_FOUND
      return
    end  if
  end if
  ResetDSP() 
  if iSbCardError then
    return
  end if
  DmaChannel = DMA8
  ResetDma(DMA8)
  if DSPVer >= 4.0 then
     ResetDma(DMA16)
  elsif DSPVer < 2.0 then
     iSbCardError = ERROR_CARD_NOT_SUPPORTED
     return
  end if
  if debug then
    WriteLogMsg(debug_file,sprintf("Found DSP version to be %1.2f",{DSPVer}))
  end if
  mem_set(DATA_SEG,0,DATA_LEN)
  poke(IRQ_NO, IRQ)
  poke(BASE_IO,{remainder(BaseIo,256),floor(BaseIo/256)})
  poke(BLOCK_SIZE,int_to_bytes(cBLOCK_SIZE))
  DmaBuffer = AllocateDMABuffer(cBUFFER_SIZE)
  lock_memory(DmaBuffer,cBUFFER_SIZE)
  poke(DMA_BUFF_ADDR,int_to_bytes(DmaBuffer))
  MixerBuffer = allocate(2*cBLOCK_SIZE)
  lock_memory(MixerBuffer,2*cBLOCK_SIZE)
  poke(MX_BUFF_ADDR,int_to_bytes(MixerBuffer))
  OrigVector = get_vector(SB_INT)
  set_vector(SB_INT,{code_segment,DSP_ISR})
  EnableIRQ(IRQ)
  WriteDSP(SPEAKER_ON)
  SetVolume(VOICE_VOLUME,VOL_MAX)
  if debug then
    WriteLogMsg(debug_file,"DSP initialisation terminated.")
  end if
end procedure -- InitDSP()

global constant -- index to access Wave info sequence
  siFMT_ID = 1,    -- format identifier
  siCHANNELS = 2,  -- number of channels
  siSPS = 3,       -- samples per seconds 
  siBITS_PS = 4,   --  bits per sample 8 or 16
  siDATA_LEN = 5,  -- data length
  siDATA_OFS = 6,  -- data offset in file
  siDATA_BUFF = 7  -- data buffer address


procedure SetChannels(integer channels)
-- set dsp for MONO or STEREO
integer temp
if channels = MONO then
  Output(FILTER_STEREO,BaseIo+MIXER_ADDR)
  temp = Input(BaseIo+MIXER_DATA)
  Output(and_bits(temp,#FD),BaseIo+MIXER_DATA)
else
  Output(FILTER_STEREO,BaseIo+MIXER_ADDR)
  temp = Input(BaseIo+MIXER_DATA)
  Output(or_bits(temp,2),BaseIo+MIXER_DATA)     
  -- reset mixer output toggle
  poke(DmaBuffer,#80)
  SetDma(DmaBuffer,1,DmaChannel,DMA_OUT+DMA_SINGLE)
  DisableIRQ(IRQ)
  WriteDSP(SET_SINGLE_CYCLE) 
  WriteDSP(0)
  WriteDSP(0)
  delay(.001)
end if
end procedure -- SetChannels()

function ComputeTC(integer SampleRate, integer NbChannels)
-- compute time constant from sample rate and number of channels.
  return 256 - floor(1000000/(NbChannels*SampleRate))
end function -- ComputeTC()


global function SoundDone(integer index)
  return peek(SOUNDS + index*REC_LEN+iSND_PLAY) = 0
end function -- SoundDone()

global function PlaySound(sequence SoundInfo, integer loop)
-- play sound from SoundInfo sequence
--  
integer index, SndPlaying
  if debug then
    WriteLogMsg(debug_file,"Entering play sound.")
  end if
  if iSbCardError then
    return -1
  end if
  SndPlaying = peek(NB_SOUNDS)
  if SndPlaying = MAX_SOUNDS then
    return -1
  end if
  index = -1
  for i = 0 to MAX_SOUNDS - 1  do -- search an empty slot
    if peek(SOUNDS+i*REC_LEN+iSND_PLAY) = 0 then
	index = i
	exit
    end if
  end for
  if index = -1 then
    return index -- not empty slot
  end if
  poke(SOUNDS + REC_LEN*index + iDATA_ADDR,int_to_bytes(SoundInfo[siDATA_BUFF]))
  poke(SOUNDS + REC_LEN*index + iPOS_IDX,int_to_bytes(SoundInfo[siDATA_BUFF]))
  poke(SOUNDS + REC_LEN*index + iDATA_LEN, int_to_bytes(SoundInfo[siDATA_LEN]))
  poke(SOUNDS + REC_LEN*index + iSND_LOOP, {loop,0})
  poke(SOUNDS + REC_LEN*index + iSND_PLAY, {1,0})
  poke(NB_SOUNDS, SndPlaying+1)
  if SndPlaying > 0 then
    return index
  end if
  if debug then
    WriteLogMsg(debug_file,"In PlaySound resetting DSP.")
  end if
  ResetDSP()
  if SoundInfo[siDATA_LEN] = 0 or iSbCardError then
    return -1
  end if
  
  if SoundInfo[siBITS_PS] = 16 and DSPVer >= 4 then
    DmaChannel = DMA16
  else
    DmaChannel = DMA8
    -- Set the playback time constant
    WriteDSP(SET_TIME_CONSTANT) 
    WriteDSP(ComputeTC(SoundInfo[siSPS],SoundInfo[siCHANNELS])) 
    if SoundInfo[siCHANNELS] = MONO then
      SetChannels(MONO)
    else
      SetChannels(STEREO)
    end if
  end if
  if debug then
    WriteLogMsg(debug_file,"In PlaySound programming DSP.")
  end if
  WriteDSP(SPEAKER_ON)
  SetVolume(MASTER_VOLUME, VOL_MAX)
  mem_set(DmaBuffer,#80,cBUFFER_SIZE)
  SetDma(DmaBuffer,cBUFFER_SIZE,DmaChannel,DMA_OUT+DMA_AUTO)
  poke(HALF_FF,0)
  poke(INT_COUNT,{0,0,0,0})
  EnableIRQ(IRQ)
  WriteDSP(SET_BLOCK_SIZE)
  WriteDSP(remainder(cBLOCK_SIZE-1,256))
  WriteDSP(floor((cBLOCK_SIZE-1)/256))
  if SoundInfo[siCHANNELS] = MONO then
    WriteDSP(SET_AUTO_INIT)  -- auto init mode
  else    -- stereo play
    WriteDSP(HSPEED_AUTO_INIT) -- high speed auto init mode.
  end if
  return index
end function -- PlaySound()

-- *******   following functions read informations in wave buffer ****
   
function ReadTag(atom buffer, integer i)
-- each block begin by a tag of 4 characters
sequence tag
    tag = peek({buffer+i,4})
    return {tag,i+4}
end function -- ReadTag()

function ReadInteger(atom buffer, integer i)
-- read an integer from buffer
-- file pointer should be on first byte of integer
integer n
    n = bytes_to_int(peek({buffer+i,4}))
    return {n,i+4}
end function --ReadInteger()

function ReadWord(atom buffer, integer i)
integer w
    w = peek(buffer+i)+peek(buffer+i+1)*256
    return {w,i+2}
end function -- ReadWord()

function ReadWavHeader(atom d_buffer)
-- read WAV file header record
-- return the following sequence of information.
-- {format id, number of channels, samples per second,
--  average bytes per second, bits per sample, data length}
-- see above for globals constants defined to access information in this
-- sequence.
--
-- structure of wave file header record
-- tagged record type file. tags are 4 characters long.
--     data              offset       data nature
--     ------------------------------------------
--     tag               [1..4]       "RIFF" file Id tag
--     integer           [5..8]       file length excluding this record
--     tag               [9..12]      "WAVE"  confirm it's a wave file
--     tag               [13..16]     "fmt " announce format record
--     integer           [17..20]     length of format record
--     word              [21..22]     format tag
--     word              [23..24]     number of channels  (mono=1,stereo=2)
--     integer           [25..28]     samples per seconds
--     integer           [29..32]     average bytes per second  
--     word              [33..34]     block align
--     word              [35..36]     bits per sample
--     tag               [37..40]     "data " or "fact "  record
--     integer           [41..44]     length of that record, if data it means
--                                    length of sound sample, if fact jump
--                                     over it and go to data record.   
--

sequence WaveInfo , data
integer number,RecLen, i

  data = {0,0}
  WaveInfo = {}
  data = ReadTag(d_buffer,0)  -- read "RIFF" tag
  if not match("RIFF", data[1]) then
    return -1 -- bad file format
  end if
  data = ReadInteger(d_buffer,data[2])  -- skip file length integer
  data = ReadTag(d_buffer,data[2])  -- read "WAV" tag
  if not match("WAVE", data[1]) then
    return -1 -- bad file format
  end if
  data = ReadTag(d_buffer,data[2])  -- read "fmt " tag
  if not match("fmt ", data[1]) then
    return -1 -- bad file format
  end if
  data = ReadInteger(d_buffer,data[2]) -- read format record length
  RecLen = data[1]
  data = ReadWord(d_buffer,data[2]) -- read format numeric id
  WaveInfo = {data[1]}
  data = ReadWord(d_buffer,data[2]) -- read number of channels
  WaveInfo = WaveInfo & data[1]
  data = ReadInteger(d_buffer,data[2]) -- read samples per second
  WaveInfo = WaveInfo & data[1] 
  data = ReadInteger(d_buffer,data[2]) -- read average bytes per second
  data = ReadWord(d_buffer,data[2]) -- skip block align word and discard it.
  data = ReadWord(d_buffer,data[2]) -- read bits per samples
  WaveInfo = WaveInfo & data[1] 
  if RecLen > #10 then
    data[2] = data[2] + RecLen - #10
  end if
  data = ReadTag(d_buffer,data[2])
  while compare(data[1],"data")  do -- skip non "data" records.
    data = ReadInteger(d_buffer,data[2])
    data[2] = data[2] + data[1]
    data = ReadTag(d_buffer,data[2])
  end while
  if not compare(data[1],"data") then
    data = ReadInteger(d_buffer,data[2])
    WaveInfo = WaveInfo & data[1] -- get sound data length
  else -- something wrong with that file.
      return -1
  end if
  return WaveInfo & data[2]
end function -- ReadWavHeader()

-- ************************************************************************

constant COPY_DATA_LEN = 16
atom COPY_CODE, COPY_DATA,
     DSK_BUFF_ADDR,
     COUNT,
     SND_BUFF_ADDR

COPY_DATA = allocate(COPY_DATA_LEN)
lock_memory(COPY_DATA,COPY_DATA_LEN)

DSK_BUFF_ADDR = COPY_DATA + 0
COUNT = COPY_DATA + 4
SND_BUFF_ADDR = COPY_DATA + 8

constant sCOPY_CODE = {
  --PROC COPY_8BITS 0
  #1E,#06,                              -- PUSH DS ES
  #60,                                  -- PUSHAD
  #8B,#35}&int_to_bytes(DSK_BUFF_ADDR)&{-- MOV  ESI, [DSK_BUFF_ADDR]
  #8B,#3D}&int_to_bytes(SND_BUFF_ADDR)&{-- MOV  EDI, [SND_BUFF_ADDR]
  #8B,#0D}&int_to_bytes(COUNT)&{        -- MOV  ECX, [COUNT]
  #0B,#C9,                              -- OR ECX, ECX
  #74,#0A,#90,#90,#90,#90,              -- JE @EXIT_CPY8
				      -- @COPY_LOOP:
  #AC,                                  -- LODSB
  #2C,#80,                              -- SUB  AL, 080
  #AA,                                  -- STOSB
  #E2,#FA,                              -- LOOP @COPY_LOOP
				      -- @EXIT_CPY8:
  #61,                                  -- POPAD
  #07,#1F,                              -- POP  ES DS
  #C3,                                  -- RET
  --PROC COPY_16BITS 39
  #1E,#06,                              -- PUSH DS ES
  #60,                                  -- PUSHAD
  #8B,#35}&int_to_bytes(DSK_BUFF_ADDR)&{-- MOV  ESI, [DSK_BUFF_ADDR]
  #8B,#3D}&int_to_bytes(SND_BUFF_ADDR)&{-- MOV  EDI, [SND_BUFF_ADDR]
  #8B,#0D}&int_to_bytes(COUNT)&{        -- MOV  ECX, [COUNT]
  #0B,#C9,                              -- OR ECX, ECX
  #74,#0B,#90,#90,#90,#90,              -- JE @EXIT_CPY16
				      -- @C16_LP:
  #66,#AD,                              -- LODSW
  #86,#E0,                              -- XCHG AH, AL
  #AA,                                  -- STOSB
  #E2,#F9,                              -- LOOP @C16_LP
				      -- @EXIT_CPY16:
  #61,                                  -- POPAD
  #07,#1F,                              -- POP  ES DS
  #C3,                                  -- RET
  0
}
COPY_CODE = allocate(length(sCOPY_CODE))
lock_memory(COPY_CODE,length(sCOPY_CODE))
poke(COPY_CODE,sCOPY_CODE)
atom Copy8bits, Copy16bits
Copy8bits = COPY_CODE
Copy16bits = COPY_CODE + 39

global function LoadWaveFile(sequence FileName)
-- load wave data in a buffer and return wave info sequence
sequence Header
object WaveInfo
integer DataLen, FHandle  
integer NbRead,j, buf_size
atom disk_buffer, DataBuf
  
  if iSbCardError then
    return -1
  end if
  FHandle = DosOpen(FileName,READ)
  if FHandle = -1 then
    return -1
  end if
  buf_size = power(2,14)
  disk_buffer = allocate_low(buf_size)
  if not disk_buffer then
    NbRead = DosClose(FHandle)
    return -1
  end if
  lock_memory(disk_buffer,buf_size)
  poke(DSK_BUFF_ADDR,int_to_bytes(disk_buffer))
  NbRead = BlockRead(FHandle,disk_buffer,buf_size)
  WaveInfo = ReadWavHeader(disk_buffer)
  if atom(WaveInfo) then
    NbRead = DosClose(FHandle)
    free_low(disk_buffer)
    return -1
  end if
  NbRead = NbRead - WaveInfo[siDATA_OFS] -- substract header
  mem_copy(disk_buffer, disk_buffer+WaveInfo[siDATA_OFS], NbRead)
  j = 0
  if WaveInfo[siBITS_PS] = 16 then -- load and convert to 8 bits
    WaveInfo[siBITS_PS] = 8
    DataLen = floor(WaveInfo[siDATA_LEN]/2)
    WaveInfo[siDATA_LEN] = DataLen
    DataBuf = allocate(DataLen)
    if not DataBuf then
	NbRead=DosClose(FHandle)
	free_low(disk_buffer)
	return -1
    end if
    lock_memory(DataBuf,DataLen)
    while NbRead > 0 do
      if NbRead > 2*(DataLen - j) then
	NbRead = 2*(DataLen -j)
      end if
      poke(COUNT,int_to_bytes(floor(NbRead/2)))
      poke(SND_BUFF_ADDR,int_to_bytes(DataBuf+j))
      call(Copy16bits)
      j = j + floor(NbRead/2)
      NbRead = BlockRead(FHandle,disk_buffer,buf_size)
    end while
  else   
    DataLen = WaveInfo[siDATA_LEN]
    DataBuf = allocate(DataLen)
    if not DataBuf then
	NbRead=DosClose(FHandle)
	free_low(disk_buffer)
	return -1
    end if
    lock_memory(DataBuf,DataLen)
    while NbRead > 0 do
      if NbRead > DataLen - j then
	 NbRead = DataLen - j
      end if
      poke(COUNT,int_to_bytes(NbRead))
      poke(SND_BUFF_ADDR,int_to_bytes(DataBuf+j))
      call(Copy8bits)
      j = j + NbRead
      NbRead = BlockRead(FHandle,disk_buffer,buf_size)
    end while
  end if
  NbRead=DosClose(FHandle)
  free_low(disk_buffer)
  return append(WaveInfo,DataBuf)
end function -- LoadWaveFile()

global procedure FreeSoundBuffer(sequence SoundInfo)
-- free allocated memory for sound buffer
   free(SoundInfo[siDATA_BUFF])
end procedure

global procedure PlayWaveFile(sequence FileName, integer volume)
object SoundInfo
integer index
    SoundInfo = LoadWaveFile(FileName)
    if atom(SoundInfo) then
	return
    end if
    SetVolume(VOICE_VOLUME,volume)
    index = PlaySound(SoundInfo,0)
    if index = -1 then
	return
    end if
    while not SoundDone(index) do
    end while
    ResetDSP()
    ResetDma(DMA8)
    FreeSoundBuffer(SoundInfo) 
end procedure -- PlayWaveFile()

global procedure StopSound(integer index)
  if index = - 1 then
     for i = 0 to MAX_SOUNDS-1 do
	poke(SOUNDS+i*REC_LEN+iSND_PLAY,{0,0})
	poke(SOUNDS+i*REC_LEN+iSND_LOOP,{0,0})
	poke(NB_SOUNDS,0)
     end for
  else  
	poke(SOUNDS+index*REC_LEN+iSND_PLAY,{0,0})
	poke(SOUNDS+index*REC_LEN+iSND_LOOP,{0,0})
	poke(NB_SOUNDS,peek(NB_SOUNDS)-1)
  end if
end procedure

global procedure CloseSfx()
  ResetDSP()
  DisableIRQ(IRQ)
  set_vector(SB_INT,OrigVector)
  free_low(DmaBuffer)
  ResetDma(DMA8)
  if DSPVer >= 4.0 then
     ResetDma(DMA16)
  end if
  iSbCardError = ERROR_NOT_INITIALISED
  if debug then
    WriteLogMsg(debug_file,"Closing sfx.")
  end if
end procedure

InitDSP()


