MODULE CRC;

  (* 32-Bit-CRC-Berechnung der Standardeingabe. Siehe Microsoft System
     Journal 3/95, Mai/Juni 1995, Seiten 145ff. *)

  (* HISTORY:
     02.05.1995 sf Erste Version.
   *)

  FROM ASCII IMPORT EOL;

  FROM Filter IMPORT Read, WriteChar, WriteLine, ParameterFound;

  FROM ErrorReporting IMPORT WriteLnError;

  FROM NumberConversion IMPORT LongIntToString;

  PROCEDURE Help;

  BEGIN
    WriteLnError ("CRC [/I]");
    WriteLnError ("Bildet eine 32-Bit-CRC-Prfsumme ber die Standardeingabe");
    WriteLnError ("und schreibt sie auf die Standardausgabe.");
    WriteLnError ("/I: Die Prfsumme wird dezimal als vorzeichenbehaftete 32-Bit-");
    WriteLnError ("    Ganzzahl anstatt als 32-Bit-Hexadezimalzahl ausgegeben.")
  END Help;

  PROCEDURE LiesParameter (VAR dezimal, ShowHelp: BOOLEAN);

    VAR Parameter: ARRAY [0..2] OF CHAR;

  BEGIN
    dezimal := FALSE;
    ShowHelp := FALSE;
    WHILE ParameterFound (TRUE, Parameter) DO
      IF (Parameter [0] = "/") AND (Parameter [1] = "I")
        AND (Parameter [2] = 0C) THEN
        dezimal := TRUE
      ELSE
        ShowHelp := TRUE
      END
    END
  END LiesParameter;

  PROCEDURE Bearbeite (dezimal: BOOLEAN);

    CONST CRCTableEntries = 256;

    TYPE
      CRCTableIndex = [0.. CRCTableEntries - 1];
      Bits8Set = SET OF [0..7];
      Bits32Set = SET OF [0..31];
      Bits32 =
        RECORD
          CASE : BOOLEAN OF
            FALSE:
              a: ARRAY [0..3] OF Bits8Set |
            TRUE:
              b: Bits32Set
          END
        END;

    VAR
      crc: Bits32;
      ch: CHAR;
      CRCTable: ARRAY CRCTableIndex OF Bits32Set;
      index: CRCTableIndex;

    PROCEDURE InitCRCTable;

    BEGIN
      CRCTable [  0] := Bits32Set ( 00000000H);
      CRCTable [  1] := Bits32Set ( 77073096H);
      CRCTable [  2] := Bits32Set (0EE0E612CH);
      CRCTable [  3] := Bits32Set ( 990951BAH);
      CRCTable [  4] := Bits32Set ( 076DC419H);
      CRCTable [  5] := Bits32Set ( 706AF48FH);
      CRCTable [  6] := Bits32Set (0E963A535H);
      CRCTable [  7] := Bits32Set ( 9E6495A3H);
      CRCTable [  8] := Bits32Set (0EDB8832H);
      CRCTable [  9] := Bits32Set ( 79DCB8A4H);
      CRCTable [ 10] := Bits32Set (0E0D5E91EH);
      CRCTable [ 11] := Bits32Set ( 97D2D988H);
      CRCTable [ 12] := Bits32Set ( 09B64C2BH);
      CRCTable [ 13] := Bits32Set ( 7EB17CBDH);
      CRCTable [ 14] := Bits32Set (0E7B82D07H);
      CRCTable [ 15] := Bits32Set ( 90BF1D91H);
      CRCTable [ 16] := Bits32Set ( 1DB71064H);
      CRCTable [ 17] := Bits32Set ( 6AB020F2H);
      CRCTable [ 18] := Bits32Set (0F3B97148H);
      CRCTable [ 19] := Bits32Set ( 84BE41DEH);
      CRCTable [ 20] := Bits32Set ( 1ADAD47DH);
      CRCTable [ 21] := Bits32Set ( 6DDDE4EBH);
      CRCTable [ 22] := Bits32Set (0F4D4B551H);
      CRCTable [ 23] := Bits32Set ( 83D385C7H);
      CRCTable [ 24] := Bits32Set ( 136C9856H);
      CRCTable [ 25] := Bits32Set ( 646BA8C0H);
      CRCTable [ 26] := Bits32Set (0FD62F97AH);
      CRCTable [ 27] := Bits32Set ( 8A65C9ECH);
      CRCTable [ 28] := Bits32Set ( 14015C4FH);
      CRCTable [ 29] := Bits32Set ( 63066CD9H);
      CRCTable [ 30] := Bits32Set (0FA0F3D63H);
      CRCTable [ 31] := Bits32Set ( 8D080DF5H);
      CRCTable [ 32] := Bits32Set ( 3B6E20C8H);
      CRCTable [ 33] := Bits32Set ( 4C69105EH);
      CRCTable [ 34] := Bits32Set (0D56041E4H);
      CRCTable [ 35] := Bits32Set (0A2677172H);
      CRCTable [ 36] := Bits32Set ( 3C03E4D1H);
      CRCTable [ 37] := Bits32Set ( 4B04D447H);
      CRCTable [ 38] := Bits32Set (0D20D85FDH);
      CRCTable [ 39] := Bits32Set (0A50AB56BH);
      CRCTable [ 40] := Bits32Set ( 35B5A8FAH);
      CRCTable [ 41] := Bits32Set ( 42B2986CH);
      CRCTable [ 42] := Bits32Set (0DBBBC9D6H);
      CRCTable [ 43] := Bits32Set (0ACBCF940H);
      CRCTable [ 44] := Bits32Set ( 32D86CE3H);
      CRCTable [ 45] := Bits32Set ( 45DF5C75H);
      CRCTable [ 46] := Bits32Set (0DCD60DCFH);
      CRCTable [ 47] := Bits32Set (0ABD13D59H);
      CRCTable [ 48] := Bits32Set ( 26D930ACH);
      CRCTable [ 49] := Bits32Set ( 51DE003AH);
      CRCTable [ 50] := Bits32Set (0C8D75180H);
      CRCTable [ 51] := Bits32Set (0BFD06116H);
      CRCTable [ 52] := Bits32Set ( 21B4F4B5H);
      CRCTable [ 53] := Bits32Set ( 56B3C423H);
      CRCTable [ 54] := Bits32Set (0CFBA9599H);
      CRCTable [ 55] := Bits32Set (0B8BDA50FH);
      CRCTable [ 56] := Bits32Set ( 2802B89EH);
      CRCTable [ 57] := Bits32Set ( 5F058808H);
      CRCTable [ 58] := Bits32Set (0C60CD9B2H);
      CRCTable [ 59] := Bits32Set (0B10BE924H);
      CRCTable [ 60] := Bits32Set ( 2F6F7C87H);
      CRCTable [ 61] := Bits32Set ( 58684C11H);
      CRCTable [ 62] := Bits32Set (0C1611DABH);
      CRCTable [ 63] := Bits32Set (0B6662D3DH);
      CRCTable [ 64] := Bits32Set ( 76DC4190H);
      CRCTable [ 65] := Bits32Set ( 01DB7106H);
      CRCTable [ 66] := Bits32Set ( 98D220BCH);
      CRCTable [ 67] := Bits32Set (0EFD5102AH);
      CRCTable [ 68] := Bits32Set ( 71B18589H);
      CRCTable [ 69] := Bits32Set ( 06B6B51FH);
      CRCTable [ 70] := Bits32Set ( 9FBFE4A5H);
      CRCTable [ 71] := Bits32Set (0E8B8D433H);
      CRCTable [ 72] := Bits32Set ( 7807C9A2H);
      CRCTable [ 73] := Bits32Set ( 0F00F934H);
      CRCTable [ 74] := Bits32Set ( 9609A88EH);
      CRCTable [ 75] := Bits32Set (0E10E9818H);
      CRCTable [ 76] := Bits32Set ( 7F6A0DBBH);
      CRCTable [ 77] := Bits32Set ( 086D3D2DH);
      CRCTable [ 78] := Bits32Set ( 91646C97H);
      CRCTable [ 79] := Bits32Set (0E6635C01H);
      CRCTable [ 80] := Bits32Set ( 6B6B51F4H);
      CRCTable [ 81] := Bits32Set ( 1C6C6162H);
      CRCTable [ 82] := Bits32Set ( 856530D8H);
      CRCTable [ 83] := Bits32Set (0F262004EH);
      CRCTable [ 84] := Bits32Set ( 6C0695EDH);
      CRCTable [ 85] := Bits32Set ( 1B01A57BH);
      CRCTable [ 86] := Bits32Set ( 8208F4C1H);
      CRCTable [ 87] := Bits32Set (0F50FC457H);
      CRCTable [ 88] := Bits32Set ( 65B0D9C6H);
      CRCTable [ 89] := Bits32Set ( 12B7E950H);
      CRCTable [ 90] := Bits32Set ( 8BBEB8EAH);
      CRCTable [ 91] := Bits32Set (0FCB9887CH);
      CRCTable [ 92] := Bits32Set ( 62DD1DDFH);
      CRCTable [ 93] := Bits32Set ( 15DA2D49H);
      CRCTable [ 94] := Bits32Set ( 8CD37CF3H);
      CRCTable [ 95] := Bits32Set (0FBD44C65H);
      CRCTable [ 96] := Bits32Set ( 4DB26158H);
      CRCTable [ 97] := Bits32Set ( 3AB551CEH);
      CRCTable [ 98] := Bits32Set (0A3BC0074H);
      CRCTable [ 99] := Bits32Set (0D4BB30E2H);
      CRCTable [100] := Bits32Set ( 4ADFA541H);
      CRCTable [101] := Bits32Set ( 3DD895D7H);
      CRCTable [102] := Bits32Set (0A4D1C46DH);
      CRCTable [103] := Bits32Set (0D3D6F4FBH);
      CRCTable [104] := Bits32Set ( 4369E96AH);
      CRCTable [105] := Bits32Set ( 346ED9FCH);
      CRCTable [106] := Bits32Set (0AD678846H);
      CRCTable [107] := Bits32Set (0DA60B8D0H);
      CRCTable [108] := Bits32Set ( 44042D73H);
      CRCTable [109] := Bits32Set ( 33031DE5H);
      CRCTable [110] := Bits32Set (0AA0A4C5FH);
      CRCTable [111] := Bits32Set (0DD0D7CC9H);
      CRCTable [112] := Bits32Set ( 5005713CH);
      CRCTable [113] := Bits32Set ( 270241AAH);
      CRCTable [114] := Bits32Set (0BE0B1010H);
      CRCTable [115] := Bits32Set (0C90C2086H);
      CRCTable [116] := Bits32Set ( 5768B525H);
      CRCTable [117] := Bits32Set ( 206F85B3H);
      CRCTable [118] := Bits32Set (0B966D409H);
      CRCTable [119] := Bits32Set (0CE61E49FH);
      CRCTable [120] := Bits32Set ( 5EDEF90EH);
      CRCTable [121] := Bits32Set ( 29D9C998H);
      CRCTable [122] := Bits32Set (0B0D09822H);
      CRCTable [123] := Bits32Set (0C7D7A8B4H);
      CRCTable [124] := Bits32Set ( 59B33D17H);
      CRCTable [125] := Bits32Set ( 2EB40D81H);
      CRCTable [126] := Bits32Set (0B7BD5C3BH);
      CRCTable [127] := Bits32Set (0C0BA6CADH);
      CRCTable [128] := Bits32Set (0EDB88320H);
      CRCTable [129] := Bits32Set ( 9ABFB3B6H);
      CRCTable [130] := Bits32Set ( 03B6E20CH);
      CRCTable [131] := Bits32Set ( 74B1D29AH);
      CRCTable [132] := Bits32Set (0EAD54739H);
      CRCTable [133] := Bits32Set ( 9DD277AFH);
      CRCTable [134] := Bits32Set ( 04DB2615H);
      CRCTable [135] := Bits32Set ( 73DC1683H);
      CRCTable [136] := Bits32Set (0E3630B12H);
      CRCTable [137] := Bits32Set ( 94643B84H);
      CRCTable [138] := Bits32Set ( 0D6D6A3EH);
      CRCTable [139] := Bits32Set ( 7A6A5AA8H);
      CRCTable [140] := Bits32Set (0E40ECF0BH);
      CRCTable [141] := Bits32Set ( 9309FF9DH);
      CRCTable [142] := Bits32Set ( 0A00AE27H);
      CRCTable [143] := Bits32Set ( 7D079EB1H);
      CRCTable [144] := Bits32Set (0F00F9344H);
      CRCTable [145] := Bits32Set ( 8708A3D2H);
      CRCTable [146] := Bits32Set ( 1E01F268H);
      CRCTable [147] := Bits32Set ( 6906C2FEH);
      CRCTable [148] := Bits32Set (0F762575DH);
      CRCTable [149] := Bits32Set ( 806567CBH);
      CRCTable [150] := Bits32Set ( 196C3671H);
      CRCTable [151] := Bits32Set ( 6E6B06E7H);
      CRCTable [152] := Bits32Set (0FED41B76H);
      CRCTable [153] := Bits32Set ( 89D32BE0H);
      CRCTable [154] := Bits32Set ( 10DA7A5AH);
      CRCTable [155] := Bits32Set ( 67DD4ACCH);
      CRCTable [156] := Bits32Set (0F9B9DF6FH);
      CRCTable [157] := Bits32Set ( 8EBEEFF9H);
      CRCTable [158] := Bits32Set ( 17B7BE43H);
      CRCTable [159] := Bits32Set ( 60B08ED5H);
      CRCTable [160] := Bits32Set (0D6D6A3E8H);
      CRCTable [161] := Bits32Set (0A1D1937EH);
      CRCTable [162] := Bits32Set ( 38D8C2C4H);
      CRCTable [163] := Bits32Set ( 4FDFF252H);
      CRCTable [164] := Bits32Set (0D1BB67F1H);
      CRCTable [165] := Bits32Set (0A6BC5767H);
      CRCTable [166] := Bits32Set ( 3FB506DDH);
      CRCTable [167] := Bits32Set ( 48B2364BH);
      CRCTable [168] := Bits32Set (0D80D2BDAH);
      CRCTable [169] := Bits32Set (0AF0A1B4CH);
      CRCTable [170] := Bits32Set ( 36034AF6H);
      CRCTable [171] := Bits32Set ( 41047A60H);
      CRCTable [172] := Bits32Set (0DF60EFC3H);
      CRCTable [173] := Bits32Set (0A867DF55H);
      CRCTable [174] := Bits32Set ( 316E8EEFH);
      CRCTable [175] := Bits32Set ( 4669BE79H);
      CRCTable [176] := Bits32Set (0CB61B38CH);
      CRCTable [177] := Bits32Set (0BC66831AH);
      CRCTable [178] := Bits32Set ( 256FD2A0H);
      CRCTable [179] := Bits32Set ( 5268E236H);
      CRCTable [180] := Bits32Set (0CC0C7795H);
      CRCTable [181] := Bits32Set (0BB0B4703H);
      CRCTable [182] := Bits32Set ( 220216B9H);
      CRCTable [183] := Bits32Set ( 5505262FH);
      CRCTable [184] := Bits32Set (0C5BA3BBEH);
      CRCTable [185] := Bits32Set (0B2BD0B28H);
      CRCTable [186] := Bits32Set ( 2BB45A92H);
      CRCTable [187] := Bits32Set ( 5CB36A04H);
      CRCTable [188] := Bits32Set (0C2D7FFA7H);
      CRCTable [189] := Bits32Set (0B5D0CF31H);
      CRCTable [190] := Bits32Set ( 2CD99E8BH);
      CRCTable [191] := Bits32Set ( 5BDEAE1DH);
      CRCTable [192] := Bits32Set ( 9B64C2B0H);
      CRCTable [193] := Bits32Set (0EC63F226H);
      CRCTable [194] := Bits32Set ( 756AA39CH);
      CRCTable [195] := Bits32Set ( 026D930AH);
      CRCTable [196] := Bits32Set ( 9C0906A9H);
      CRCTable [197] := Bits32Set (0EB0E363FH);
      CRCTable [198] := Bits32Set ( 72076785H);
      CRCTable [199] := Bits32Set ( 05005713H);
      CRCTable [200] := Bits32Set ( 95BF4A82H);
      CRCTable [201] := Bits32Set (0E2B87A14H);
      CRCTable [202] := Bits32Set ( 7BB12BAEH);
      CRCTable [203] := Bits32Set ( 0CB61B38H);
      CRCTable [204] := Bits32Set ( 92D28E9BH);
      CRCTable [205] := Bits32Set (0E5D5BE0DH);
      CRCTable [206] := Bits32Set ( 7CDCEFB7H);
      CRCTable [207] := Bits32Set ( 0BDBDF21H);
      CRCTable [208] := Bits32Set ( 86D3D2D4H);
      CRCTable [209] := Bits32Set (0F1D4E242H);
      CRCTable [210] := Bits32Set ( 68DDB3F8H);
      CRCTable [211] := Bits32Set ( 1FDA836EH);
      CRCTable [212] := Bits32Set ( 81BE16CDH);
      CRCTable [213] := Bits32Set (0F6B9265BH);
      CRCTable [214] := Bits32Set ( 6FB077E1H);
      CRCTable [215] := Bits32Set ( 18B74777H);
      CRCTable [216] := Bits32Set ( 88085AE6H);
      CRCTable [217] := Bits32Set (0FF0F6A70H);
      CRCTable [218] := Bits32Set ( 66063BCAH);
      CRCTable [219] := Bits32Set ( 11010B5CH);
      CRCTable [220] := Bits32Set ( 8F659EFFH);
      CRCTable [221] := Bits32Set (0F862AE69H);
      CRCTable [222] := Bits32Set ( 616BFFD3H);
      CRCTable [223] := Bits32Set ( 166CCF45H);
      CRCTable [224] := Bits32Set (0A00AE278H);
      CRCTable [225] := Bits32Set (0D70DD2EEH);
      CRCTable [226] := Bits32Set ( 4E048354H);
      CRCTable [227] := Bits32Set ( 3903B3C2H);
      CRCTable [228] := Bits32Set (0A7672661H);
      CRCTable [229] := Bits32Set (0D06016F7H);
      CRCTable [230] := Bits32Set ( 4969474DH);
      CRCTable [231] := Bits32Set ( 3E6E77DBH);
      CRCTable [232] := Bits32Set (0AED16A4AH);
      CRCTable [233] := Bits32Set (0D9D65ADCH);
      CRCTable [234] := Bits32Set ( 40DF0B66H);
      CRCTable [235] := Bits32Set ( 37D83BF0H);
      CRCTable [236] := Bits32Set (0A9BCAE53H);
      CRCTable [237] := Bits32Set (0DEBB9EC5H);
      CRCTable [238] := Bits32Set ( 47B2CF7FH);
      CRCTable [239] := Bits32Set ( 30B5FFE9H);
      CRCTable [240] := Bits32Set (0BDBDF21CH);
      CRCTable [241] := Bits32Set (0CABAC28AH);
      CRCTable [242] := Bits32Set ( 53B39330H);
      CRCTable [243] := Bits32Set ( 24B4A3A6H);
      CRCTable [244] := Bits32Set (0BAD03605H);
      CRCTable [245] := Bits32Set (0CDD70693H);
      CRCTable [246] := Bits32Set ( 54DE5729H);
      CRCTable [247] := Bits32Set ( 23D967BFH);
      CRCTable [248] := Bits32Set (0B3667A2EH);
      CRCTable [249] := Bits32Set (0C4614AB8H);
      CRCTable [250] := Bits32Set ( 5D681B02H);
      CRCTable [251] := Bits32Set ( 2A6F2B94H);
      CRCTable [252] := Bits32Set (0B40BBE37H);
      CRCTable [253] := Bits32Set (0C30C8EA1H);
      CRCTable [254] := Bits32Set ( 5A05DF1BH);
      CRCTable [255] := Bits32Set ( 2D02EF8DH);
    END InitCRCTable;

    PROCEDURE SchreibeErgebnis (crc: Bits32;
                                dezimal: BOOLEAN);

      VAR
        s: ARRAY [0..15] OF CHAR;
        i: [0..3];

      PROCEDURE SchreibeNibble (Nibble: CARDINAL);

      BEGIN
        IF Nibble < 10 THEN
          WriteChar (CHR (Nibble + ORD ("0")))
        ELSE
          WriteChar (CHR (Nibble - 10 + ORD ("A")))
        END
      END SchreibeNibble;

    BEGIN (* SchreibeErgebnis *)
      IF dezimal THEN
        LongIntToString (LONGINT (crc.b), s, 0);
        WriteLine (s)
      ELSE
        FOR i := 3 TO 0 BY -1 DO
          SchreibeNibble (ORD (CHAR (crc.a [i])) DIV 16);
          SchreibeNibble (ORD (CHAR (crc.a [i])) MOD 16)
        END;
        WriteChar (EOL)
      END
    END SchreibeErgebnis;

  BEGIN (* Bearbeite *)
    InitCRCTable;
    crc.b := Bits32Set {0..31};
    WHILE Read (ch) DO
      index := ORD (CHAR (crc.a [0] / Bits8Set (ch)));
      crc.a [0] := crc.a [1];
      crc.a [1] := crc.a [2];
      crc.a [2] := crc.a [3];
      crc.a [3] := Bits8Set {};
      crc.b := crc.b / CRCTable [index]
    END;
    crc.b := Bits32Set {0..31} / crc.b;
    SchreibeErgebnis (crc, dezimal)
  END Bearbeite;

  VAR dezimal, ShowHelp: BOOLEAN;

BEGIN
  LiesParameter (dezimal, ShowHelp);
  IF ShowHelp THEN
    Help
  ELSE
    Bearbeite (dezimal)
  END
END CRC.
