{


 Visionix Advanced Power Management (VAPM) Unit
   Version 0.7
 Copyright 1991,92,93 Visionix
 ALL RIGHTS RESERVED



 Revision history in reverse chronological order:

 Initials  Date      Comment
 --------  --------  -------------------------------------------------------

 jrt       10/27/93  Renamed from VAPM to VAPMU for beta 0.30

 mep       03/26/93  Rewritten and added many new functions.

 lpg       03/15/93  Added Source Documentation

 mep       02/11/93  Cleaned up code for beta release

 jrt       02/08/93  Sync with beta 0.12 release

 lpg       01/22/93  Wrote VAPMPresent and VPAMGetPowerStatus in ASM

 jrt       12/07/92  Sync with beta 0.11 release

 jrt       12/07/92  First logged revision.


}

(*-

[SECTION: Section 3: The Operating System Services Libraries]
[CHAPTER: Chapter 1: The Advanced Power Management Unit]

[TEXT]

<Overview>

This unit is an interface to the DOS Advanced Power Management functions.

For more information, see the DOS Avanced Power Management
specification which is available from Intel.

The documentation for this unit will be enhanced in the next BETA release.

<Interface>

-*)


Unit VAPMU;

Interface

Uses

  VTypesu,
  DOS,
  VGenu;

{}

{}


Const

  {-----------------------------}
  { Generic error for functions }
  {-----------------------------}

  apmError = $C8;

  {------------------------------------------------}
  { Installation Flags - use with VAPMInstallCheck }
  {------------------------------------------------}

  apmif16PModeAllowed     = 0; { 16-bit protected mode interface supported }
  apmif32PModeAllowed     = 1; { 32-bit protected mode interface supported }
  apmifCPUIdleSlower      = 2; { CPU idle decreases processor speed        }
  apmifBIOSPowerManageOff = 3; { BIOS power management disabled            }

  {-----------------------------------------}
  { Device IDs - use with VAPMSetPowerState }
  {-----------------------------------------}

  apmidSysBIOS                = $0000;
  apmidAllSysBIOSDevices      = $0001;
  apmidDisplay                = $0100; { [00..FE] devices allowed }
  apmidAllDisplay             = $01FF;
  apmidSecStorage             = $0200; { [00..FE] devices allowed }
  apmidSecStorageDevices      = $02FF;
  apmidParallelPort           = $0300; { [00..FE] devices allowed }
  apmidAllParallelPortDevices = $03FF;
  apmidSerialPort             = $0400; { [00..FE] devices allowed }
  apmidAllSerialPortDevices   = $04FF;

  {--------------------------------------------------}
  { Set Power State IDs - use with VAPMSetPowerState }
  {--------------------------------------------------}

  apmpsReady   = $0;
  apmpsStandBy = $1;
  apmpsSuspend = $2;
  apmpsOff     = $3;


Type

  TAPMError = BYTE;

{}

Function  VAPMErrorToStr(              ErrorCode      : TAPMError): STRING;

Function  VAPMInstalled                                           : BOOLEAN;

Procedure VAPMInstallCheck(        Var Version        : STRING;
                                   Var Flags          : WORD;
                                   Var ErrorCode      : TAPMError);

Function  VAPMConRealModeIntr                                     : TAPMError;

Function  VAPMCon16PModeInter(     Var RM16CodeSeg    : WORD;
                                   Var EntryOfs       : WORD;
                                   Var RM16DataSeg    : WORD    ) : TAPMError;

Function  VAPMCon32PModeInter(     Var RM32CodeSeg    : WORD;
                                   Var EntryOfs       : LONGINT;
                                   Var RM16CodeSeg    : WORD;
                                   Var RM16DataSeg    : WORD    ) : TAPMError;

Function  VAPMDisConInter                                         : TAPMError;

Function  VAPMCPUIdle                                             : TAPMError;

Function  VAPMCPUBusy                                             : TAPMError;

Function  VAPMSetPowerState(           DeviceID       : WORD;
                                       State          : WORD    ) : TAPMError;

Function  VAPMSysStandby                                          : TAPMError;

Function  VAPMSuspendSys                                          : TAPMError;

Function  VAPMSetPowerManager(         OnOff          : BOOLEAN ) : TAPMError;

Function  VAPMResetAsPowerOn                                      : TAPMError;

Function  VAPMGetPowerStatus(      Var ACLineStatus   : WORD;
                                   Var BatteryStatus  : WORD;
                                   Var BatteryPercent : WORD    ) : TAPMError;

Function  VAPMGetACLineStatus                                     : WORD;

Function  VAPMGetBatteryStatus                                    : WORD;

Function  VAPMGetBatteryPercent                                   : WORD;

Function  VAPMGetACLineStatusText                                 : STRING;

Function  VAPMGetBatteryStatusText                                : STRING;

Function  VAPMGetPowerEvent(       Var Event          : WORD    ) : TAPMError;

{}

Implementation

{}

(*-

[FUNCTION]

Function  VAPMErrorToStr(              ErrorCode      : TAPMError ) : STRING;

[PARAMETERS]

ErrorCode   APM Error code

[RETURNS]

String

[DESCRIPTION]

Converts APM errorcodes to string format.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMErrorToStr(              ErrorCode      : TAPMError ) : STRING;

Var

  S : STRING;

BEGIN

  S := '';

  Case ErrorCode of

    $01 : S := 'Power management functionality disabled';
    $02 : S := 'Interface connection already in effect';
    $03 : S := 'Interface not connected';
    $04 : S := 'Real-mode interface not connected';
    $05 : S := '16-bit protected-mode already connected';
    $06 : S := '16-bit protected-mode not supported';
    $07 : S := '32-bit protected-mode already connected';
    $08 : S := '32-bit protected-mode not supported';
    $09 : S := 'Unrecognized device ID';
    $0A : S := 'Invalid value for state parameter';
    $60 : S := 'Can not enter requested state';
    $80 : S := 'No power management events pending';
    $86 : S := 'APM not present';
    $87..$9F : S := 'Reserved for other power management event errors';
    $0B..$1F : S := 'Reserved for other interface and general errors';
    $20..$3F : S := 'Reserved for CPU errors';
    $40..$5F : S := 'Reserved for device errors';
    $61..$7F : S := 'Reserved for other system errors';
    $81..$85 : S := 'Reserved for other power management event errors';

  End;

  VAPMErrorToStr := S;

END;

{}

(*-

[FUNCTION]

Function  VAPMInstalled                                         : BOOLEAN;

[PARAMETERS]

(none)

[RETURNS]

TRUE  if an APM manager can be found,
FALSE if an APM manager can NOT be found.

[DESCRIPTION]

Checks to see if an APM manager is installed.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMInstalled                                         : BOOLEAN;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5300;
  R.BX := $0000;
  R.DS := 0;
  R.ES := 0;

  Intr( $15, R );

  VAPMInstalled := ( R.AH <> $86 );

END;  { VAPMInstalled }

{}

(*-

[FUNCTION]

Procedure VAPMInstallCheck(        Var Version        : STRING;
                                   Var Flags          : WORD;
                                   Var ErrorCode      : TAPMError );

[PARAMETERS]

(none)

[RETURNS]

Version     Version of APM.
Flags       Support flags of APM (see Installation Flags)
ErrorCode   Condition of APM (active or not present).

[DESCRIPTION]

This function returns information about the APM manager.

The flags field will return any combination of the following values:

  apmif16PModeAllowed     = 0; { 16-bit protected mode interface supported }
  apmif32PModeAllowed     = 1; { 32-bit protected mode interface supported }
  apmifCPUIdleSlower      = 2; { CPU idle decreases processor speed        }
  apmifBIOSPowerManageOff = 3; { BIOS power management disabled            }

[SEE-ALSO]

[EXAMPLE]

-*)

Procedure VAPMInstallCheck(        Var Version        : STRING;
                                   Var Flags          : WORD;
                                   Var ErrorCode      : TAPMError );

Var

  R : REGISTERS;

BEGIN

  R.AX := $5300;
  R.BX := $0000;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    ErrorCode := R.AH;

  END
  Else
  BEGIN

    Version   := IntToStr(BCDtoDec(R.AH)) + '.' +
                 IntToStr(BCDtoDec(R.AL));
    Flags     := R.CX;
    ErrorCode := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMConRealModeIntr                                     : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

Condition of interface.

[DESCRIPTION]

Makes a connection to the real-mode interface. Checks using the device ID of
the system BIOS.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMConRealModeIntr                                     : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5301;
  R.BX := $0000;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMConRealModeIntr := R.AH;

  END
  Else
  BEGIN

    VAPMConRealModeIntr := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMCon16PModeInter(     Var RM16CodeSeg    : WORD;
                                   Var EntryOfs       : WORD;
                                   Var RM16DataSeg    : WORD    ) : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

RM16CodeSeg  Real-mode segment base address of the protected-mode 16-bit code
             segment.
EntryOfs     Offset of entry point.
RM16DataSeg  Real-mode segment base address of the protected-mode 16-bit data
             segment.

[DESCRIPTION]

Makes a connection to the 16-bit protected-mode interface.  The caller must
initialize two consecutive descriptors with the returned segment base
addresses; these descriptors must be valid whenever the protected-mode
interface is called, and will have their limits arbitrarily set to 64K.

The protected mode interface is invoked by making a far call with the same
register values as for INT 15; it must be invoked while CPL=0, the code
segment descriptor must have a DPL of 0, the stack must be in a 16-bit
segment and have enough room for BIOS use and possible interrupts, and the
current I/O permission bit map must allow access to the I/O ports used for
power management.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMCon16PModeInter(     Var RM16CodeSeg    : WORD;
                                   Var EntryOfs       : WORD;
                                   Var RM16DataSeg    : WORD    ) : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5302;
  R.BX := $0000;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMCon16PModeInter := R.AH;

  END
  Else
  BEGIN

    VAPMCon16PModeInter := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMCon32PModeInter(     Var RM32CodeSeg    : WORD;
                                   Var EntryOfs       : LONGINT;
                                   Var RM16CodeSeg    : WORD;
                                   Var RM16DataSeg    : WORD    ) : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

RM32CodeSeg  Real-mode segment base address of protected-mode 32-bit code
             segment.
EntryOfs     Offset of entry point.
RM16CodeSeg  Real-mode segment base address of protected-mode 16-bit code
             segment.
RM16DataSeg  Real-mode segment base address of protected-mode 16-bit data
             segment.

[DESCRIPTION]

Makes a connection to the 32-bit protected-mode interface.  The caller must
initialize three consecutive descriptors with the returned segment base
addresses for 32-bit code, 16-bit code, and 16-bit data, respectively; these
descriptors must be valid whenever the protected-mode interface is called,
and will have their limits arbitrarily set to 64K.

The protected mode interface is invoked by making a far call to the 32-bit
code segment with the same register values as for INT 15; it must be invoked
while CPL=0, the code segment descriptor must have a DPL of 0, the stack must
be in a 32-bit segment and have enough room for BIOS use and possible
interrupts, and the current I/O permission bit map must allow access to the
I/O ports used for power management.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMCon32PModeInter(     Var RM32CodeSeg    : WORD;
                                   Var EntryOfs       : LONGINT;
                                   Var RM16CodeSeg    : WORD;
                                   Var RM16DataSeg    : WORD    ) : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5303;
  R.BX := $0000;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMCon32PModeInter := R.AH;

  END
  Else
  BEGIN

    VAPMCon32PModeInter := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMDisConInter                                         : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

Condition of APM connection.

[DESCRIPTION]

Disconnects the installed APM interface using the device ID of the system
BIOS.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMDisConInter                                         : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5304;
  R.BX := $0000;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMDisConInter := R.AH;

  END
  Else
  BEGIN

    VAPMDisConInter := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMCPUIdle                                             : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

(clear)

[DESCRIPTION]

Places the CPU into idle state.  Call this when the system is idle and
should be suspended until the next system event or interrupt.  Also, if an
interrupt causes the system to resume normal processing, the interrupt may
or may not have been handled when the BIOS returns from this call; thus,
the caller should allow interrupts on return.

Interrupt handlers may not retain control if the BIOS allows interrupts while
in idle mode even if they are able to determine that they were called from
idle mode.  The caller should issue this call continuously in a loop until
it needs to perform some processing of its own.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMCPUIdle                                             : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5305;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  VAPMCPUIdle := 0;

END;

{}

(*-

[FUNCTION]

Function  VAPMCPUBusy                                             : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

(clear)

[DESCRIPTION]

Places the CPU into a busy state.  This is called to ensure that the system
runs at full speed even on systems where the BIOS is unable to recognize
increased activity (especially if interrupts are hooked by other programs and
not chained to the BIOS).  This call may be made even when the system is
already running at full speed, but it will create unnecessary overhead.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMCPUBusy                                             : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5306;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  VAPMCPUBusy := 0;

END;

{}

(*-

[FUNCTION]

Function  VAPMSetPowerState(           DeviceID       : WORD;
                                       State          : WORD    ) : TAPMError;

[PARAMETERS]

DeviceID     Device ID to apply new power state (see Device IDs).
State        New power state for device (see Set Power State IDs).

[RETURNS]

Result of function.

[DESCRIPTION]

Sets the system state for a specific device.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMSetPowerState(           DeviceID       : WORD;
                                       State          : WORD    ) : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5307;
  R.BX := DeviceID;
  R.CX := State;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMSetPowerState := R.AH;

  END
  Else
  BEGIN

    VAPMSetPowerState := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMSysStandby                                          : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

(clear)

[DESCRIPTION]

Puts the entire system into stand-by mode; normally called in response to a
System Stand-by Request notification after any necessary processing, but may
also be invoked at the caller's discretion.

The stand-by state is typically exited on an interrupt.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMSysStandby                                          : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5307;
  R.BX := $0001;
  R.CX := $0001;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  VAPMSysStandby := 0;

END;

{}

(*-

[FUNCTION]

Function  VAPMSuspendSys                                          : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

(clear)

[DESCRIPTION]

Puts the entire system into a low-power suspended state; normally called in
response to a Suspend System Request notification after any necessary
processing, but may also be invoked at the caller's discretion.  The caller
may need to update its date and time values because the system could have
been suspended for a long period of time.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMSuspendSys                                          : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5307;
  R.BX := $0001;
  R.CX := $0002;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  VAPMSuspendSys := 0;

END;

{}

(*-

[FUNCTION]

Function  VAPMSetPowerManager(         OnOff          : BOOLEAN ) : TAPMError;

[PARAMETERS]

OnOff        Condition to put APM into.

[RETURNS]

APM errorcode.

[DESCRIPTION]

When power management is disabled, the system BIOS will not automatically
power down devices, enter stand-by or suspended mode, or perform any
power-saving actions in response to CPU idle calls.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMSetPowerManager(         OnOff          : BOOLEAN ) : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5308;
  R.BX := $FFFF;
  R.CX := Byte(OnOff);
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMSetPowerManager := R.AH;

  END
  Else
  BEGIN

    VAPMSetPowerManager := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMResetAsPowerOn                                      : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

Condition of function.

[DESCRIPTION]

Restores power-on defaults for the APM.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMResetAsPowerOn                                      : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $5309;
  R.BX := $FFFF;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMResetAsPowerOn := R.AH;

  END
  Else
  BEGIN

    VAPMResetAsPowerOn := 0;

  END;

END;

{}

(*-

[FUNCTION]

Function  VAPMGetPowerStatus(      Var ACLineStatus   : WORD;
                                   Var BatteryStatus  : WORD;
                                   Var BatteryPercent : WORD    ) : TAPMError;

[PARAMETERS]

(none)

[RETURNS]

ACLineStatus   Status of A/C power supply.
BatteryStatus  Status of the battery supply.
BatteryPercent Percentage of remaining battery life.

[DESCRIPTION]

Gets status of power supply from APM.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMGetPowerStatus(      Var ACLineStatus   : WORD;
                                   Var BatteryStatus  : WORD;
                                   Var BatteryPercent : WORD    ) : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $530A;
  R.BX := $0001;
  R.DS := 0;
  R.ES := 0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMGetPowerStatus := R.AH;

  END
  Else
  BEGIN

    ACLineStatus       := R.BH;
    BatteryStatus      := R.BL;
    BatteryPercent     := R.CX;
    VAPMGetPowerStatus := 0;

  END;

END;  { VAPMGetPowerStatus }

{}

(*-

[FUNCTION]

Function VAPMGetACLineStatus                                  : WORD;

[PARAMETERS]

(none)

[RETURNS]

Status of A/C line.

[DESCRIPTION]

Returns condition of A/C line power supply.

[SEE-ALSO]

[EXAMPLE]

-*)

Function VAPMGetACLineStatus                                  : WORD;

Var

  ACStat  : WORD;
  BatStat : WORD;
  BatPer  : WORD;
  Err     : TAPMError;

BEGIN

  Err := VAPMGetPowerStatus( ACStat, BatStat, BatPer );
  If (Err <> 0) Then
    VAPMGetACLineStatus := apmError
  Else
    VAPMGetACLineStatus := ACStat;

END;  { VAPMGetACLineStatus }

{}

(*-

[FUNCTION]

Function VAPMGetBatteryStatus                                 : WORD;

[PARAMETERS]

(none)

[RETURNS]

Status of Battery.

[DESCRIPTION]

Returns condition of battery supply.

[SEE-ALSO]

[EXAMPLE]

-*)

Function VAPMGetBatteryStatus                                 : WORD;

Var

  ACStat  : WORD;
  BatStat : WORD;
  BatPer  : WORD;
  Err     : TAPMError;

BEGIN

  Err := VAPMGetPowerStatus( ACStat, BatStat, BatPer );
  If (Err <> 0) Then
    VAPMGetBatteryStatus := apmError
  Else
    VAPMGetBatteryStatus := BatStat;

END;  { VAPMGetBatteryStatus }

{}

(*-

[FUNCTION]

Function VAPMGetBatteryPercent                                : WORD;

[PARAMETERS]

(none)

[RETURNS]

Percentage of battery.

[DESCRIPTION]

Returns the percentage of remaining battery life.

[SEE-ALSO]

[EXAMPLE]

-*)

Function VAPMGetBatteryPercent                                : WORD;

Var

  ACStat  : WORD;
  BatStat : WORD;
  BatPer  : WORD;
  Err     : TAPMError;

BEGIN

  Err := VAPMGetPowerStatus( ACStat, BatStat, BatPer );
  If (Err <> 0) Then
    VAPMGetBatteryPercent := apmError
  Else
    VAPMGetBatteryPercent := BatPer;

END;  { VAPMGetBatteryPower }

{}

(*-

[FUNCTION]

[PARAMETERS]

[RETURNS]

[DESCRIPTION]

[SEE-ALSO]

[EXAMPLE]

-*)

Function VAPMGetACLineStatusText                              : STRING;

Var

  S : STRING;

BEGIN

  S := '';

  Case VAPMGetACLineStatus of

    $00 : S := 'Off-line (running from battery)';
    $01 : S := 'On-line  (running from AC line)';
    $FF : S := '(unknown)';

  ELSE

    S := '(error/unknown)';

  END;  { Case VAPMGetACLineStatus }

  VAPMGetACLineStatusText := S;

END;  { VAPMGetACLineStatusText }

{}

(*-

[FUNCTION]

[PARAMETERS]

[RETURNS]

[DESCRIPTION]

[SEE-ALSO]

[EXAMPLE]

-*)

Function VAPMGetBatteryStatusText                             : STRING;

Var

  S : STRING;

BEGIN

  S := '';

  Case VAPMGetACLineStatus of

    $00 : S := 'High';
    $01 : S := 'Low';
    $02 : S := 'Critical';
    $03 : S := 'Charging';
    $FF : S := '(unknown)';

  ELSE

    S := '(error/unknown)';

  END;  { Case VAPMGetACLineStatus }

  VAPMGetBatteryStatusText := S;

END;  { VAPMGetBatteryStatusText }

{}

(*-

[FUNCTION]

[PARAMETERS]

[RETURNS]

[DESCRIPTION]

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VAPMGetPowerEvent(       Var Event          : WORD    ) : TAPMError;

Var

  R : REGISTERS;

BEGIN

  R.AX := $530B;
  R.DS := $0;
  R.ES := $0;

  Intr( $15, R );

  If (R.Flags AND FCarry <> 0) Then
  BEGIN

    VAPMGetPowerEvent := R.AH;

  END
  Else
  BEGIN

    Event             := R.BX;
    VAPMGetPowerEvent := 0;

  END;

END;

{}

(*-

[FUNCTION]

[PARAMETERS]

[RETURNS]

[DESCRIPTION]

[SEE-ALSO]

[EXAMPLE]

-*)

Function VAPMGetPowerEventText                                : STRING;

Var

  S     : STRING;
  Event : WORD;
  Err   : TAPMError;

BEGIN

  S   := '';
  Err := VAPMGetPowerEvent(Event);

  If (Err = 0) Then
  BEGIN

    Case Event of

      $01 : S := 'System stand-by request';
      $02 : S := 'System suspend request';
      $03 : S := 'Normal resume system notification';
      $04 : S := 'Critical resume system notification';
      $05 : S := 'Battery low notification';

    Else

      S := '(unknown)';

    END;

  END
  Else
    S := '(error)';

  VAPMGetPowerEventText := S;

END;  { VAPMGetPowerEventText }

{}
{}
{}

BEGIN
END.
