{
 

 Visionix CD-ROM Audio Library (VCDROM)
   Version 0.6
 Copyright 1991,92,93 Visionix
 ALL RIGHTS RESERVED

 

 Revision history in reverse chronological order:

 Initials  Date      Comment
     

 jrt       10/27/93  renamed from VCDROM to VCDROMU for beta 0.30

 rob       09/20/93  Changed code to use MSCDEX.
                     Added VCD_GetNumDrives
                           VCD_GetFirstDrive
                           VCD_LoadMSF

 jrt       09/20/93  Changed TMSF to TFSM.

 lpg       03/15/93  Added Source Documentation

 mep       02/11/93  Cleaned up code for beta release

 jrt       02/08/93  Sync with beta 0.12 release

 jrt       12/07/92  Sync with beta 0.11 release

 jrt       11/21/92  Sync with beta 0.08

 jrt       09/15/92  First logged revision.

 

 INCOMPLETE FUNCTIONS
 --------------------

   VCD_GetCaps
   VCD_Close
   VCD_GetDriveCaps
}


(*-

[SECTION: Section 3: Operating System Services Libraries
[CHAPTER: Chapter 4: The Generic CD-ROM Unit]

[TEXT]

<Overview>

VCDROMu is a generic CD-ROM unit which currently works on MSCDEX,
but which will be enhanced in the next VisionTools release to
also support direct control of SCSI-2 CD-ROM drives without
requiring MSCDEX.

VCDROMus is handle based.  Before you call any VCDROMu functions,
you should call VCD_Open.  VCD_Open takes two paramaters: "flags"
which are currently unsed, and a "method list string" which tells
VCDROMu which CD-ROM methods you wish to use.  Currently, only
MSCDEX is supported, so the method list string must be 'MSCDEX'.

VCD_Open will return a handle which must be used with subsequent
calls to VCDROMu functions.  If the handle is NIL, the CD-ROM
unit could not be opened successfully.  This is typically because
the specified method (in this case, MSCDEX) is not present.

<CD-ROM Addressing types>

CD-ROM devices support two addressing types: logical block addressing
and Minute/Second/Frame addressing.

Logical block addresses specify a logical block position on the CD-ROM
media.  CD-ROM logical blocks are 2048 bytes in size.

Minute/Second/Frame addresses specify a minute/second/frame offset
from either the beggining of the disk or the beggining of the current
track.  A frame is 1/75 of a second.  A frame is also 2048 bytes in
size, making a frame the same as a logical block.

<Interface>

-*)


Unit VCDROMu;

Interface

Uses

  vmscdexu,
  VTypesu,
  VGenu,
  DOS;

{}

Const

  cCdMethodMSCDEX = 1;
  cCdMethodSCSI2  = 2;

Type

  TError = WORD;

  {-----}

  TCDLibCaps = RECORD

    A : BYTE;

  END;

  PCDLibCaps = ^TCDLibCaps;

  {-----}

  TCDDriveCaps = RECORD

    Caps         : LONGINT;

  END;

  CONST

    cDoorClosed                = $001;
    cDoorLocked                = $002;
    cSupportsCookedAndRaw      = $004;
    cSupportsWrite             = $008;
    cSupportsAudio             = $010;
    cSupportsISO9660           = $020;
    cSupportsPrefetching       = $040;
    cSupportsAudioChanMani     = $080;
    cSupportsHSGAndMSF         = $100;
    cReserved                  = $200;
    cDiscInDrive               = $400;


TYPE


  PCDDriveCaps = ^TCDDriveCaps;

  {-----}

  TCDDriveInfo = RECORD

    Method        : WORD;

    MethodBus     : WORD;
    MethodUnit    : WORD;

  END;

  {-----}

  TCDInstanceData = RECORD

    NumDrives : BYTE;

    DriveInfo : Array[1..8] of TCDDriveInfo;

  END;

  PCDInstanceData = ^TCDInstanceData;

  {-----}

  TCDHandle = PCDInstanceData;

  {-----}

  TCDRequestHeader = RECORD

    HeaderSize     : BYTE;
    Subunit        : BYTE;
    CommandCode    : BYTE;
    Status         : WORD;
    Reserved       : Array[1..8] of BYTE;

  END;



{}

{----------------}
{ Math Functions }
{----------------}

Procedure VCD_AddMSF(             Value1         : TFSM;
                                  Value2         : TFSM;
                              Var Result         : TFSM      );

Procedure VCD_SubMSF(             Value1         : TFSM;
                                  Value2         : TFSM;
                              Var Result         : TFSM      );

Function  VCD_MSFtoBlock(         MSF            : TFSM      ) : LONGINT;

Procedure VCD_BlockToMSF(         Block          : LONGINT;
                              Var MSF            : TFSM      );

Function  VCD_MSFtoTotalFrames(   MSF            : TFSM      ) : LONGINT;

Procedure VCD_TotalFramesToMSF(   TotalFrames    : LONGINT;
                              Var MSF            : TFSM      );

Procedure VCD_LoadMSF(        Var MSF            : TFSM;
                                  M              : BYTE;
                                  S              : BYTE;
                                  F              : BYTE      );

Function  VCD_MSFToStr(           MSF            : TFSM      ) : STRING;


{--------------------------}
{ Library/System functions }
{--------------------------}

Procedure VCD_GetCaps(            Caps           : PCDLibCaps );

Function  VCD_Open(               Flags          : WORD;
                                  MethodListStr  : ST80      ) : TCDHandle;

Procedure VCD_Close(              Handle         : TCDHandle );

Function  VCD_GetDriveStatus(     Handle         : TCDHandle;
                                  Drive          : WORD         ) : LONGINT;

Function  VCD_GetNumDrives(       Handle         : TCDHandle ) : WORD;


Function  VCD_GetDriveMethod(     Handle         : TCDHandle;
                                  Drive          : WORD      ) : WORD;


{--------------------}
{ VCD ASCII Name Get }
{--------------------}

Function  VCD_GetMethodNames(     Handle         : TCDHandle;
                                  Flags          : WORD        ):ST80;

Function  VCD_GetDriveName(       Handle         : TCDHandle;
                                  Flags          : WORD;
                                  Drive          : WORD        ):ST80;


{--------------------}
{ CD Audio Functions }
{--------------------}

Function  VCD_GetAudioDiskInfo(   Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var FirstTrack     : BYTE;
                              Var LastTrack      : BYTE;
                              Var StartLeadOut   : TFSM      ) : TError;

Function  VCD_GetAudioTrackInfo(  Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Track          : BYTE;
                              Var Start          : TFSM;
                              Var TrackConInfo   : BYTE      ) : TError;

Function  VCD_GetAudioQChanInfo(  Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var ConAdr         : BYTE;
                              Var TrackNum       : BYTE;
                              Var IndexNum       : BYTE;
                              Var TrkRelTime     : TFSM;
                              Var DskRelTime     : TFSM      ) : TError;

Function  VCD_GetAudioStatusInfo( Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var AudioStatus    : WORD;
                              Var PlayStart      : TFSM;
                              Var PlayEnd        : TFSM      ) : TError;

Function  VCD_PlayMSFfor(         Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Start          : TFSM;
                                  PlayFor        : TFSM      ) : TError;

Function  VCD_PlayMSFto(          Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Start          : TFSM;
                                  PlayTo         : TFSM      ) : TError;

Function  VCD_Pause(              Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

Function  VCD_Stop(               Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

Function  VCD_Resume(             Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;



{---------------------------}
{ High-Level PLAY Functions }
{---------------------------}

(*
Function  VCD_PlayTracks(         Handle         : TCDHandle;
                                  Drive          : WORD;
                                  FirstTrack     : BYTE;
                                  NumTracks      : BYTE      ) : TError;
*)

Function  VCD_PlayNextTrack(      Handle         : TCDHandle;
                                  Drive          : WORD       ) : TError;

Function  VCD_PlayPrevTrack(      Handle         : TCDHandle;
                                  Drive          : WORD       ) : TError;


Function  VCD_PlayForwardMSF(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveForward    : TFSM       ) : TError;

Function  VCD_PlayReverseMSF(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveReverse    : TFSM       ) : TError;

(*
Function  VCD_PlayForwardLBA(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveForward    : LONGINT    ) : TError;


Function  VCD_PlayReverseLBA(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveReverse    : LONGINT    ) : TError;
*)




{}

Implementation

{}

(*-

[FUNCTION]

Procedure VCD_AddMSF(             Value1         : TFSM;
                                  Value2         : TFSM;
                              Var Result         : TFSM      );

[PARAMETERS]

Value1      1st Min/Sec/Frame Value
Value2      2nd Min/Sec/Frame Value
Result      VAR Returned Min/Sec/Frame Sum

[RETURNS]

(Function : None)
(VAR      : [Result] Min/Sec/Frame Sum)

[DESCRIPTION]

Adds the 1st Min/Sec/Frame Value to the 2nd Min/Sec/Frame Value to
produce the a new Min/Sec/Frame Value.

Adds two Min/sec/frame values and puts the result in "Result".


[SEE-ALSO]

VCD_SubMSF

[EXAMPLE]

-*)

Procedure VCD_AddMSF(             Value1         : TFSM;
                                  Value2         : TFSM;
                              Var Result         : TFSM      );

Var
  Val1Frames : LONGINT;
  Val2Frames : LONGINT;
  Z          : INTEGER;

BEGIN

  Val1Frames := VCD_MSFtoTotalFrames( Value1 );
  Val2Frames := VCD_MSFtoTotalFrames( Value2 );

  Inc( Val1Frames, Val2Frames );

  VCD_TotalFramesToMSF( Val1Frames, Result );

END;  { VCD_AddMSF }

{}

(*-

[FUNCTION]

Procedure VCD_SubMSF(             Value1         : TFSM;
                                  Value2         : TFSM;
                              Var Result         : TFSM      );

[PARAMETERS]

Value1      1st Min/Sec/Frame Value
Value2      2nd Min/Sec/Frame Value
Result      VAR Returned Min/Sec/Frame Difference

[RETURNS]

(Function : None)
(VAR      : [Result] Min/Sec/Frame Difference)

[DESCRIPTION]

Subtracts the 2nd Min/Sec/Frame Value from the 1st Min/Sec/Frame Value
to produce the Difference (a new Min/Sec/Frame Value).

Subs MSF "Value2" from MSF "value1" and puts result in "Result".

[SEE-ALSO]

VCD_AddMSF

[EXAMPLE]

-*)

Procedure VCD_SubMSF(             Value1         : TFSM;
                                  Value2         : TFSM;
                              Var Result         : TFSM      );

Var
  Val1Frames : LONGINT;
  Val2Frames : LONGINT;
  Z          : INTEGER;

BEGIN

  Val1Frames := VCD_MSFtoTotalFrames( Value1 );
  Val2Frames := VCD_MSFtoTotalFrames( Value2 );

  Dec( Val1Frames, Val2Frames );

  VCD_TotalFramesToMSF( Val1Frames, Result );

END;  { VCD_SubMSF }

{}

(*-

[FUNCTION]

Function  VCD_MSFtoBlock(         MSF            : TFSM      ) : LONGINT;

[PARAMETERS]

MSF         Source Min/Sec/Frames Value

[RETURNS]

Total Blocks represented by Min/Sec/Frame Value.

[DESCRIPTION]

Converts a min/sec/frame address to a logical block address.

[SEE-ALSO]


[EXAMPLE]

-*)

Function  VCD_MSFtoBlock(         MSF            : TFSM      ) : LONGINT;

BEGIN

  VCD_MSFToBlock := (  ( Longint(MSF.M) * 4500 ) +
                       ( Longint(MSF.S) * 75   ) +
                       ( Longint(MSF.F)        )   ) - 150;

END;  { VCD_MSFtoBlock }

{}

(*-

[FUNCTION]

Procedure VCD_BlockToMSF(         Block          : LONGINT;
                              Var MSF            : TFSM      );

[PARAMETERS]

Block       Source Number of Blocks
MSF         VAR Returned Min/Sec/Frame Value

[RETURNS]

(Function : None)
(VAR      : [MSF] Min/Sec/Frame Value)

[DESCRIPTION]

Converts a Logical Block Address to a Min/Sec/Frame Address.

[SEE-ALSO]

VCD_MSFtoBlock

[EXAMPLE]

-*)

Procedure VCD_BlockToMSF(         Block          : LONGINT;
                              Var MSF            : TFSM      );

BEGIN

  MSF.M := ( (Block+150) DIV 4500 );
  MSF.S := ( (Block+150) MOD 4500 ) DIV 60;
  MSF.F := ( (Block+150) MOD 60   );

END;  { VCD_BlockToMSF }

{}

(*-

[FUNCTION]

Function  VCD_MSFtoTotalFrames(   MSF            : TFSM      ) : LONGINT;

[PARAMETERS]

MSF         Min/Sec/Frame Value

[RETURNS]

Total Frames from Min/Sec/Frames Value

[DESCRIPTION]

Converts a Min/Sec/Frames Value into the Total Number of Frames it
represents.

Converts a min/sec/frame value to a count of the total frames.

[SEE-ALSO]

VCD_TotalFramesToMSF

[EXAMPLE]

-*)

Function  VCD_MSFtoTotalFrames(   MSF            : TFSM      ) : LONGINT;

BEGIN

  VCD_MSFToTotalFrames := ( Longint(MSF.M) * 4500 ) +
                          ( Longint(MSF.S) * 75   ) +
                          ( Longint(MSF.F)        );

END;  { VCD_MSFtoTotalFrames }

{}

(*-

[FUNCTION]

Procedure VCD_TotalFramesToMSF(   TotalFrames    : LONGINT;
                              Var MSF            : TFSM      );

[PARAMETERS]

TotalFrames
MSF         VAR Returned Frames converted to Min/Sec/Frames

[RETURNS]

(Function : None)
(VAR      : [MSF] Frame converted to Min/Sec/Frames)

[DESCRIPTION]

Converts a Frames Value into Min/Sec/Frames.

Converts a total frames value to a min/sec/frame value.

[SEE-ALSO]

VCD_MSFtoTotalFrames

[EXAMPLE]

-*)

Procedure VCD_TotalFramesToMSF(   TotalFrames    : LONGINT;
                              Var MSF            : TFSM      );

BEGIN

  MSF.M := (TotalFrames DIV 4500);
  MSF.S := (TotalFrames MOD 4500) DIV 75;
  MSF.F := (TotalFrames MOD 75);

END;  { VCD_TotalFramesToMSF }

{}

(*-

[FUNCTION]

Procedure VCD_LoadMSF(        Var MSF            : TFSM;
                                  M              : BYTE;
                                  S              : BYTE;
                                  F              : BYTE      );

[PARAMETERS]

MSF         VAR Returned Min/Sec/Frame Value
M           Source Minutes
S           Source Seconds
F           Source Frames

[RETURNS]

(Function : None)
(VAR      : [MSF] Min/Sec/Frame Value)

[DESCRIPTION]

Converts time Values into Min/Sec/Frames

Loads a min/sec/frame record with values.

[SEE-ALSO]

[EXAMPLE]

-*)

Procedure VCD_LoadMSF(        Var MSF            : TFSM;
                                  M              : BYTE;
                                  S              : BYTE;
                                  F              : BYTE      );

BEGIN

   MSF.M := M;
   MSF.S := S;
   MSF.F := F;
   MSF.P := 0;

END;  { VCD_LoadMSF }

{}


(*-

[FUNCTION]

[PARAMETERS]

MSF       min/sec/frame to convert to a string

[RETURNS]

string representation of the MSF in the form m:s.f

[DESCRIPTION]

Converts a min/sec/frame record into at string.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_MSFToStr(           MSF            : TFSM      ) : STRING;

BEGIN

  VCD_MSFToStr := IntToStr(MSF.M) + ':' +
                  IntToStr(MSF.S) + '.' +
                  IntToStr(MSF.F);

END;

{}

(*-

[FUNCTION]

Procedure VCD_GetCaps(            Caps           : PCDLibCaps );

[PARAMETERS]

Caps        Pointer to ?

[RETURNS]

(Function : None)
(Ptr      : [Caps] Pointer to ?)

[DESCRIPTION]

Gets the capabilities of the CD-rom lib in the current environment.

[SEE-ALSO]

[EXAMPLE]

-*)

Procedure VCD_GetCaps(            Caps           : PCDLibCaps );

BEGIN

END;  { VCD_GetCaps }

{}

(*-

[FUNCTION]

Function  VCD_CheckForMethod(     Method         : BYTE      ) : BOOLEAN;

[PARAMETERS]

Method      ?

[RETURNS]

[DESCRIPTION]

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_CheckForMethod(     Method         : BYTE      ) : BOOLEAN;
Var
   MSCDEXVerNumber   : Word;

BEGIN

   Case Method of
      ccdmethodMSCDEX :
      Begin
         asm
            mov bx,0
            mov ax,$150c
            int $2f
            mov MSCDEXVerNumber,bx
         end;
         If MSCDEXVerNumber < $200 then
            VCD_CheckForMethod := False
         Else
            VCD_CheckForMethod := True;

      End;
      ccdMethodSCSI2  : VCD_CheckForMethod := False;  {not supported }
   else
      VCD_CheckForMethod := False;
   End;

END;  { VCD_CheckForMethod }

{}

(*-

[FUNCTION]

Function  VCD_Open(               Flags          : WORD;
                                  MethodListStr  : ST80      ) : TCDHandle;

[PARAMETERS]

Flags         ?
MethodListStr ?

[RETURNS]

[DESCRIPTION]

Opens an instance of the cd-rom library.  "MethodListStr" should
currently be 'MSCDEX'.  "Flags" should be 0.  Returns a Non-0 (NIL)
handle if successfull, 0/NIL if failure.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_Open(               Flags          : WORD;
                                  MethodListStr  : ST80      ) : TCDHandle;

Var
  Method  : BYTE;
  NCDI    : PCDInstanceData;


  {}

  Function GetNextMethod : BYTE;

  Var
    NMLen     : BYTE;
    MethodStr : STRING;

  BEGIN

    {------------------------}
    { Get next method string }
    {------------------------}

    NMLen := Pos( ',', MethodListStr );

    If NMLen <> 0 Then
    BEGIN

      MethodStr := Copy( MethodListStr, 1, NMLen-1 );
      Delete( MethodListStr, 1, NMLen );

    END  { If NMLen }

    ELSE
    BEGIN

      MethodStr     := MethodListStr;
      MethodListStr := '';

    END;  { If NMLen / Else }

    {--------------------------}
    { Convert Method into byte }
    {--------------------------}

    If MethodStr='' Then
      GetNextMethod := $FF

    Else
    If MethodStr='MSCDEX' Then
      GetNextMethod := cCDmethodMSCDEX

    Else
    If MethodStr='SCSI-2' Then
      GetNextMethod := cCDmethodSCSI2

    Else
      GetNextMethod := $FE;

  END; { GetNextMethod }

  {}

BEGIN

  {----------------------------------}
  { Loop through method string until }
  { the end, an error, or we find    }
  { a valid method.                  }
  {----------------------------------}

  Repeat

    Method := GetNextMethod;

  Until ( Method=$FF                         ) or
        ( Method=$FE                         ) or
        ( VCD_CheckForMethod( Method )=TRUE  );

  {-------------------}
  { Was it an error ? }
  {-------------------}

  If (Method=$FF) or (Method=$FE) Then
  BEGIN

    VCD_Open := NIL;

  END  { If Method }

  ELSE
  BEGIN

    {----------------------------}
    { No error.  We have found a }
    { valid method.              }
    {----------------------------}

    New( NCDI );

    NCDI^.NumDrives := 0;

    Case Method Of

      {----}

      cCDmethodMSCDEX:
      BEGIN

         NCDI^.NumDrives := NCDI^.NumDrives + Lo(MSCDEX_GetNumDrives);

         NCDI^.DriveInfo[1].MethodUnit := MSCDEX_GetFirstDrive;
         NCDI^.DriveInfo[1].Method     := ccdMethodMSCDEX;
         NCDI^.DriveInfo[1].MethodBus  := 0;

      END;  { cCDMethodMSCDEX }

      {----}

      cCDMethodSCSI2:
      BEGIN

      END;  { cCDMethodSCSI2 }

      {----}

    END; { Case Method }

    VCD_Open := NCDI;

  END; { If Method / Else }

END; { VCD_Open }

{}

(*-

[FUNCTION]

Procedure VCD_Close(              Handle         : TCDHandle );

[PARAMETERS]

Handle      ?

[RETURNS]

(None)

[DESCRIPTION]

Closes an instance of the cd-rom library.  The "Handle" should be a
handle previously obtained from a call to VCD_Open.

[SEE-ALSO]

[EXAMPLE]

-*)

Procedure VCD_Close(              Handle         : TCDHandle );

BEGIN

   If Handle <> NIL Then
      Dispose( Handle );

END;  { VCD_Close }

{}

(*-

[FUNCTION]

Procedure VCD_GetDriveCaps(       Handle         : TCDHandle;
                                  Drive          : WORD;
                                  CDDriveCaps    : PCDDriveCaps );

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
CDDriveCaps Pointer to CD-ROM Drive Capacity Data

[RETURNS]

(Function : None)
(Ptr      : [CDDriveCaps] Pointer to CD-ROM Drive Capacity Data)

[DESCRIPTION]

Gets the capabilities of a specific cd-rom drive.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetDriveStatus(     Handle         : TCDHandle;
                                  Drive          : WORD         ) : LONGINT;

Var

  Err    : TError;
  Status : LONGINT;

BEGIN


  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

      Status := 0;

      Err := MSCDEX_GetDevStatus( Handle^.DriveInfo[Drive].MethodUnit,
                                  Status                                 );

      VCD_GetDriveStatus := Status;

    END;  { cCDMethodMSCDEX: }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END;  { VCD_GetDriveStatus }

{}

(*-

[FUNCTION]

Function  VCD_GetNumDrives(       Handle         : TCDHandle ) : WORD;

[PARAMETERS]

Handle      ?

[RETURNS]

[DESCRIPTION]

Returns the number of CD-rom drives installed in the system "Handle"
should be a handle previously obtained from VCD_Open.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetNumDrives(       Handle         : TCDHandle ) : WORD;

BEGIN

   VCD_GetNumDrives := Handle^.NumDrives;

END;  { VCD_GetNumDrives }

{}

(*-

[FUNCTION]

Function  VCD_GetDriveMethod(     Handle         : TCDHandle;
                                  Drive          : WORD      ) : WORD;

[PARAMETERS]

Handle      ?

[RETURNS]

[DESCRIPTION]

Returns the method that is used by the specified drive.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetDriveMethod(     Handle         : TCDHandle;
                                  Drive          : WORD      ) : WORD;

BEGIN

  VCD_GetDriveMethod := Handle^.DriveInfo[Drive].Method;

END;  { VCD_GetFirstDrive }

{}

(*-

[FUNCTION]

Function  VCD_GetMethodNames(     Handle         : TCDHandle;
                                  Flags          : WORD        ):ST80;

[PARAMETERS]

Handle      ?

[RETURNS]

[DESCRIPTION]


[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_GetMethodNames(     Handle         : TCDHandle;
                                  Flags          : WORD        ):ST80;

VAR
   S      : ST80;
   Z      : WORD;

BEGIN

  FOR Z := 1 to Handle^.NumDrives Do

    Case Handle^.DriveInfo[Z].Method Of

      cCDMethodMSCDEX:
      BEGIN

         {If it isn't already in list add it }

         If Pos ( 'MSCDEX', S ) = 0 then
            S := S + 'MSCDEX';

      END;  { cCDMethodMSCDEX: }

      cCDMEthodSCSI2:
      BEGIN
         IF Pos ( 'SCSI-2', S ) = 0 Then
            S := S + 'SCSI-2';

      END;  { cCDMethodSCSI2 }

    END; { Case Handle^.DriveInfo[Drive].Method }

END;

{}

(*-

[FUNCTION]

Function  VCD_GetDriveName(       Handle         : TCDHandle;
                                  Flags          : WORD;
                                  Drive          : WORD        ):ST80;

[PARAMETERS]

Handle      ?

[RETURNS]

[DESCRIPTION]


[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_GetDriveName(       Handle         : TCDHandle;
                                  Flags          : WORD;
                                  Drive          : WORD        ):ST80;


BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       VCD_GetDriveName := 'UnDetermined';

    END;  { cCDMethodMSCDEX: }

    cCDMEthodSCSI2:
    BEGIN

    END;  { cCDMethodSCSI2 }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END;

{}

(*-

[FUNCTION]

Function  VCD_GetAudioDiskInfo(   Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var FirstTrack     : WORD;
                              Var LastTrack      : WORD;
                              Var StartLeadOut   : TFSM      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
FirstTrack   VAR Returned First Track Number
LastTrack    VAR Returned Last Track Number
StartLeadOut VAR Returned Starting Lead Out Time

[RETURNS]

(Function : Status Code)
(VAR      : [FirstTrack] First Track Number)
(VAR      : [LastTrack] Last Track Number)
(VAR      : [StartLeadOut] Starting Lead Out Time)

[DESCRIPTION]

Gets a audio cd's disk information.  Includes information on the first
and last audio track on the disk, and the start of the lead out track
(end of the audio) as a min/sec/frame value.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetAudioDiskInfo(   Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var FirstTrack     : BYTE;
                              Var LastTrack      : BYTE;
                              Var StartLeadOut   : TFSM      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

      VCD_GetAudioDiskInfo :=

              MSCDEX_GetAudioDIskInfo( Handle^.DriveInfo[Drive].MethodUnit,
                                       FirstTrack,
                                       LastTrack,
                                       Longint(StartLeadOut)              );


    END;  { cCDMethodMSCDEX: }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END; { VCD_GetAudioDiskInfo }

{}

(*-

[FUNCTION]

Function  VCD_GetAudioTrackInfo(  Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Track          : WORD;
                              Var Start          : TFSM;
                              Var TrackConInfo   : WORD      ) : TError;

[PARAMETERS]

Handle       ?
Drive        CD-ROM Drive Number
Track        ?
Start        VAR Returned Starting Time
TrackConInfo VAR Returned ?

[RETURNS]

(Function : Status Code)
(VAR      : [Start] Starting Time)
(VAR      : [TrackConInfo] ?)

[DESCRIPTION]

Gets information about an audio track on a cd.  "track" is the track
number to get the information about.  "Start" returns as the Min/sec/frame
start time of the track.  "TrackConInfo" are the track control info flags.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetAudioTrackInfo(  Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Track          : BYTE;
                              Var Start          : TFSM;
                              Var TrackConInfo   : BYTE      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

      VCD_GetAudioTrackInfo := MSCDEX_GetAudioTrackInfo (
                                Handle^.DriveInfo[Drive].MethodUnit,
                                Track,
                                Longint(Start),
                                TrackConInfo                         );

    END; { cCDMethodMSCDEX }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END; { VCD_GetAudioTrackInfo }

{}

(*-

[FUNCTION]

Function  VCD_GetAudioQChanInfo(  Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var ConAdr         : WORD;
                              Var TrackNum       : WORD;
                              Var IndexNum       : WORD;
                              Var TrkRelTime     : TFSM;
                              Var DskRelTime     : TFSM      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
ConAdr      VAR Returned ?
TrackNum    VAR Returned Current Track Number
IndexNum    VAR Returned Current Index Number
TrkRelTime  VAR Returned Current Track Relative Time
DskRelTime  VAR Returned Current Disk Relative Time

[RETURNS]

(Function : Status Code)
(VAR      : [ConAdr] ?)
(VAR      : [TrackNum] Current Track Number)
(VAR      : [IndexNum] Current Index Number)
(VAR      : [TrkRelTime] Current Track Relative Time)
(VAR      : [DskRelTime] Current Disk Relative Time)

[DESCRIPTION]

Gets an audio tracks "Q-Channel" information from the currently playing
track.  The Q-Channel contains information about the currently playing
audio track.  The returned info includes "ConAdr" (not used), "TrackNum",
the track number that is currently being played; "IndexNum" the index
within the track that is being played; "TrkRelTime", the current
min/sec/frame being played, relative to the start of the track; and
"DskRelTime" which is the current min/sec/frame being played relative
to the start of the disk.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetAudioQChanInfo(  Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var ConAdr         : BYTE;
                              Var TrackNum       : BYTE;
                              Var IndexNum       : BYTE;
                              Var TrkRelTime     : TFSM;
                              Var DskRelTime     : TFSM      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       VCD_GetAudioQChanInfo := MSCDEX_GetAudioQChanInfo (
                                   Handle^.DriveInfo[Drive].MethodUnit,
                                   ConAdr,
                                   TrackNum,
                                   IndexNum,
                                   TrkRelTime,
                                   DskRelTime                          );


    END; { cCDMethodMSCDEX }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END;  { VCD_GetAudioQChanInfo }

{}

(*-

[FUNCTION]

Function  VCD_GetAudioStatusInfo( Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var AudioStatus    : WORD;
                              Var PlayStart      : TFSM;
                              Var PlayEnd        : TFSM      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
AudioStatus VAR Returned CD-ROM Audio Status Code
PlayStart   VAR Returned Last Play Start Time
PlayEnd     VAR Returned Last Play End Time

[RETURNS]

(Function : Status Code)
(VAR      : [AudioStatus] CD-ROM Audio Status Code)
(VAR      : [PlayStart] Last Play Start Time)
(VAR      : [PlayEnd] Last Play End Time)

[DESCRIPTION]

Gets information about the currently "audio play".  This info includes
"AudioStatus" (TBD), "PlayStart", the min/sec/frame at which the current
audio play operation started, and PlayEnd, the min/sec/frame at which
the audio play operation will end.							  }

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_GetAudioStatusInfo( Handle         : TCDHandle;
                                  Drive          : WORD;
                              Var AudioStatus    : WORD;
                              Var PlayStart      : TFSM;
                              Var PlayEnd        : TFSM      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       VCD_GetAudioStatusInfo := MSCDEX_GetAudioStatusInfo (
                                   Handle^.DriveInfo[Drive].MethodUnit,
                                   AudioStatus,
                                   Longint(PlayStart),
                                   Longint(PlayEnd)                      );

    END; { cCDMethodMSCDEX }

  END; { Case Handle^.DriveInfo[Drive].Method }

END;  { VCD_GetAudioStatusInfo }

{}

(*-

[FUNCTION]

Function  VCD_PlayMSFfor(         Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Start          : TFSM;
                                  PlayFor        : TFSM      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
Start       Min/Sec/Frame to Start Playing From
PlayFor     Number of Min/Sec/Frame to Play

[RETURNS]

[DESCRIPTION]

Plays from a specified min/sec/frame for a specified number  of
mins/secs/frames.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_PlayMSFfor(         Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Start          : TFSM;
                                  PlayFor        : TFSM      ) : TError;

Var
   L      : LONGINT;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       L := VCD_MSFtoTotalFrames ( PlayFor );

       VCD_PlayMSFfor := MSCDEX_PlayAudio (
                           Handle^.DriveInfo[Drive].MethodUnit,
                           cMscdexMSFMode,
                           Longint(Start),
                           L                                   );

    END; { cCDMethodMSCDEX }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END; { VCD_PlayMSFfor }

{}

(*-

[FUNCTION]

Function  VCD_PlayMSFto(          Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Start          : TFSM;
                                  PlayTo         : TFSM      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
Start       Min/Sec/Frame to Start Playing from
PlayTo      Min/Sec/Frame to End Playing at

[RETURNS]

[DESCRIPTION]

Plays from a specified min/sec/frame to a specified min/sec/frame.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_PlayMSFto(          Handle         : TCDHandle;
                                  Drive          : WORD;
                                  Start          : TFSM;
                                  PlayTo         : TFSM      ) : TError;

Var
  PlayFor : TFSM;

BEGIN

  VCD_SubMSF( PlayTo, Start, PlayFor );
  VCD_PlayMSFto := VCD_PlayMSFfor( Handle, Drive, Start, PlayFor );

END; { VCD_PlayMSFto }

{}

(*-

[FUNCTION]

Function VCD_Pause(               Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Pauses the audio play of a specified drive.

[SEE-ALSO]

[EXAMPLE]

-*)

Function VCD_Pause(               Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       VCD_Pause := MSCDEX_StopAudio( Handle^.DriveInfo[Drive].MethodUnit );

    END; { cCDMethodMSCDEX }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END; { VCD_Pause }

{}

(*-

[FUNCTION]

Function  VCD_Stop(               Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Stops the audio play of a specified drive.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_Stop(               Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       VCD_Stop := MSCDEX_StopAudio( Handle^.DriveInfo[Drive].MethodUnit );
       VCD_Stop := MSCDEX_StopAudio( Handle^.DriveInfo[Drive].MethodUnit );

    END; { cCDMethodMSCDEX }

    {----}

  END; { Case Handle^.DriveInfo[Drive].Method }

END; { VCD_Stop }

{}

(*-

[FUNCTION]

Function  VCD_Resume(             Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Resumes the audio play of a specified drive.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_Resume(             Handle         : TCDHandle;
                                  Drive          : WORD      ) : TError;

BEGIN

  Case Handle^.DriveInfo[Drive].Method Of

    {----}

    cCDMethodMSCDEX:
    BEGIN

       VCD_Resume := MSCDEX_ResumeAudio(Handle^.DriveInfo[Drive].MethodUnit);

    END; { cCDMethodMSCDEX }

    {----}

  END;  { Case Handle^.DriveInfo[Drive].Method }

END; { VCD_Resume }

{}

(*-

[FUNCTION]

Function  VCD_PlayTracks(         Handle         : TCDHandle;
                                  Drive          : WORD;
                                  FirstTrack     : BYTE;
                                  NumTracks      : BYTE      ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number
FirstTrack  Audio track to start the play at
NumTracks   Number of audio tracks to play

[RETURNS]

[DESCRIPTION]

Plays the specified "numtracks" number of audio tracks, starting at
"firsttrack"

[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_PlayTracks(         Handle         : TCDHandle;
                                  Drive          : WORD;
                                  FirstTrack     : BYTE;
                                  NumTracks      : BYTE      ) : TError;
BEGIN

END;

{}

(*-

[FUNCTION]

Function  VCD_PlayNextTrack(      Handle         : TCDHandle;
                                  Drive          : WORD       ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Skips forward to the track following the currently playing track.

[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_PlayNextTrack(      Handle         : TCDHandle;
                                  Drive          : WORD       ) : TError;

VAR
   TrackNum       : BYTE;
   IndexNum       : BYTE;
   TrackTime      : tFSM;
   DiskTime       : tFSM;
   Start          : tFSM;
   TrackCon       : BYTE;
   ConAddr        : BYTE;
   StartTrack     : BYTE;
   EndTrack       : BYTE;
   LeadOut        : tFSM;
   Length         : tFSM;
   Status         : tError;
BEGIN

  Status := VCD_GetAudioDiskInfo  ( Handle, Drive, StartTrack, EndTrack, LeadOut );

  Status := VCD_GetAudioQChanInfo ( Handle, Drive, conaddr,tracknum,indexnum,tracktime,disktime);

  Inc(TrackNum);
  If TrackNum > EndTrack Then
     TrackNum := StartTrack;

  Status := VCD_GetAudioTrackInfo ( Handle, Drive, TrackNum, start, trackcon );

  VCD_SubMSF ( LeadOut, Start, Length );

  Status := VCD_stop( Handle, drive );

  Status := VCD_PlayMSFFor ( Handle, Drive , start, Length );
  VCD_PlayNextTrack := Status;

END;

{}

(*-

[FUNCTION]

Function  VCD_PlayPrevTrack(      Handle         : TCDHandle;
                                  Drive          : WORD       ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Plays the track prior to the currently playing audio track.

[SEE-ALSO]

[EXAMPLE]

-*)

Function  VCD_PlayPrevTrack(      Handle         : TCDHandle;
                                  Drive          : WORD       ) : TError;

VAR
   TrackNum       : BYTE;
   IndexNum       : BYTE;
   TrackTime      : tFSM;
   DiskTime       : tFSM;
   Start          : tFSM;
   TrackCon       : BYTE;
   ConAddr        : BYTE;
   StartTrack     : BYTE;
   EndTrack       : BYTE;
   LeadOut        : tFSM;
   Length         : tFSM;
   Status         : tError;

BEGIN

  Status := VCD_GetAudioDiskInfo  ( Handle, Drive, StartTrack, EndTrack, LeadOut );

  Status := VCD_GetAudioQChanInfo ( Handle, Drive, conaddr,tracknum,indexnum,tracktime,disktime);

  Dec(TrackNum);
  If TrackNum < StartTrack Then
     TrackNum := EndTrack;

  Status := VCD_GetAudioTrackInfo ( Handle, Drive, TrackNum, start, trackcon );

  VCD_SubMSF ( LeadOut, Start, Length );

  Status := VCD_stop( Handle, drive );

  Status := VCD_PlayMSFFor ( Handle, Drive , start, Length );
  VCD_PlayPrevTrack := Status;

END;

{}

(*-

[FUNCTION]

Function  VCD_PlayForwardMSF(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveForward    : TFSM       ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Skips the current audio play forward by the specified Min:Sec.Frame
count.  Can be used to implement a "fast-forward" control.

[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_PlayForwardMSF(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveForward    : TFSM       ) : TError;
VAR
   TrackNum       : BYTE;
   IndexNum       : BYTE;
   TrackTime      : tFSM;
   DiskTime       : tFSM;
   Start          : tFSM;
   TrackCon       : BYTE;
   ConAddr        : BYTE;
   StartTrack     : BYTE;
   EndTrack       : BYTE;
   LeadOut        : tFSM;
   Length         : tFSM;
   TempMSF        : tFSM;
   Status         : tError;
BEGIN

  Status := VCD_GetAudioDiskInfo  ( Handle, Drive, StartTrack, EndTrack, LeadOut );
  Status := VCD_GetAudioQChanInfo ( Handle, Drive, conaddr,tracknum,indexnum,tracktime,disktime);

  VCD_AddMSF ( DiskTime,MoveForward ,TempMSF );

  If VCD_MSFToTotalFrames(TempMSF) <= VCD_MSFToTotalFrames(Leadout) Then
  Begin
     VCD_SubMSF ( Leadout, TempMSF, Length );
     Status := VCD_PlayMSFFor ( Handle, Drive , TempMSF, Length );
  End
  else
  Begin
     Status := VCD_GetAudioTrackInfo ( Handle, Drive, 1, start, trackcon );
     VCD_SubMSF(LeadOut,Start,Length);
     VCD_PlayMSFFor( Handle,Drive,Start,Length);
  End;

  VCD_PlayForwardMSF := Status;

END;

{}


(*-

[FUNCTION]

Function  VCD_PlayForwardLBA(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveForward    : LONGINT    ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Skips the current audio play forward by the specified number of blocks.
Can be used to implement a "fast-forward" control.

[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_PlayForwardLBA(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveForward    : LONGINT    ) : TError;

Begin
END;

{}

(*-

[FUNCTION]

Function  VCD_PlayReverseMSF(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveReverse    : TFSM       ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Skips the current audio play backward by the specified Min:Sec.Frame
count.  Can be used to implement a "fast-rewind" control.

[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_PlayReverseMSF(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveReverse    : TFSM       ) : TError;
VAR
   TrackNum       : BYTE;
   IndexNum       : BYTE;
   TrackTime      : tFSM;
   DiskTime       : tFSM;
   Start          : tFSM;
   TrackCon       : BYTE;
   ConAddr        : BYTE;
   StartTrack     : BYTE;
   EndTrack       : BYTE;
   LeadOut        : tFSM;
   Length         : tFSM;
   TempMSF        : tFSM;
   Status         : tError;
BEGIN

  Status := VCD_GetAudioDiskInfo  ( Handle, Drive, StartTrack, EndTrack, LeadOut );
  Status := VCD_GetAudioQChanInfo ( Handle, Drive, conaddr,tracknum,indexnum,tracktime,disktime);
  Status := VCD_GetAudioTrackInfo ( Handle, Drive, 1, Start, TrackCon );

  If VCD_MSFToTotalFrames(DiskTime) > VCD_MSFToTotalFrames(MoveReverse) Then
  Begin
     VCD_SubMSF ( DiskTime,MoveReverse ,TempMSF );
     VCD_SubMSF ( Leadout, TempMSF, Length );
     Status := VCD_PlayMSFFor ( Handle, Drive , TempMSF, Length );
  End
  Else
  Begin
     Status := VCD_GetAudioTrackInfo ( Handle, Drive, EndTrack, start, trackcon );
     VCD_SubMSF(Leadout,MoveReverse,TempMSF);
     VCD_SubMSF(LeadOut,TempMSF,Length);
     VCD_PlayMSFFor( Handle,Drive,TempMSF,Length);
  End;

  VCD_PlayReverseMSF := Status;

END;

{}


(*-

[FUNCTION]

Function  VCD_PlayReverseLBA(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveReverse    : LONGINT    ) : TError;

[PARAMETERS]

Handle      ?
Drive       CD-ROM Drive Number

[RETURNS]

[DESCRIPTION]

Skips the current audio play backward by the specified # of logical blocks.
Can be used to implement a "fast-rewind" control.

[SEE-ALSO]

[EXAMPLE]

-*)


Function  VCD_PlayReverseLBA(     Handle         : TCDHandle;
                                  Drive          : WORD;
                                  MoveReverse    : LONGINT    ) : TError;

BEGIN

END;

{}
{}

BEGIN

END.
