#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gen.h"
#include "sbstuff.h"

#ifdef __SC__
#include <int.h>
#endif

#define MAX_MIX_SAMPLES     8       /*  Max number of samples to mix  */

#define NUM_OF_SAMPLES      9       /*  Number of samples in raw lib  */

/*  To get music working the CMF player needs to be invoked from a timer  */
/*  interrupt service routine. This routine needs to perform the task of  */
/*  arbitrating what will most likely be a fast timer interrupt down to   */
/*  the rate the music player needs to be called at. The SBSTUFF object   */
/*  declares a couple of variables to enable you to do this. It also      */
/*  needs to have access to a variable indicating the current timer tick  */
/*  rate. We'll declare that here as a volatile int and you should look   */
/*  at the code pertaining to the control of the timer (and the interrupt */
/*  service routine) for how these things should be used.                 */

volatile int            nTimer_Rate;

/*  We'll need to access these from the SBSTUFF object.  Declare as extern  */

extern char             szSBInfo[];
extern volatile int     nSB_Music_TickRate;
extern volatile int     nSB_Music_TickLimit;

/*  Our own variables for the demo  */

SB_SAMPLE               samples[NUM_OF_SAMPLES];
volatile int            nTimer_TickLimit;

void (interrupt far *OldTimerInterrupt)( void );

/*  Here's our own timer interrupt service routine  */

#ifdef __SC__       /*  For Symantec only  */
static int TimerServiceRoutine( struct INT_DATA *pID )
#else
static void interrupt TimerServiceRoutine( void )
#endif      /*  __SC__  */
{
    /*  This routine will handle the arbitration of the timer interrupt.  */
    /*  Most games will reprogram the timer to interrupt at a faster rate */
    /*  than the default 18.2 times a second. What we do here is to keep  */
    /*  a check on the number of times we have been interrupted and then  */
    /*  (when appropriate) to pass on the interrupt to the original code  */
    /*  handler. This is good because the original code keeps the DOS     */
    /*  clock up to date, etc. If we didn't do this and the user ran our  */
    /*  game for a couple of hours, say, the time would then be 2 hours   */
    /*  slow. If we just passed on all interrupts to the original code    */
    /*  the clock (and date) would be well out of sync!.                  */
    /*                                                                    */
    /*  The other thing this timer service routine does is to arbitrate   */
    /*  the calling of the CMF music player. Like the DOS clock, it too   */
    /*  needs to be called at a predetermined rate. This rate, however,   */
    /*  depends upon the tempo of the song being played. What we do with  */
    /*  the music player is much the same as the DOS clock. Just keep a   */
    /*  count of the number of interrupts we've had and when appropriate  */
    /*  call the music player.                                            */

    static volatile unsigned int    ticks=0;        /*  Ticks to call original timer code  */
    static volatile unsigned int    mticks=0;       /*  Ticks to call music player         */

    if ( SB_IsCMFPlaying() && ++mticks>=nSB_Music_TickLimit )
    {
        mticks=0;
        SB_Music_Player();
    }

    if ( ++ticks>=nTimer_TickLimit )
    {
        ticks     = 0;

#ifdef __SC__               /*  Symantec has different code here!   */
        return( 0 );        /*  Okay to call original handler       */
#else
        (*OldTimerInterrupt)();
#endif                      /*  __SC__  */
    }
    else
    {
        outp( 0x20,0x20 );
    }

#ifdef __SC__
    return( 1 );            /*  Don't call original handler  */
#endif
}

/*  Here are our helper functions  */

void SetTimerRate( unsigned short int nNewTimerRate )
{
    /*  Reprogram the timer to our new rate and recalculate all ticker  */
    /*  arbitration limits...                                           */

    WORD    wRate;

    _disable();
    nTimer_Rate         = nNewTimerRate;
    nTimer_TickLimit    = nTimer_Rate/18;
    nSB_Music_TickLimit = (nSB_Music_TickRate==0) ? 0 : (nTimer_Rate+nSB_Music_TickRate-1)/nSB_Music_TickRate;

    /*  Determine the timer programming factor  */

    wRate = (nNewTimerRate==18) ? 0 : ((WORD) ( 1193180L / ((unsigned long)nNewTimerRate) ));

    outp( 0x43,0x34 );
    outp( 0x40,wRate&0xff );
    outp( 0x40,wRate>>8 );
    _enable();
}

void main( int argc, char *argv[] )
{
    int         err, errDing, nSoundCount, key, nVol=8, nRate=11000,
                nMasterVol, nWaveVol, nFMVol, bForce8Bit=FALSE,
                bPrintHelp=TRUE;
    SB_SAMPLE   sample;

    printf( "\nUsage: TestSB  [/8]      where /8 forces 8-bit DMA\n" );

    /*  Scan command line for forcing of 8-bit DMA  */

    for ( key=1; key<argc; key++ )
    {
        if ( strcmp(argv[key],"/8")==0 ) bForce8Bit=TRUE;
    }

    /*  Start off by intercepting the timer interrupt and installing  */
    /*  our own handler...                                            */

#ifdef __SC__
    int_intercept( 0x08,&TimerServiceRoutine,0 );
#else
    OldTimerInterrupt = _dos_getvect( 0x08 );
    _dos_setvect( 0x08,&TimerServiceRoutine );
#endif      /*  __SC__  */

    /*  Program a faster timer rate  */

    SetTimerRate( 1000 );

    /*  Enable the SB sub-system and away we gooooooo...  */

    err=SB_Enable( MAX_MIX_SAMPLES,nRate,bForce8Bit );
    if ( err!=0 ) printf( "SB_Enable() failed! Error=%d\n",err );

    /*  Display some diagnostic information about the chosen mode, etc  */

    printf( "\nszSBInfo=%s DMA Mode=%s\n",szSBInfo,(SB_Is16BitDMA()?"16-bit":"8-bit") );

    /*  Load in a single raw sample manually  */

    errDing=SB_LoadRawSample( &sample,"DING.RAW" );
    if ( errDing!=0 ) printf( "Error loading DING.RAW=%d\n",errDing ); else
    {
        /*  If no errors, set this sample to continuous play at half volume  */

        err=SB_PlaySample( &sample,1L,SB_MAX_VOLUME/2,-1 );
        if ( err!=0 ) printf( "Error playing=%d\n",err );
    }

    /*  Load in samples from raw library for interactive playing  */

    nSoundCount=SB_LoadRawSounds( samples,NUM_OF_SAMPLES,"SOUNDS.RLB" );
    if ( nSoundCount==0 ) printf( "Error loading from raw library!\n" );

    /*  Reset all mixer control volumes to 3/4 maximum. The volume  */
    /*  levels are controlled for the left and right channels       */
    /*  independantly and range from 0 (off) to 15 (0x0f) maximum.  */

    nMasterVol = SB_SetMixerMasterVolume( 0x0c,0x0c );
    nFMVol     = SB_SetMixerFMVolume( 0x0c,0x0c );

    /*  16-bit DMA looses volume for the sake of precision. Set volume to  */
    /*  maximum if 16-bit in effect...                                     */

    if ( SB_Is16BitDMA() )
      nWaveVol = SB_SetMixerWaveVolume( 0x0f,0x0f );
    else
      nWaveVol   = SB_SetMixerWaveVolume( 0x0c,0x0c );

    printf( "\n" );
    printf( "Original Master Volume = %02X\n",nMasterVol );
    printf( "Original Wave   Volume = %02X\n",nWaveVol );
    printf( "Original FM     Volume = %02X\n",nFMVol );

    /*  Main loop for that little bit of interaction  */

    while ( TRUE )
    {
        if ( bPrintHelp )
        {
            printf( "\nESCape to quit\n" );
            printf( "Press R to toggle repeating DING\n" );
            printf( "Press P to play DING\n" );
            printf( "Press M to toggle music\n" );
#ifdef _DEBUG
            printf( "Press D to display debug info\n" );
#endif
            printf( "Press V to change sample volume level\n" );
            printf( "Press < to reduce playback rate\n" );
            printf( "Press > to increase playback rate\n" );
            if ( nSoundCount>0 ) printf( "Press 1 to %d to play a sound\n",nSoundCount );
            printf( "Current Volume To Play Sounds At: %d\n",nVol );
            printf( "Current Playback Rate           : %d\n",nRate );
            bPrintHelp = FALSE;
        }

        /*  Show number of samples playing  */

        printf( "\r%d",SB_NumSamplesPlaying() );

        if ( kbhit() )
        {
            key=getch();
            if ( key==27 ) break;       /*  This is ESCape  */

#ifdef _DEBUG
            /*  Diagnostics in DEBUG mode. Dump out mixing list  */

            if ( key=='d' || key=='D' ) SB_Dump();
#endif

            /*  P to play single ding at current volume  */

            if ( key=='p' || key=='P' )
            {
                if ( errDing==0 ) SB_PlaySample( &sample,2L,nVol,1 );
            }

            /*  R to toggle play of repeating ding at half volume  */

            if ( key=='r' || key=='R' )
            {
                if ( errDing==0 )
                {
                    if ( SB_IsSamplePlaying(1L) )
                      SB_StopSample( 1L );
                    else
                      SB_PlaySample( &sample,1L,SB_MAX_VOLUME/2,-1 );
                }
            }

            /*  1 to N to play sample at current volume  */

            if ( key>='1' && key<'1'+nSoundCount )
            {
                SB_PlaySample( samples+key-'1',3L+key-'1',nVol,1 );
            }

            /*  M to toggle CMF music  */

            if ( key=='m' || key=='M' )
            {
                /*  Set up a CMF playing continuously in the background at 50% volume  */

                if ( SB_IsCMFPlaying() ) SB_StopCMF(); else
                {
                    err=SB_PlayCMF( "ALTONLY2.CMF",50,TRUE );
                    if ( err!=0 ) printf( "\nCMF error=%d\n",err );
                }
            }

            /*  V to change volume level to play samples at  */

            if ( key=='v' || key=='V' )
            {
                printf( "\nEnter New Volume Level (1-8): " );
                key=getche();
                if ( key>='1' && key<='8' )
                {
                    nVol = key-'0';
                    bPrintHelp = TRUE;
                }
                printf( "\n" );
            }

            /*  < or > to change playback rate  */

            if ( key=='<' || key=='>' )
            {
                if ( key=='<' && nRate>5000 ) nRate-=500;
                if ( key=='>' && nRate<44000 ) nRate+=500;
                SB_SetSampleRate( nRate );
                bPrintHelp = TRUE;
            }
        }
    }

    /*  Stop all samples from playing  */

    SB_StopSample( SB_ALL_SAMPLES );

    /*  Stop the CMF  */

    SB_StopCMF();

    /*  Free up our sample data memory  */

    SB_FreeRawSample( &sample );
    SB_FreeRawSounds( samples,NUM_OF_SAMPLES );

    /*  Be a well behaved program and reset the volume levels to the user's  */
    /*  original values.                                                     */

    SB_SetMixerMasterVolume( nMasterVol>>4,nMasterVol&0x0f );
    SB_SetMixerWaveVolume( nWaveVol>>4,nWaveVol&0x0f );
    SB_SetMixerFMVolume( nFMVol>>4,nFMVol&0x0f );

    /*  Disable the SB sub-system. You MUST remember to do this or build  */
    /*  it into any atexit() function you use!!!                          */

    SB_Disable();

    /*  Restore the original timer  */

    SetTimerRate( 18 );

#ifdef __SC__
    int_restore( 0x08 );
#else
    _dos_setvect( 0x08,OldTimerInterrupt );
#endif
}
