' $Header:   D:/misc/midi/vcs/dumpster.bas   1.1   07 Nov 1994 23:42:20   DAVEC  $

' Public domain by David Churcher. No rights reserved. Use at your own risk.

Option Explicit

Type midiDev
	DeviceNumber As Integer
	DeviceName   As String
	Handle As Integer
End Type

Type InstrumentType
     sDescription As String     ' Instrument description
     sRequest As String         ' Request string in hex
     bAutoEnd As Integer        ' End reception automatically
     nReceiveTimeout As Integer ' End reception after timeout (secs)
     sAcknowledge As String     ' Sent at end of reception
     nBufferCount As Integer    ' Number of buffers to allocate
     nBufferSize As Integer     ' Size of each buffer
     nCtrStart As Integer       ' Counter start for multiple request strings
     nCtrEnd As Integer         ' Counter end for multiple request strings
     sTransform As String       ' Transformation for incoming data
End Type

'-----------------------------------------------------------------------

' Dump request macros
Global Instrument() As InstrumentType

' MIDI input and output devices
Global gInputdev As midiDev
Global gOutputdev As midiDev

' Current instrument
Global gnInstrument As Integer

' Current filename
Global gsFilename As String

' Mode variable and constants, used in midiInputArrived to decide what to do with input
Global gnState As Integer
Global Const STATE_OFF = 0
Global Const STATE_RECEIVE = 1
Global Const STATE_SEND = 2

' File dirty flag
Global gbFileModified As Integer

' Sysex send block size and delay
Global gnSendBlockSize As Integer
Global gnSendDelay As Integer

' INI filename (was a constant, but want file in same dir as app)
Global INI_FILENAME As String

'-----------------------------------------------------------------------
' INI file functions (correct versions from the Knowledgebase tips file)
Declare Function GetPrivateProfileString Lib "Kernel" (ByVal lpApplicationName As String, ByVal lpKeyName As Any, ByVal lpDefault As String, ByVal lpReturnedString As String, ByVal nSize As Integer, ByVal lpFileName As String) As Integer
Declare Function GetPrivateProfileInt Lib "Kernel" (ByVal lpAppName As String, ByVal lpKeyName As String, ByVal nDefault As Integer, ByVal lpFileName As String) As Integer
Declare Function WritePrivateProfileString Lib "Kernel" (ByVal lpApplicationName As String, ByVal lpKeyName As Any, ByVal lpString As Any, ByVal lplFileName As String) As Integer

' Help function
Declare Function WinHelp Lib "User" (ByVal hWnd As Integer, ByVal lpHelpFile As String, ByVal wCommand As Integer, dwData As Any) As Integer

'-----------------------------------------------------------------------

' Special bytes used in request macros
Global Const MACRO_OR = &HFC
Global Const MACRO_CHANNEL = &HFD
Global Const MACRO_PATCH = &HFE
Global Const MACRO_COUNTER = &HFF

Global Const APP_NAME = "Dumpster"

Global Const UNTITLED_NAME = "[untitled]"

Global Const SYSEX_SEND_PACKET_SIZE = 1024
Global Const SYSEX_SEND_DELAY_MS = 60

' MsgBox parameters
Global Const MB_OK = 0                 ' OK button only
Global Const MB_OKCANCEL = 1           ' OK and Cancel buttons
Global Const MB_ABORTRETRYIGNORE = 2   ' Abort, Retry, and Ignore buttons
Global Const MB_YESNOCANCEL = 3        ' Yes, No, and Cancel buttons
Global Const MB_YESNO = 4              ' Yes and No buttons
Global Const MB_RETRYCANCEL = 5        ' Retry and Cancel buttons

Global Const MB_ICONSTOP = 16          ' Critical message
Global Const MB_ICONQUESTION = 32      ' Warning query
Global Const MB_ICONEXCLAMATION = 48   ' Warning message
Global Const MB_ICONINFORMATION = 64   ' Information message

Global Const MB_APPLMODAL = 0          ' Application Modal Message Box
Global Const MB_DEFBUTTON1 = 0         ' First button is default
Global Const MB_DEFBUTTON2 = 256       ' Second button is default
Global Const MB_DEFBUTTON3 = 512       ' Third button is default
Global Const MB_SYSTEMMODAL = 4096      'System Modal

' MsgBox return values
Global Const IDOK = 1                  ' OK button pressed
Global Const IDCANCEL = 2              ' Cancel button pressed
Global Const IDABORT = 3               ' Abort button pressed
Global Const IDRETRY = 4               ' Retry button pressed
Global Const IDIGNORE = 5              ' Ignore button pressed
Global Const IDYES = 6                 ' Yes button pressed
Global Const IDNO = 7                  ' No button pressed

' MousePointer constants
Global Const MP_DEFAULT = 0
Global Const MP_WAIT = 11

' Help constants
Global Const HELP_CONTENTS = &H3

'-----------------------------------------------------------------------

'-----------------------------------------------------------------------
' Add trailing backslash to string if it's not a relative path ending with
' a drive.
Function addslash (ByVal sPath As String) As String

    sPath = Trim$(sPath)
    If Right$(sPath, 1) <> "\" And Right$(sPath, 1) <> ":" Then
	sPath = sPath & "\"
    End If

    addslash = sPath

End Function

Sub Alert (sMsg As String)
    MsgBox sMsg, 48
End Sub

'-----------------------------------------------------------------------
Sub CloseDev ()

	If gInputdev.Handle >= 0 Then

		gInputdev.Handle = vbMidiInClose(gInputdev.Handle)
		gInputdev.Handle = -1

	End If

	If gOutputdev.Handle >= 0 Then
		
		gOutputdev.Handle = vbMidiOutClose(gOutputdev.Handle)
		gOutputdev.Handle = -1

	End If

End Sub

'-----------------------------------------------------------------------
Sub dumpsterCleanup ()

	Call CloseDev

	HugeClear

End Sub

'-----------------------------------------------------------------------
Sub dumpsterSetup ()
	
	' Init MIDI handles to invalid number
	gInputdev.Handle = -1
	gOutputdev.Handle = -1

	If Not LoadSetup() Then
		Unload Dumpster
	End If

	Dumpster.lstInstrument.ListIndex = 0

	SetFilename UNTITLED_NAME
	gbFileModified = False

End Sub

'-----------------------------------------------------------------------
' Convert list of hex bytes (space-delimited or contiguous) to a string of characters
' Note: ignores hex values > FF
' NB This is currently different from the versions in Junolibr and Miditest in that it doesn't
' require space-delimited bytes
Function HexListToString (ByVal sHexList As String) As String

	Dim sRetString As String, sThisByte As Integer
	Dim nStartPos As Integer

	nStartPos = 1
	sRetString = ""

	Do While nStartPos < Len(sHexList)

		sThisByte = Val("&H" & Mid$(sHexList, nStartPos, 2))

		If sThisByte >= 0 And sThisByte < 256 Then
			' Convert to character
			sRetString = sRetString & Chr$(sThisByte)
		End If

		nStartPos = nStartPos + 2
		If Mid$(sHexList, nStartPos, 1) = " " Then nStartPos = nStartPos + 1

	Loop

	HexListToString = sRetString

End Function

'-----------------------------------------------------------------------
' Load configuration from INI file and set globals
' gInputDev.deviceName, gOutputdev.deviceName, gMidiInDevNo, gMidiOutDevNo
Function LoadIni ()

	LoadIni = LoadIniDev("Input", gInputdev, 0, incaps(0).szPname) And LoadIniDev("Output", gOutputdev, 0, outcaps(0).szPname) And LoadIniDRM() And LoadIniSend()

End Function

'-----------------------------------------------------------------------
' Load and validate a configured MIDI input or output device from the INI file
' sIniKey: Name of key to load from INI file e.g. "Input"
' nDefaultDevNumber: Number of default MIDI device if no configured device or invalid device name
' sDefaultDevName: Name of default MIDI device
' Returns: True if successful
Function LoadIniDev (sIniKey As String, dev As midiDev, nDefaultDevNumber, sDefaultDevName As String)

	Dim sDevName As String, nRetcode As Integer
	sDevName = Space$(32)

	' Load settings to get names of configured devices
	' (ByVal Appname As String, ByVal KeyName As String, ByVal DEFAULT As String, ByVal ReturnedString As String, ByVal MaxSize, ByVal FileName As String)
	nRetcode = GetPrivateProfileString(APP_NAME, sIniKey, "Not configured", sDevName, 32, INI_FILENAME)

	sDevName = szTrim(sDevName)
	If sDevName = "Not configured" Then
		MsgBox "No " & LCase$(sIniKey) & " device configured, defaulting to " & Trim(sDefaultDevName)
		dev.DeviceName = Trim(sDefaultDevName)
		dev.DeviceNumber = nDefaultDevNumber
	Else
		' Find matching device number
		dev.DeviceNumber = findDeviceByName(sDevName)
		If dev.DeviceNumber >= 0 Then
			' Configured device is OK
			dev.DeviceName = Trim(sDevName)
		Else
			MsgBox "Can't find configured MIDI input device " & sDevName & ", defaulting to " & Trim(sDefaultDevName)
			dev.DeviceNumber = nDefaultDevNumber
			dev.DeviceName = sDefaultDevName
		End If
	End If

	LoadIniDev = True ' No error return just yet

End Function

'-----------------------------------------------------------------------
' Purpose: Load dump request macros from INI file
Function LoadIniDRM ()

    Const BUF_SIZE = 32767
    Const LINE_SIZE = 1024

    Dim sMacroBuffer As String, iPtr As Integer, iEnd As Integer, iMacroctr As Integer
    Dim nRetcode As Integer, nStart As Integer, nEnd As Integer
    Dim sThisMacro As String, sThisName As String, sArgument As String, nArgument As Integer
    Dim nCtrLow As Integer, nCtrHigh As Integer
    Dim DefaultInstrument As InstrumentType

    ' Set up default values for new instrument defs
    DefaultInstrument.bAutoEnd = True
    DefaultInstrument.nBufferCount = 10
    DefaultInstrument.nBufferSize = 10000

    ReDim Instrument(1)
    Dumpster.lstInstrument.Clear

    ' Put the manual request in as the first item
    Instrument(0).sDescription = "< Start dump on instrument >"
    Dumpster.lstInstrument.AddItem Instrument(0).sDescription

    sMacroBuffer = Space$(BUF_SIZE)

    ' Get names of all entries in the [Dump Request Macros] section of the INI file.
    ' (ByVal Appname As String, ByVal KeyName As String, ByVal DEFAULT As String, ByVal ReturnedString As String, ByVal MaxSize, ByVal FileName As String)
    nRetcode = GetPrivateProfileString("Dump Request Macros", 0&, "", sMacroBuffer, BUF_SIZE, INI_FILENAME)

    If nRetcode > 2 Then
	
	' Step through the list of entries separated by binary 0's
	iPtr = 1
	iEnd = nRetcode - 2
	iMacroctr = 0
	Do While iPtr < iEnd
	    
	    nStart = iPtr
	    nEnd = InStr(iPtr, sMacroBuffer, Chr$(0))
	    If nEnd = 0 Then Exit Do

	    sThisMacro = Trim$(Mid$(sMacroBuffer, nStart, nEnd - nStart))
	    iMacroctr = iMacroctr + 1
	    ReDim Preserve Instrument(iMacroctr)

	    ' Default values and flags for this instrument
	    Instrument(iMacroctr) = DefaultInstrument
	    
	    ' Set description
	    Instrument(iMacroctr).sDescription = sThisMacro

	    ' Add to main form's list of request macros
	    Dumpster.lstInstrument.AddItem sThisMacro

	    ' Next INI file entry
	    iPtr = iPtr + (nEnd - iPtr) + 1

	Loop

	' Now get values for each entry
	iEnd = iMacroctr
	For iMacroctr = 1 To iEnd

	    sThisMacro = Space$(LINE_SIZE)
	    nRetcode = GetPrivateProfileString("Dump Request Macros", Instrument(iMacroctr).sDescription, "", sThisMacro, LINE_SIZE, INI_FILENAME)
	    sThisMacro = Left$(sThisMacro, nRetcode)

	    ' Do some cursory validation
	    nEnd = InStr(sThisMacro, "F7")

	    If nEnd > 0 Then

		Instrument(iMacroctr).sRequest = HexListToString(Left$(sThisMacro, nEnd + 1))
		
		' Interpret parameters after the basic dump request
		sThisMacro = Trim$(Mid$(sThisMacro, nEnd + 2))

		Do While Len(sThisMacro) > 0
		    nEnd = InStr(sThisMacro, " ")
		    If nEnd = 0 Then
			nEnd = Len(sThisMacro)
		    Else
			nEnd = nEnd - 1
		    End If
		    sArgument = Trim$(Mid$(sThisMacro, 2, nEnd - 1))

		    Select Case Left$(sThisMacro, 1)
		    Case "m"    ' Manual finish
			Instrument(iMacroctr).bAutoEnd = False
		    Case "t"    ' Timeout finish
			Instrument(iMacroctr).nReceiveTimeout = Val(sArgument)
			Instrument(iMacroctr).bAutoEnd = False
		    Case "s"    ' Buffer size
			Instrument(iMacroctr).nBufferSize = Val(sArgument)
		    Case "n"    ' number of buffers
			Instrument(iMacroctr).nBufferCount = Val(sArgument)
		    Case "a"    ' Acknowledge string
			Instrument(iMacroctr).sAcknowledge = HexListToString(sArgument)
		    Case "c"    ' Counter limits
			nArgument = Val("&H" + sArgument)
			nCtrLow = nArgument \ 256
			nCtrHigh = nArgument Mod 256
			If nCtrHigh = 0 Or nCtrHigh < nCtrLow Then
			    Alert "Wrong counter values in " & Instrument(iMacroctr).sDescription & Chr$(13) & Chr$(10) & "Enter counter parameter as cLLHH where LL and HH are low and high limits in hex"
			Else
			    If InStr(Instrument(iMacroctr).sRequest, Chr$(MACRO_COUNTER)) = 0 Then
				Alert "No counter byte found in request string for " & Instrument(iMacroctr).sDescription
			    End If
			    Instrument(iMacroctr).nCtrStart = nCtrLow
			    Instrument(iMacroctr).nCtrEnd = nCtrHigh
			End If

		    Case "x"    ' Transformation
			Instrument(iMacroctr).sTransform = sArgument

		    End Select
		    
		    If nEnd = 0 Then
			Exit Do
		    Else
			sThisMacro = Trim$(Mid$(sThisMacro, nEnd + 1))
		    End If
		Loop

	    Else

		Call Alert("Don't understand this dump request macro:" + Chr$(13) + Chr$(10) + Left$(sThisMacro, 50))

	    End If
			
	Next
	
    End If

    LoadIniDRM = True

End Function

' Load send block size and delay from INI file
Function LoadIniSend () As Integer

    gnSendBlockSize = GetPrivateProfileInt(APP_NAME, "SendBlockSize", 1, INI_FILENAME)
    If gnSendBlockSize > 32 Then
	gnSendBlockSize = 32
    ElseIf gnSendBlockSize < 1 Then
	gnSendBlockSize = 1
    End If
    
    gnSendDelay = GetPrivateProfileInt(APP_NAME, "SendDelay", 60, INI_FILENAME)
    If gnSendDelay > 10000 Then
	gnSendDelay = 10000
    ElseIf gnSendDelay < 60 Then
	gnSendDelay = 60
    End If

    LoadIniSend = True

End Function

'-----------------------------------------------------------------------
' Load and validate the setup configuration
' Returns: True if successful
Function LoadSetup ()

SetIniFilename

If LoadCaps() = True And LoadIni() = True Then

	LoadSetup = True

Else

	LoadSetup = False

End If


End Function

'-----------------------------------------------------------------------
Sub OpenDev ()
	Dim nRetcode As Integer
	If gInputdev.Handle < 0 Then
		' Open the device
		gInputdev.Handle = vbMidiInOpen(Dumpster.hWnd, gInputdev.DeviceNumber, Dumpster.MsgBlaster1, Dumpster.MidiNotifyControl)
		If (gInputdev.Handle > 0) Then
			' Start MIDI input on the device. NB: Short input is disabled!
			nRetcode = vbMidiInStart(gInputdev.Handle, Dumpster.MsgBlaster1, False)
		End If
	End If
	If gOutputdev.Handle < 0 Then
		gOutputdev.Handle = vbMidiOutOpen(Dumpster.hWnd, gOutputdev.DeviceNumber)
	End If
End Sub

'-----------------------------------------------------------------------
Sub SaveIniEntry (ByVal keyName As String, ByVal keyValue As String)

Dim nRetcode As Integer

	nRetcode = WritePrivateProfileString(APP_NAME, keyName, keyValue, INI_FILENAME)

End Sub

' Select text box contents on entry. Called from GotFocus event of text box
Sub SelectOnEntry ()
    If TypeOf Screen.ActiveForm.ActiveControl Is TextBox Then
	If Len(Screen.ActiveForm.ActiveControl.Text) Then
	    Screen.ActiveForm.ActiveControl.SelStart = 0
	    Screen.ActiveForm.ActiveControl.SelLength = Len(Screen.ActiveForm.ActiveControl.Text)
	End If
    End If
End Sub

'-----------------------------------------------------------------------
Sub SetFilename (sFilename As String)

	gsFilename = sFilename
	SetMainWindowCaption

End Sub

'-----------------------------------------------------------------------
Sub SetIniFilename ()

	INI_FILENAME = addslash(App.Path) & APP_NAME & ".INI"

End Sub

'-----------------------------------------------------------------------
Sub SetMainWindowCaption ()

	Dumpster.Caption = APP_NAME & " - " & gsFilename

End Sub

'-----------------------------------------------------------------------
Sub statusupdate ()
    Dim lLen As Long
    lLen = HugeLen()
    If lLen = 0 Then
	Dumpster.lblStatus.Caption = "Buffer empty"
    Else
	Dumpster.lblStatus.Caption = lLen & " bytes in buffer"
    End If
    Dumpster.cmdSend.Enabled = (lLen > 0)
End Sub

'-----------------------------------------------------------------------
' Purpose: Convert string to space-delimited list of hex values
Function StringToHexList (sTheString As String) As String

	Dim sRetString As String
	Dim ictr As Integer

	For ictr = 1 To Len(sTheString)

		' Convert to two hex characters (the Right$() adds a leading 0 if necessary)
		sRetString = sRetString & Right$("0" + Hex$(Asc(Mid$(sTheString, ictr, 1))), 2) & " "

	Next

	sRetString = Trim$(sRetString)

	StringToHexList = sRetString

End Function

