////////////////////////////////////////////////////////////////////////////////////////////
// midPlay.cpp
// Written by Kevin Hoffman.
// Simple app to play .MID file.
#include <windows.h>
#include <mmsystem.h>
#include <commdlg.h>
#include "resource.h"

//Application instance handle
HINSTANCE hInst=NULL;
//Application window handle
HWND hMainWindow=NULL;
//TRUE if the song is playing.
BOOL bSongPlaying = FALSE;
//TRUE if we should loop forever!
BOOL bLoopForever = FALSE;
//TRUE if MID device is open.
BOOL bMIDOpen=FALSE;
//The name of the open .MID file.
char sMidFile[300];
//Our app icon
HICON iconApp=NULL;
//Our accellerator table
HACCEL accelTable=NULL;
//Our timer ID
UINT gTimerID=0;

//Registers our window class. Returns TRUE on success
BOOL InitApp(HINSTANCE hInstance);
//Creates our window. Returns TRUE on success
BOOL InitInst(HINSTANCE hInstance,int nShowWindow,LPSTR sCmdLine);
//The window procedure that handles the messages.
long FAR PASCAL __export WindowProc(HWND hWnd,UINT nMsg,WPARAM wParm,LPARAM lParm);
//Issues a MCI device command. Returns TRUE on success or displays an error message and returns FALSE.
//See the mciSendString function.
BOOL TellMCI(LPSTR sCmd,LPSTR sRetStr,UINT nRetLen,HANDLE hCallback,LPSTR sErr,BOOL bReportError=TRUE);
//Shows an error message box.
void ErrorMessage(LPSTR sErr);
//Updates the window text based on status.
void UpdateWinText(BOOL bRepaint=TRUE);
//Loads options from disk.
void LoadOptions();
//Saves options to disk.
void SaveOptions();
//Updates menu items based on options
void UpdatedMenuOptions();

//Functions to manipulate .MID files.
BOOL LoadMID(LPSTR sFile);
BOOL SeekMIDToStart();
BOOL PlayMID();
BOOL StopMID();
BOOL CloseMID();
BOOL GetStatusMID(LPSTR sStat,LPSTR RetBuf,UINT iLen);

//Startup function
int PASCAL WinMain(HINSTANCE hInstance,HINSTANCE hPrevInst,LPSTR sCmdLine,int nShowWindow)
{
	//Load app icon
	iconApp=LoadIcon(hInstance, MAKEINTRESOURCE(IDI_MAIN));	
	//Set instance handle
	hInst = hInstance;
	//Do one time app initialization.
	if (!hPrevInst)
	{
		if (!InitApp(hInstance))
		{
			//Tell the user.
			ErrorMessage("We failed to initialize MidPlay!(0)");
			//We failed!
			return -1;
		}
	}
	//Do every time initialization.
	if (!InitInst(hInstance,nShowWindow,sCmdLine))
	{
		//Tell the user
		ErrorMessage("We failed to initialize MidPlay(1)");
		//We failed!
		return -1;
	}
	//Enter message loop.
	MSG msg;
	while (GetMessage(&msg,NULL,0,0))
	{
		//Process the accelerator keys :)
		TranslateAccelerator(hMainWindow,accelTable,&msg);
		//Translate and dispatch the message.
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	//Success!
	return 0;
}

//Registers our window class. Returns TRUE on success
BOOL InitApp(HINSTANCE hInstance)
{
	//The structure describing the window class
	WNDCLASS  wc;
	
    //Setup class information
    wc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;	//Class style
    wc.lpfnWndProc = WindowProc;					//Window Procedure
    wc.cbClsExtra = 0;								//Nothing
    wc.cbWndExtra = 0;                              //Nothing
    wc.hInstance = hInstance;                       //Instance Handle
    wc.hIcon = iconApp;								//Icon
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);		//Cursor
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); //Background Brush
    wc.lpszMenuName =  MAKEINTRESOURCE(IDR_MAINMENU);//Menu
    wc.lpszClassName = "MidPlayClass";              //Window class name

    //Try to register the class
    return (RegisterClass(&wc));
}

//Creates our window. Returns TRUE on success
BOOL InitInst(HINSTANCE hInstance,int nShowWindow,LPSTR sCmdLine)
{
	//Load accellerator table
	accelTable=LoadAccelerators(hInst,MAKEINTRESOURCE(IDR_MAINACCEL));
	
	//Try to create the window
	hMainWindow = CreateWindow(	"MidPlayClass",
                        		"MidPlay",
                        		WS_OVERLAPPEDWINDOW,
                        		CW_USEDEFAULT,
                        		CW_USEDEFAULT,
                        		CW_USEDEFAULT,
                        		CW_USEDEFAULT,
		                        NULL,
		                        NULL,
		                        hInstance,
		                        NULL);
    //Were we successfull?
    if (!hMainWindow)
        return (FALSE);
        
    //Try to install timer :)
    gTimerID = SetTimer(hMainWindow,5,100,NULL);    

    //If the command line is valid try to load a file.
    if (lstrlen(sCmdLine) > 0)
    {
    	//Try to load the file.
    	LoadMID(sCmdLine);
    }    
    //Show the window
    ShowWindow(hMainWindow, nShowWindow);
    //Paint it.
    UpdateWindow(hMainWindow);
    //Update Text
    UpdateWinText();
    //Load our options.
    LoadOptions();    
    //Update the menu.
    UpdatedMenuOptions();
    return (TRUE);
}

//The window procedure that handles the messages.
long FAR PASCAL __export WindowProc(HWND hWnd,UINT nMsg,WPARAM wParam,LPARAM lParam)
{
	HDC hDC;

    //What should we do? :)
    switch (nMsg)
    {
        //MM Notify messages!
        case MM_MCINOTIFY:
        	//if it was succesfull then restart the song :)
			if ((wParam & MCI_NOTIFY_SUCCESSFUL) != 0)
			{
	        	//First stop the song :)
	        	StopMID();
	        	//If loop mode is on then restart.
	            if (TRUE == bLoopForever)
	            {
	            	//This will restart and play :)
	            	SeekMIDToStart();
	            	PlayMID();
	            }        	
	        }
        	break;
        //Timer(every half second)!
        case WM_TIMER:        	
        	//if we are playing then update everything :)
        	if (TRUE == bSongPlaying)
        	{
        		UpdateWinText(FALSE);        		
        	}
        	break;
        //Menu messages :)
        case WM_COMMAND:
            switch(wParam)
            {
           	case IDM_OPEN:
           	{
           		OPENFILENAME cOFInfo;
           		char sFileName[300]={0,0};           		           		
           		//Get the name of the file through a common dialog.
           		//Init info
           		cOFInfo.lStructSize		=sizeof(cOFInfo);		//Size of structure
           		cOFInfo.hwndOwner		=hMainWindow;           //Parent window
           		cOFInfo.hInstance		=NULL;                  //Not used
           		cOFInfo.lpstrFilter		="Music Files(*.mid;*.rmi)\0*.mid;*.rmi\0";//Filter to go into combo box
           		cOFInfo.lpstrCustomFilter=NULL;					//Not used
           		cOFInfo.nMaxCustFilter	=0;						//Not used
           		cOFInfo.nFilterIndex	=1;						//Default filter
           		cOFInfo.lpstrFile		=sFileName;				//Filename buffer
           		cOFInfo.nMaxFile		=299;					//Filename buffer size
           		cOFInfo.lpstrFileTitle	=NULL;					//Not used
           		cOFInfo.nMaxFileTitle	=0;						//Not used
           		cOFInfo.lpstrInitialDir	=NULL;					//Not used
           		cOFInfo.lpstrTitle		="Choose a music file";	//Dialog window title text
           		cOFInfo.Flags			=OFN_FILEMUSTEXIST;		//Options.
           		cOFInfo.nFileOffset		=0;						//Not used
           		cOFInfo.nFileExtension	=0;						//Not used
           		cOFInfo.lpstrDefExt		="mid";					//Default extension
           		cOFInfo.lCustData		=NULL;					//Not used
           		cOFInfo.lpfnHook		=NULL;					//Not used
           		cOFInfo.lpTemplateName	=NULL;					//Not used
           		//Try to get the filename.
           		if (TRUE == GetOpenFileName(&cOFInfo))
           		{
           			//Try to open the music file now :)           			
    				LoadMID(sFileName);
           		}
           		break;
           	}
           	case IDM_PLAY:
           		PlayMID();
           		break;
           	case IDM_STOP:
           		//Stop playing.
           		StopMID();
           		break;
           	case IDM_QUIT:
           		//Tell us to quit :)
           		DestroyWindow(hWnd);
           		break;
           	case IDM_MIDLOOP:
           		//Toggle option.
           		bLoopForever = !bLoopForever;
           		//Save options.
           		SaveOptions();
           		//Update the menu.
           		UpdatedMenuOptions();
           		//Update the txt.
           		UpdateWinText();
           	default:
           		//Do the default stuff :)
           		DefWindowProc(hWnd, nMsg, wParam, lParam);
           		break;
            }                        
            break;
            
        case WM_ACTIVATE:
        	//Update cursor if needed :)
            if (!GetSystemMetrics(SM_MOUSEPRESENT))
            {
                if (!HIWORD(lParam))
                {
                    if (wParam)
                    {
                        SetCursor(LoadCursor(NULL, IDC_ARROW));                        
                    }
                    ShowCursor(wParam);
                }
            }
            break;

        case WM_PAINT:
            PAINTSTRUCT     ps;

            //Init DC for painting
            hDC = BeginPaint (hWnd, &ps);
            char sPaintBuf[320];
            //If a file is loaded paint the name of it!
            if (TRUE == bMIDOpen)
            {
            	wsprintf(sPaintBuf,sMidFile);
            }
            else
            {
            	//Else tell user to use file|open to open a file.
            	wsprintf(sPaintBuf,"Push Ctrl+O or select Open from the File menu to choose a .MID file.");
            }
            //Actually paint the name of the file.
            TextOut (hDC, 1, 1, sPaintBuf, lstrlen(sPaintBuf));
            //Get the status.            
            if (TRUE == bSongPlaying)
            {
            	wsprintf(sPaintBuf,"Playing - ");
            }
            else
            {
            	wsprintf(sPaintBuf,"Stopped - ");
            }
            //Paint the current status if a file is open.
            if (TRUE == bMIDOpen)
            {
            	//First append the position and length of song before painting.
            	//Position
            	char statbuf[256];
				if (TRUE == GetStatusMID("position",statbuf,255))
				{
					//Add on the position :)
					lstrcat(sPaintBuf,statbuf);					
					//Length
					if (TRUE == GetStatusMID("length",statbuf,255))
					{
						lstrcat(sPaintBuf," of ");
						lstrcat(sPaintBuf,statbuf);
					}
				}            	
            	//Now pain :)
            	TextOut (hDC, 1, 15, sPaintBuf, lstrlen(sPaintBuf));
            }
            //Paint if looping is on.
            if (TRUE == bLoopForever)
            {
            	wsprintf(sPaintBuf,"Looping is On");
            }
            else
            {
            	wsprintf(sPaintBuf,"Looping is Off");
            }
            if (TRUE == bMIDOpen)
            {
            	TextOut (hDC, 1, 30, sPaintBuf, lstrlen(sPaintBuf));
            }
            //Paint other information
            //Author
            wsprintf(sPaintBuf,"Programmed by Kevin Hoffman");
            TextOut (hDC, 1, 45, sPaintBuf, lstrlen(sPaintBuf));            
            wsprintf(sPaintBuf,"Distribute freely to all!");
            TextOut (hDC, 1, 60, sPaintBuf, lstrlen(sPaintBuf));
            //Version
            wsprintf(sPaintBuf,"Version 1.00");
            TextOut (hDC, 1, 75, sPaintBuf, lstrlen(sPaintBuf));
            //End painting
            EndPaint (hWnd, &ps);
            break;
            
        case WM_QUERYDRAGICON:
        	//Return our icon :)
        	return iconApp;        	

        case WM_DESTROY:        	
        	//Make sure MCI device is stopped and closed.
        	StopMID();
        	CloseMID();
        	//Kill the timer!
        	if (0 != gTimerID)
        	{
        		KillTimer(hMainWindow,5);
        	}
        	//Tell application to quit.
            PostQuitMessage(0);
            break;

        default:
        	//Do default stuff :)
            return (DefWindowProc(hWnd, nMsg, wParam, lParam));
    }
    //Success!
    return (NULL);
}

//Issues a MCI device command. Returns TRUE on success or displays an error message and returns FALSE.
//See the mciSendString function.
BOOL TellMCI(LPSTR sCmd,LPSTR sRetStr,UINT nRetLen,HANDLE hCallback,LPSTR sErr,BOOL bReportError)
{
	//First try to send the command to the MCI device
	DWORD dwRet = mciSendString(sCmd,sRetStr,nRetLen,hCallback);
	if (0 != dwRet)
	{
		//If we should report error then report.
		if (TRUE == bReportError)
		{
			//Error! So format message
			char sBuf[300];
			char sErrMsg[350];
			if (TRUE != mciGetErrorString(dwRet,sBuf,299))
			{
				wsprintf(sBuf,"we don't know why!");
			}
			wsprintf(sErrMsg,"%s(%s)",sErr,sBuf);
			//Show the message!
			ErrorMessage(sErrMsg);
		}
		//Failure!
		return FALSE;
	}
	//Success!
	return TRUE;
}

//Shows an error message box.
void ErrorMessage(LPSTR sErr)
{
	//Show the message.
	::MessageBox(NULL,sErr,"MidPlay",MB_ICONEXCLAMATION | MB_OK | MB_TASKMODAL);
}

//Updates the window text based on status.
void UpdateWinText(BOOL bRepaint)
{
	//Build the status string.
	char sText[350];
	//If a file is loaded then build more info. Else build basic info
	if (TRUE == bMIDOpen)
	{
		//If a song is playing then use 1 string else use another :)
		wsprintf(sText,"MidPlay - ");
		lstrcat(sText,sMidFile);		
		if (TRUE == bSongPlaying)
		{
			lstrcat(sText," [Playing] <");			
		}
		else
		{
			lstrcat(sText," [Stopped] <");
		}
		char statbuf[256];
		if (TRUE == GetStatusMID("position",statbuf,255))
		{
			//Add on the position :)
			lstrcat(sText,statbuf);
			if (TRUE == GetStatusMID("length",statbuf,255))
			{
				lstrcat(sText,"-");
				lstrcat(sText,statbuf);
				lstrcat(sText,">");
			}
		}
	}
	else
	{
		wsprintf(sText,"MidPlay");
	}
	//Update the window text.
	SetWindowText(hMainWindow,sText);	
	if (TRUE == bRepaint)
	{
		//Redraw all of the window too.
		InvalidateRect(hMainWindow,NULL,TRUE);
	}
	else
	{
		//Redraw only parts of window.
		RECT rInvalid;
		//Get size of client area.
		GetClientRect(hMainWindow,&rInvalid);
		//Only draw the new position to avoid heavy flickering.
		rInvalid.left+=60;
		rInvalid.top=15;
		rInvalid.bottom=29;
		InvalidateRect(hMainWindow,&rInvalid,TRUE);
	}
	//Update window period.
	UpdateWindow(hMainWindow);	
	//Now its time to update the menu :)
	//First get it.
	HMENU menuWin = GetMenu(hMainWindow);
	if (NULL != menuWin)
	{
		//(if nothing is open then disable both)
		if (TRUE == bMIDOpen)
		{
			//If we are playing then disable play option and enable stop.
			//Else do the oppisite.				
			EnableMenuItem(menuWin,IDM_PLAY, ( bSongPlaying ? MF_GRAYED : MF_ENABLED ) | MF_BYCOMMAND );
			EnableMenuItem(menuWin,IDM_STOP, ( bSongPlaying ? MF_ENABLED : MF_GRAYED ) | MF_BYCOMMAND );
		}
		else
		{
			//Disable both.
			EnableMenuItem(menuWin,IDM_PLAY, MF_GRAYED | MF_BYCOMMAND );
			EnableMenuItem(menuWin,IDM_STOP, MF_GRAYED | MF_BYCOMMAND );			
		}
		DrawMenuBar(menuWin);
	}
}

//Functions to manipulate .MID files.
BOOL LoadMID(LPSTR sFile)
{
	if (TRUE == bMIDOpen)
	{
		TellMCI("stop MIDPLAY_MIDI wait",NULL,0,NULL,"Couldn't stop music device",FALSE);
		//Make sure device is closed first.
		TellMCI("close MIDPLAY_MIDI wait",NULL,0,NULL,"Couldn't close music device",FALSE);		
	}
	//Try to open the MIDI device	
	char sBuf[350];
	wsprintf(sBuf,"open sequencer!%s alias MIDPLAY_MIDI wait",sFile);
	if (!TellMCI(sBuf,NULL,0,NULL,"Couldn't load music file"))
	{
		//Failure!
		return FALSE;
	}
	//MID is open now :)
	bMIDOpen=TRUE;
	wsprintf(sMidFile,sFile);
	//We are not playing yet :)
	bSongPlaying=FALSE;
	if (!SeekMIDToStart())
	{
		UpdateWinText();		
	}
	//Now that its loaded. Tell it to start playing.
	if (!PlayMID())
	{
		//Update the window text anyway :)
		UpdateWinText();		
	}
	//Success!
	return TRUE;
}
BOOL SeekMIDToStart()
{
	//If MID isn't open don't do anything
	if (FALSE == bMIDOpen) return TRUE;
	//Rewind music first.	
	if (!TellMCI("seek MIDPLAY_MIDI to start wait",NULL,0,NULL,"Couldn't rewind music"))
	{
		//Failure!
		return FALSE;
	}	
	//Success!
	return TRUE;
}
BOOL PlayMID()
{
	//If MID isn't open don't do anything
	if (FALSE == bMIDOpen) return TRUE;
	//If we are playing then stop.
	if (TRUE == bSongPlaying)
	{
		//Try to stop the MIDI device	
		if (TRUE == TellMCI("stop MIDPLAY_MIDI wait",NULL,0,NULL,"Couldn't stop music device",FALSE))
		{
			//We are not playing now :)
			bSongPlaying=FALSE;
		}
		//Update the window text :)
		UpdateWinText();
	}
	//Try to play the music
	if (!TellMCI("play MIDPLAY_MIDI notify",NULL,0,hMainWindow,"Couldn't play music"))
	{
		//Failure!
		return FALSE;
	}
	//We are playing now :)
	bSongPlaying=TRUE;
	//Update the window text :)
	UpdateWinText();	
	//Success!
	return TRUE;
}
BOOL StopMID()
{
	//If MID isn't open don't do anything
	if (FALSE == bMIDOpen) return TRUE;
	//If we aren't playing then don't bother.
	if (FALSE == bSongPlaying) return TRUE;	
	//Try to stop the MIDI device	
	if (!TellMCI("stop MIDPLAY_MIDI wait",NULL,0,NULL,"Couldn't stop music device"))
	{
		//Failure!
		return FALSE;
	}
	//We are not playing now :)
	bSongPlaying=FALSE;
	//Update the window text :)
	UpdateWinText();	
	//Success!
	return TRUE;
}
BOOL CloseMID()
{
	//If MID isn't open don't do anything
	if (FALSE == bMIDOpen) return TRUE;
	//If we are playing then stop.
	if (TRUE == bSongPlaying)
	{
		//Try to stop the MIDI device	
		if (TRUE == TellMCI("stop MIDPLAY_MIDI wait",NULL,0,NULL,"Couldn't stop music device",FALSE))
		{
			//We are not playing now :)
			bSongPlaying=FALSE;
		}
		//Update the window text :)
		UpdateWinText();
	}
	//Try to close the MIDI device	
	if (!TellMCI("close MIDPLAY_MIDI",NULL,0,NULL,"Couldn't close music device"))
	{
		//Failure!
		return FALSE;
	}
	//MID is not open now :)
	bMIDOpen=FALSE;
	wsprintf(sMidFile,"");
	//Update the window text :)
	UpdateWinText();	
	//Success!
	return TRUE;
}
BOOL GetStatusMID(LPSTR sStat,LPSTR RetBuf,UINT iLen)
{
	//Try to get the status of the device :)
	
	//First build MCI command string.
	char tempbuf[200];
	wsprintf(tempbuf,"status MIDPLAY_MIDI ");
	lstrcat(tempbuf,sStat);
	lstrcat(tempbuf," wait");
	
	//Then ask for status.
	if (!TellMCI(tempbuf,RetBuf,iLen,NULL,"Couldn't get status of music device",FALSE))
	{
		//Failure!
		return FALSE;
	}
	//Success!
	return TRUE;
}

//Loads options from disk.
void LoadOptions()
{
	//Try to load if we should loop at end of song :)
    bLoopForever=(BOOL)GetPrivateProfileInt("Settings","Loop",2,"MIDPLAY.INI");
    if (1 != bLoopForever && 0 != bLoopForever)
    {
    	bLoopForever=1;
    	//Write correct setting back to INI file.
    	char numbuf[20];
    	wsprintf(numbuf,"%d",(int)(bLoopForever));
    	WritePrivateProfileString("Settings","Loop",numbuf,"MIDPLAY.INI");
    }
}

//Saves options to disk.
void SaveOptions()
{
	//Write if we should loop forever.	
	char numbuf[20];
	wsprintf(numbuf,"%d",(int)(bLoopForever));
	WritePrivateProfileString("Settings","Loop",numbuf,"MIDPLAY.INI");
}

//Updates menu items based on options
void UpdatedMenuOptions()
{
	//Now its time to update the menu :)
	//First get it.
	HMENU menuWin = GetMenu(hMainWindow);
	if (NULL != menuWin)
	{
		//Check or uncheck option based on bLoopForever.
		CheckMenuItem(menuWin,IDM_MIDLOOP, ( bLoopForever ? MF_CHECKED : MF_UNCHECKED ) | MF_BYCOMMAND );
		//Make sure menu is updated.
		DrawMenuBar(menuWin);
	}
}