/* BootRst.c */

#include <dos.h>
#include <fcntl.h>
#include <stat.h>
#include <stdio.h>
#include <stdlib.h>

#define TRUE -1
#define FALSE 0

#define DESCLEN 25
#define BOOTAREAS 7

int Drive;
int Handle;
unsigned char FileBuf[512];
unsigned char BSbuffer[512];

int TEST = FALSE;
int Recognized;
int i, j, k;

typedef struct { int End;
                 char Desc[DESCLEN];
               } bootarea;

#define JUMP 0
#define OEM  1
#define BIOS 2
#define FORMAT 3
#define CODE 4
#define PARTITION 5
#define CODE2 6

bootarea CopyBlock[BOOTAREAS] = { {FALSE, "JUMP"}, 
                                  {FALSE, "OEM"},
                                  {FALSE, "BIOS"},
                                  {FALSE, "FORMAT"},
                                  {FALSE, "CODE"},
                                  {FALSE, "PARTITION"},
                                  {FALSE, "CODE"} };
/* Parm-selectable restore options: */
/* -JUMP   = JMP op-code            */
/* -OEM    = OEM Name and version   */
/* -BIOS   = BIOS parameter block   */
/* -FORMAT = FORMAT information     */
/* -CODE   = Boot code              */
/* -PART   = Parition info          */

bootarea BootArea[BOOTAREAS]={ {0x0002, "JMP op code"},
                               {0x000A, "OEM name & version"},
                               {0x0017, "BIOS parameter block"},
                               {0x001D, "Format information"},
                               {0x01BD, "Boot code"},
                               {0x01CD, "Partition info"},
                               {0x01FF, "Boot code"} };

void read_boot_sector()
{
   static union REGS IReg, OReg;
   static struct SREGS SReg;
   int i, Stat;
   static int AL_id[] = {0x00, 0x01, 0x02, 0x04, 0x06, 0x07, 0x08,
           0x0a, 0x0b, 0x0c};
   static char *AL_err[] = { "Write protected\n", "Invalid drive\n",
      "Drive not ready\n", "CRC/Parity error\n", "Seek error\n", 
      "Unknown media\n", "Sector not found\n", "Write error\n", 
      "Read error\n", "General error\n" };
   static int AH_id[]={0x00, 0x02, 0x03, 0x04, 0x08, 0x10, 0x20,
           0x40, 0x80};
   static char *AH_err[] = { "Other errors\n", "Bad address, sector not found\n",
      "Write protect error\n", "DMA failure\n", "Bad CRC\n", "Controller failed\n",
      "Bad seek\n", "Time out\n"};

   IReg.x.ax = Drive-'A';   
   IReg.x.cx = 1; /* read 1 sector */
   IReg.x.dx = 0; /* read sector 0 */
   SReg.ds = _CS; /* segment of target buffer */
   IReg.x.bx = BSbuffer; /* offset of target buffer */

   int86x( 0x25, &IReg, &OReg, &SReg );
   _SP-=2;

   if(OReg.x.cflag > 0)   {
      if(Stat=biosdisk( 2, (Drive>='C')?(0x80+Drive-'C'):(Drive-'A'), 0, 0, 1, 1, (void *) BSbuffer)) {
            fputs("Error: unable to read boot sector.\n", stderr);
            for(i=0; (i<10) && (AL_id[i]!=IReg.h.al); i++)
               ;
            if( i<10 ) { fputs( AL_err[i], stderr ); }
            for(i=0; (i<9) && (AH_id[i]!=IReg.h.ah); i++)
               ;
            if( i<9 ) { fputs( AH_err[i], stderr ); }
            
            fputs("Bios read failed too.\n", stderr);
            exit(1);
      }
   }
}


void write_boot_sector()
{
   static union REGS IReg, OReg;
   static struct SREGS SReg;
   int i, Stat;
   static int AL_id[] = {0x00, 0x01, 0x02, 0x04, 0x06, 0x07, 0x08,
           0x0a, 0x0b, 0x0c};
   static char *AL_err[] = { "Write protected\n", "Invalid drive\n",
      "Drive not ready\n", "CRC/Parity error\n", "Seek error\n", 
      "Unknown media\n", "Sector not found\n", "Write error\n", 
      "Read error\n", "General error\n" };
   static int AH_id[]={0x00, 0x02, 0x03, 0x04, 0x08, 0x10, 0x20,
           0x40, 0x80};
   static char *AH_err[] = { "Other errors\n", "Bad address, sector not found\n",
      "Write protect error\n", "DMA failure\n", "Bad CRC\n", "Controller failed\n",
      "Bad seek\n", "Time out\n"};

   IReg.x.ax = Drive-'A';   
   IReg.x.cx = 1; /* read 1 sector */
   IReg.x.dx = 0; /* read sector 0 */
   SReg.ds = _CS; /* segment of target buffer */
   IReg.x.bx = BSbuffer; /* offset of target buffer */

   int86x( 0x26, &IReg, &OReg, &SReg );
   _SP-=2;

   if(OReg.x.cflag > 0)   {
      if(Stat=biosdisk(3, (Drive>='C')?(0x80+Drive-'C'):(Drive-'A'), 0, 0, 1, 1, (void *) BSbuffer)) {
            fputs("Error: unable to write boot sector.\n", stderr);
            for(i=0; (i<10) && (AL_id[i]!=IReg.h.al); i++)
               ;
            if( i<10 ) { fputs( AL_err[i], stderr ); }
            for(i=0; (i<9) && (AH_id[i]!=IReg.h.ah); i++)
               ;
            if( i<9 ) { fputs( AH_err[i], stderr ); }
            
            fputs("Bios write failed too.\n", stderr);
            exit(1);
      }
   }
}


void restore()
{
   int i, c, j;
   int Base=0;

   for(i=0; i<BOOTAREAS; i++)  {
      if (CopyBlock[i].End==TRUE) {   
         for(j=Base;j<=BootArea[i].End;j++)
            BSbuffer[j]=FileBuf[j];
         printf("   %s restored.\n",BootArea[i].Desc);
      }
      Base=BootArea[i].End+1;
   }
}


void show_help()
{
   puts("HELP SCREEN\n");
   puts("   BootRst is designed to aid the recovery of a hard disk after a");
   puts("virus infection in the boot sector. The utility allows you to");
   puts("copy portions (or all) of a saved copy of the boot sector back");
   puts("to the hard disk. The boot sector should have been saved to a");
   puts("file (preferably via bootchk (c) Bryan Leaman 1990).");
   puts("\n   To be effective, the boot sector will need to have been");
   puts("saved while intact on removable media and kept safe.");
   puts("\nCommand syntax:");
   puts("       bootrst file.ext d: [/J{UMP}] [/O{EM}] [/B{IOS}] [/F{ORMAT}] [/C{ODE}] [/P{ARTITION}]");
   puts(" -or-  bootrst file.ext d: /A{LL}");
   puts("d:         is the drive whose boot sector you want to restore.");
   puts("file.ext   is the copy of the boot sector that you want to restore");
   puts("           from.");
   puts("parameters:"); 
   puts("\t/JUMP      - restore the jump op-code of the boot sector");
   puts("\t/OEM       - restore the OEM name & version");
   puts("\t/BIOS      - restore BIOS parameter block");
   puts("\t/FORMAT    - restore FORMAT information");
   puts("\t/CODE      - restore Boot Program Code");
   puts("\t/PARTITION - restore disk partition");
   puts("\t/ALL       - restore entire boot sector");
}


int main( int argc, char *argv[] )
{

   fputs("BootRst v1.0: Copyright (c) 1995 by Bryan L. Leaman. All rights reserved.\n", stderr);

   if ((argc==1) || (argc==2 && argv[1][0]=='?' && argv[1][1]=='\0'))  {
      show_help();
      exit(0);
   }
   if (argc<3) {
      fputs("Error: wrong number of parameters.\n", stderr);
      show_help();
      exit(1);
   }

   Drive=*argv[2] & 0xDF; /* upper case drive letter */
   if ((Drive<'A') || (Drive>'Z') ||
       (argv[2][1]!=':' || argv[2][2]!='\0')) {
       fputs("Error: invalid command syntax.\n", stderr);
       show_help();
       exit(1);
   }

   read_boot_sector();
   if( (Handle=_open( argv[1], O_RDONLY )) >= 0 )  {
       _read( Handle, FileBuf, 512 );
       _close(Handle);
   }

   for(i=3;i<argc;i++) {
      for(j=0;j<BOOTAREAS;j++) {
         if (argv[i][0]=='-' || argv[i][0]=='/') {
            Recognized=FALSE;
            for (k=0;k<BOOTAREAS;k++) {
                if (strnicmp(&(argv[i][1]), CopyBlock[k].Desc, 
                    min(strlen(&(argv[i][1])), strlen(CopyBlock[k].Desc))) == 0) {
                    CopyBlock[k].End=TRUE;
                    Recognized=TRUE;
                }
            }
            if ((Recognized==FALSE) && (strnicmp(&(argv[i][1]),"all",
                 min(strlen(&(argv[i][1])), strlen("all")))==0) ) {
               for(k=0;k<BOOTAREAS;k++) CopyBlock[k].End=TRUE;
               Recognized=TRUE;
            }
            if (Recognized==FALSE) {
               if (stricmp(&(argv[i][1]), "TEST")==0) {
                  TEST=TRUE;
               }
               else {
                  fprintf(stderr,"Error: unrecognized parm $s\n",argv[i]);
                  show_help();
                  exit(1);
               }
            }
         }
         else {
            fprintf(stderr,"Error: unrecognized parameter %s\n", argv[i]);
            show_help();
            exit(1);
         }
      }
   }

   fputs("Creating new boot sector...\n",stderr);
   restore();
   if (TEST!=TRUE) {
      write_boot_sector();
      puts("Boot sector written to disk.\n");
   }
   else {
      if( (Handle=open( "TESTRST.MBR", O_WRONLY|O_CREAT|O_BINARY, S_IWRITE )) >= 0 )  {
          _write( Handle, FileBuf, 512 );
          _close(Handle);
          puts("Boot sector saved as TESTRST.MBR.\n");
      }
      else {
         fputs("Error: unable to write test boot sector.\n",stderr);
         exit(1);
      }
   }
}

