/* TCGI.C     Advanced Example program for Help Access Library
              Generates HTML-Output for a Topic as a CGI-Program

   The RTF-Parser contained herein is limited to the subset of RTF-Coding
   applied by Help Access Library. It cannot be used to Convert fully
   featured RTF-Files from Word processing Applications into HTML format.

   This is only an EXAMPLE PROGRAM for Help Access Library and therefore
   it is not a fully featured Help to HTML-Converter, it's just intented do
   give you an Idea of how to deal with Fonts, Colors and Formatting in
   Help Access Library.

   Interesting limitations:
   - Anchors inside of a Topic <a href="...#id"> cannot be handled because
     we'd have to use the HTO_CONTEXT-Flag to do so, and this would drastically
     reduce performance here.
   - SHG-Files built from Metafiles cannot be processed as we'd have to add
     code to convert the Metafile to a Bitmap. This code is part of our DaVinci
     Package...
   - MAPs/SHGs are not supported.

   (C)1996 Herd Software Development, Rudolf-Virchow-Str. 8 / 68642 Buerstadt / Germany"
*/


#include <windows.h>
#pragma argsused
#include <fcntl.h>
#include <io.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include "hlpacces.h"           // Help Access Library Functions
#include "shgimpor.h"           // Import of Bitmaps in Help File Format


#ifdef __WIN32__
# define _export	// Microsoft C can't stand that
#endif


#define ERROR_FONT "<h2><font color=#ff0000>"
#define NORMAL_FONT "</h2><font color=#0><p>\n"



#define MAX_FONTS 1000



/* The ENUMERATTION_PRIVATE_DATA structur contains fields that are needed by
   the callback-function during enumeration of a Topic. */
typedef struct
        {
          BOOL     TitleEnumerated;             /* TRUE after Title has been enumerated first */
          LPCSTR   HlpFileName;                 /* Filename of Help-File currently processed */
          DWORD    TopicOffset;                 /* TopicOffset of Topic currently processed */

          BOOL     Hidden,                      /* text is currently hidden (RTF \v)*/
                   InAnchor,                    /* An anchor-Tag <a ...> has is currently open */
                   HtmlTable,                   /* A table-Tag <table> is currently open */
                   HtmlRow,                     /* A Table-Row <tr><td> is currently open */
                   RtfTable,                    /* RTF opened table with \intbl (will be closed by \row)
                                                   needed to collect several lines of a RTF-Tables to ONE
                                                   HTML-Table     
                                                */
		   InBrdrb;			/* Display line below */	
          LOGFONT  lf[MAX_FONTS],               /* Font-Name from RTF \fonttbl */
                   lflast,                      /* Last font set by <font...> tag */
                   lfnow;                       /* Font that need to be set for next Character */
          RGBQUAD  cl[MAX_FONTS],               /* Color-Informations from RTF \colortbl */
                   cllast,                      /* Last color set b< <font...> tag */
                   clnow;                       /* Color that need to be set for next Character */
        }
          ENUMERATION_PRIVATE_DATA, FAR *LPENUMERATION_PRIVATE_DATA;


/* ------ Local Function prototypes ---------------------------------- */
void    htmlputs(LPCSTR src);
void    SimpleRtfToHtml(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszRtf);
HGLOBAL HDib2HGif(HGLOBAL hDIB, LPDWORD dwGIFSize);




/* CreateAnchor
 *
 * Creates an anchor referencing to the Specified information.
 */
void CreateAnchor(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszTopicText)
{
  char  HlpFileName[MAX_PATH],
        WindowName[50],
        ContextId[50];
  LPSTR dst=ContextId;
  HIFS  NewIFS;
  DWORD ContextIdTopicOffset,
        TitleTopicOffset;

  /* Seperate Jump destination into Context-Id, Window-Name and Filename */
  strcpy(HlpFileName, dta->HlpFileName);

  while (*lpszTopicText)
  {
    switch (*lpszTopicText)
    {
       case '@': *dst=0; dst=HlpFileName; break;
       case '>': *dst=0; dst=WindowName; break;
       default : *dst++=*lpszTopicText;
    }
    lpszTopicText++;
  }
  *dst=0;

  /* Look if jump-Destination is currently available */
  NewIFS=IFSOpen(HlpFileName);
  if (NewIFS)
  {
     if (0xffffffffL != (ContextIdTopicOffset = HlpIsTopic(NewIFS, HlpGenerateHashValue(ContextId))) &&   // Context If can be found.
         HlpGetTopicTitle(NewIFS, ContextIdTopicOffset, NULL, 0, &TitleTopicOffset)&&                    // Need context-Id for start of Topic
         TitleTopicOffset!=dta->TopicOffset                                                              // Do not refer inside of html-file else we'd have to use HTO_CONTEXT and this is slow...
        )
     {
        /* Setup HTML anchor */
        printf("<a href=\"/cgi-bin/tcgi?FILENAME=%s&TOPICOFFSET=0x%08lx\">",HlpFileName,TitleTopicOffset);
        dta->InAnchor=TRUE;
     }

     IFSClose(NewIFS);
  }
}




/* CreateImageRef
 *
 * Creates an reference to an image
 */
void CreateImageRef(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszTopicText)
{
  LPCSTR p=strchr(lpszTopicText,' '),
         szAlign="";

  switch (*(p-1))
  {
    case 'l' : szAlign=" align=left"; break;
    case 'r' : szAlign=" align=right"; break;
  }

  p++;
  *(strchr(p,'}'))=0;

  printf("<img src=\"/cgi-bin/tcgi?FILENAME=%s&IMAGE=%s\"%s>",dta->HlpFileName,p,szAlign);
}




        /* TopicEnumerationProcedure
         *
         * This is the callback-Function this application defined for the call to
         * HlpTopicEnumerate to Enumerate the TOPIC-Files contents
         */
	BOOL FAR PASCAL _export TopicEnumerationProcedure(
                LPCSTR    lpszTopicText,   // Text to be transfered to the Application program
                DWORD     dummy,           // resevred
                TEXTTYPE  eTextType,       // type of Text-String transfered to the Application
                LPARAM    lParam)          // Application-Defined user Data
        #pragma argsused
	{
          LPENUMERATION_PRIVATE_DATA dta = (LPENUMERATION_PRIVATE_DATA) lParam;

          switch (eTextType)
          {
             case TOPIC_RTF_INIT:                     // RTF Control strings
             case TOPIC_RTF_TEXT:                     // RTF Control strings
             case TOPIC_RTF_EXIT:                     // RTF Control strings
                     SimpleRtfToHtml(dta, lpszTopicText);
                     break;

             case TOPIC_TEXT:                   // Normal Text...

                     if (dta->HtmlTable && !dta->RtfTable)
                     {  /* Theoretically a Table always spans only one line in RTF, so we have to collect them */
                        dta->HtmlTable=FALSE;
                        if (dta->HtmlRow)
                           puts("</td></tr>");
                        dta->HtmlRow=FALSE;
                        puts("</table>");
                     }

                     if (dta->HtmlTable && !dta->HtmlRow)
                     {
                        puts("<tr><td valign=top>");
                        dta->HtmlRow=TRUE;
                     }

                     if (memcmp(&dta->clnow,&dta->cllast,sizeof(RGBQUAD))||
                         memcmp(&dta->lfnow,&dta->lflast,sizeof(LOGFONT)))
                     {
                       printf("</font><font size=%d color=#%02x%02x%02x face=\"%s\">"
                              ,
                              dta->lfnow.lfHeight/7,
                              dta->clnow.rgbRed,dta->clnow.rgbGreen,dta->clnow.rgbBlue,
                              dta->lfnow.lfFaceName
                              );

                       memcpy(&dta->clnow,&dta->cllast,sizeof(RGBQUAD));
                       memcpy(&dta->lfnow,&dta->lflast,sizeof(LOGFONT));
                     }

                     htmlputs(lpszTopicText);
                     break;

             case TOPIC_TITLE:                  // Next topic's title
                     if (dta->TitleEnumerated)
                          return FALSE;         // Cancel Enumeration, No more Topics to Enumerate.
                     else
                          dta->TitleEnumerated=TRUE;
                     break;

             case TOPIC_HOTSPOT_START:          // reference to another topic or Anchor
                     if (*lpszTopicText=='%' || // Invisible Hotspots...
                         *lpszTopicText=='*')
                          lpszTopicText++;

                     if (*lpszTopicText!='!')
                        CreateAnchor(dta, lpszTopicText);
                     break;

             case TOPIC_HOTSPOT_END:            // reference to another topic or Anchor END
                     if (dta->InAnchor)
                        puts("</a>");
                     dta->InAnchor=FALSE;
                     break;

             case TOPIC_BMX:                    // Reference to a Bitmap (or Metafile :-( )
                     CreateImageRef(dta, lpszTopicText);
          }

	  return TRUE;                          // Continue enumeration
	}




/* EnumerateTopic
 *
 * Enumerates the Contents of one Topic of a Help File.
 *
 */
void EnumerateTopic(LPCSTR HlpFileName, LPCSTR HlpTopicOffset)
{ HIFS                     ifs= IFSOpen(HlpFileName);
  HTOPIC                   topic;
  DWORD                    TopicOffset;
  ENUMERATION_PRIVATE_DATA dta;
  char                     szTopicTitle[512];
  memset(&dta, 0, sizeof(dta));
  puts(   "Content-type: text/html\n\n"
          "<html><head><title>");

  if (ifs)
  {
      sscanf(HlpTopicOffset+2, "%lx", &TopicOffset);

      /* ------------- Inclue Topic Title as HTML-Title ------------------*/
      HlpGetTopicTitle(ifs, TopicOffset, szTopicTitle, sizeof(szTopicTitle), NULL);
      htmlputs(szTopicTitle);
      puts("</title></head> <body bgcolor=#ffffff>");

      /* ------------- Enumerate the Topic Contents ----------------------*/
      topic = HlpTopicOpen(ifs, HTO_FONTS /*| HTO_CONTEXT | HTO_KEYWORDS*/);          // Open TOPIC-File and load Phrases etc.
      if (topic)
      {
        dta.TitleEnumerated = FALSE;
        dta.HlpFileName     = HlpFileName;
        dta.TopicOffset     = TopicOffset;

        printf("<font size=3>");        // open ANY font-Tag

        //------------ Enumerate the Topic ------------------------------
        HlpTopicSeek(     topic, TopicOffset);
        HlpTopicEnumerate(topic, TopicEnumerationProcedure, (LPARAM)&dta);
        HlpTopicClose(topic);

        puts("</font>");                // close ANY font-Tag
      }
      else
        printf(ERROR_FONT "HlpTopicOpen: Topic-File could not be opened. Maybe it is not a Help-File?\n" NORMAL_FONT);

      puts("</body></html>");

    IFSClose(ifs);

  } else printf(ERROR_FONT "Sorry, can't open File %s" NORMAL_FONT, HlpFileName);


}





/* GetQueryVariable --------------------------------------------*
 * Get a Data-Block signifieing the contents of the Form query
 * variable. returns NULL if Variable nit defined.
 *--------------------------------------------------------------*/
LPCSTR GetQueryVariable(LPCSTR QueryString, LPCSTR VarName)
{   int    VarNameLength= strlen(VarName);
    LPSTR  VarContents  = NULL,
    	   dst;

    int    cint;
    char   c;
    
    /*---------------- Scan for "KEYWORDS"-Variable ------------------*/
    while (QueryString &&
           memcmp(QueryString, VarName, strlen(VarName))
           )
    {
                if (NULL!=(QueryString = strchr(QueryString, '&')))
                           QueryString++;
    }

    /*---------------- If Variable has been found, return contents ---*/
    if (QueryString)
    {  VarContents = dst = (LPSTR) malloc(strlen(QueryString));

       for (QueryString+=VarNameLength;              /* Step over "KEYWORDS="... */
           *QueryString&&*QueryString!='&';
            )
       {
          switch (c=(*QueryString++))
          { case '+': c=' '; break;
            case '%': sscanf(QueryString, "%02x", &cint);
            	      c=(char) cint;
                      QueryString+=2;
          }

          *dst++=c;
       }

       *dst=0;
    }

    return VarContents;
}



/* htmlputs ----------------------------------------------------*
 * Display a string that may contain special Characters as HTML
 *--------------------------------------------------------------*/
void htmlputs(LPCSTR src)
{ LPSTR block = malloc(strlen(src)*6+1000),
        dst;

  if (block)
  {
     for (dst=block;
          *src;
          ++src)
       switch (((unsigned char) *src) & 0xff)
       {
         case 0xe4: strcpy(dst, "&auml;"); dst+=6; break;
         case 0xf6: strcpy(dst, "&ouml;"); dst+=6; break;
         case 0xfc: strcpy(dst, "&uuml;"); dst+=6; break;
         case 0xc4: strcpy(dst, "&Auml;"); dst+=6; break;
         case 0xd6: strcpy(dst, "&Ouml;"); dst+=6; break;
         case 0xdc: strcpy(dst, "&Uuml;"); dst+=6; break;
         case 0xdf: strcpy(dst, "&szlig;"); dst+=7; break;
         case '<' : strcpy(dst, "&lt;"); dst+=4; break;
         case '>' : strcpy(dst, "&gt;"); dst+=4; break;
         case '&' : strcpy(dst, "&amp;"); dst+=5; break;
         case '\"': strcpy(dst, "&quot;"); dst+=6; break;
         default:   if (*src & 0x80)
                    {
                       sprintf(dst, "&#%d;", ((unsigned char) *src) & 0xff);
                       dst+=strlen(dst);
                    }
                    else
                    if (*src >= ' ')
                       *dst++=*src;
       }
     *dst=0;

     fwrite(block, 1, strlen(block), stdout);

     free(block);
  }
}




LPCSTR SimpleReadRtfCommand(LPCSTR lpszRtf, LPSTR dst, LPSTR par);


LPCSTR SimpleReadFontTable(LPCSTR lpszRtf, LPLOGFONT lf)
{
  int   count=0;
  LPSTR dst;

  while (*lpszRtf && *lpszRtf!='}' && count<MAX_FONTS)
  {
    while (*lpszRtf && *lpszRtf!='{' && *lpszRtf!='}') ++lpszRtf;

    if (*lpszRtf=='{')
    {
       while (*lpszRtf && *lpszRtf!=' ') ++lpszRtf;

       if (*lpszRtf==' ')
       {  lpszRtf++;
          for (dst=lf[count++].lfFaceName;
               *lpszRtf && *lpszRtf!=';';
               )
               *dst++=*lpszRtf++;
          *dst=0;

       }

       if (*lpszRtf==';') lpszRtf++;
       if (*lpszRtf=='}') lpszRtf++;
    }
  }

  return lpszRtf;
}


LPCSTR SimpleReadColorTable(LPCSTR lpszRtf, LPRGBQUAD cl)
{
  int   count=0;
  char  szRtfCommand[100],
        szRtfPar[100];

  while (*lpszRtf && *lpszRtf!='}' && count<MAX_FONTS-1)
  {
    switch (*lpszRtf)
    {
      case ';':  { count++; ++lpszRtf; } break;
      case '\\': lpszRtf=SimpleReadRtfCommand(lpszRtf+1, szRtfCommand, szRtfPar);
                      if (!stricmp(szRtfCommand,"red"  )) cl[count].rgbRed  =(BYTE)atoi(szRtfPar);
                 else if (!stricmp(szRtfCommand,"green")) cl[count].rgbGreen=(BYTE)atoi(szRtfPar);
                 else if (!stricmp(szRtfCommand,"blue" )) cl[count].rgbBlue =(BYTE)atoi(szRtfPar);
                 break;
      default:   lpszRtf++;
    }
  }

  return lpszRtf;
}


LPCSTR SimpleReadRtfCommand(LPCSTR lpszRtf, LPSTR szRtfCommand, LPSTR szRtfPar)
{   LPSTR dst;

    for  ( dst=szRtfCommand;
           (*lpszRtf>='a' && *lpszRtf<='z')||
           (*lpszRtf>='A' && *lpszRtf<='Z');
         )
         *dst++=*lpszRtf++;
    *dst=0;

    for  ( dst=szRtfPar;
           (*lpszRtf>='0' && *lpszRtf<='9')||
            *lpszRtf=='-';
         )
         *dst++=*lpszRtf++;
    *dst=0;

    if (*lpszRtf && strchr("\r\n\t ", *lpszRtf))
       ++lpszRtf;

    return lpszRtf;
}


/* SimpleRTfToHtml
 *
 * This routine parses the RTF-Codes transfered by HlpEnumTopic callback
 * to find out about fonts, tables etc. The Coding only supports the
 * subset of RTF used by Help Access Library, it would fail on a
 * RTF-File created by a Word-Processing application, mainly due
 * to the fact that we havent supported restaurtaion of font-
 * attrubutes on { } parantheses.
 *
 * 
 */
void SimpleRtfToHtml(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszRtf)
{
  char  szRtfCommand[100],
        szRtfPar[100];

  while (*lpszRtf)
  {
    if (*lpszRtf=='\\')
    {  lpszRtf=SimpleReadRtfCommand(lpszRtf+1, szRtfCommand, szRtfPar);

       if (*lpszRtf=='}') dta->Hidden=FALSE;

            if (!stricmp(szRtfCommand,"b"))    printf(szRtfPar[0]=='0' ? "</b>" : "<b>");
       else if (!stricmp(szRtfCommand,"v"))    dta->Hidden=szRtfPar[0]!='0';
       else if (!stricmp(szRtfCommand,"ul"))   printf(szRtfPar[0]=='0' ? "</u>" : "<u>");
       else if (!stricmp(szRtfCommand,"i"))    printf(szRtfPar[0]=='0' ? "</i>" : "<i>");
       else if (!stricmp(szRtfCommand,"pard")) dta->InBrdrb=FALSE;
       else if (!stricmp(szRtfCommand,"par"))  printf(dta->InBrdrb ? "<hr>\n" : "<p>\n");
       else if (!stricmp(szRtfCommand,"line")) printf("<br>\n");
       else if (!stricmp(szRtfCommand,"tab"))  printf(" ");
       else if (!stricmp(szRtfCommand,"fonttbl"))  lpszRtf=SimpleReadFontTable(lpszRtf, dta->lf);
       else if (!stricmp(szRtfCommand,"colortbl")) lpszRtf=SimpleReadColorTable(lpszRtf, dta->cl);
       else if (!stricmp(szRtfCommand,"fs"))       dta->lfnow.lfHeight=atoi(szRtfPar);
       else if (!stricmp(szRtfCommand,"cf"))       dta->clnow=dta->cl[atoi(szRtfPar)];
       else if (!stricmp(szRtfCommand,"f" ))       strcpy(dta->lfnow.lfFaceName, dta->lf[atoi(szRtfPar)].lfFaceName);
       else if (!stricmp(szRtfCommand,"brdrb"))    dta->InBrdrb=TRUE;
       else if (!stricmp(szRtfCommand,"brdrt"))    puts("<hr>");

       else if (!stricmp(szRtfCommand,"intbl" ))   { if (!dta->HtmlTable)
                                                         { dta->HtmlTable=TRUE;
                                                           puts("<table border=3>");
                                                         }
                                                     dta->RtfTable=TRUE;
                                                   }
       else if (!stricmp(szRtfCommand,"row" ))     { dta->RtfTable=FALSE;
                                                     puts("</td></tr>");
                                                     dta->HtmlRow=FALSE;
                                                   }
       else if (!stricmp(szRtfCommand,"cell" ))    puts("</td><td valign=top>");
    }
      else
    if (!strchr("\r\n\t{}",*lpszRtf) && !dta->Hidden)
    {
        szRtfCommand[0] = *lpszRtf;
        szRtfCommand[1] = 0;
        htmlputs(szRtfCommand);
        lpszRtf++;
    }
    else
        lpszRtf++;

  }
}



/* ============= Accessing Bitmaps ============================================*/

/* GetImage
 *
 * Access Contents of an Image
 *
 * Graphics are stored as Internal Files named BMnumber or |BMnumber in
 * the Internal File System. These files have the same structure as MRB and
 * SHG-Files created outside of the Help-Files, but there are two other
 * Compression schemes.
 *
 * The SHGImport routine takes a Memory-Block in this SHG-File format
 * and converts it to a DIB (Device Independend Bitmap) or a Metafile
 * depending on the contents of the File.
 *
 * The Segmentation Parameters Handle defined in the last Parameter of
 * SHGImport has the same structure defined in seg.h first introduced with
 * ThinHelp Version 1.0 and contains the hotspot rectangle and
 * Macro/Jump informations.
 *
 * Things get much more complex when an embedded Window DLL is to be
 * used by the Help File. We don't want to discuss this here...
 */
void GetImage(LPCSTR HlpFileName, LPCSTR HlpImage)
{
  HIFS      ifs = IFSOpen(HlpFileName);
  HIFSFILE  ifsfile;
  DWORD     dwSHGSize, dwGIFSize;
  HGLOBAL   hgSHG, hDIB=NULL, hGIF;
  LPSTR     lpSHG,            lpGIF;


  /* MIME content-type information */
  fputs( "Content-type: image/gif\n\n",stdout);

  if (ifs)
  {
    /* Read the "bm0.." or "|bm0.." - File, which is always in SHG-File format */
    ifsfile = IFSOpenFile(ifs, HlpImage);
    if (ifsfile)
    {
       dwSHGSize = IFSSeekFile(ifsfile, 0, SEEK_END);
                   IFSSeekFile(ifsfile, 0, SEEK_SET);

       hgSHG = GlobalAlloc(GHND, dwSHGSize);
       lpSHG = (LPSTR) GlobalLock(hgSHG);
       IFSReadFile(ifsfile, lpSHG, dwSHGSize);

       /* Convert SHG-File to standard DIB format */
       SHGImport(NULL,                            // Parent-Window for error Messages
		 lpSHG,                   	  // Input Data Block
                 &hDIB,                     	  // DIB-Handle
                 NULL,                  	  // Metafile-Handle for SHG containing WMF
                 NULL,                            // Will be filled with Informations on placeable Metafiles
                 NULL                             // Segmentation Informations
		);

       /* Note: SHGs can consist either of Bitmaps or of
          Metafiles. As we didn't want to make things too
          complex, we are only supporting Bitmaps in
          this Example Application.

          Bitmaps pasted into Microsoft Word RTF-Code
          are converted to Metafiles by Mircrosoft Word and
          therefore will not be supported by this example
          Program.

          To Convert Metafiles to Bitmaps you might like
          to use our DaVinci package.
       */

       GlobalUnlock(hgSHG);
       GlobalFree(hgSHG);

       /* for PCs set stdout to binary mode... */
       fflush(stdout);
       setmode( fileno(stdout), O_BINARY);

       if (hDIB)
       {
         /* Convert the Standard format DIB to an LZW-Compressed GIF image */ 
         hGIF = HDib2HGif(hDIB, &dwGIFSize);

         if (hGIF)
         {
            lpGIF = (LPSTR) GlobalLock(hGIF);

            /* Write the GIF-Image */
            fwrite(lpGIF, dwGIFSize, 1, stdout);
             
            GlobalUnlock(hGIF);
            GlobalFree(hGIF);
         }

         GlobalFree(hDIB);
       }


       IFSCloseFile(ifsfile);
    }


    IFSClose(ifs);

  } else printf(ERROR_FONT "Sorry, can't open File %s" NORMAL_FONT, HlpFileName);
}







/* ============= GIF image creation ===========================================*
 *
 * Most of this code has been taken from:
 *
 * Copyright (C) 1991-1994, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 */

#define	MAX_LZW_BITS	12	/* maximum LZW code size (4096 symbols) */

typedef short code_int;		/* must hold -1 .. 2**MAX_LZW_BITS */

#define LZW_TABLE_SIZE	((code_int) 1 << MAX_LZW_BITS)

#define HSIZE		5003	/* hash table size for 80% occupancy */

typedef int hash_int;		/* must hold -2*HSIZE..2*HSIZE */

#define MAXCODE(n_bits)	(((code_int) 1 << (n_bits)) - 1)


/*
 * The LZW hash table consists of two parallel arrays:
 *   hash_code[i]	code of symbol in slot i, or 0 if empty slot
 *   hash_value[i]	symbol's value; undefined if empty slot
 * where slot values (i) range from 0 to HSIZE-1.  The symbol value is
 * its prefix symbol's code concatenated with its suffix character.
 *
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / suffix character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.
 *
 * The hash_value[] table is allocated from FAR heap space since it would
 * use up rather a lot of the near data space in a PC.
 */

typedef DWORD hash_entry;	/* must hold (code_int<<8) | byte */

#define HASH_ENTRY(prefix,suffix)  ((((hash_entry) (prefix)) << 8) | (suffix))
#define MEMZERO(p,l) memset(p,0,l)


/* Private version of data destination object */

typedef struct {
//  struct djpeg_dest_struct pub;	/* public fields */

//  j_decompress_ptr cinfo;	/* back link saves passing separate parm */

  /* State for packing variable-width codes into a bitstream */
  int n_bits;			/* current number of bits/code */
  code_int maxcode;		/* maximum code, given n_bits */
  int init_bits;		/* initial n_bits ... restored after clear */
  long cur_accum;		/* holds bits not yet output */
  int cur_bits;			/* # of bits in cur_accum */

  /* LZW string construction */
  code_int waiting_code;	/* symbol not yet output; may be extendable */
  BOOL first_byte;		/* if TRUE, waiting_code is not valid */

  /* State for LZW code assignment */
  code_int ClearCode;		/* clear code (doesn't change) */
  code_int EOFCode;		/* EOF code (ditto) */
  code_int free_code;		/* first not-yet-used symbol code */

  /* LZW hash table */
  code_int *hash_code;		/* => hash table of symbol codes */
  hash_entry FAR *hash_value;	/* => hash table of symbol values */

  /* GIF data packet construction buffer */
  int bytesinpkt;		/* # of bytes in current packet */
  char packetbuf[256];		/* workspace for accumulating packet */

  /* GIF data destionation memory block */
  LPBYTE lpGIFdst;
  LPBITMAPINFOHEADER lpbmi;

} gif_dest_struct;

typedef gif_dest_struct * gif_dest_ptr;


#define GIFput(c,dinfo) (*(dinfo->lpGIFdst++)=c)

/*
 * Routines to package compressed data bytes into GIF data blocks.
 * A data block consists of a count byte (1..255) and that many data bytes.
 */

void flush_packet (gif_dest_ptr dinfo)
/* flush any accumulated data */
{
  if (dinfo->bytesinpkt > 0) {	/* never write zero-length packet */
    dinfo->packetbuf[0] = (char) dinfo->bytesinpkt++;

    memcpy(dinfo->lpGIFdst, dinfo->packetbuf, dinfo->bytesinpkt);
           dinfo->lpGIFdst +=                 dinfo->bytesinpkt;

    dinfo->bytesinpkt = 0;
  }
}


/* Add a character to current packet; flush to disk if necessary */
#define CHAR_OUT(dinfo,c)  \
	{ (dinfo)->packetbuf[++(dinfo)->bytesinpkt] = (char) (c);  \
	    if ((dinfo)->bytesinpkt >= 255)  \
	      flush_packet(dinfo);  \
	}


/* Routine to convert variable-width codes into a byte stream */

void output (gif_dest_ptr dinfo, code_int code)
/* Emit a code of n_bits bits */
/* Uses cur_accum and cur_bits to reblock into 8-bit bytes */
{
  dinfo->cur_accum |= ((long) code) << dinfo->cur_bits;
  dinfo->cur_bits += dinfo->n_bits;

  while (dinfo->cur_bits >= 8) {
    CHAR_OUT(dinfo, dinfo->cur_accum & 0xFF);
    dinfo->cur_accum >>= 8;
    dinfo->cur_bits -= 8;
  }

  /*
   * If the next entry is going to be too big for the code size,
   * then increase it, if possible.  We do this here to ensure
   * that it's done in sync with the decoder's codesize increases.
   */
  if (dinfo->free_code > dinfo->maxcode) {
    dinfo->n_bits++;
    if (dinfo->n_bits == MAX_LZW_BITS)
      dinfo->maxcode = LZW_TABLE_SIZE; /* free_code will never exceed this */
    else
      dinfo->maxcode = MAXCODE(dinfo->n_bits);
  }
}


/* The LZW algorithm proper */


void clear_hash (gif_dest_ptr dinfo)
/* Fill the hash table with empty entries */
{
  /* It's sufficient to zero hash_code[] */
  MEMZERO(dinfo->hash_code, HSIZE * sizeof(code_int));
}


void clear_block (gif_dest_ptr dinfo)
/* Reset compressor and issue a Clear code */
{
  clear_hash(dinfo);			/* delete all the symbols */
  dinfo->free_code = dinfo->ClearCode + 2;
  output(dinfo, dinfo->ClearCode);	/* inform decoder */
  dinfo->n_bits = dinfo->init_bits;	/* reset code size */
  dinfo->maxcode = MAXCODE(dinfo->n_bits);
}


void compress_init (gif_dest_ptr dinfo, int i_bits)
/* Initialize LZW compressor */
{
  /* init all the state variables */
  dinfo->n_bits = dinfo->init_bits = i_bits;
  dinfo->maxcode = MAXCODE(dinfo->n_bits);
  dinfo->ClearCode = ((code_int) 1 << (i_bits - 1));
  dinfo->EOFCode = dinfo->ClearCode + 1;
  dinfo->free_code = dinfo->ClearCode + 2;
  dinfo->first_byte = TRUE;	/* no waiting symbol yet */
  /* init output buffering vars */
  dinfo->bytesinpkt = 0;
  dinfo->cur_accum = 0;
  dinfo->cur_bits = 0;
  /* clear hash table */
  clear_hash(dinfo);
  /* GIF specifies an initial Clear code */
  output(dinfo, dinfo->ClearCode);
}


void compress_byte (gif_dest_ptr dinfo, int c)
/* Accept and compress one 8-bit byte */
{
  register hash_int i;
  register hash_int disp;
  register hash_entry probe_value;

  if (dinfo->first_byte) {	/* need to initialize waiting_code */
    dinfo->waiting_code = c;
    dinfo->first_byte = FALSE;
    return;
  }

  /* Probe hash table to see if a symbol exists for
   * waiting_code followed by c.
   * If so, replace waiting_code by that symbol and return.
   */
  i = ((hash_int) c << (MAX_LZW_BITS-8)) + dinfo->waiting_code;
  /* i is less than twice 2**MAX_LZW_BITS, therefore less than twice HSIZE */
  if (i >= HSIZE)
    i -= HSIZE;

  probe_value = HASH_ENTRY(dinfo->waiting_code, c);
  
  if (dinfo->hash_code[i] != 0) { /* is first probed slot empty? */
    if (dinfo->hash_value[i] == probe_value) {
      dinfo->waiting_code = dinfo->hash_code[i];
      return;
    }
    if (i == 0)			/* secondary hash (after G. Knott) */
      disp = 1;
    else
      disp = HSIZE - i;
    for (;;) {
      i -= disp;
      if (i < 0)
	i += HSIZE;
      if (dinfo->hash_code[i] == 0)
	break;			/* hit empty slot */
      if (dinfo->hash_value[i] == probe_value) {
	dinfo->waiting_code = dinfo->hash_code[i];
	return;
      }
    }
  }

  /* here when hashtable[i] is an empty slot; desired symbol not in table */
  output(dinfo, dinfo->waiting_code);
  if (dinfo->free_code < LZW_TABLE_SIZE) {
    dinfo->hash_code[i] = dinfo->free_code++; /* add symbol to hashtable */
    dinfo->hash_value[i] = probe_value;
  } else
    clear_block(dinfo);
  dinfo->waiting_code = c;
}


void compress_term (gif_dest_ptr dinfo)
/* Clean up at end */
{
  /* Flush out the buffered code */
  if (! dinfo->first_byte)
    output(dinfo, dinfo->waiting_code);
  /* Send an EOF code */
  output(dinfo, dinfo->EOFCode);
  /* Flush the bit-packing buffer */
  if (dinfo->cur_bits > 0) {
    CHAR_OUT(dinfo, dinfo->cur_accum & 0xFF);
  }
  /* Flush the packet buffer */
  flush_packet(dinfo);
}


/* GIF header construction */


void put_word (gif_dest_ptr dinfo, unsigned int w)
/* Emit a 16-bit word, LSB first */
{
  GIFput(w & 0xFF, dinfo);
  GIFput((w >> 8) & 0xFF, dinfo);
}


void put_3bytes (gif_dest_ptr dinfo, int val)
/* Emit 3 copies of same byte value --- handy subr for colormap construction */
{
  GIFput(val, dinfo);
  GIFput(val, dinfo);
  GIFput(val, dinfo);
}


void emit_header (gif_dest_ptr dinfo)
/* Output the GIF file header, including color map */
/* If colormap==NULL, synthesize a gray-scale colormap */
{
  int BitsPerPixel, ColorMapSize, InitCodeSize, FlagByte, num_colors;
  int i;

  LPBITMAPINFOHEADER lpbmi=dinfo->lpbmi;
  LPRGBQUAD          lprgbq=(LPRGBQUAD)(lpbmi+1);   

  num_colors = (WORD)(1<<lpbmi->biBitCount); 

  if (num_colors > 256)
    exit(1);

  /* Compute bits/pixel and related values */
  BitsPerPixel = 1;
  while (num_colors > (1 << BitsPerPixel))
    BitsPerPixel++;
  ColorMapSize = 1 << BitsPerPixel;
  if (BitsPerPixel <= 1)
    InitCodeSize = 2;
  else
    InitCodeSize = BitsPerPixel;
  /*
   * Write the GIF header.
   * Note that we generate a plain GIF87 header for maximum compatibility.
   */
  GIFput('G', dinfo);
  GIFput('I', dinfo);
  GIFput('F', dinfo);
  GIFput('8', dinfo);
  GIFput('7', dinfo);
  GIFput('a', dinfo);
  /* Write the Logical Screen Descriptor */
  put_word(dinfo, (unsigned int) lpbmi->biWidth);
  put_word(dinfo, (unsigned int) lpbmi->biHeight);
  FlagByte = 0x80;		/* Yes, there is a global color table */
  FlagByte |= (BitsPerPixel-1) << 4; /* color resolution */
  FlagByte |= (BitsPerPixel-1);	/* size of global color table */
  GIFput(FlagByte, dinfo);
  GIFput(0, dinfo); /* Background color index */
  GIFput(0, dinfo); /* Reserved (aspect ratio in GIF89) */
  /* Write the Global Color Map */
  /* If the color map is more than 8 bits precision, */
  /* we reduce it to 8 bits by shifting */
  for (i=0; i < ColorMapSize; i++) {
    if (i < num_colors) {
	  /* Normal case: RGB color map */
	  GIFput(lprgbq[i].rgbRed   , dinfo);
	  GIFput(lprgbq[i].rgbGreen , dinfo);
	  GIFput(lprgbq[i].rgbBlue  , dinfo);
    } else {
      /* fill out the map to a power of 2 */
      put_3bytes(dinfo, 0);
    }
  }
  /* Write image separator and Image Descriptor */
  GIFput(',', dinfo); /* separator */
  put_word(dinfo, 0);		/* left/top offset */
  put_word(dinfo, 0);
  put_word(dinfo, (unsigned int) lpbmi->biWidth); /* image size */
  put_word(dinfo, (unsigned int) lpbmi->biHeight);
  /* flag byte: not interlaced, no local color map */
  GIFput(0x00, dinfo);
  /* Write Initial Code Size byte */
  GIFput(InitCodeSize, dinfo);

  /* Initialize for LZW compression of image data */
  compress_init(dinfo, InitCodeSize+1);
}




/*
 * Finish up at the end of the file.
 */

void finish_output_gif (gif_dest_ptr dest)
{
  /* Flush LZW mechanism */
  compress_term(dest);
  /* Write a zero-length data block to end the series */
  GIFput(0, dest);
  /* Write the GIF terminator mark */
  GIFput(';', dest);
}


/* HDib2HGif
 *
 * A simple GIF-Creation procedure that does not bother much on
 * details of GIF-Coding and is limited in the maximum file size
 * to be transfered, but it's quite OK as all this GIF-Stuff
 * does not have nothing to do With Help Access Library and
 * you shurely do not think us to explain GIF-Coding in an
 * example program about Help Access Library don't you ?
 */
HGLOBAL HDib2HGif(HGLOBAL hDIB, LPDWORD dwGIFSize)
{
  LPBITMAPINFOHEADER lpbmi = (LPBITMAPINFOHEADER) GlobalLock(hDIB);
  LPBYTE             lpbits= ((LPBYTE)(lpbmi+1)) + ((WORD)(1<<lpbmi->biBitCount)) * sizeof(RGBQUAD),
                     lpsrc;

  gif_dest_struct    dinfo;

  HGLOBAL            hGIF = GlobalAlloc(GHND, 0x2000L); /* a really ugly hack... */
  LPBYTE             lpGIF = (LPBYTE) GlobalLock(hGIF);

  UINT               perline = (lpbmi->biWidth * lpbmi->biBitCount +31)/32 * 4,
                     x,y,
                     pixel;

  memset(&dinfo, 0, sizeof(dinfo));
  dinfo.lpbmi=lpbmi;
  dinfo.lpGIFdst=lpGIF;

  /* Allocate space for hash table */
  dinfo.hash_code  = (code_int *)malloc(HSIZE * sizeof(code_int));
  dinfo.hash_value = (hash_entry FAR *)malloc(HSIZE * sizeof(hash_entry));

  emit_header(&dinfo);

  for (y=lpbmi->biHeight;y-->0;)
  {
    lpsrc = lpbits + perline * y;
    for (x=0;x<lpbmi->biWidth;++x)
    {
      switch (lpbmi->biBitCount)
      {
        case 4: pixel = (x&1) ? (*lpsrc++) & 0xf
                              :((*lpsrc  ) >> 4) & 0xf;
                break;
        case 8: pixel = *lpsrc++;
                break;
        case 1: pixel = (*lpsrc >> (7-x%8)) & 1;
                if (x%8==7)
                          lpsrc++;
                break;
        case 24: pixel=*lpsrc; lpsrc+=3; /* hack */
                 break;
      }

      compress_byte(&dinfo, pixel);
    }
  }

  finish_output_gif(&dinfo);
  free(dinfo.hash_code);
  free(dinfo.hash_value);

  GlobalUnlock(hDIB);
  *dwGIFSize=dinfo.lpGIFdst-lpGIF;

  return hGIF;
}



/* ================ Enumerate Help file Titles ======================= */

#define MAX_TITLES   100   

typedef struct
        { LPCSTR HlpKeyWord,
                 HlpFileName;
          int    TitleCounter;
        } TITLEDATA, FAR *LPTITLEDATA;



/* TitleEnumerationProcedure
 *
 * Callback-Routine called by EnumerateContents in call to HlpEnumTitles
 */
BOOL FAR PASCAL TitleEnumerationProcedure(LPCSTR lpszTitle, DWORD dwTopicOffset, LPARAM lParam)
	{
          LPTITLEDATA lptit = (LPTITLEDATA) lParam;
          LPCSTR      HlpKeyWord = lptit->HlpKeyWord,
                      p=NULL;
          LPSTR       dup;

          if (!HlpKeyWord || !*HlpKeyWord ||
              NULL!=(p=strstr(lpszTitle, HlpKeyWord)))
          {
            printf("<LI><a href=\"tcgi?FILENAME=%s&TOPICOFFSET=0x%08lx\">", lptit->HlpFileName, dwTopicOffset);

            if (p)
            {
              dup=strdup(lpszTitle);
              dup[p-lpszTitle]=0;
              htmlputs(dup);
              free(dup);
              printf("<b>");
              dup=strdup(p);
              dup[strlen(HlpKeyWord)]=0;
              htmlputs(dup);
              free(dup);
              printf("</b>");
              htmlputs(p+strlen(HlpKeyWord));
            }
            else
              htmlputs(*lpszTitle ? lpszTitle : "<No Title - maybe popup...>");


            printf("</a>\n");
          }

	  return lptit->TitleCounter++ < MAX_TITLES;
	}




/* EnumerateContents
 *
 * Enumerate the Contents of the Help-File
 */
void EnumerateContents(LPCSTR HlpFileName, LPCSTR HlpKeyWord)
{
  HIFS      ifs = IFSOpen(HlpFileName);
  TITLEDATA       tit;
  LPHELPFILETITLE lpHelpTitle;

  if (ifs)
  {
    /* ----------- Get Help file Title -------------------*/
    lpHelpTitle = HlpGetHelpTitle(ifs);
    if (lpHelpTitle)
    {
      /* -------- MIME Content-Type ----------------------*/
      printf("Content-type: text/html\n\n");

      /* -------- Include Help-File Title as Title -------*/
      printf("<html><head><title>");
      htmlputs(lpHelpTitle->HelpTitle ? lpHelpTitle->HelpTitle : HlpFileName);
      printf("</title></head><body bgcolor=#ffffff>");

      /* -------- Display some Help File informations ----*/
      puts(   "<b>File:</b>"); htmlputs(lpHelpTitle->HelpTitle ? lpHelpTitle->HelpTitle : HlpFileName);
      if (lpHelpTitle->Citation)
         { puts("<p><b>Citation:</b>"); htmlputs(lpHelpTitle->Citation); }
      if (lpHelpTitle->Copyright)
         { puts("<p><b>Copyright:</b>"); htmlputs(lpHelpTitle->Copyright); }

      /* -------- Enumerate the Contents of the Help File-*/
      printf("<hr><OL>");

      memset(&tit, 0, sizeof(tit));

      tit.HlpKeyWord=HlpKeyWord;
      tit.HlpFileName=HlpFileName;

      HlpEnumTitles(ifs, TitleEnumerationProcedure, (LPARAM) &tit);

      /* -------- Write a Trailer ------------------------*/
      printf("</OL><hr>");
      printf(tit.TitleCounter < MAX_TITLES
                ? "Contents listing is complete"
                : "Maximum Number of Titles in list reached. Listing is incomplete");

      printf("<p><center><font size=2>Created using Help Access Library (C) 1996 Herd Software Development</font></center>"
             "</body></html>");

      HlpFreeHelpTitle(lpHelpTitle);
    }
    else printf(ERROR_FONT "Sorry, can't retrieve Title Data for %s" NORMAL_FONT, HlpFileName);

    IFSClose(ifs);

  } else printf(ERROR_FONT "Sorry, can't open File %s" NORMAL_FONT, HlpFileName);
}





/* ============= Main Program Control =========================================*/


/* TCGI Main Program
 *
 */
void TCGI(LPCSTR QueryString)
{
  LPCSTR HlpFileName,
         HlpTopicOffset,
         HlpKeyWord,
         HlpImage;

  HlpFileName    = GetQueryVariable(QueryString, "FILENAME=");
  HlpTopicOffset = GetQueryVariable(QueryString, "TOPICOFFSET=");
  HlpKeyWord     = GetQueryVariable(QueryString, "KEYWORD=");
  HlpImage       = GetQueryVariable(QueryString, "IMAGE=");

  if (HlpFileName)
  {
    if (HlpTopicOffset)
         EnumerateTopic(HlpFileName, HlpTopicOffset);
    else
    if (HlpImage)
         GetImage(HlpFileName, HlpImage);
    else EnumerateContents(HlpFileName, HlpKeyWord);
  } else printf(ERROR_FONT "Sorry, Neccessarry Variables are Missing" NORMAL_FONT);
}





main(int argc, char **argv)
#pragma argsused
{ LPCSTR QueryString=getenv("QUERY_STRING");

  if (QueryString)
  { TCGI(*QueryString=='\"'?QueryString+1:QueryString);
  }
  else
    printf("TCGI    : Advanced Example program for Help Access Library\n"
           "          Generates HTML-Output for a Topic as a CGI-Program\n\n"
           "USAGE   : This Program is intented to be used by a HTTP (WWW-) Server\n"
           "          for Internet Purposes.\n\n"
           "          it is based on the CGI (Common Gateway Interface) standard\n"
           "          for Internet Programming\n\n"
           "          You can simply run the HTTPSERV-Server delivered with Help\n"
           "          Access Library to run This Demo. Use html-formular tcgi.htm\n"
           "EXAMPLE : Run HTTPSERV\n"
           "          Open web browser (Microsoft Internet Explorer, Mosaic ...)\n"
           "          Open page: http://localhost/tcgi.htm\n"
           "(C)1996 Herd Software Development, Rudolf-Virchow-Str. 8 / 68642 Buerstadt / Germany");

  return 0;
}

