/* TRTF.C    Advanced Example program for Help Access Library
             A Microsoft Word 6.0 Import filter for Help Files.

             16-Bit only

   To recompile this example you need to download the Word Text
   Converter SDK from Microsofts ftp-Site, which is available
   for download as GC1039.EXE and includes an example converter
   for both 16 and 32-Bit Word for Windows.

   To install the compiled example, either place an INI-String
   into WINWORD6.INi:

   [MSWord Text Converters]
   Help File=Help File, E:\OFFICE\WINWORD\THLP.CNV,  hlp
   Media Viewer File=Media Viewer File, E:\OFFICE\WINWORD\THLP.CNV,  mvb hlp

   Or Place the TCNV.CNV-File into your Word For Windows directory,
   close word, restart again and open a Help File.

   (C)1996 Herd Software Development, Rudolf-Virchow-Str. 8 / 68642 Buerstadt / Germany"
*/


#include <windows.h>
#pragma argsused
#include <time.h>
#include <stdio.h>
#include <string.h>
#include "hlpacces.h"           // Help Access Libarary Access
#include "shgimpor.h"           // SHG-Import functions

#include "convapi.h"            // Word Converter API
#include "crwin.h"		// for MessageBox wrapper
#include "crmgr.h"		// for WCallOtherStack
#include "util.h"
#include "debug.h"

#ifdef __WIN32__
# define HUGE
#else
# define HUGE huge
#endif

typedef char HUGE *HPSTR;


// Coroutine manager stack
char cdecl rgbStack[8192] = { 0 };


typedef struct _FOREIGNTORTFPARM
	{
	short PercentComplete;
	short cbPass;
	} FOREIGNTORTFPARM;


/* The ENUMERATION_PRIVATE_DATA-Struktucture is just used to show how the
   Application can Transfer a Pointer to an application-defined object
   to the callback-Function by use of the lParam-Parameter.

   The HlpTopicEnumerate-Function does not stop enumeration until the
   Callback-Functions tells it to stop by returning Zero.

   In This example we want to enumerate only once Topics contents,
   so we have to cancel enumeration when the seconds topic Title is
   to be enumerated.

   So we set our marker when the First enumeration took place.
*/

typedef struct
        {
          BOOL      TitleEnumerated;                /* TRUE after Title has been enumerated first */
          HIFS      ifs;                            /* Handle to the Help file internal File System */
          HGLOBAL   ghBuff;                         /* Handle used to pass RTF-Strings to WinWord */
          PFN_CRMGR lpfnOut;                        /* WinWord data destination callback routine */
          int       fceRet;                         /* Word canceled this Process? */
          DWORD     LastTopicOffset,
                    NowTopicOffset;    
        }
          ENUMERATION_PRIVATE_DATA, FAR *LPENUMERATION_PRIVATE_DATA;

void RtfPutc(BYTE ch, NPSTR *dst);
void RtfPuts(LPENUMERATION_PRIVATE_DATA dta, LPCSTR String);

void IncludeImage(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszTopicText);
void IncludeKeyword(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszTopicText);

void CreateIndexField(LPENUMERATION_PRIVATE_DATA dta);
void CreateStyleSheet(LPENUMERATION_PRIVATE_DATA dta);

void TransferMeta2Word(LPENUMERATION_PRIVATE_DATA dta, HMETAFILE hMetaFile, LPMETAFILEHEADER lpmfh);
void TransferDIB2Word(LPENUMERATION_PRIVATE_DATA dta, HGLOBAL hDIB);

     /*
      * TransferToWord
      *
      * Transfer chunks of RTF-Data to Microsoft Word
      */
     int Transfer2Word(LPENUMERATION_PRIVATE_DATA dta, LPCSTR istring, DWORD cbFile)
     {
        int              fceRet = 0,
                         cbPass = 2048;
        HPSTR            string = (HPSTR) istring;
        LPSTR            lpBuff;
	static FOREIGNTORTFPARM ForeignToRtfParm;


	// do the actual file import conversion.  We've already read the prefix,
	// all that remains is to copy the remainder of the file.
	for (; cbFile > 0 && fceRet >= 0; cbFile -= (long)cbPass)
		{
		// write at most 2K of Rtf on each WinWord callback
		cbPass = (cbFile > 2048L) ? 2048 : (short)cbFile;

		// we must resize the buffer, each time, ourselves.
		if (GlobalReAlloc(dta->ghBuff, (long)cbPass+1, GMEM_MOVEABLE) == NULL)
			{
			return fceNoMemory;
			}


		if ((lpBuff = (char FAR *)GlobalLock(dta->ghBuff)) == NULL)
			{
			return fceNoMemory;
			}

                hmemcpy(lpBuff, string, cbPass);
                                string+=cbPass;
                lpBuff[cbPass]=0;
		GlobalUnlock(dta->ghBuff);

		// build argument list for callback
		ForeignToRtfParm.cbPass = cbPass;
		ForeignToRtfParm.PercentComplete = MulDiv(HIWORD(dta->NowTopicOffset),100,HIWORD(dta->LastTopicOffset));


		// pass the Rtf to WinWord.  Word will check for a user abort and
		// may return any of our fce values to us.  If negative, we need to
		// ourselves abort and return that same fce.
		fceRet = (FCE)WCallOtherStack(dta->lpfnOut, hStackCaller,
				(WORD STACKSEG *)&ForeignToRtfParm,
				sizeof(FOREIGNTORTFPARM) / sizeof(short));
		}

         return fceRet;
     }


        /* TopicEnumerationProcedure
         *
         * This is the callback-Function this application defined for the call to
         * HlpTopicEnumerate to Enumerate the TOPIC-Files contents
         */
	BOOL FAR PASCAL _export TopicEnumerationProcedure(
                LPCSTR    lpszTopicText,   // Text to be transfered to the Application program
                DWORD     dummy,           // resevred
                TEXTTYPE  eTextType,       // type of Text-String transfered to the Application
                LPARAM    lParam)          // Application-Defined user Data
	{
          LPENUMERATION_PRIVATE_DATA dta = (LPENUMERATION_PRIVATE_DATA) lParam;
          dta->NowTopicOffset = dummy;

          switch (eTextType)
          {
             case TOPIC_RTF_EXIT:                // RTF Control strings: End of File Marker
                     CreateIndexField(dta);
                     // Fall thru...

             case TOPIC_RTF_TEXT:                // RTF Control strings: Text Data
                     dta->fceRet = Transfer2Word(dta, lpszTopicText, lstrlen(lpszTopicText));        // Just forward it.
                     break;

             case TOPIC_RTF_INIT:                // RTF Control strings: Initialization
                     dta->fceRet = Transfer2Word(dta, lpszTopicText, lstrlen(lpszTopicText));        // Just forward it.
                     if (dta->fceRet>=0)
                         CreateStyleSheet(dta);
                     break;

             case TOPIC_TEXT:                   // Normal Text...
                     RtfPuts(dta, lpszTopicText);
                     break;

             case TOPIC_BMX:                    // Reference to a Bitmap or Metafile
                     IncludeImage(dta, lpszTopicText);

             case TOPIC_KEYWORD:
                     IncludeKeyword(dta, lpszTopicText);

/*             case TOPIC_TITLE:                  // Next topic's title
                     if (dta->TitleEnumerated)
                          return FALSE;         // Cancel Enumeration, No more Topics to Enumerate.
                     else
                          dta->TitleEnumerated=TRUE;
                     break;*/
          }

	  return dta->fceRet >= 0;              // Continue enumeration ?
	}






/* RtfPutc -----------------------------------------------------*
 * Write Char as RTF-Character					*
 *--------------------------------------------------------------*/
void RtfPutc(BYTE ch, NPSTR *dst)
{ if (ch > 0x80)
       { wsprintf(*dst, "\\'%02x", ch); *dst+=4; }
  else switch (ch)
       {
	 case '\\': strcpy(*dst, "\\\\") ;   *dst+=2; break;
	 case '{' : strcpy(*dst, "\\{")  ;   *dst+=2; break;
	 case '}' : strcpy(*dst, "\\}")  ;   *dst+=2; break;
	 case '\t': strcpy(*dst, "\\tab ");  *dst+=5; break;
	 case '\n': strcpy(*dst, "\\par ");  *dst+=5; break;
	 case '\r': strcpy(*dst, "\\line "); *dst+=6; break;
	 default  : if (ch<' ')
			 { wsprintf(*dst, "(0x%02x)", ch); *dst+=7; }
		    else { *(*dst)++=ch; **dst=0; }
       }
}


void RtfPuts(LPENUMERATION_PRIVATE_DATA dta, LPCSTR String)
{
  static char  buf[512];
  NPSTR start=buf,
        dst=start;

  while (*String && dta->fceRet>=0)
  {
     RtfPutc((BYTE) *String++, &dst);
     if (!(*String) || (dst-start)>500)
     {
        dta->fceRet = Transfer2Word(dta, buf, lstrlen(buf));        // Just forward it.
        dst=start;
     }
  }
}

/* IncludeKeyword
 *
 * Include Keyword Data into the RTF
 *
 * lpszTopicText containts the Keyword List Name (K/A...), a Tab-character and
 * list of Keywords assigned to this paragraph.
 */
void IncludeKeyword(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszTopicText)
{
  NPSTR dup = (NPSTR) LocalAlloc(LPTR, lstrlen(lpszTopicText)+1),
        kw,
        KeyWordList;
  if (dup)
  {
     lstrcpy(dup, lpszTopicText);

     KeyWordList=(NPSTR) strchr(dup, '\t');
     if (KeyWordList)
     {
       *KeyWordList++=0;

       for (kw=KeyWordList;
            kw;
            kw=(NPSTR) strchr(kw,';'), kw=kw?kw+1:NULL)
       {
         Transfer2Word(dta, "{\\pard\\plain\\v\\xe {", 19);
         RtfPuts(dta, kw);
         Transfer2Word(dta, "}}", 2);
       }
     }

     LocalFree((HLOCAL) dup);
  }
}


void CreateStyleSheet(LPENUMERATION_PRIVATE_DATA dta)
{
  static NPSTR StyleSheet =
                "{\\stylesheet{\\f0\\fs20\\lang1031 \\snext0 Normal;}\n"
                "{\\s1\\sb240\\sa60\\keepn \\b\\f0\\fs28\\lang1031\\kerning28 \\sbasedon0\\snext0 heading 1;}\n"
                "{\\s2\\sb240\\sa60\\keepn \\b\\i\\f0\\lang1031 \\sbasedon0\\snext0 heading 2;}\n"
                "{\\*\\cs10 \\additive Default Paragraph Font;}\n"
                "{\\s15\\fi-200\\li200\\tqr\\tx4176 \\f0\\fs18\\lang1031 \\sbasedon0\\snext0 index 1;}\n"
                "{\\s16\\fi-200\\li400\\tqr\\tx4176 \\f0\\fs18\\lang1031 \\sbasedon0\\snext0 index 2;}\n"
                "{\\s24\\sb360\\sa240\\brdrt\\brdrth\\brdrw15 \\b\\i\\f0\\fs26\\lang1031 \\sbasedon0\\snext15 index heading;}\n"
                "{\\s25\\sb360\\tqr\\tx9072 \\b\\caps\\f0\\lang1031 \\sbasedon0\\snext0 toc 1;}\n"
                "{\\s26\\li200\\sb240\\tqr\\tx9072 \\b\\f0\\fs20\\lang1031 \\sbasedon0\\snext0 toc 2;}\n"
               "}";
  if (dta->fceRet>=0)
     dta->fceRet=Transfer2Word(dta, StyleSheet, lstrlen(StyleSheet));
}


/* CreateIndexField
 *
 * Create an Index-Field telling word to collect all the \\xe-entries
 * found...
 */
void CreateIndexField(LPENUMERATION_PRIVATE_DATA dta)
{
  static NPSTR IndexField = "\\page{\\b\\f0\\fs40\\s24 Index}\\par\\par{\\field{\\*\\fldinst{\\f0\\cf1 INDEX \\\\h \"\\emdash A\\emdash \" \\\\c \"2\" }}}";

  if (dta->fceRet>=0)
     dta->fceRet=Transfer2Word(dta, IndexField, lstrlen(IndexField));
}



/* IncludeImage
 *
 * Include a Image-reference into the RTF-Code. Note that Images
 * transfered to microsoft wird are coded 8 Bit, while images stored
 * into an RTF-File are coded in hexadecimal.
 */
void IncludeImage(LPENUMERATION_PRIVATE_DATA dta, LPCSTR lpszTopicText)
{
  NPSTR     dup      = (NPSTR) LocalAlloc(LPTR, lstrlen(lpszTopicText)+1),
            sep,
            FileName;
  char      Placement;

  HIFSFILE  ifsfile;
  DWORD     dwSHGSize;
  HGLOBAL   hgSHG;
  static HGLOBAL        hDIB;
         LPSTR          lpSHG;
  static METAFILEHEADER MetaFileHeader;
  static HMETAFILE      hMetaFile;

  lstrcpy(dup, lpszTopicText);

  /* Seperate Filename and placement from {mx fname}-String */
  if (NULL!=(sep = (NPSTR) strchr(dup,' ')))
  { Placement=*(sep-1);
    FileName = sep+1;
    FileName[strlen(FileName)-1]=0;
  }

    hDIB=NULL;
    hMetaFile=NULL;

    /* Read the "bm0.." or "|bm0.." - File, which is always in SHG-File format */
    ifsfile = IFSOpenFile(dta->ifs, FileName);
    if (ifsfile)
    {
       dwSHGSize = IFSSeekFile(ifsfile, 0, SEEK_END);
                   IFSSeekFile(ifsfile, 0, SEEK_SET);

       hgSHG = GlobalAlloc(GHND, dwSHGSize);
       lpSHG = (LPSTR) GlobalLock(hgSHG);
       IFSReadFile(ifsfile, lpSHG, dwSHGSize);

       /* Convert SHG-File to standard DIB format */
       SHGImport(NULL,                            // Parent-Window for error Messages
		 lpSHG,                   	  // Input Data Block
                 &hDIB,                     	  // DIB-Handle
                 &hMetaFile,                  	  // Metafile-Handle for SHG containing WMF
                 &MetaFileHeader ,                // Will be filled with Informations on placeable Metafiles
                 NULL                             // Segmentation Informations
		);

       GlobalUnlock(hgSHG);
       GlobalFree(hgSHG);

            if (hDIB)      TransferDIB2Word(dta, hDIB);
       else if (hMetaFile) TransferMeta2Word(dta, hMetaFile, &MetaFileHeader); 

       IFSCloseFile(ifsfile);
    }
    else
       MessageBox(NULL,"Can't open internal Bitmap File",FileName, MB_OK | MB_ICONSTOP);

  LocalFree((HLOCAL)dup);
}



/* TransferDIB2Word --------------------------------------------*
 * Put Picture in RTF-File					*
 *--------------------------------------------------------------*/
void TransferDIB2Word(LPENUMERATION_PRIVATE_DATA dta, HGLOBAL hDIB)
{ BITMAPINFOHEADER bih, FAR *lpbmi;

  METAHEADER       mh;
  METARECORD NEAR *mr = (METARECORD NEAR *) LocalAlloc(LPTR, sizeof(METARECORD) + 1000);
  WORD *           par;
  DWORD            SizeOfDIBData,
                   SizeOfSetDiEntry,
                   BinarySizeInWords;
  NPSTR            buf = (NPSTR) LocalAlloc(LPTR, 1000);


  lpbmi=(LPBITMAPINFOHEADER) GlobalLock(hDIB);

  if (lpbmi)
  {
       if (!lpbmi->biSizeImage )
               lpbmi->biSizeImage = lpbmi->biHeight * ((lpbmi->biWidth * lpbmi->biBitCount + 31)/32*4);

       bih=*lpbmi;

       SizeOfDIBData    = bih.biSizeImage+sizeof(BITMAPINFOHEADER)+
                                (bih.biBitCount<=8 ? (1<<bih.biBitCount) : 0)*sizeof(RGBQUAD);
       SizeOfSetDiEntry = 3+11+SizeOfDIBData/2;
       BinarySizeInWords=(sizeof(METAHEADER) + 3*(sizeof(METARECORD)-2) +
			   (1+2+2)*sizeof(WORD))/2+3+SizeOfSetDiEntry;

       wsprintf(buf,"{\\pict\\wmetafile8\\picw%ld\\pich%ld{\\bin%ld ",
		     bih.biWidth * 100000L/ (bih.biXPelsPerMeter ? bih.biXPelsPerMeter : 3840),
		     bih.biHeight* 100000L/ (bih.biYPelsPerMeter ? bih.biYPelsPerMeter : 3840),
                     BinarySizeInWords*2);
       Transfer2Word(dta, (BYTE*) buf, lstrlen(buf));

       memset(&mh,0,sizeof(mh));
       mh.mtType        = 1;
       mh.mtHeaderSize  = sizeof(mh)/2;
       mh.mtVersion     = 0x0300;
       mh.mtSize        = BinarySizeInWords;
       mh.mtMaxRecord   = SizeOfSetDiEntry;
       Transfer2Word(dta, (BYTE*) &mh, sizeof(mh));

       mr->rdSize     = 3+1;
       mr->rdFunction = META_SETMAPMODE;
       mr->rdParm[0]  = MM_ANISOTROPIC;
       Transfer2Word(dta, (BYTE*) mr, mr->rdSize * sizeof(WORD) );

       mr->rdSize     = 3+2;
       mr->rdFunction = META_SETWINDOWORG;
       mr->rdParm[0]  = 0;
       mr->rdParm[1]  = 0;
       Transfer2Word(dta, (BYTE*) mr, mr->rdSize * sizeof(WORD) );

       mr->rdSize     = 3+2;
       mr->rdFunction = META_SETWINDOWEXT;
       mr->rdParm[0]  = bih.biHeight;
       mr->rdParm[1]  = bih.biWidth;
       Transfer2Word(dta, (BYTE*) mr, mr->rdSize * sizeof(WORD) );

       mr->rdSize     = SizeOfSetDiEntry;
       mr->rdFunction = META_STRETCHDIB;
       par=(WORD *) mr->rdParm;
       *((DWORD FAR *) par)++= SRCCOPY;			// fdwRop
       *(              par)++= DIB_RGB_COLORS;		// foColorUse
       *(	       par)++= (WORD) bih.biHeight;  	// Height of Source-Rect
       *(	       par)++= (WORD) bih.biWidth;   	// Width of Source-Rect
       *(	       par)++= 0;	 		// x-coord
       *(	       par)++= 0; 			// y-coord
       *(	       par)++= (WORD) bih.biHeight;    	// Height of Dest-Rect
       *(	       par)++= (WORD) bih.biWidth;     	// Width of Dest-Rect
       *(	       par)++= 0;	 		// x-coord Dest
       *(	       par)++= 0; 			// y-coord Dest
       Transfer2Word(dta, (BYTE*) mr, (3+11) * sizeof(WORD) );

       LocalFree((HLOCAL)mr);


    // --------------- Copy Hex Data Strings -----------------
    Transfer2Word(dta, (BYTE*) lpbmi, SizeOfDIBData );

    Transfer2Word(dta, (BYTE*) "\3\0\0\0\0\0}}", 8);

    GlobalUnlock(hDIB);
    GlobalFree(hDIB);
  }

  LocalFree((HLOCAL)buf);
}

/* TransferMeta2Word -------------------------------------------*
 * Put Picture in RTF-File					*
 *--------------------------------------------------------------*/
void TransferMeta2Word(LPENUMERATION_PRIVATE_DATA dta, HMETAFILE hMetaFile, LPMETAFILEHEADER lpmfh)
{
  NPSTR            buf = (NPSTR) LocalAlloc(LPTR, 1000);
  HGLOBAL          hMetaBits = GetMetaFileBits(hMetaFile);
  LPSTR            lpMetaBits;

  if (hMetaBits)
  {
       lpMetaBits = (LPSTR) GlobalLock(hMetaBits);

       wsprintf(buf,"{\\pict\\wmetafile8\\picw%d\\pich%d{\\bin%ld ",
		     MulDiv((lpmfh->bbox.right -lpmfh->bbox.left), 2540, lpmfh->inch),
		     MulDiv((lpmfh->bbox.bottom-lpmfh->bbox.top) , 2540, lpmfh->inch),
                     GlobalSize(hMetaBits));   
       Transfer2Word(dta, (BYTE*) buf, lstrlen(buf));

       Transfer2Word(dta, lpMetaBits, GlobalSize(hMetaBits) );
       Transfer2Word(dta, (BYTE*) "}}", 2 );

     GlobalUnlock(hMetaBits);
     GlobalFree(hMetaBits);
  }

  LocalFree((HLOCAL)buf);
}



DeclareFileName         // Debug Support


// prefix to add to Rtf on export and strip on import
static char szFilePrefix[] = { 0x3f, 0x5f, 0x03, 0x00 };
#define cchFilePrefix (sizeof(szFilePrefix))

// WIN.INI entry information
char szzIniNames[] = 
	"Help File\0Media Viewer File\0";
char szzIniExts[] = "^.hlp\0^.mvb ^.hlp\0^.mvb\0";


enum
	{
	errMemory = 0,
	// other values for other conditions
	};

static char *rgszErrorMsg[] =
	{
	"Insufficient global memory",
        "Writing is not supported by this converter",
	// other messages for other conditions
	};

/* E R R O R */
/*----------------------------------------------------------------------------
	%%Function: Error

	Purpose:
	display an error message box containing text identified by error code.

	Parameters:
	iError :	index of error text
----------------------------------------------------------------------------*/
int Error(short iError)
{
	// DLLs should always use system modal message boxes
	return (MessageBox((HWND)NULL, (LPSTR)(rgszErrorMsg[iError]),
			(LPSTR)"Conversion Error", MB_OK | MB_SYSTEMMODAL));
}


/* G E T  I N I  E N T R Y */
/*----------------------------------------------------------------------------
	%%Function: GetIniEntry

	Purpose:
	Return information appropriate for this converter's WIN.INI entries

	Parameters:
	ghIniName :	global handle to destination buffer for converter names
	ghIniExt :	global handle to destination buffer for file extensions
----------------------------------------------------------------------------*/
void FAR PASCAL GetIniEntry(HANDLE ghIniName, HANDLE ghIniExt)
{
	// No SaveState to initialize coroutine manager since Word doesn't
	// call us through it.  We can do no wrapped Windows calls.

	if (!FCopySzzToGhsz(szzIniNames, ghIniName))
		return;

	if (!FCopySzzToGhsz(szzIniExts, ghIniExt))
		return;

	return;
}


/* I S  F O R M A T  C O R R E C T */
/*----------------------------------------------------------------------------
	%%Function: IsFormatCorrect

	Purpose:
	Return fTrue if ghszFile refers to a file in a format understood by this
	converter.  If this converter deals with multiple formats, return the
	specific format of this file in *ghszDescrip.  Return an empty
	string in *ghszDescrip if version is unknown.

	Parameters:
	ghszFile :	global handle to '\0'-terminated filename
	ghszDescrip :	global handle to return file's version

	Returns:
	0 if format not recognized, 1 if format recognized, fce if error
----------------------------------------------------------------------------*/
FCE FAR PASCAL IsFormatCorrect(HANDLE ghszFile, HANDLE ghszDescrip)
{
	HFILE hFile;
	char szFileName[260];
	char rgb[cchFilePrefix];

	// Each entrypoint calls SaveState with the sum of the sizes of its
	// arguments to initialize coroutine manager
	SaveState(sizeof(ghszFile) + sizeof(ghszDescrip));

	// copy filename locally; file open doesn't want a global handle
	AssertSz(ghszFile != NULL, "NULL filename in IsFormatCorrect");
	if (!FCopyGhszToSz(ghszFile, szFileName))
		return fFalse;
	// translate filename from the oem character set to the windows set
	OemToAnsi((LPCSTR)szFileName, (LPSTR)szFileName);

	// open file, read sufficient bytes to identify file type, and close.
	if ((hFile = _lopen((LPSTR)szFileName, OF_READWRITE)) == HFILE_ERROR)
		return fFalse;
	_lread(hFile, (LPSTR)rgb, cchFilePrefix);
	_lclose(hFile);

	// if the file begins with our szFilePrefix, it's most likely in our
	// file format
	if (strncmp(szFilePrefix, rgb, cchFilePrefix) != 0)
		return fFalse;

	// this converter always claims files are in the 1.5 version format
	FCopySzToGhsz("Help File", ghszDescrip);

	return fTrue;
}




/* F O R E I G N  T O  R T F */
/*----------------------------------------------------------------------------
	%%Function: ForeignToRtf

	Purpose:
	Convert a file to Rtf using the format specified in ghszDescrip.

	Parameters:
	ghszFile :	global handle to '\0'-terminated filename to be read
	ghBuff :	global handle to buffer in which chunks of Rtf are passed
				to WinWord.
	ghszDescrip: identifies which file format to translate.  This
				string is the one selected by the user in the 'Confirm
				Conversions' dialog box in Word.
	ghszSubset:	identifies which subset of the file is to be converted.
				Typically used by spreadsheet converters to identify
				subranges of the entire spreadsheet to import.
	lpfnOut:	callback function provided by WinWord to be called whenever
				we have a chunk of Rtf to return.

	Returns:
	fce indicating success or failure and cause
----------------------------------------------------------------------------*/
FCE FAR PASCAL ForeignToRtf(HANDLE ghszFile, HANDLE ghBuff, HANDLE ghszDescrip, HANDLE ghszSubset, PFN_CRMGR lpfnOut)
#pragma argsused
{
        /* ---------- Variables used for Help Access Library --------------*/
        HIFS   ifs;
        HTOPIC topic;
        ENUMERATION_PRIVATE_DATA dta;

        /* ---------- Variables used for Word Converter Interface ---------*/
	char szFileName[260];
	FCE fceRet = fceNoErr;

	// Each entrypoint calls SaveState with the sum of the sizes of its
	// arguments to initialize coroutine manager
	SaveState(12);	// 12 == sizeof(ghszFile) + sizeof(ghBuff)+ ...

	// copy filename locally; file open doesn't want a global handle
	AssertSz(ghszFile != NULL, "NULL filename in ForeignToRtf");
	if (!FCopyGhszToSz(ghszFile, szFileName))
		return fceNoMemory;
	// translate filename from the oem character set to the windows set
	OemToAnsi((LPCSTR)szFileName, (LPSTR)szFileName);

	// open file and read sufficient bytes to identify file type.
        // Note that Help Access Library automatically verifys that the file
        // is really a Help-File when opening the Handles...

        /* ---------- Initialize Variables --------------------------------*/
        memset(&dta, 0, sizeof(dta));
        dta.ghBuff =ghBuff;                        /* Handle used to pass RTF-Strings to WinWord */
        dta.lpfnOut=lpfnOut;                       /* WinWord data destination callback routine */

        dta.ifs = ifs = IFSOpen(szFileName);                                                      // Open the .HLP-File file system
        if (ifs)
        {
            topic = HlpTopicOpen(ifs, HTO_FONTS | HTO_CONTEXT | HTO_KEYWORDS);          // Open TOPIC-File and load Phrases etc.
            if (topic)
            {
              //------------ Get Last Topic in File ---------------------------
              HlpGetTopicTitle(ifs, 0x7fffffffL, NULL, 0, &dta.LastTopicOffset);

              //------------ Enumerate the Topic ------------------------------
              HlpTopicSeek(     topic, 0);
              HlpTopicEnumerate(topic, TopicEnumerationProcedure, (LPARAM)&dta);
              HlpTopicClose(    topic);

              fceRet = dta.fceRet;

            }
            else
              fceRet = fceWrongFileType;;

            IFSClose(ifs);
        } else
            fceRet = fceWrongFileType;


	return fceRet;
}


typedef struct _RTFTOFOREIGNPARM
	{
	short PercentComplete;
	BOOL fPicture;
	} RTFTOFOREIGNPARM;


/* R T F  T O  F O R E I G N */
/*----------------------------------------------------------------------------
	%%Function: RtfToForeign

	Purpose:
	Convert a stream of Rtf from Word to a file in the format specified in
	ghszDescrip.

	Parameters:
	ghszFile :	global handle to '\0'-terminated filename to be written
	ghBuff :	global handle to buffer in which chunks of Rtf are passed
				to us from WinWord.
	ghszDescrip: identifies which file format to translate to.  This
				string is the one selected by the user in the 'Save File
				as Type' drop down in Word's Save dialog.
	lpfnIn:		callback function provided by WinWord to be called whenever
				we are ready for more Rtf.

	Returns:
	fce indicating success or failure and cause
----------------------------------------------------------------------------*/
FCE FAR PASCAL RtfToForeign(HANDLE ghszFile, HANDLE ghBuff, HANDLE ghszDescrip, PFN_CRMGR lpfnIn)
#pragma argsused
{
        Error(1);
	return fceWrongFileType;
}



