/*------------------------------------------------------------------

      L E O N A R D O : Statische Linkbibliothek mit Routinen zur
			Handhabung von DIBs (Device-Independend-Bitmaps)
			im Anwendungsprogramm

	  (C) 1993-1995	Dipl. Ing. Bernd Herd
			Heildelberger Landstr. 316
			64297 Darmstadt

    L O N A R D O . H : Header-Datei mit Funktionsprototypen f
			Anwendungsprogramm

  ------------------------------------------------------------------*/

#ifndef __LEONARDO_H
#define __LEONARDO_H

#ifdef __cplusplus
extern "C" {            /* Assume C declarations for C++ */
#endif  /* __cplusplus */


#ifndef API
# ifdef __CRTDLL__
  #define API FAR PASCAL _export	 	// Markiert Funktionen exportierbar (gut fr DLLs)
# else
  #define API PASCAL			// Soll angeblich effizienter sein...
# endif
#endif

typedef HGLOBAL HDIB;			// Ist nur eine Umbenennung ...

/* OpenDIB ----------------------------------------------------------------
   .BMP-Datei lesen und als Speicherhandle in CF_DIB-Format zurckliefern
   Parameter : szFile		Dateiname der Bitmap-Datei, blich Endungen .BMP oder .DIB
   Rckgabe  : NULL	      : Konvertierung gescheitert, Datei nicht vorhanden ...
	       sonst	      : Speicherhandle in CD_DIB-Format
  -------------------------------------------------------------------------*/
HDIB API OpenDIB (LPSTR szFile);






/* WriteDIB----------------------------------------------------------------
   Speicherhandle in CF_DIB-Format in .BMP-Datei umwandeln
   Parameter : szFile		Dateiname der Bitmap-Datei, blich Endungen .BMP oder .DIB
	       hdib		Handle der DIB
   Rckgabe  : TRUE	      : Erfolgreich gespeichert
	       FALSE	      : Fehler
  -------------------------------------------------------------------------*/
BOOL API WriteDIB (LPSTR szFile, HDIB hdib);




/* DibInfo ----------------------------------------------------------------
   Extrahiert den BITMAPINFOHEADER aus einem Speicherhandle in CF_DIB-Format
   Parameter : hbi		Speicherhandle
	       lpbi		Zieladresse fr BITMAPINFOHEADER
   Rckgabe  : TRUE	      : Erfolgreich 
	       FALSE	      : Fehler
  -------------------------------------------------------------------------*/
BOOL API DibInfo (HDIB hbi, LPBITMAPINFOHEADER lpbi);


/* CreateBIPalette --------------------------------------------------------
   Erzeugt eine GDI-Palette aus einem Zeiger auf einen BITMAPINFOHEADER
   Parameter : lpbi		Adresse des BITMAPINFOHEADER
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Paletten-Handle
  -------------------------------------------------------------------------*/
HPALETTE API CreateBIPalette (LPBITMAPINFOHEADER lpbi);


/* CreateDIBPalette -------------------------------------------------------
   Erzeugt eine GDI-Palette aus einem Speicherhandle in CF_DIB-Format
   Parameter : hdib 		Speicherhandle in CF_DIB-Format
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Paletten-Handle
  -------------------------------------------------------------------------*/
HPALETTE API CreateDibPalette (HDIB hdib);



WORD API DibNumColors (VOID FAR *pv);		// Anzahl der Farben bei BITMAPINFO oder BITMAPINFOHEADER
WORD API PaletteSize (VOID FAR *pv);		// Gre der Farbpalette in Bytes bei BITMAPINFO oder BITMAPINFOHEADER



/* DibFromBitmap ----------------------------------------------------------
   Umwandlung einer Bitmap in eine DIB (die Bitmap bleibt gltig)
   Parameter : hbm  		Handler der Quell-Bitmap (Device-Dependend Bitmap)
	       biStyle          Sollte BI_RGB sein
	       biBits		Anzahl der Bits je pixel
				0 - Wie Device-Dependend-Bitmap
				1 - Monochrom
				4 - 16-Farbig
				8 - 256-Farbig
				24- TrueColor
	       hpal		Handle der zugrundeliegenden Farbpalette
				oder NULL, falls Keine bzw. Systempalette
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Speicherhandle in CF_DIB-Format
  -------------------------------------------------------------------------*/
HDIB API DibFromBitmap (HBITMAP hbm, DWORD biStyle, WORD biBits, HPALETTE hpal);




/* BitmapFromDib ----------------------------------------------------------
   ein Speicherhandle in CF_DIB-Format in Eine Einheitanabhngige Bitmap
   umwandeln
   Parameter : hdib		Speicherhandle in CF_DIB-Format
	       hpal		Handle der zugrundeliegenden Farbpalette
				oder NULL, falls Keine bzw. Systempalette
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Handle der Einheitenabhngigen Bitmap
  -------------------------------------------------------------------------*/
HBITMAP API BitmapFromDib (HDIB hdib, HPALETTE hpal);




/* CreateDIB --------------------------------------------------------------
   erzeugt eine leere (schwarze) DIB mit vorgegebener Gre und Farbauflsung.
   Wichtig: Es werden nur standard-Farben in die Farbpalette eingetragen (16 VGA-Farben)
   Parameter : Width		Neue DIB-Breite
	       Height		Neue DIB-Hhe
	       BitCount	        Anzahl Bits / Pixel
				1  - Monochrom
				4  - 16 VGA Farben
				8  - 256 Farben (Fr Paletten)
				24 - TrueColor
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Handle der Einheitenunabhngigen Bitmap
  -------------------------------------------------------------------------*/
HDIB API CreateDIB(UINT Width, UINT Height, UINT BitCount);




/* CreateCompatibleDIB ----------------------------------------------------
   erzeugt eine leere (schwarze) DIB mit derselben Farbauflsung und Palette
   wie eine andere, aber mit einer anderen Gre. Die Quell-DIB bleibt gltig
   Parameter : hdib		Speicherhandle in CF_DIB-Format
	       Width		Neue DIB-Breite
	       Height		Neue DIB-Hhe
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Handle der Einheitenunabhngigen Bitmap
  -------------------------------------------------------------------------*/
HDIB API CreateCompatibleDIB(HDIB hdib, UINT Width, UINT Height);




/* DrawBitmap -------------------------------------------------------------
   Eineitenabhngige Bitmap in Device-Context zeichnen
   Parameter : hdc		Ziel-Device-Context
	       x,y		Koordinaten im Ziel-Einheitenkontext
	       hbm		Handle der Device-Dependend-Bitmap
	       rop		AUszufhrende Rasteroperation, normal SRCCOPY	 	
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Ok
  -------------------------------------------------------------------------*/
BOOL API DrawBitmap (HDC hdc, int x, int y, HBITMAP hbm, DWORD rop);



/* DibBlt -----------------------------------------------------------------
   Ausgabe einer DIB in einen Device-Context. Parameter wie bei BitBlt
   Parameter : hdc		Ziel-Device-Context
	       x0,y0		Linke oebere Koordinaten im Ziel-Einheitenkontext
               dx,dy		Breite und Hhe in Ziel-Einheiten
	       hdib		Handle der DIB
	       x1,y1		Linke obere Ecke der Quell-Bitmap (sinnvoll, falls nur Ausschnitt gezeichnet wird).
	       rop		Auszufhrende Rasteroperation, normal SRCCOPY
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Ok
  -------------------------------------------------------------------------*/
BOOL API DibBlt (HDC hdc, int x0, int y0, int dx, int dy, HDIB hdib, int x1, int y1, LONG rop);



/* StrechDibBlt -----------------------------------------------------------
   Ausgabe einer DIB in einen Device-Context. Parameter wie bei StretchBlt
   Parameter : hdc		Ziel-Device-Context
	       x,y		Linke obere Koordinaten im Ziel-Einheitenkontext
               dx,dy		Breite und Hhe in Ziel-Einheiten
	       hdib		Handle der DIB
	       x0,y0		Linke obere Ecke der Quell-Bitmap
               dx0,dy0		Breite und Hhe des Quell-Rechtecks
	       rop		Auszufhrende Rasteroperation, normal SRCCOPY
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Ok
  -------------------------------------------------------------------------*/
BOOL API StretchDibBlt (HDC hdc, int x, int y, int dx, int dy, HDIB hdib, int x0, int y0, int dx0, int dy0, LONG rop);



/* CopyHandle -------------------------------------------------------------
   Globales Speicherhandle kopieren
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Handle der Kopie
  -------------------------------------------------------------------------*/
HGLOBAL API CopyHandle (HGLOBAL h);



DWORD API lwrite (HFILE fh, VOID FAR *pv, DWORD ul);	// wie _hread, aber auch WIN 3.0
DWORD API lread (HFILE fh, VOID far *pv, DWORD ul);	// wie _hwrite, aber auch WIN 3.0



/* LoadDib ----------------------------------------------------------------
   Wie LoadBitmap, aber liefert HDIB statt HBITMAP. Damit sind 256 Farben mglich
   Parameter : hInstance	: Instanz des Moduls, in dem die Bitmap-Ressource
				  zu finden sein sollte
	       lpBitmapName	: Zeigt auf einen nullterminierten String, der den
	       			  Namen der Bitmap enthlt			  	
   Rckgabe  : NULL		Bei Fehlern
	       sonst		Speicherhandle in CF_DIB-Format
  -------------------------------------------------------------------------*/
HDIB API LoadDib(HINSTANCE hInstance, LPSTR lpBitmapName);



/* GrayDIB ----------------------------------------------------------------
   Umwandeln der Paletteninformationen in der angegebenen DIB in Grauwerte
   Die Eigenschaften des menschlichen Auges werden bercksichtigt
   Paameter  : hDIB            : Zu konvertierende Bitmap

   RckGabe  : NULL	         Wenn erfolglos
	       hDIB		 Wenn erfolgreich

  -------------------------------------------------------------------------*/
HDIB API GrayDIB(HDIB hDIB);


/* DitherTo4 -------------------------------------------------------------
   schnelles Dithern einer 24-Bit-DIB oder 8-Bit-Dib in eine 4-Bit-Dib mit
   Windows-Standard-Systempalette

   RckGabe  : NULL	         Wenn erfolglos
				 Grnde: DIB hat < 8 Bit Farbauflsung
				 Speichermangel
				 DIB hat mehr als 65534 Pixel in X-Richtung
	       hDIB		 Wenn erfolgreich

  -------------------------------------------------------------------------*/
HDIB FAR PASCAL DitherTo4(LPBITMAPINFOHEADER lpDib);





/* DitherTo8 -------------------------------------------------------------
   schnelles Dithern einer 24-Bit-DIB in eine 8-Bit-Dib (256 Farben) mit
   Farbpalette.

   Parameter:  lpDIB		 Zeiger auf einen Speicherblock im CF_DIB-
				 Datenformat, der die DIB-Daten mit 24 Bit
				 Farbtiefe enthlt.

	       std		 Steuert die Anzahl der nichtoptimierten Farben
				 in der Farbpalette der neuen DIB:
				 0 - Keine
				 2 - 2*2*2 = 8 Farben
				 3 - 3*3*3 = 27 Farben
				 4 - 4*4*4 = 64 Farben
				 5 - 5*5*5 = 125 Farben
				 6 - 6*6*6 = 216 Farben

	       More		 Anzahl der zustzlichen optimierten Farben in
				 der Farbpalette der neuen DIB

	       lpPal		 Zeiger auf eine DIB mit 8 Bit Farbtiefe, deren
				 Farbpalette fr die Erstellung der neuen DIB
                                 benutzt werden soll.


   RckGabe  : NULL	         Wenn erfolglos
				 Grnde: DIB hat <> 24 Bit Farbauflsung
				 Speichermangel
				 DIB hat mehr als 65534 Pixel in X-Richtung
	       hDIB		 Wenn erfolgreich


   Verwendung von std,More und lpPal:
				   std  More   lpPal
   1. Schnellstmgliches Dithern    6    0     NULL
   2. Optimale Farbpalette 	    0   236    NULL
   3. Farbpalette einer anderen	    0    0     lpdib
      DIB bernehmen
   4. Kombination mehrerer DIBs     5    50    NULL	125 farben fr alle DIBs gleich,
      zur gleichzeitigen Anzeige			50  zustzliche Farben zum Optimieren

  -------------------------------------------------------------------------*/
#ifndef __WIN32__		// Funktion z.Zt. nur f. 16-Bit-Windows untersttzt
HDIB FAR PASCAL DitherTo8(LPBITMAPINFOHEADER lpDib, int std, int More, LPBITMAPINFOHEADER lpPal);
#endif




/* RotateDIB -------------------------------------------------------------
   Erzeugt eine neue DIB gleichen Aufbaus, die eine gedrehte Darstellung
   der bergebenen DIB enthlt. Benutzt TransformDIB

   hdib      : Handle der zu drehenden DIB
   Angle     : Winkel in 1/10 Grad, Positiv fr Linksdrehung
   clBkColor : Hintergrundfarbe fr den Rand der neuen DIB

   RckGabe  : NULL	         Wenn erfolglos
	       hDIB		 Wenn erfolgreich

  -------------------------------------------------------------------------*/
HDIB API        RotateDIB(HDIB hdib,
			  int Angle,
			  COLORREF clBkColor);



/* StretchDIB ------------------------------------------------------------
   Erzeugt eine neue DIB gleichen Aufbaus, die vergrerte oder
   verkleinerte Darstellung der bergebenen DIB enthlt.

   Dient nicht zuletzt Dazu, den StretchBlt - Bug von Windows 3.x zu umgehen.

   hdib      : Handle der Quell-DIB
   cxDest    : Breite der neuen DIB
   cyDest    : Hhe der neuen DIB
   rc	     : Zeiger auf RECT-Struktur mit den Koordinaten des zu verarbeitenden
	       Ausschnitts der Quell-DIB oder NULL, wenn die gesamte Quell-DIB
	       bearbeitet wird.		

   RckGabe  : NULL	         Wenn erfolglos
	       hDIB		 Wenn erfolgreich

  -------------------------------------------------------------------------*/
HDIB API        StretchDIB(HDIB  hdib,
			  int    cxDest,
			  int    cyDest,
			  LPRECT rc);



/* -------------------------------------------------------------*
 *								*
 *--------------------------------------------------------------*/

// ------------ Callback-Routine fr Koordinatenberechn. -
typedef BOOL (CALLBACK *FNTRANSFORMPOINT)(UINT msg, LPPOINT lppt, LONG lParam);
 #define TRFM_INIT   0x001		// Verarbeitung einer DIB beginnt
 #define TRFM_EXIT   0x002		// Verarbeitung einer DIB abgschlossen
 #define TRFM_LINE   0x003		// Verarbeitung der Nchsten (Ziel-) Zeile beginnt
 #define TRFM_PIXEL  0x004		// Pixel-Koordinate umrechnen

typedef COLORREF (CALLBACK *FNTRANSFORMCOLOR)(UINT msg, LPPOINT lppt, LONG lParam, COLORREF clPixelColor);

/* TransformDIB ------------------------------------------------*
 * Transformieren einer DIB in eine andere			*
 * Rckgabewert: TRUE = OK, FALSE = Problem			*
 * Hinweis: TransformDIB ist auch auf die neuen DIB-Formate	*
 *          mit 16 und 32 Bit Farbtiefe vorbereitet		*
 *--------------------------------------------------------------*/
BOOL API TransformDIB(
	    HDIB 	     hDibSource,// Quell-DIB
	    HDIB 	     hDibDest,	// Ziel-DIB
	    FNTRANSFORMPOINT fnPoint,	// Koordinaten-Transformationsroutine
            FNTRANSFORMCOLOR fnColor,	// Farbwerte-Transformationsroutine
	    LONG	     lParam,	// Applikations-Privater Parameter
	    COLORREF	     clBkColor	// Hintergrundfarbe
	    );







#ifdef __cplusplus
}			/* Assume C declarations for C++ */
#endif  /* __cplusplus */

#endif	/* __LEONARDO_H */

