with Xref_Types;

package body Scanner is

  --
  --  provides the char-by-char file scanning capability
  --  to find each reserved word, keyword, or identifier;
  --  returning with each new word found
  --
  --         by John H. Heidema  (December, 1985)
  --
  --        [ with minor changes in October 1990 ]
  --



  procedure Classify(
      Char   : in     Character;              -- byte/char to classify
      F      : in     Text_IO.File_Type;      -- file being processed
      TClass :    out Xref_Types.InputClass;  -- classification to return
      Dash   : in out Boolean ) is            -- "half-comment" flag
    --
    --  a internal procedure serving Get_Word to classify
    --  each file byte and detect/skip over comments
    --
    use Xref_Types;
  begin

    if (Char in 'A'..'Z') or (Char in 'a'..'z') then
        TClass := Letter;
        Dash := False;
      elsif (Char in '0'..'9') or (Char = '_') then
        TClass := Di_git;
        Dash := False;
      elsif (Char = '-') and Dash then   --  two successive dashes
        TClass := CR;                    --  means comment detected
        Dash := False;                   --          so
        Text_IO.Skip_Line(F);            --    skip rest of line
      elsif (Char = '-') then
        TClass := Junk;
        Dash := True;                 -- 1st dash; set "half-comment" flag
      else
        TClass := Junk;
        Dash := False;
    end if;

  end Classify;


  procedure Get_Word
     ( F        : in     Text_IO.File_Type;  -- the file being searched
       ThisWord :    out Text_Handler.Text;  -- the word found, if any
       EOL      :    out Boolean;            -- reports jump to next line
       EOF      :    out Boolean )           -- signals end_of_file reached
                                   is
    --
    -- does a char-by-char scan of the Ada source code file from
    -- the current file position until a reserved word, keyword,
    -- or Ada identifier is found.  Returns to caller at that
    -- point (or at EOL or EOF) with the word found (if any)
    --

    use Xref_Types;

    Char         : Character := ' ';    -- for char fetched from file
    PreviousChar : Character := ' ';    -- for storing last character

    ThisClass    : InputClass := Junk;  -- classification for this char
    PresentState : State   := Start;    -- current state within the FSM
    ThisEntry    : Transitions;         -- FSM destination & output action
    NewAction    : Action  := Nothing;  -- for given fetched character

    TempWord     : Text_Handler.Text;   -- local word during build
    Dash         : Boolean := False;    -- the "half-comment" flag


    AdaText      : FSM_Table :=         -- output state and action
                                        -- for row and column input

    -- entries for current state = Start,   current char =

    (( ( Build,    StartWord ),              -- Letter
       ( Start,    Nothing   ),              -- Di_get
       ( Finish,   BumpLine  ),              -- CR
       ( Start,    Nothing   ) ),            -- Junk

    -- entries for current state = Build,   current char =

     ( ( Build,    AddLetter ),              -- Letter
       ( Build,    AddLetter ),              -- Di_get
       ( Finish,   BumpLine  ),              -- CR
       ( Finish,   Nothing   ) ),            -- Junk

    -- entries for current state = Finish,  current char =

     ( ( Finish,   Nothing   ),              -- Letter
       ( Finish,   Nothing   ),              -- Di_get
       ( Finish,   Nothing   ),              -- CR
       ( Finish,   Nothing   ) ));           -- Junk


    procedure Handle_Quote is
      --
      --  dealing with possible string literals
      --
    begin

      Text_IO.Get(F, Char);
      if (Char = ''') and             --
        (PreviousChar = ''') then     -- avoid '"' character literal
           return;                    --
      end if;
                                    --[ note that proper Ada syntax  ]
      Skip_Over_Literal:            --[ is required for this to work ]
      loop
        exit Skip_Over_Literal    -- stop discarding characters
           when Char = '"';        -- at end of string literal
        Text_IO.Get(F, Char);
      end loop Skip_Over_Literal;

    end Handle_Quote;


  begin           --  main character processing procedure Get_Word()

    EOL := False;                     --     initializations
    EOF := False;                     --
    Text_Handler.Clear(TempWord);     -- (initialize to length zero)

    Main_Word_Finding_Loop:
    loop

      if Text_IO.End_Of_File(F) then
        EOF := True;
        exit Main_Word_Finding_Loop;           -- quit at end of file
      end if;

      if Text_IO.End_Of_Line(F) then         --
          Text_IO.Skip_Line(F);              -- jump over the Ada
          ThisClass := CR;                   -- end_of_line roadblocks
          Dash := False;                     --
        else
          Text_IO.Get(F, Char);        -- fetch next character from file

          if Char = '"' then           -- if possible string literal
            Handle_Quote;
          end if;

          Classify(Char, F, ThisClass, Dash);
      end if;

      ThisEntry := AdaText(PresentState, ThisClass);
      NewAction := ThisEntry.ThisAction;

      case NewAction is       -- four possibilities to consider:
        when Nothing   =>
          null;
        when StartWord =>
          TempWord := Text_Handler.MakeText(Char);      -- begin new identifier
        when AddLetter =>
          TempWord := Text_Handler."&"(TempWord, Char); -- add to current one
        when BumpLine  =>
          EOL := True;
      end case;

      PreviousChar := Char;               -- store old character

      PresentState := ThisEntry.NewState;

      exit Main_Word_Finding_Loop
        when PresentState = Finish;               -- quit at end of identifier

    end loop Main_Word_Finding_Loop;

    Text_Handler.Copy(ThisWord, TempWord);   -- copy local to output parameter
                                             --     just before returning
  end Get_Word;


end Scanner;

