

with TEXT_IO;
package body TEXT_HANDLER is

  --
  --  version of a string (text) processing package
  --  like that outlined in the Ada LRM (Section 7.6)
  --
  --    coding details by Michael B. Feldman
  --          with modifications by
  --     Steven Gutknecht and John Heidema
  --
  --
  --           Status:  Public Domain
  --



  function LENGTH(T : in TEXT) return INDEX is
    begin
      return T.POS;
    end LENGTH;

--
  function VALUE(T : in TEXT) return STRING is
    begin
      if T.POS = 0 then
        return "";
      else
        return T.VALUE(1..T.POS);
      end if;
    end VALUE;

--
  function EMPTY(T : in TEXT) return BOOLEAN is
    begin
      return T.POS=0;
    end EMPTY;

--
  procedure CLEAR(T : in out TEXT) is
    begin
      T.POS := 0;
    end CLEAR;

--
  function MAKETEXT(S : in STRING) return TEXT is
    T : TEXT;

    begin
      if S'LENGTH > MAXIMUM then
        raise STRING_OVERFLOW;
      end if;
      T.POS := S'LENGTH;
      T.VALUE(1..S'LENGTH) := S;
      return T;
    end MAKETEXT;

--
  function MAKETEXT(C : in CHARACTER) return TEXT is
    T : TEXT;

    begin
      T.POS := 1;
      T.VALUE(1) := C;
      return T;
    end MAKETEXT;

--
  procedure COPY(DESTINATION : out TEXT;
    SOURCE : in TEXT) is

    begin
      DESTINATION.VALUE(1..SOURCE.POS) := SOURCE.VALUE(1..SOURCE.POS);
      DESTINATION.POS := SOURCE.POS;
    end COPY;

--
  function "&"(T1, T2 : in TEXT) return TEXT is
    T : TEXT;

    begin
      if EMPTY(T1) then
        return T2;
      end if;
      if EMPTY(T2) then
        return T1;
      end if;

      if (T1.POS + T2.POS) > MAXIMUM then
        raise STRING_OVERFLOW;
      end if;

      T.POS := T1.POS + T2.POS;
      T.VALUE(1..T.POS) := T1.VALUE(1..T1.POS) & T2.VALUE(1..T2.POS);
      return T;
    end "&";

--
  function "&"(T1 : in TEXT;
    C : in CHARACTER) return TEXT is

    begin
      return T1 & MAKETEXT(C);
    end "&";

--

  function "&"(C : in CHARACTER;
    T1 : in TEXT) return TEXT is

    begin
      return MAKETEXT(C) & T1;
    end "&";

--

  function "&"(T1 : in TEXT;
    S : in STRING) return TEXT is

    begin
      return T1 & MAKETEXT(S);
    end "&";

--

  function "&"(S : in STRING;
    T1 : in TEXT) return TEXT is

    begin
      return MAKETEXT(S) & T1;
    end "&";


--
  function Equal(T1, T2 : in TEXT) return BOOLEAN is
    begin
      return T1.VALUE(1..T1.POS) = T2.VALUE(1..T2.POS);
    end Equal;

--
  function "<="(T1, T2 : in TEXT) return BOOLEAN is
    begin
      return T1.VALUE(1..T1.POS) <= T2.VALUE(1..T2.POS);
    end "<=";

--
  function "<"(T1, T2 : in TEXT) return BOOLEAN is
    begin
      return T1.VALUE(1..T1.POS) < T2.VALUE(1..T2.POS);
    end "<";

--
  function ">="(T1, T2 : in TEXT) return BOOLEAN is
    begin
      return T1.VALUE(1..T1.POS) >= T2.VALUE(1..T2.POS);
    end ">=";

--
  function ">"(T1, T2 : in TEXT) return BOOLEAN is
    begin
      return T1.VALUE(1..T1.POS) > T2.VALUE(1..T2.POS);
    end ">";

--
  function LOCATE(SUB : in TEXT;
    WITHIN : in TEXT) return INDEX is

    begin
      return LOCATE(SUB.VALUE(1..SUB.POS), WITHIN);
    end LOCATE;

--
  function LOCATE(SUB : in STRING;
    WITHIN : in TEXT) return INDEX is

    begin
      if SUB'LENGTH = 0 or WITHIN.POS = 0 or SUB'LENGTH > WITHIN.POS then
          return 0;
      end if;
      for START in 1..(WITHIN.POS - SUB'LENGTH + 1) loop
        if WITHIN.VALUE(START..(START + SUB'LENGTH - 1)) =     --cont
        SUB(1..SUB'LENGTH) then
          return START;
        end if;
      end loop;
      return 0;
    end LOCATE;

--
  function LOCATE(SUB : in CHARACTER;
    WITHIN : in TEXT) return INDEX is

    begin
      if WITHIN.POS = 0 then
        return 0;
      end if;
      for START in 1..WITHIN.POS loop
        if SUB = WITHIN.VALUE(START) then
          return START;
        end if;
      end loop;
      return 0;
    end LOCATE;

--
  function SUBSTR(T : in TEXT;
    START, SIZE : in INDEX) return TEXT is

    T1 : TEXT;
    LIMIT : INTEGER;
    HOWBIG : INTEGER := SIZE;

    begin
      if START > T.POS then
        raise INVALID_ARGUMENTS;
      end if;
      if (SIZE + START - 1) <= T.POS then
        LIMIT := SIZE + START - 1;
      else
        LIMIT := T.POS;
        HOWBIG := T.POS - START + 1;
      end if;

      T1.VALUE(1..HOWBIG) := T.VALUE(START..LIMIT);
      T1.POS := HOWBIG;
      return T1;
    end SUBSTR;

--
  procedure PUT(T : in TEXT) is
    begin
      if not EMPTY(T) then
        TEXT_IO.PUT(T.VALUE(1..T.POS));
      end if;
    end PUT;

end TEXT_HANDLER;
