
with Text_IO;

package body Tree_Handler is

  --
  --  contains the binary search tree management/processing
  --  facilities needed for this cross-referencer
  --
  --      built by John H. Heidema  (December, 1985)
  --
  --        [ with minor changes in October 1990]
  --



  procedure Update(
         T : in out Xref_Types.BTree;             -- binary tree pointer
         K : in     Text_Handler.Text;            -- new word to insert
         V : in     Xref_Types.Line_Number ) is   -- its line number value
  --
  --  This procedure adds one word node (with data of type Text)
  --  to the binary tree indicated by access variable T
  --  If a node for this word already exists, then just the
  --  new line number V is added to the tree data structure
  --  (which has a linked list of line numbers attached to
  --  each tree node).  New words get attached to existing
  --  tree leaves, using recursion to move left if earlier
  --  in the alphabet and right if later.
  --

    use Text_Handler, Xref_Types;


    function MakeNode( K : Text;             -- the new word
                       V : Line_Number )     -- its line number value
                           return BTree is   -- the returned nodeptr
      --
      --  making a new binary tree node for a new word
      --  and returning a pointer to it
      --
      Result : BTree;      -- the word nodeptr to return
      L      : List;       -- its line nodeptr
    begin
      Result := new TreeNode;         -- allocate new treenode
      Copy(Result.Key, K);
      L := new OneWayListNode;        -- allocate new line listnode
      L.Value := V;
      Result.Value.Head := L;         -- attaching new line_number
      Result.Value.Tail := L;         --   list to the tree node
      return Result;
    end MakeNode;


    procedure ProcessDuplicate (
                  T : in out BTree;            -- correct node ptr
                  V : in     Line_Number ) is  -- new line number value
      --
      --  for adding a new line number when
      --  word is already present in tree
      --
      L : List;  -- line number nodeptr
    begin
      L := new OneWayListNode;      -- allocate new line listnode
      L.Value := V;
      T.Value.Tail.Next := L;       -- attach to
      T.Value.Tail      := L;       -- list tail
    end ProcessDuplicate;


  begin       -- procedure Update

    if T = null then                       -- if tree doesn't yet exist,

      T := MakeNode(K,V);                  --    then create one

    elsif K < T.Key then                   -- else find appropriate
      if T.Left = null then                -- location to insert a new
        T.Left := MakeNode(K,V);           -- wordnode or, if word is
      else                                 -- already present, to insert
        Update(T.Left, K, V);              -- the new line number
      end if;
    elsif K > T.Key then
      if T.Right = null then
        T.Right := MakeNode(K,V);
      else
        Update(T.Right, K, V);
      end if;
    else
      ProcessDuplicate(T, V);              -- when word is already present
    end if;

  end Update;



  procedure Report( T : in     Xref_Types.Btree ) is

    --
    --   Called at the end of file processing.
    --   A recursive, inorder traversal routine
    --   to print the data found in input tree T
    --   in alphabetical order
    --

    use Text_IO,
        Text_Handler,
        Xref_Types;

    L : List;         -- line_number nodeptr

  begin

    if T = null then
      return;         -- quit if current tree is empty
    end if;

    Report(T.Left);              -- recurse to left


    -- Print data at root node:

    Put("  ");                      -- indent
    Put(T.Key);                     -- print word
    Text_IO.Set_Col(15);            -- align first number

    L := T.Value.Head;                   --
    while L /= null loop                 -- get and print
      Put(Line_Number'Image(L.Value));   -- line numbers
      L := L. Next;                      --
      if L /= null then
        Put(",");                -- add comma after each except last
      end if;
    end loop;
    New_Line;    -- end of node processing


    Report(T.Right);             -- recurse to right

  end Report;


end Tree_Handler;
