with Text_IO,
     Text_Handler,
     Xref_Types,
     Tree_Handler,
     Scanner;

procedure Xref is

  --
  --              Ada Cross-Referencing Program
  --
  --                   by  John H. Heidema
  --
  --                     (December, 1985)
  --
  --
  --     Adapted by JHH for metrics demo in October of 1990
  --

  --  This was a first Ada programming project, done as a course
  --  assignment for Prof. Michael B. Feldman at George Washington
  --  University.  It is acknowledged that large chunks of the code
  --  were borrowed almost verbatum from program fragments
  --  designed for similar purposes and appearing in the textbook
  --  "Data Structures with Ada" by Michael B. Feldman
  --  (Reston Publishing Company, Inc., 1985).
  --
  --  The student was responsible for decisions about how to
  --  distribute program features among packages, for generating
  --  some unprovided segments of Ada code, for adding features
  --  necessary to deal with the peculiar aspects of Ada syntax
  --  that differ from written English, and for molding the
  --  various fragments into a functioning whole.

  --  Function:
  --    This program provides a cross-reference of all reserved
  --    words, keywords, and identifiers found in any Ada source
  --    code file.  Words are reported alphabetically according
  --    to the ASCII collating sequence, followed by the line
  --    numbers of all lines in which they appear.

  --  Peculiarities:
  --    The same line number will be reported sequentially as
  --    many times as the word appears in the line.
  --    The alphabetizing is case sensitive, which (depending
  --    upon preferences and coding styles) may or may not
  --    be desirable.

  --  Limitations:
  --    * The program currently processes only one file at a time.
  --    * The output is currently sent only to the console.
  --    * The user interface and error handling is crude.
  --    * The Ada source file is assumed to use correct Ada syntax.
  --    * Reserved words are not distinguished from programmer
  --      selected identifiers (except by capitalization style).
  --    * No detailed categorization of identifiers is done.
  --    * Files are read one_character_at_a_time (a bit slow).
  --    * Character literals get reported as identifiers.

  --  Algorithms:
  --    The program uses the finite state machine method found
  --    in Feldman's text to select identifiers, modified to
  --    exclude comments and string literals.  The program stores
  --    the words in a binary search tree of dynamically allocated
  --    nodes (one per unique word), with a linked list of
  --    dynamically allocated line_number nodes attached to each
  --    word node.  Deallocation is simply done by the system
  --    when the program terminates.  An "inorder" traversal of
  --    the binary search tree in the Report procedure gives the
  --    alphabetized output.  (See Feldman's book or any other
  --    Data Structures text for details on these common tree
  --    processing methods.)
  --
  --



  F         : Text_IO.File_Type;                -- file handle

  File_Name : String(1..80) := (1..80 => ' ');  -- Get_Line()
  Length    : Natural := 0;                     -- parameters

  T         : Xref_Types.BTree;                 -- pointer to binary tree
  Line      : Xref_Types.Line_Number := 1;      -- current line number
  ThisWord  : Text_Handler.Text;                -- current identifier

  EOL  ,                                -- the end_of_line flag
  EOF       : Boolean := False;         -- the end_of_file flag


  use Text_IO;       -- for standard screen and file I/O routines


begin                       -- begin Main Program

  New_Line(2);                                 --
  Put("Which Ada source file, please?  ");     -- get desired file
  Get_Line(File_Name, Length);                 --   and open it
  Open(F, In_File, File_Name(1..Length));      --

  File_Processing:
  loop

    Scanner.Get_Word(F, ThisWord, EOL, EOF);    -- find next identifier

    if not Text_Handler.Empty(ThisWord) then
      Tree_Handler.Update(T, ThisWord, Line);   -- store found word data
    end if;

    if EOL then
      Line := Xref_Types."+"(Line, 1);         -- simple incrementing
    end if;

    exit File_Processing when EOF;             -- quit file processing when
                                               --  end_of_file is detected
  end loop File_Processing;

  Close(F);

  New_Line(4);
  Put(" A Cross-Reference Listing for the ada file:  ");  -- output
  Put( File_Name(1..Length) );                            -- header
  New_Line(3);

  Tree_Handler.Report(T);                 -- call for output of results

  New_Line(2);

 exception
   when Text_IO.Name_Error =>                     -- if file open fails
     New_Line(2);
     Put(" -- Sorry.  File (as specified) was not found. --");
     New_Line(2);

end Xref;                   -- end Main Program










