Attribute VB_Name = "Module1"
'=======================================================================
'
' This module provides event handling and general routines for DBSEARCH.
'
'=======================================================================
'
' Original Author:   Barth Riley
'
' Originally called LISTSRCH - this module allowed the developer to set
' up a populated listbox and search the contents of the listbox by typing
' into a textbox. This allowed "Quicken" style searching with the module
' finding the first match for what the user had typed and filling in the
' remainder.
'
'=======================================================================
'
' Modified : Martin Colloby - Harlequin Computer Services Limited 7/7/97
'
' Now dynamically creates a recordset to allow searching of a table in a
' database. The old approach worked OK but the systems in which I used it
' didn't display the list boxes so it made no sense to have them present
' on the form.
'
'=======================================================================
'
' Modified : Martin Colloby - Harlequin Computer Services Limited 14/8/97
'
' Now initialises just about everything in the call to DBSRCH_GotFocus
' Added option to use + and - keys to scroll within the current range
'
' e.g. if table contains three records :
'
'           MARTIN A
'           MARTIN B
'           MARTIN C
'
'      when the users presses the M key, MARTIN A is displayed.
'      Instead of having to type MARTIN C to get to the third value, the
'      user simply presses + twice.
'
'==========================================================================
'
' Example
'
' You have a database (DB) which contains a table ("USERS") which contains
' a field ("USER"). You want to create a text box on a form that the user
' can type into, but you want the text box to check what the user has typed
' so far against the USER field and fill in what the user hasn't yet entered.
'
' In the GotFocus event of the text box, place a call to DBSRCH_GotFocus,
'   passing it the database, the name of the table, the name of the field
'   and an optional condition for record selection.
'
' e.g. DBSRCH_GotFocus txtSearch, db, "USERS", "USER", ""
'
' If you wanted to restrict the search to users with a certain access level,
' simply pass the WHERE clause
'
' e.g. DBSRCH_GotFocus txtSearch, db, "USERS", "USER", "LEVEL > 2"
'
' In the KeyDown event of the text box, place a call to DBSRCH_KeyDown,
'   passing it the key pressed and the text box.
'
' e.g. DBSRCH_KeyDown KeyAscii

' In the KeyPress event of the text box, place a call to DBSRCH_KeyPress,
'   passing it the key pressed and the text box.
'
' e.g. DBSRCH_KeyPress KeyAscii, txtSearch
'
' In the Change event of the text box, place a call to DBSRCH_Change,
'   passing it the text box
'
' e.g. DBSRCH_Change txtSearch
'
' That's all there is to it!
'==========================================================================
  
Option Explicit

'--Key Code Constants
Const KEY_BACK = &H8
Const KEY_DELETE = &H2E
Const KEY_CLEAR = &HC

Dim mintNumKeys As Integer          ' Number of keys pressed by user
Dim mintPluses As Integer           ' Number of times user has pressed plus or minus keys

Dim mstrTable As String             ' Name of table to search
Dim mstrField As String             ' Name of field to search
Dim mstrWhere As String             ' Conditional clause for record selection

Dim mdb As Database                 ' Database object

Dim mblnPluses As Boolean           ' Flag to indicate
Dim mblnScrolling As Boolean        ' Control reentry into change routine
Dim mblnKeepKey As Boolean          ' False if user hit delete/backspace


Sub DBSRCH_Change(txtSearch As TextBox)

    ' Parameters -  txtSearch - Text box into which user has typed some text
  
    SearchForMatch txtSearch        ' The text in txtSearch has changed, so search for a new match

End Sub

Sub DBSRCH_GotFocus(txtSearch As TextBox, db As Database, strTable As String, strField As String, strWhere As String)

    ' Initialise everything on entering the text box
    '
    ' Parameters - txtSearch - Text box that the user will type into
    '              db        - Database object
    '              strTable  - Table in db
    '              strField  - Field in strTable
    '              strWhere  - Optional conditional clause (e.g. "DELETED = 0")
    
    mintNumKeys = 0                             ' Clear number of keys entered
    mintPluses = 0                              ' Clear count of plus and minus keys

    Set mdb = db                                ' Store the database
    mstrTable = strTable                        ' Store the table to be searched
    mstrField = strField                        ' Store the field within the table
    mstrWhere = strWhere                        ' Store the conditional clause
    
    txtSearch.SelStart = 0                      ' Start selection at start of text
    txtSearch.SelLength = Len(txtSearch.Text)   ' End selection at end of text

End Sub

Sub DBSRCH_KeyDown(ByVal KeyCode As Integer)
  
    '=====================================================
    ' Determines if a valid (printable) character has been pressed.  If the character is printable, the
    ' the txtSearch_Change event handler will search for a matching item.
    '=======================================================
    ' Parameters - KeyCode  - Key pressed by user

    Select Case KeyCode
        Case vbKeyBack, vbKeyDelete, vbKeyClear
            mblnKeepKey = False                             ' Don't want to keep these keys
            mintPluses = 0                                  ' Clear count of plus and minus keys
            mblnPluses = False                              ' Flag to the search routine that user has not pressed plus or minus

        Case vbKeyAdd, vbKeySubtract
            mblnKeepKey = False                             ' Don't keep these keys
            mblnPluses = True                               ' Flag to the search routine that user has pressed plus or minus

        Case Else
            mblnKeepKey = True                              ' Keep this key
            mintPluses = 0                                  ' Clear count of plus and minus keys
            mblnPluses = False                              ' Flag to the search routine that user has not pressed plus or minus

    End Select

End Sub

Sub DBSRCH_KeyPress(KeyAscii As Integer, txtSearch As TextBox)
  
    '===============================================================
    ' Keeps track of number of keys pressed
    '===============================================================
  
    ' Parameters - KeyAscii  - Key pressed by user
    '              txtSearch - Text box into which user has typed some text

    Select Case KeyAscii
        Case Asc("+")                                                       ' Plus key pressed
            mintPluses = mintPluses + 1                                     ' Increment count of plus keys
            SearchForMatch txtSearch                                        ' Search for the new value

        Case Asc("-")                                                       ' Minus key pressed
            If mintPluses > 0 Then
                mintPluses = mintPluses - 1                                 ' Decrement count of plus keys
            End If
            SearchForMatch txtSearch                                        ' Search for the new value

        Case Else
            If mblnKeepKey Then                                             ' Are we keeping this key
                mintNumKeys = Len(txtSearch.Text) + 1                       ' Increment the count of keys pressed
            End If
    End Select

End Sub

Sub SearchForMatch(txtSearch As TextBox)

    ' Parameters -  txtSearch - Text box into which user has typed some text
    '
    '=====================================================
    ' Called when the user presses a character to change the typed selection, or if the user presses the plus or minus keys
    ' This procedure creates a recordset which will hold any records that match the typed text, subject to the WHERE clause
    ' passed to the DBSRCH_GotFocus routine.
    '
    ' As the recordset is ordered by the specified field the first record will hold the first alphabetical match.
    ' If the recordset is empty there are no matches.
    '
    ' If the user has accumulated plus and minus keys, mblnPluses will be true, and the record pointer in the recordset is moved
    ' accordingly.
    '
    ' The content of the record is then selected and the portion of the text NOT typed by the user is highlighted in the text box.
    '
    ' Note that mfScrolling is used to prevent re-entry into this event handler as storing the new value in the
    ' text box retriggers this procedure.
    '=====================================================

    Dim intTxtLen As Integer        ' Length of search string
    Dim intCount As Integer         ' Counting variable
    Dim intRecords As Integer       ' Number of records that match criteria
    
    Dim strSql As String            ' String to be found
    Dim strSrchText As String       ' Contents of text box

    Dim rst As Recordset            ' Dynaset for retrieving matching records

    If (mblnKeepKey And Not mblnScrolling) Or mblnPluses Then                   ' Make sure we don't reenter
        intTxtLen = Len(txtSearch.Text)                                         ' Store the initial length of the text
        If intTxtLen Then                                                       ' Is there any text?
            mintNumKeys = IIf(mintNumKeys < intTxtLen, mintNumKeys, intTxtLen)  ' Adjust the number of keys
            strSrchText = txtSearch.Text                                        ' Store the initial text
      
            strSql = "SELECT " & mstrField & " FROM " & mstrTable               ' Start the SQL string - we will be selecting strField from the strTable table
            strSql = strSql & " WHERE " & mstrField & " LIKE '" & Left$(strSrchText, mintNumKeys) & "*'" ' Add the search criteria - WHERE strField LIKE 'ABC*'
            If mstrWhere <> "" Then
                strSql = strSql & " AND " & mstrWhere
            End If
            strSql = strSql & " ORDER BY " & mstrField                          ' Finally, order the resulting recordset
            
            Set rst = mdb.OpenRecordset(strSql)                                 ' Create the recordset
            If rst.RecordCount > 0 Then
                rst.MoveLast                                                    ' Skip to the last record ...
                intRecords = rst.RecordCount                                    ' ... so we can record the true record count ...
                rst.MoveFirst                                                   ' ... and then get back to the top
            Else
                intRecords = 0                                                  ' No records found
            End If
      
            mblnScrolling = True                                                ' Set the flag to prevent reentry
            If intRecords = 0 Then                                              ' Did we get any records?
                txtSearch.Text = Left$(txtSearch.Text, mintNumKeys)             ' No, so set the text box to show only what the user has typed
            Else
                If mintPluses >= intRecords Then                                 ' Limit count of plus keys to number of records
                    mintPluses = intRecords - 1
                End If
                For intCount = 1 To mintPluses                                  ' If the user has pressed the plus or minus keys
                    If Not rst.EOF Then                                         ' Make sure we are not at the end of the file
                        rst.MoveNext                                            ' Skip to the next record
                        If rst.EOF Then                                         ' If we are now at the end of the file ...
                            Beep                                                ' ... beep at the user ...
                            rst.MovePrevious                                    ' ... and step back
                        End If
                    Else
                        Beep                                                    ' At end of file, so beep at the user
                    End If
                Next ' intCount
                txtSearch = rst.Fields(mstrField)                               ' Perfect match was found
            End If
        
            txtSearch.SelStart = mintNumKeys                                    ' Set the start of the selected text to the letter after what the user has typed
            txtSearch.SelLength = (Len(txtSearch.Text) - mintNumKeys)           ' Set the end of the selected text
            mblnScrolling = False                                               ' Allow entry next time the user changes the text

            rst.Close                                                           ' Close the dynaset before exiting
        End If
    End If


End Sub

