Attribute VB_Name = "Astronomy_Funcs"
'---------------------------------------------------------------------
'
'   ===========
'   ASTRO32.BAS
'   ===========
'
' Interface declarations for the Astronomy Library. Drop this into
' any VB project to get access to the astronomical support functions
' in astro32.dll
'
' Written:  18-Jul-96   Robert B. Denny <rdenny@dc3.com>
'
' Edits:
'
' 18-Jul-96 rbd Initial edit
'---------------------------------------------------------------------
Option Explicit

' You know what this is!
Public Const PI = 3.14159265358979
' Ratio of from synodic (solar) to sidereal (stellar) rate
Public Const SIDRATE = 0.9972695677
' Seconds per day
Public Const SPD = 86400#
'
' Modified Julian Date (MJD) calculations. The epoch for MJD is
'
Public Const MJD0 = 2415020#    ' MJD Julian epoch (JD = MJD + MJD0)
Public Const J2000 = 36525#     ' MJD for 2000 (2451545.0 - MJD0)
'
' Manifest names for planets as used in calls to plans() and
' pelement().
'
Public Const MERCURY = 0
Public Const VENUS = 1
Public Const MARS = 2
Public Const JUPITER = 3
Public Const SATURN = 4
Public Const URANUS = 5
Public Const NEPTUNE = 6
Public Const PLUTO = 7
'
' Date formatting preferences for fmt_mjd() and scn_date()
'
Public Const DATE_YMD = 0
Public Const DATE_MDY = 1
Public Const DATE_DMY = 2

'
' Timezone name preferences for tz_name()
'
Public Const DATE_UTCTZ = 3
Public Const DATE_LOCALTZ = 4

'
' =================
' LIBRARY FUNCTIONS
' =================
'
' NOTES:
'
' (1) For whatever reason, the authors of the original C functions chose
'     to pass back and forth via parameters only for most functions.
'
' (2) The descriptive comments below were lifted straight out of the C
'     functions. Some variables are listed with the C dereferening '*'.
'     Note that these are passed ByRef in the declarations, then forget
'     about the '*'.
'
' (3) Modified Julian Dates (number of days elapsed since 1900 jan 0.5,)
'     are used for most times. Several functions are provided for converting
'     between mjd and other time systems (C runtime, VB, Win32).
'
'
' given latitude (n+, radians), lat, altitude (up+, radians), alt, and
' azimuth (angle around to the east from north+, radians),
' return hour angle (radians), ha, and declination (radians), dec.
'
Declare Sub aa_hadec Lib "astro32" (ByVal lat As Double, ByVal Alt As Double, ByVal Az As Double, ByRef ha As Double, ByRef dec As Double)
'
' apply aberration correction to ecliptical coordinates *lam and *bet
' (in radians) for a given time mjd and handily supplied longitude of sun,
' lsn (in radians)
'
Declare Sub ab_ecl Lib "astro32" (ByVal mjd As Double, ByVal lsn As Double, ByRef lam As Double, ByRef bet As Double)
'
' apply aberration correction to equatoreal coordinates *ra and *dec
' (in radians) for a given time mjd and handily supplied longitude of sun,
' lsn (in radians)
'
Declare Sub ab_eq Lib "astro32" (ByVal mjd As Double, ByVal lsn As Double, ByRef RA As Double, ByRef dec As Double)
'
' given the mean anomaly, ma, and the eccentricity, s, of elliptical motion,
' find the true anomaly, *nu, and the eccentric anomaly, *ea.
' all angles in radians.
'
Declare Sub anomaly Lib "astro32" (ByVal ma As Double, ByVal s As Double, ByRef nu As Double, ByRef ea As Double)
'
' given the apparent ha and dec, aha and adec, the geographical latitude, phi,
' the height above sea-level (as a fraction of the earths radius, 6378.16km),
' ht, and the equatorial horizontal parallax, ehp, find the true ha and dec,
' tha and tdec allowing for parallax.
' all angles in radians. ehp is the angle subtended at the body by the
' earth's equator.
'
' uses ta_par() iteratively: find a set of true ha/dec that converts back
' to the given apparent ha/dec.
'
Declare Sub at_par Lib "astro32" (ByVal aha As Double, ByVal adec As Double, ByVal phi As Double, ByVal ht As Double, ByVal ehp As Double, ByRef tha As Double, ByRef tdec As Double)
'
' given a date in months, mn, days, dy, years, yr,
' return the modified Julian date (number of days elapsed since 1900 jan 0.5),
' *mjd.
'
Declare Sub cal_mjd Lib "astro32" (ByVal mn As Long, ByVal dy As Double, ByVal yr As Long, ByRef mjd As Double)
'
' transformation from cartesian to spherical coordinates
'   x, y, z:    source: rectangular coordinates
'   l, b, r:    result: spherical coordinates
'
Declare Sub cartsph Lib "astro32" (ByVal X As Double, ByVal Y As Double, ByVal z As Double, ByRef l As Double, ByRef b As Double, ByRef r As Double)
'
' given a modified Julian date, mjd, and a set of heliocentric parabolic
' orbital elements referred to the epoch of date (mjd):
'   ep:   epoch of perihelion,
'   inc:  inclination,
'   ap:   argument of perihelion (equals the longitude of perihelion minus the
'         longitude of ascending node)
'   qp:   perihelion distance,
'   om:   longitude of ascending node;
' find:
'   lpd:  heliocentric longitude,
'   psi:  heliocentric latitude,
'   rp:   distance from the sun to the planet,
'   rho:  distance from the Earth to the planet,
'   lam:  geocentric ecliptic longitude,
'   bet:  geocentric ecliptic latitude,
'         none are corrected for light time, ie, they are the true values for
'         the given instant.
'
' all angles are in radians, all distances in AU.
' mutual perturbation corrections with other solar system objects are not
' applied. corrections for nutation and abberation must be made by the caller.
' The RA and DEC calculated from the fully-corrected ecliptic coordinates are
' then the apparent geocentric coordinates. Further corrections can be made,
' if required, for atmospheric refraction and geocentric parallax.
'
Declare Sub comet Lib "astro32" (ByVal mjd As Double, ByVal ep As Double, ByVal inc As Double, ByVal ap As Double, ByVal qp As Double, ByVal om As Double, ByRef lpd As Double, ByRef psi As Double, ByRef rp As Double, ByRef rho As Double, ByRef lam As Double, ByRef bet As Double)
'
' given the difference in two RA's, in rads, return their difference,
'   accounting for wrap at 2*PI. caller need *not* first force it into the
'   range 0..2*PI.
'
Declare Function delra Lib "astro32" (ByVal dra As Double) As Double
'
' given the modified Julian date, mjd, and a geocentric ecliptic latitude,
'   *lat, and longititude, *lng, each in radians, find the corresponding
'   equitorial ra and dec, also each in radians. Correction for the
'   effect on the angle of the obliquity due to nutation is included.
'
Declare Sub ecl_eq Lib "astro32" (ByVal mjd As Double, ByVal lat As Double, ByVal lng As Double, ByRef RA As Double, ByRef dec As Double)
'
' given the modified Julian date, mjd, and an equitorial ra and dec, each in
'   radians, find the corresponding geocentric ecliptic latitude, *lat, and
'   longititude, *lng, also each in radians. Correction for the effect on
'   the angle of the obliquity due to nutation is included.
'
Declare Sub eq_ecl Lib "astro32" (ByVal mjd As Double, ByVal RA As Double, ByVal dec As Double, ByRef lat As Double, ByRef lng As Double)
'
' given ra and dec, each in radians, for the given epoch, find the
'   corresponding galactic latitude, *lat, and longititude, *lng, also
'   each in radians.
'
Declare Sub eq_gal Lib "astro32" (ByVal mjd As Double, ByVal RA As Double, ByVal dec As Double, ByRef lat As Double, ByRef lng As Double)
'
' Format a date string into buf, given a modified julian date and the
' selected format (m/d/y, etc.). Typically mm/dd.ddd/yyyy (note the
' fractional days).
'
Declare Sub fmt_mjd Lib "astro32" (ByVal buf As String, ByVal mjd As Double, ByVal pref As Long)
'
' format the Double (e.g., mjd, lst) in sexagesimal format into buf[].
' w is the number of spaces for the whole part.
' fracbase is the number of pieces a whole is to broken into; valid options:
'  360000: <w>:mm:ss.ss
'  36000:  <w>:mm:ss.s
'  3600:   <w>:mm:ss
'  600:    <w>:mm.m
'  60:     <w>:mm
'
Declare Sub fmt_sexa Lib "astro32" (ByVal buf As String, ByVal val As Double, ByVal w As Long, ByVal fracbase As Long)
'
' given galactic latitude, lat, and longititude, lng, each in radians, find
' the corresponding equitorial ra and dec, also each in radians, at the
' given epoch.
'
Declare Sub gal_eq Lib "astro32" (ByVal mjd As Double, ByVal lat As Double, ByVal lng As Double, ByRef RA As Double, ByRef dec As Double)
'
' given a modified julian date, mjd, and a greenwich mean siderial time, gst,
' return universally coordinated time, *utc.
'
Declare Sub gst_utc Lib "astro32" (ByVal mjd As Double, ByVal gst As Double, ByRef utc As Double)
'
' given latitude (n+, radians), lat, hour angle (radians), ha, and declination
'   (radians), dec, return altitude (up+, radians), alt, and azimuth (angle
'   round to the east from north+, radians),
'
Declare Sub hadec_aa Lib "astro32" (ByVal lat As Double, ByVal ha As Double, ByVal dec As Double, ByRef Alt As Double, ByRef Az As Double)
'
' return the Modified Julian Date of the epoch 2000
'
Declare Function mjd_2000 Lib "astro32" () As Double
'
' given the modified Julian date, mjd, return the calendar date in months, *mn,
' days, *dy, and years, *yr.
'
Declare Sub mjd_cal Lib "astro32" (ByVal mjd As Double, ByRef mn As Long, ByRef dy As Double, ByRef yr As Long)
'
' given an mjd, truncate it to the beginning of the whole day
'
Declare Function mjd_day Lib "astro32" (ByVal jd As Double) As Double
'
' given an mjd, set *dow to 0..6 according to which day of the week it falls
' on (0=sunday). return 0 if ok else -1 if can't figure it out.
'
Declare Function mjd_dow Lib "astro32" (ByVal mjd As Double, ByRef dow As Long) As Long
'
' given a mjd, return the the number of days in the month.
'
Declare Sub mjd_dpm Lib "astro32" (ByVal mjd As Double, ByRef ndays As Long)
'
' given an mjd, return the number of hours past midnight of the
' whole day
'
Declare Function mjd_hr Lib "astro32" (ByVal jd As Double) As Double
'
' Return the Visual Basic Date given a Modified Julian Date
'
Declare Function mjd_vb Lib "astro32" (ByVal mjd As Double) As Date
'
' given a mjd, return the year as a double.
'
Declare Sub mjd_year Lib "astro32" (ByVal mjd As Double, ByRef yr As Double)
'
' given the mjd, find the geocentric ecliptic longitude, lam, and latitude,
'   bet, and horizontal parallax, hp for the moon. also return the sun's
'   mean anomaly, *msp, and the moon's mean anomaly, *mdp.
' N.B. series for long and lat are good to about 10 and 3 arcseconds. however,
'   math errors cause up to 100 and 30 arcseconds error, even if use double.
'   why?? suspect highly sensitive nature of difference used to get m1..6.
' N.B. still need to correct for nutation. then for topocentric location
'   further correct for parallax and refraction.
'
Declare Sub moon Lib "astro32" (ByVal mjd As Double, ByRef lam As Double, ByRef bet As Double, ByRef hp As Double, ByRef msp As Double, ByRef mdp As Double)
'
' given a Julian date and a lunar location, find selenographic colongitude of
' rising sun, lunar latitude of subsolar point, illuminated fraction, and alt
' of sun at the given location. Any pointer may be 0 if not interested.
' From Bruning and Talcott, October 1995 _Astronomy_, page 76.
' N.B. lunar coordinates use +E, but selenograhic colongs are +W.
'
Declare Sub moon_colong Lib "astro32" (ByVal mjd As Double, ByVal lt As Double, ByVal lg As Double, ByRef cp As Double, ByRef kp As Double, ByRef ap As Double, ByRef sp As Double)
'
' given a modified Julian date, mjd, return the mjd of the new
' and full moons about then, mjdn and mjdf.
' TODO: exactly which ones does it find? eg:
'   5/28/1988 yields 5/15 and 5/31
'   5/29             6/14     6/29
'
Declare Sub moonnf Lib "astro32" (ByVal mjd As Double, ByRef mjdn As Double, ByRef mjdf As Double)
'
' Return the current Local Sidereal Time (LST) from the clock and longitude
'
Declare Function now_lst Lib "astro32" (ByVal lng As Double) As Double
'
' Return the current Modified Julian Date derived from the system clock
'
Declare Function now_mjd Lib "astro32" () As Double
'
' given the modified JD, mjd, correct, IN PLACE, the right ascension *ra
' and declination *dec (both in radians) for nutation.
'
Declare Sub nut_eq Lib "astro32" (ByVal mjd As Double, ByRef RA As Double, ByRef dec As Double)
'
' given the modified JD, mjd, find the nutation in obliquity, *deps, and
' the nutation in longitude, *dpsi, each in radians.
'
Declare Sub nutation Lib "astro32" (ByVal mjd As Double, ByRef deps As Double, ByRef dpsi As Double)
'
' given the modified Julian date, mjd, find the mean obliquity of the
' ecliptic, *eps, in radians.
'
Declare Sub obliquity Lib "astro32" (ByVal mjd As Double, ByRef eps As Double)
'
' given a modified Julian date, mjd, and a planet, p, find:
'   lpd0: heliocentric longitude,
'   psi0: heliocentric latitude,
'   rp0:  distance from the sun to the planet,
'   rho0: distance from the Earth to the planet,
'         none corrected for light time, ie, they are the true values for the
'         given instant.
'   lam:  geocentric ecliptic longitude,
'   bet:  geocentric ecliptic latitude,
'         each corrected for light time, ie, they are the apparent values as
'         seen from the center of the Earth for the given instant.
'   dia:  angular diameter in arcsec at 1 AU,
'   mag:  visual magnitude when 1 AU from sun and earth at 0 phase angle.
'
' all angles are in radians, all distances in AU.
'
' corrections for nutation and abberation must be made by the caller. The RA
'   and DEC calculated from the fully-corrected ecliptic coordinates are then
'   the apparent geocentric coordinates. Further corrections can be made, if
'   required, for atmospheric refraction and geocentric parallax.
'
' NOTE: Planet numbers (p) are defined as constants above
'
Declare Sub plans Lib "astro32" (ByVal mjd As Double, ByVal p As Long, ByRef lpd0 As Double, ByRef psi0 As Double, ByRef rp0 As Double, ByRef rho0 As Double, ByRef lam As Double, ByRef bet As Double, ByRef dia As Double, ByRef mag As Double)
'
' Same as plans() except uses faster single-pass algorithm, yielding less
' accuracy. Check your results if you use the fast version.
'
Declare Sub plans_fast Lib "astro32" (ByVal mjd As Double, ByVal p As Long, ByRef lpd0 As Double, ByRef psi0 As Double, ByRef rp0 As Double, ByRef rho0 As Double, ByRef lam As Double, ByRef bet As Double, ByRef dia As Double, ByRef mag As Double)
'
' corrects ra and dec, both in radians, for precession from epoch 1 to epoch 2.
' the epochs are given by their modified JDs, mjd1 and mjd2, respectively.
' N.B. ra and dec are modifed IN PLACE.
'
Declare Sub precess Lib "astro32" (ByVal mjd1 As Double, ByVal mjd2 As Double, ByRef RA As Double, ByRef dec As Double)
'
' Same as precess() except uses faster single-pass algorithm, yielding
' less accuracy. Check your results if you use the fast version.
'
Declare Sub precess_fast Lib "astro32" (ByVal mjd1 As Double, ByVal mjd2 As Double, ByRef RA As Double, ByRef dec As Double)
'
' insure 0 <= *v < r. Used to range angles and times
'
Declare Sub range Lib "astro32" (ByRef v As Double, ByVal r As Double)
'
' convert those orbital elements that change from epoch mjd0 to epoch mjd.
'   mjd0:   initial epoch
'   mjd:    desired epoch
'   inc0:   initial inclination, rads
'   ap0:    initial argument of perihelion, as an mjd
'   om0:    initial long of ascending node, rads
'   inc:    desired inclination, rads
'   ap:     desired epoch of perihelion, as an mjd
'   om:     desired long of ascending node, rads
'
Declare Sub reduce_elements Lib "astro32" (ByVal MJD0 As Double, ByVal mjd As Double, ByVal inc0 As Double, ByVal ap0 As Double, ByVal om0 As Double, ByRef inc As Double, ByRef ap As Double, ByRef om As Double)
'
' correct the true altitude, ta, for refraction to the apparent altitude, aa,
' each in radians, given the local atmospheric pressure, pr, in mbars, and
' the temperature, tr, in degrees C.
'
Declare Sub refract Lib "astro32" (ByVal pr As Double, ByVal tr As Double, ByVal ta As Double, ByRef aa As Double)
'
' given the true geocentric ra and dec of an object, the observer's latitude,
'   lat, and a horizon displacement correction, dis, all in radians, find the
'   local sidereal times and azimuths of rising and setting, lstr/s
'   and azr/s, also all in radians, respectively.
' dis is the vertical displacement from the true position of the horizon. it
'   is positive if the apparent position is lower than the true position.
'   said another way, it is positive if the shift causes the object to spend
'   longer above the horizon. for example, atmospheric refraction is typically
'   assumed to produce a vertical shift of 34 arc minutes at the horizon; dis
'   would then take on the value +9.89e-3 (radians). On the other hand, if
'   your horizon has hills such that your apparent horizon is, say, 1 degree
'   above sea level, you would allow for this by setting dis to -1.75e-2
'   (radians).
'
' algorithm:
'   the situation is described by two spherical triangles with two equal angles
'    (the right horizon intercepts, and the common horizon transverse):
'   given lat, d(=d1+d2), and dis find z(=z1+z2) and rho, where      /| eq pole
'     lat = latitude,                                              /  |
'     dis = horizon displacement (>0 is below ideal)             / rho|
'     d = angle from pole = PI/2 - declination                /       |
'     z = azimuth east of north                            /          |
'     rho = polar rotation from down = PI - hour angle    /           |
'   solve simultaneous equations for d1 and d2:         /             |
'     1) cos(d) = cos(d1+d2)                           / d2           | lat
'            = cos(d1)cos(d2) - sin(d1)sin(d2)        /               |
'     2) sin(d2) = sin(lat)sin(d1)/sin(dis)          /                |
'   then can solve for z1, z2 and rho, taking       /                 |
'     care to preserve quadrant information.       /                 -|
'                                              z1 /        z2       | |
'                      ideal horizon ------------/--------------------|
'                                         | |   /                     N
'                                          -|  / d1
'                                       dis | /
'                                           |/
'                  apparent horizon  ---------------------------------
'
' note that when lat=0 this all breaks down (because d2 and z2 degenerate to 0)
'   but fortunately then we can solve for z and rho directly.
'
' status: 0: normal; 1: never rises; -1: circumpolar; 2: trouble.
'
Declare Sub riset Lib "astro32" (ByVal RA As Double, ByVal dec As Double, ByVal lat As Double, ByVal dis As Double, ByRef lstr As Double, ByRef lsts As Double, ByRef azr As Double, ByRef azs As Double, ByRef status As Long)
'
' crack a floating date string, bp, of the form X/Y/Z determined by the
'   DATE_DATE_FORMAT preference into its components. allow the day to be a
'   floating point number. A lone component is always a year if it contains
'   a decimal point or pref is MDY or DMY and it is not a reasonable month
'   or day value, respectively. Leave any unspecified component unchanged.
'   ( actually, the slashes may be anything but digits or a decimal point)
'   'pref' indicates the format of the date (DATE_xxx).
'
Declare Function scn_date Lib "astro32" (ByVal dtstr As String, ByRef m As Long, ByRef d As Double, ByRef Y As Long, ByVal pref As Long)
'
' scan a sexagesimal string and update a double. the string, bp, is of the form
'   H:M:S. a negative value may be indicated by a '-' char before any
'   component. All components may be integral or real. In addition to ':' the
'   separator may also be '/' or ';' or ',' or '-'.
' any components not specified in bp[] are copied from old's in 'o'.
'   eg:  ::10  only changes S
'        10    only changes H
'        10:0  changes H and M
'
Declare Function scn_sexa Lib "astro32" (ByVal o As Double, ByVal sexa As String) As Double
'
' round a time in days, *t, to the nearest second, IN PLACE.
'
Declare Sub rnd_second Lib "astro32" (ByRef t As Double)
'
' solve a spherical triangle:
'           A
'          /  \
'         /    \
'      c /      \ b
'       /        \
'      /          \
'    B ____________ C
'           a
'
' given A, b, c find B and a in range 0..B..2PI and 0..a..PI, respectively..
' Bp may be NULL if not interested in B.
' N.B. we pass in cos(c) and sin(c) because in many problems one of the sides
'   remains constant for many values of A and b.
'
Declare Sub solve_sphere Lib "astro32" (ByVal A As Double, ByVal b As Double, ByVal cosc As Double, ByVal sinc As Double, ByRef cosap As Double, ByRef Bp As Double)
'
' transformation from spherical to cartesian coordinates
'
'  l, b, r:     source: spherical coordinates
'  x, y, z:     result: rectangular coordinates
'
Declare Sub sphcart Lib "astro32" (ByVal l As Double, ByVal b As Double, ByVal r As Double, ByRef X As Double, ByRef Y As Double, ByRef z As Double)
'
' given the modified JD, mjd, return the true geocentric ecliptic longitude
'   of the sun for the mean equinox of the date, *lsn, in radians, the
'   sun-earth distance, *rsn, in AU, and the latitude *bsn, in radians
'   (since this is always <= 1.2 arcseconds, in can be neglected by
'   calling with bsn = 0& (VB style null)).
'
' if the APPARENT ecliptic longitude is required, correct the longitude for
'   nutation to the true equinox of date and for aberration (light travel time,
'   approximately  -9.27e7/186000/(3600*24*365)*2*pi = -9.93e-5 radians).
'
Declare Sub sunpos Lib "astro32" (ByVal mjd As Double, ByRef lsn As Double, ByRef rsn As Double, ByRef bsn As Any)
'
' given true ha and dec, tha and tdec, the geographical latitude, phi, the
' height above sea-level (as a fraction of the earths radius, 6378.16km),
' ht, and the equatorial horizontal parallax, ehp, find the apparent
' ha and dec, aha and adec allowing for parallax.
' all angles in radians. ehp is the angle subtended at the body by the
' earth's equator.
'
Declare Sub ta_par Lib "astro32" (ByVal tha As Double, ByVal tdec As Double, ByVal phi As Double, ByVal ht As Double, ByVal ehp As Double, ByRef aha As Double, ByRef adec As Double)
'
' Fill buffer with the name of the current timezone, given a preference
' flag, pref, (DATE_UTCTZ = always "UTC", DATE_LOCALTZ = e.g., "PDT")
'
Declare Sub tz_name Lib "astro32" (ByVal buf As String, ByVal pref As Long)
'
' correct the apparent altitude, aa, for refraction to the true altitude, ta,
' each in radians, given the local atmospheric pressure, pr, in mbars, and
' the temperature, tr, in degrees C.
'
Declare Sub unrefract Lib "astro32" (ByVal pr As Double, ByVal tr As Double, ByVal aa As Double, ByRef ta As Double)
'
' given a modified julian DATE, mjd, and a universally coordinated time, utc,
' return greenwich mean siderial time, *gst.
' NOTE: mjd must be at the beginning of the day!
'
Declare Sub utc_gst Lib "astro32" (ByVal mjd As Double, ByVal utc As Double, ByRef gst As Double)
'
' Return the current UTC offset (+ = West) in seconds
'
Declare Function utc_offs Lib "astro32" () As Long
'
' Return a Modified Julian Date given a Visual Basic Date
'
Declare Function vb_mjd Lib "astro32" (ByVal d As Date) As Double
'
'
' given a decimal year, return mjd
'
Declare Sub year_mjd Lib "astro32" (ByVal Y As Double, ByRef mjd As Double)

'---------------------------------------------------------------------
'   Degrees to Radians
'---------------------------------------------------------------------
Public Function degrad(d As Double) As Double
    degrad = (d * PI) / 180#
End Function

'---------------------------------------------------------------------
'   Radians to Degrees
'---------------------------------------------------------------------
Public Function raddeg(r As Double) As Double
    raddeg = (r * 180#) / PI
End Function

'---------------------------------------------------------------------
'   Hours to Degrees
'---------------------------------------------------------------------
Public Function hrdeg(h As Double)
    hrdeg = h * 15#
End Function

'---------------------------------------------------------------------
'   Degrees to Hours
'---------------------------------------------------------------------
Public Function deghr(d As Double) As Double
    deghr = d / 15#
End Function

'---------------------------------------------------------------------
'   Hours to Radians
'---------------------------------------------------------------------
Public Function hrrad(h As Double) As Double
    hrrad = degrad(hrdeg(h))
End Function

'---------------------------------------------------------------------
'   Radians to Hours
'---------------------------------------------------------------------
Public Function radhr(r As Double) As Double
    radhr = deghr(raddeg(r))
End Function
