#include <owl\owlpch.h>
#pragma hdrstop
#include <owl\framewin.h>
#include <owl\scroller.h>
#include <owl\opensave.h>
#include <string.h>
#include <dir.h>
#include <time.h>
#include <stdio.h>
#include "viewdlg.h"
#include "imagevw.h"
#include "imagelib.h"

#define MAXAPPNAME  20
static const char AppName[] = "ImageView";

class TViewWindow : virtual public TWindow {
  public:
		char      FileName[MAXPATH];
		TBitmap*  Bitmap;
		TPalette* Palette;
		TBrush*   BkgndBrush;
		long      Rop;
		int       PixelWidth;
		int       PixelHeight;
		WORD      Colors;
		BOOL      Fit;

		TViewWindow();
	  ~TViewWindow();

  protected:
		void      CmOpenJpg();
		void      CmOpenBmp();
		void      CmOpenGif();
		void      CmOpenPcx();

		void      CmFit();
		void      CeFit(TCommandEnabler& ce);

		void      Paint(TDC&, BOOL erase, TRect&);
		void      EvSize(UINT sizeType, TSize&);

		void      EvPaletteChanged(HWND hWndPalChg);
		BOOL      EvQueryNewPalette();
		void      EvSetFocus(HWND); // should use above when working
		void      EvDestroy();

		BOOL      UpdatePalette(BOOL alwaysRepaint);
		void      AdjustScroller();
		void      SetCaption(const char*);
		void      CmSaveToJpg();
		void      CmSaveToBmp();

  private:
		DecTransfer DecBuffer;
		ComTransfer ComBuffer;

  DECLARE_RESPONSE_TABLE(TViewWindow);
};

DEFINE_RESPONSE_TABLE1(TViewWindow, TWindow)
  EV_COMMAND(CM_OPENJPG, CmOpenJpg),
  EV_COMMAND(CM_OPENBMP, CmOpenBmp),
  EV_COMMAND(CM_OPENGIF, CmOpenGif),
  EV_COMMAND(CM_OPENPCX, CmOpenPcx),
  EV_COMMAND(CM_FIT, CmFit),
  EV_COMMAND(CM_SAVETOJPG, CmSaveToJpg),
  EV_COMMAND(CM_SAVETOBMP, CmSaveToBmp),
  EV_COMMAND_ENABLE(CM_FIT, CeFit),
  EV_WM_SIZE,
  EV_WM_PALETTECHANGED,
  EV_WM_QUERYNEWPALETTE,
  EV_WM_SETFOCUS,
  EV_WM_DESTROY,
END_RESPONSE_TABLE;

//
// Constructor for a TViewWindow, sets scroll styles andconstructs
// the Scroller object.  Also sets the Rop based on whether thedisplay
// is monochrome (two-color) or polychrome.
//
TViewWindow::TViewWindow()
  : TWindow(0, 0, 0)
{
  Attr.Style |= WS_VSCROLL | WS_HSCROLL;
  Bitmap = 0;
  Palette = 0;
  BkgndBrush = new TBrush(::GetSysColor(COLOR_WINDOW));
  Scroller = new TScroller(this, 1, 1, 200, 200);
  Fit = FALSE;

  TScreenDC screenDC;
  if (screenDC.GetDeviceCaps(NUMCOLORS) < 3 )
	 Rop = NOTSRCCOPY;
  else
	 Rop = SRCCOPY;
  memset(&DecBuffer, 0, sizeof(DecBuffer));
  DecBuffer.Res8But = BF_CHECKED;
  DecBuffer.Rad2DBut = BF_CHECKED;
  DecBuffer.Scl1But = BF_CHECKED;
  memset(&ComBuffer, 0, sizeof(ComBuffer));
  strcpy(ComBuffer.Cquality, "75");
  strcpy(ComBuffer.Csmooth, "00");

  SetCaption(0);
}

TViewWindow::~TViewWindow()
{
  delete Bitmap;
  delete Palette;
  delete BkgndBrush;
  delete Scroller;
}

//
// Build up a caption based on a filename, and set it into thetitle.
//
void
TViewWindow::SetCaption(const char* name)
{
  char caption[MAXPATH + MAXAPPNAME + 2 + 1];
  strcpy(FileName, name ? name : "(Untitled)");
  strcpy(caption, GetApplication()->GetName());
  strcat(caption, " - ");
  strcat(caption, FileName);
  if (Parent)
	 Parent->SetCaption(caption);
}

//
// Toggle Fit member variable & adjust scroller as needed
//
void
TViewWindow::CmFit()
{
  Fit = !Fit;
  AdjustScroller();
}

//
// The fit menu item is checked if the Fit member is true
//
void
TViewWindow::CeFit(TCommandEnabler& ce)
{
  ce.SetCheck(Fit ? TCommandEnabler::Checked : TCommandEnabler::Unchecked);
}

void
TViewWindow::CmOpenJpg()
{
  static TOpenSaveDialog::TData data (
		OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOREADONLYRETURN,
		"JPEG Files (*.JPG)|*.jpg|",
		0,
		"",
		"JPG"
  );
  int resolution, option, scale;
  // Display options dialog for JPG to Bitmap option choices
  unsigned int hddb, hpal;

  if (TFileOpenDialog(this, data).Execute() == IDOK)
	{
	 char fileTitle[MAXPATH];
	 TOpenSaveDialog::GetFileTitle(data.FileName, fileTitle,MAXPATH);
	 TDecDialog* pDlg = new TDecDialog(this, &DecBuffer);
	 if (pDlg->Execute() == IDOK)
	  {
		SetCursor(0, IDC_WAIT);

		// Set Dib output resolution  - 24, 8, or 4 bit
		if (DecBuffer.Res24But == BF_CHECKED)
		  resolution = 24;
		else if (DecBuffer.Res8But == BF_CHECKED)
		  resolution = 8;
		else
		  resolution = 4;

		// Set Color Quantization and dithering methods for output.
		// This is used if Resolution != 24

		if (DecBuffer.Rad1But == BF_CHECKED)
		  option = 1;
		else if (DecBuffer.Rad1DBut == BF_CHECKED)
		  option = 2;
		else if (DecBuffer.Rad2But == BF_CHECKED)
		  option = 3;
		else
		  option = 4;

		// Set output Dib scale factor either 1/1, 1/2, 1/4, or 1/8
		// of origional size
		if (DecBuffer.Scl1But == BF_CHECKED)
		  scale = 1;
		else if (DecBuffer.Scl2But == BF_CHECKED)
		  scale = 2;
		else if (DecBuffer.Scl4But == BF_CHECKED)
		  scale = 4;
		else
		  scale = 8;

		int password = 0;

		readjpgfile(data.FileName, resolution, scale, option, password,
						&hddb, &hpal);
		delete Palette;
		Palette = new TPalette((HPALETTE)hpal);
		delete Bitmap;
		Bitmap = new TBitmap((HBITMAP)hddb);
		PixelWidth = Bitmap->Width();
		PixelHeight = Bitmap->Height();
		UpdatePalette(TRUE);
	   AdjustScroller();
		SetCursor(0, IDC_ARROW);
		SetCaption(strlwr(fileTitle));
	  }
	}
}

void
TViewWindow::CmOpenGif()
{
  static TOpenSaveDialog::TData data (
		OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOREADONLYRETURN,
		"GIF Files (*.GIF)|*.gif|",
		0,
		"",
		"GIF"
  );
  // Display options dialog for JPG to Bitmap option choices
  unsigned int hddb, hpal;

  if (TFileOpenDialog(this, data).Execute() == IDOK)
	{
	 char fileTitle[MAXPATH];
	 TOpenSaveDialog::GetFileTitle(data.FileName, fileTitle,MAXPATH);
	 SetCursor(0, IDC_WAIT);

	 int password = 0;

	 readgiffile(data.FileName,  password, &hddb, &hpal);
	 delete Palette;
	 Palette = new TPalette((HPALETTE)hpal);
	 delete Bitmap;
	 Bitmap = new TBitmap((HBITMAP)hddb);
	 PixelWidth = Bitmap->Width();
	 PixelHeight = Bitmap->Height();
	 UpdatePalette(TRUE);
	 AdjustScroller();
	 SetCursor(0, IDC_ARROW);
	 SetCaption(strlwr(fileTitle));
	}
}

void
TViewWindow::CmOpenPcx()
{
  static TOpenSaveDialog::TData data (
		OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOREADONLYRETURN,
		"PCX Files (*.PCX)|*.pcx|",
		0,
		"",
		"PCX"
  );
  unsigned int hddb, hpal;

  if (TFileOpenDialog(this, data).Execute() == IDOK)
	{
	 char fileTitle[MAXPATH];
	 TOpenSaveDialog::GetFileTitle(data.FileName, fileTitle,MAXPATH);
	 SetCursor(0, IDC_WAIT);

	 int password = 0;

	 readpcxfile(data.FileName,  password, &hddb, &hpal);
	 delete Palette;
	 Palette = new TPalette((HPALETTE)hpal);
	 delete Bitmap;
	 Bitmap = new TBitmap((HBITMAP)hddb);
	 PixelWidth = Bitmap->Width();
	 PixelHeight = Bitmap->Height();
	 UpdatePalette(TRUE);
	 AdjustScroller();
	 SetCursor(0, IDC_ARROW);
	 SetCaption(strlwr(fileTitle));
	}
}

void
TViewWindow::CmOpenBmp()
{
  static TOpenSaveDialog::TData data (
		OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOREADONLYRETURN,
		"Bitmap Files (*.BMP)|*.bmp|",
		0,
		"",
		"BMP"
  );
  if (TFileOpenDialog(this, data).Execute() == IDOK)
	{
	 char fileTitle[MAXPATH];
	 TOpenSaveDialog::GetFileTitle(data.FileName, fileTitle,MAXPATH);
	 TDib* newDib;
	 try
	  {
		newDib = new TDib(data.FileName);
	  }
	 catch (TGdiBase::TXGdi)
	  {
		MessageBox("Cannot open bitmap file", GetApplication()->GetName(), MB_OK);
		return;
	  }
	 delete Palette;
	 Palette = 0;
	 try
	  {
		Palette = new TPalette(*newDib);
	  }
	 catch (...)
	  {
		Palette = new TPalette((HPALETTE)::GetStockObject(DEFAULT_PALETTE));
	  }
	 delete Bitmap;
	 Bitmap = new TBitmap(*newDib, Palette);
	 PixelWidth  = newDib->Width();
	 PixelHeight = newDib->Height();
	 UpdatePalette(TRUE);
	 AdjustScroller();
	 SetCaption(strlwr(fileTitle));
	}
}

//
// Adjust the Scroller range so that the the origin is the
// upper-most scrollable point and the corner is the
// bottom-most.
//
void
TViewWindow::AdjustScroller()
{
  TRect  clientRect = GetClientRect();

  // only show scrollbars when image is larger than
  // the client area and we are not stretching to fit.
  //
  if (Fit)
	 Scroller->SetRange(0, 0);

  else
	{
	 TPoint Range(Max(PixelWidth-clientRect.Width(), 0),
									Max(PixelHeight-clientRect.Height(),0));
	 Scroller->SetRange(Range.x, Range.y);
	}
  Scroller->ScrollTo(0, 0);
  if (!GetUpdateRect(clientRect, FALSE))
	 Invalidate(FALSE);
}

//
// Reset scroller range.
//
void
TViewWindow::EvSize(UINT SizeType, TSize& Size)
{
  TWindow::EvSize(SizeType, Size);
  if (SizeType != SIZEICONIC)
	{
	 AdjustScroller();
	 Invalidate(FALSE);
	}
}

//
// Somebody changed the palette. If its not us, then we need toupdate.
//
void
TViewWindow::EvPaletteChanged(HWND hWndPalChg)
{
  if (hWndPalChg != HWindow)
	 UpdatePalette(TRUE);    // pass FALSE to UpdateColors()instead of repaint
}

//
// We need to re-realize the logical palette each time
// we regain the input focus
//
BOOL
TViewWindow::EvQueryNewPalette()
{
  return UpdatePalette(TRUE);
}

//
// Use this message temporarily until the palette msgs get routedto us
//
void
TViewWindow::EvSetFocus(HWND)
{
  UpdatePalette(TRUE);
}

void
TViewWindow::EvDestroy()
{
  TWindow::EvDestroy();
}

BOOL
TViewWindow::UpdatePalette(BOOL alwaysRepaint)
{
  if (Palette)
	{
	 TClientDC clientDC(*this);
	 #if !defined(__WIN32__)
		Palette->UnrealizeObject();
	 #endif
	 clientDC.SelectObject(*Palette, FALSE);
	 if (alwaysRepaint)
		Invalidate(FALSE);
	 else
		clientDC.UpdateColors();
	 return TRUE;
	}
  return FALSE;
}

void
TViewWindow::Paint(TDC& dc, BOOL, TRect&)
{
  TRect clientRect = GetClientRect();
  dc.SelectObject(*BkgndBrush);

  if (Bitmap)
	{
	 TMemoryDC memoryDC(dc);
	 memoryDC.SelectObject(*Bitmap);
	 dc.SetStretchBltMode(COLORONCOLOR);
	 if (Palette)
	  {
		dc.SelectObject(*Palette, FALSE);
		dc.RealizePalette();
		memoryDC.SelectObject(*Palette, FALSE);
	  }

	 TRect imageRect(0,0, PixelWidth, PixelHeight);
	 if (Parent->IsIconic())
	  {
		dc.StretchBlt(Parent->GetClientRect(), memoryDC,imageRect, Rop);
	  }
	 else
	  {
		clientRect += TPoint((int)Scroller->XPos,(int)Scroller->YPos);
		if (Fit)
		  dc.StretchBlt(clientRect, memoryDC,imageRect, Rop);
		else
		  dc.BitBlt(imageRect, memoryDC, TPoint(0,0), Rop);
		// Clear borders here for no flicker
		//
		if (!Fit)
		 {
		  dc.PatBlt(TRect(TPoint(PixelWidth,0),clientRect.BottomRight()));
		  dc.PatBlt(TRect(TPoint(0,PixelHeight),clientRect.BottomRight()));
       }
	  }
	}
  else
	dc.PatBlt(clientRect, PATCOPY);
}


void TViewWindow::CmSaveToBmp()
{
 static TOpenSaveDialog::TData bmpdata (
		OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOREADONLYRETURN,
		"Bmp Files (*.BMP)|*.BMP|",
		0,
		0,
		"*.bmp"
	  );
 if ((TFileSaveDialog(this, bmpdata)).Execute() == IDOK)
  {
	// dump bmp into newdata.FileName

	if (Bitmap)
	 {
	  BOOL ret;
	  TDib *tmpdib = new TDib(*Bitmap, Palette);
	  ret = tmpdib->WriteFile(bmpdata.FileName);
	  if (!ret)
		 MessageBox(bmpdata.FileName, "Error Writing BMP",MB_OK);
	  else
		 MessageBox(bmpdata.FileName, "BMP Written", MB_OK);
	 }
	else
	  MessageBox("No Image Loaded!", "Error Writing BMP", MB_OK);
  }
}

void TViewWindow::CmSaveToJpg()
{
 static TOpenSaveDialog::TData jpgdata (
		OFN_HIDEREADONLY|OFN_FILEMUSTEXIST|OFN_NOREADONLYRETURN,
		"Jpg Files (*.JPG)|*.JPG|",
		0,
		0,
		"*.jpg"
	  );
 HBITMAP hddb;
 HPALETTE hpal;
 if (Bitmap)
  {
	if ((TFileSaveDialog(this, jpgdata)).Execute() == IDOK)
	 {
	  // dump into jpg file newdata.FileName

	  TComDialog* pDlg = new TComDialog(this, &ComBuffer);

	  // Display the options for writing a JPEG file
	  if (pDlg->Execute() == IDOK)
		{
		 SetCursor(0, IDC_WAIT);

		 // Set the type of marker used to identify the type
		 // of JPEG file it is.

		 int quality, smooth;

		 // Set the quality of the JPEG image.  A higher quality
		 // yields a better JPEG image that takes more space.

		 sscanf(ComBuffer.Cquality, "%d", &quality);

		 // Set a smoothing factor on the JPEG image.
		 sscanf(ComBuffer.Csmooth, "%d", &smooth);
		 hddb = *Bitmap;
		 hpal = *Palette;

		 writejpegfile(jpgdata.FileName, quality, smooth, 0,
							(unsigned int)hddb, (unsigned int)hpal);

		 // Call the JPeg Class to compress the Bitmap to the JPEG file
		 SetCursor(0, IDC_ARROW);
		}
	 }
  }
 else
	MessageBox("No Image Loaded", "Information", MB_OK);
}

//----------------------------------------------------------------------------

class TViewApp : public TApplication {
  public:
		TViewApp(const char far* name) : TApplication(name) {}
		void InitMainWindow()
		 {
		  MainWindow = new TFrameWindow(0, Name, new TViewWindow);
		  MainWindow->AssignMenu(JPGMENU);
		  MainWindow->SetIcon(this, JPGVIEW);
		 }
};

//----------------------------------------------------------------------------

int
OwlMain(int /*argc*/, char* /*argv*/ [])
{
  return TViewApp(AppName).Run();
}
