/******************************************************************************
*                                                                             *
*  This example registers a database on an SQL Server with the local ODBC     *
*  driver, opens the database and creates a table in the database.            *
*                                                                             *
******************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include "windows.h"
#include "sql.h"
#include "sqlext.h"
#include "odbcinst.h"
#include "stdlib.h"

// ============================================================================

BOOL WINAPI ODBCDriverDlgProc( HWND, UINT, WPARAM, LPARAM );
BOOL PrepareDriverConnection( HENV *, HDBC * );
BOOL CleanDriverConnection( HENV, HDBC );
BOOL GetODBCDrivers( HWND );
BOOL CreateTable( HENV, char * );

// ============================================================================

HINSTANCE hInst;

// ============================================================================

int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd,
                      int nShow )
{
   HWND hDlg;
   MSG msg;


   hInst = hInstance;
   hDlg = CreateDialog( hInst, MAKEINTRESOURCE( 10000 ), NULL,
                        ODBCDriverDlgProc );
   if( ! hDlg )
      return( FALSE );

   while( GetMessage( &msg, NULL, 0, 0 ) )
   {
      if( IsDialogMessage( hDlg, &msg ) )
         continue;

      TranslateMessage( &msg );
      DispatchMessage( &msg );
   }
   return( FALSE );
}

// ============================================================================

BOOL WINAPI ODBCDriverDlgProc( HWND hDlg, UINT msg, WPARAM wParam,
                               LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:
      {
         HENV hEnv;
         HDBC hDbc;
                           //  Open an ODBC environment and database connection
         if( ! PrepareDriverConnection( &hEnv, &hDbc ) )
         {
            DestroyWindow( hDlg );
            PostQuitMessage( 0 );
            break;
         }
                                        //  List all the drivers in the listbox
         if( ! GetODBCDrivers( GetDlgItem( hDlg, 100 ) ) )
         {
            CleanDriverConnection( hEnv, hDbc );           //  If not, clean up
            DestroyWindow( hDlg );
            PostQuitMessage( 0 );
            break;
         }
         SetProp( hDlg, "henv", hEnv );         //  Save ODBC handles for later
         SetProp( hDlg, "hdbc", hDbc );
         return( TRUE );
      }


      case WM_COMMAND:
         if( wParam == IDCANCEL )
         {                                  //  Clean up when program is closed
            CleanDriverConnection( GetProp( hDlg, "henv" ), 
                                   GetProp( hDlg, "hdbc" ) );
            DestroyWindow( hDlg );
            PostQuitMessage( 0 );
            break;
         }
         if( wParam == IDOK )
         {
            char szDriver[ 256 ], szDBQ[ 512 ];
            char szDBName[ 64 ], szServer[ 128 ];
            char szUser[ 64 ], szPwd[ 64 ];
            char *pszCurrent;
            int iItem;

                                //  First, get the driver listbox item selected
            iItem = SendDlgItemMessage( hDlg, 100, LB_GETCURSEL, 0, 0 );
            if( iItem == LB_ERR )
               break;                               //  Now get the driver name
            if( ! SendDlgItemMessage( hDlg, 100, LB_GETTEXT, iItem,
                                      (LPARAM)szDriver ) )
               break;
                                       //  Get name to use for the new database
            if( ! GetDlgItemText( hDlg, 101, szDBName, 64 ) )
               break;
                                             //  Get server name (if specified)
            GetDlgItemText( hDlg, 102, szServer, 64 );
                                                  //  Get username and password
            GetDlgItemText( hDlg, 103, szUser, 64 );
            GetDlgItemText( hDlg, 104, szPwd, 64 );

            strcpy( szDBQ, "DSN=" );          //  Construct a connection string
            strcat( szDBQ, szDBName );                //  Each item is ended by
            pszCurrent = szDBQ + strlen( szDBQ ) + 1;           //  by a (null)

            strcpy( pszCurrent, "SERVER=" );     //  The entire string is ended
            strcat( pszCurrent, szServer );             //  with a (null)(null)
            pszCurrent = pszCurrent + strlen( pszCurrent ) + 1;

            strcpy( pszCurrent, "UID=" );
            if( strlen( szUser ) )
               strcat( pszCurrent, szUser );
            pszCurrent = pszCurrent + strlen( pszCurrent ) + 1;

            strcpy( pszCurrent, "PWD=" );
            if( strlen( szPwd ) )
               strcat( pszCurrent, szPwd );
            pszCurrent = pszCurrent + strlen( pszCurrent ) + 1;

            *pszCurrent = '\0';            //  Add second null to end of string
                                   //  Attempt to add data source to local ODBC
            if( ! SQLConfigDataSource( NULL, ODBC_ADD_DSN, szDriver,  szDBQ ) )
               break;
                                            //  If it worked, go make the table
            CreateTable( GetProp( hDlg, "henv" ), szDBName );
            break;
         }
         break;
   }
   return( FALSE );
}

// === Obtain an HENV and an HDBC from ODBC ===================================

BOOL PrepareDriverConnection( HENV *phEnv, HDBC *phDbc )
{
   if( SQLAllocEnv( phEnv ) != SQL_SUCCESS )
   {
      *phEnv = NULL;
      return( FALSE );
   }

   if( SQLAllocConnect( *phEnv, phDbc ) != SQL_SUCCESS )
   {
      SQLFreeEnv( *phEnv );
      *phEnv = NULL;
      *phDbc = NULL;
      return( FALSE );
   }
   return( TRUE );
}

// === Free HDBC and HENV =====================================================

BOOL CleanDriverConnection( HENV hEnv, HDBC hDbc )
{
   if( hDbc )
      SQLFreeConnect( hDbc );
   if( hEnv )
      SQLFreeEnv( hEnv );
   return( TRUE );
}

// === Add ODBC drivers to a listbox ==========================================

BOOL GetODBCDrivers( HWND hList )
{
   LPSTR lpStr, lpCurrent;
   USHORT wLen;

   lpStr = malloc( 4096 );
   if( ! lpStr )
      return( FALSE );

   if( ! SQLGetInstalledDrivers( lpStr, 4096, &wLen ) )
   {
      free( lpStr );
      return( FALSE );
   }

   lpCurrent = lpStr;
   while( *lpCurrent )
   {
      SendMessage( hList, LB_ADDSTRING, 0, (LPARAM)lpCurrent );
      lpCurrent = lpCurrent + lstrlen( lpCurrent ) + 1;
   }

   free( lpStr );
   return( TRUE );
}

// === Create a table in a specified database =================================

BOOL CreateTable( HENV hEnv, char *szDatabase )
{
   RETCODE RetCode;
   char szSQL[ 512 ], szName[ 32 ];
   SDWORD sdwLen;
   SWORD swLen;
   HDBC hDbc;
   HSTMT hStmt;


   RetCode = SQLAllocConnect( hEnv, &hDbc );
   if( RetCode != SQL_SUCCESS )
   {
      SQLError( NULL, hDbc, NULL, szName, &sdwLen, szSQL, 256, &swLen );
      MessageBox( NULL, szSQL, "CreateDB", MB_OK );
      return( FALSE );
   }

   RetCode = SQLConnect( hDbc, szDatabase, SQL_NTS, "sa", SQL_NTS,
                               "", SQL_NTS );
   if( RetCode != SQL_SUCCESS && RetCode != SQL_SUCCESS_WITH_INFO )
   {
      SQLError( NULL, hDbc, NULL, szName, &sdwLen, szSQL, 256, &swLen );
      MessageBox( NULL, szSQL, "CreateDB", MB_OK );
      SQLFreeConnect( hDbc );
      return( FALSE );
   }

   RetCode = SQLAllocStmt( hDbc, &hStmt );
   if( RetCode != SQL_SUCCESS )
   {
      SQLError( NULL, hDbc, NULL, szName, &sdwLen, szSQL, 256, &swLen );
      MessageBox( NULL, szSQL, "CreateDB", MB_OK );
      SQLFreeConnect( hDbc );
      return( FALSE );
   }

   strcpy( szSQL, "CREATE TABLE Test (Name varchar,Age integer)" );
   RetCode = SQLExecDirect( hStmt, szSQL, SQL_NTS );
   if( RetCode != SQL_SUCCESS )
   {
      SQLError( NULL, hDbc, NULL, szName, &sdwLen, szSQL, 256, &swLen );
      MessageBox( NULL, szSQL, "CreateDB", MB_OK );
      SQLFreeStmt( hStmt, SQL_DROP );
      SQLFreeConnect( hDbc );
      return( FALSE );
   }

   SQLFreeStmt( hStmt, SQL_DROP );
   SQLFreeConnect( hDbc );
   return( TRUE );
}
  
     
