// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||     Event Log example for Windows NT. Public domain by Mark Gamber     ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

#include "windows.h"
#include "events.h"


// ----------------------------------------------------------------------------


const char *TITLE = "Event Log";                            //  Used everywhere
HINSTANCE hInst;
HWND hMainWin;
HANDLE hEvent;       //  Kept as global since it's open the life of the program


// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||                          Program Entry Point                           ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++


int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd,
							 int nShow )
{
   WNDCLASS wc;
   MSG msg;
	char str[ 128 ];

   hInst = hInstance;

   wc.style = 0;                             //  Register the main window class
   wc.lpfnWndProc = MainWndProc;
   wc.cbClsExtra = 0;
   wc.cbWndExtra = 0;
   wc.hInstance = hInstance;
   wc.hCursor = LoadCursor( NULL, IDC_ARROW );
   wc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
   wc.hbrBackground = (HBRUSH)( COLOR_WINDOW + 1 );
   wc.lpszMenuName = "MainMenu";
   wc.lpszClassName = "events";

   if( ! RegisterClass( &wc ) )
      return( FALSE );
                                          //  Get registry ready for our events
   if( ! InitializeRegistry( hInstance ) )
      return( FALSE );
                                             //  Tell event system what we want
   hEvent = RegisterEventSource( NULL, "EventApp" );
   if( ! hEvent )
   {
      LoadString( hInst, 103, str, 128 );
      MessageBox( NULL, str, TITLE, MB_OK | MB_ICONSTOP );
      return( FALSE );
   }

   hMainWin = CreateWindow( "events", TITLE,
                            WS_OVERLAPPEDWINDOW,
                            CW_USEDEFAULT, CW_USEDEFAULT,
                            CW_USEDEFAULT, CW_USEDEFAULT,
                            NULL, NULL, hInstance, NULL );
   if( ! hMainWin )
   {
      DeregisterEventSource( hEvent );            //  Remove event registration
      return( FALSE );
   }

   ShowWindow( hMainWin, nShow );
   UpdateWindow( hMainWin );

   while( GetMessage( &msg, NULL, 0, 0 ) )
   {
      TranslateMessage( &msg );
      DispatchMessage( &msg );
   }

   DeregisterEventSource( hEvent );       //  Remove event registration on exit
   return( msg.wParam );
}

// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||                          Main Window Procedure                         ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

LONG WINAPI MainWndProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_CREATE:
      {
         RECT Rect;
         UINT uTabs[ 3 ];
         HWND hList;
                                 //  Create main display listbox for event list	
         GetClientRect( hWnd, &Rect );
         hList = CreateWindow( "listbox", NULL, WS_CHILD | WS_BORDER |
                               WS_VISIBLE | LBS_USETABSTOPS,
                               0, 0, Rect.left, Rect.bottom, hWnd,
                               (HMENU)1000, hInst, NULL );
         if( ! hList )
            return( -1 );

         uTabs[ 0 ] = 45;                     //  Set up tab spaces for listbox
         uTabs[ 1 ] = 110;
         uTabs[ 2 ] = 170;
         SendMessage( hList, LB_SETTABSTOPS, 3, (LPARAM)(UINT *)&uTabs );
         break;
      }


      case WM_SIZE:
      {
         RECT Rect;                     //  Resize listbox to fit window client

         GetClientRect( hWnd, &Rect );
         SetWindowPos( GetDlgItem( hWnd, 1000 ), NULL, 0, 0, Rect.right,
                       Rect.bottom, SWP_NOZORDER | SWP_NOMOVE );
         break;
      }


      case WM_DESTROY:
         PostQuitMessage( 0 );                                   //  Later, bud
         break;


      case WM_COMMAND:
      {
         if( wParam == IDM_EXIT )                        //  If time to exit...
         {
            PostQuitMessage( 0 );
            break;
         }

         if( wParam == IDM_ABOUT )                      //  The user is curious
         {
            DialogBox( hInst, "ABOUT", hWnd, AboutDlgProc );
            break;
         }

         if( wParam == IDM_ADDEVENT )             //  Add new event to log file
         {
            DialogBox( hInst, "ADDEVENT", hWnd, AddEventProc );
            break;
         }
                                                  //  View one of the log types
         if( wParam == IDM_VIEWAPPS ||
             wParam == IDM_VIEWSYSTEM ||
             wParam == IDM_VIEWSECURITY )
         {
            DisplayEvents( hWnd, wParam );
            break;
         }

         if( wParam == IDM_CLEARLOG )                       //  Clear Event Log
         {
            HANDLE hLog;

            hLog = OpenEventLog( NULL, "EventApp" );           //  Open our log
            if( ! hLog )
            {
               MessageBox( hWnd, "Error opening this program's Event Log!",
                           TITLE, MB_OK | MB_ICONEXCLAMATION );
               break;
            }

            if( ClearEventLog( hLog, NULL ) )           //  Attempt to clear it
            {
               MessageBox( hWnd, "Program's Event Log Cleared", TITLE,
                           MB_OK | MB_ICONINFORMATION );
               SendMessage( GetDlgItem( hWnd, 1000 ), LB_RESETCONTENT, 0, 0 );
            }
            else
               MessageBox( hWnd, "Unable to clear Event Log!", TITLE,
                           MB_OK | MB_ICONEXCLAMATION );

            CloseEventLog( hLog );           //  Close event log and we're done
            break;
         }
         break;
      }


      default:
         return( DefWindowProc( hWnd, msg, wParam, lParam ) );
   }
}

// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||                       About Dialog Procedure                           ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

BOOL WINAPI AboutDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:
         return( TRUE );

      case WM_COMMAND:
         if( wParam == IDOK || wParam == IDCANCEL )
         {
            EndDialog( hDlg, TRUE );
            return( TRUE );
         }
         break;
   }
   return( FALSE );
}

// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||                   Dialog Box to Add a New Event                        ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

BOOL WINAPI AddEventProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:        //  Initialize multiline edit and radiobuttons
         SendMessage( GetDlgItem( hDlg, 100 ), EM_LIMITTEXT, 255, 0 );
         CheckRadioButton( hDlg, 100, 104, 100 );
         return( TRUE );


      case WM_COMMAND:
      {
         if( wParam == IDOK || wParam == IDCANCEL )
         {
            char str[ 256 ];
            WORD wEvent;
            DWORD dwEventID;
            BOOL junk;

            if( IsDlgButtonChecked( hDlg, 100 ) )      //  Determine event type
                wEvent = EVENTLOG_ERROR_TYPE;         //  based on radiobuttons

            if( IsDlgButtonChecked( hDlg, 101 ) )
                wEvent = EVENTLOG_WARNING_TYPE;

            if( IsDlgButtonChecked( hDlg, 102 ) )
                wEvent = EVENTLOG_INFORMATION_TYPE;

            if( IsDlgButtonChecked( hDlg, 103 ) )
                wEvent = EVENTLOG_AUDIT_SUCCESS;

            if( IsDlgButtonChecked( hDlg, 104 ) )
                wEvent = EVENTLOG_AUDIT_FAILURE;

            if( wParam == IDOK )      //  If OK was pressed, get multiline text
            {
               if( GetDlgItemText( hDlg, 105, str, 256 ) )
               {                                 //  Also get optional event ID
                  dwEventID = GetDlgItemInt( hDlg, 106, &junk, FALSE );

                                                      //  If event was added...
                  if( AddNewEvent( wEvent, dwEventID, str ) )
                  {                     //  Inform user and let dialog box live
                     LoadString( hInst, 101, str, 256 );
                     MessageBox( hDlg, str, TITLE,
                                 MB_OK | MB_ICONINFORMATION );
                     break;
                  }
                  else                    //  If it failed, inform user and die
                  {
                     wsprintf( str, "Error #%d occurred adding the event!",
                               GetLastError() );
                     MessageBox( hDlg, str, TITLE,
                                 MB_OK | MB_ICONEXCLAMATION );
                  }
               }
            }
            EndDialog( hDlg, TRUE );
            return( TRUE );
         }
         break;
      }
   }
   return( FALSE );
}

// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||              Create and/or Open required Registry Keys                 ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

BOOL InitializeRegistry( HINSTANCE hInstance )
{
   HKEY hKey;
   DWORD dwTypes;
   char str[ MAX_PATH ];
	
                               //  Registry path stored in resource stringtable
   LoadString( hInstance, 100, str, 128 );           //  Create our key or exit
   if( RegCreateKey( HKEY_LOCAL_MACHINE, str, &hKey ) != ERROR_SUCCESS )
   {
      LoadString( hInstance, 102, str, 128 );
      MessageBox( NULL, str, TITLE, MB_OK | MB_ICONSTOP );
      return( FALSE );
   }
                      //  Get path to this executable to use as an event source
   GetModuleFileName( hInstance, str, MAX_PATH );
                                               //  Add path to registry or exit
   if( RegSetValueEx( hKey, "EventMessageFile", 0, REG_EXPAND_SZ,
                      str, lstrlen( str ) + 1 ) != ERROR_SUCCESS )
   {
      LoadString( hInstance, 104, str, 128 );
      MessageBox( NULL, str, TITLE, MB_OK | MB_ICONSTOP );
      RegCloseKey( hKey );
   }

                                             //  These are the types we support
   dwTypes = EVENTLOG_ERROR_TYPE | EVENTLOG_WARNING_TYPE |
             EVENTLOG_INFORMATION_TYPE | EVENTLOG_AUDIT_SUCCESS |
             EVENTLOG_AUDIT_FAILURE;
                                             //  Add them to the registry, also
   if( RegSetValueEx( hKey, "TypesSupported", 0, REG_DWORD,
                      (LPBYTE)&dwTypes, sizeof(DWORD) ) != ERROR_SUCCESS )
   {
      LoadString( hInstance, 105, str, 128 );
      MessageBox( NULL, str, TITLE, MB_OK | MB_ICONSTOP );
      RegCloseKey( hKey );
      return( FALSE );
   }
		
   RegCloseKey( hKey );              //  Close key and register this program as
                        //  an event source using the same name as with the key
   return( TRUE );
}

// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// ||           Friendlier call to ReportEvent() to add the Event            ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

BOOL AddNewEvent( WORD wEventType, DWORD IDEvent, LPSTR lpMsg )
{
   return( ReportEvent( hEvent, wEventType, 0, IDEvent, NULL, 1, 0,
                        &lpMsg, NULL ) );
}

// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++
// ||                                                                        ||
// || Displays events listed in a type determined by the menu item selected  ||
// ||                                                                        ||
// ++------------------------------------------------------------------------++
// ++------------------------------------------------------------------------++

BOOL DisplayEvents( HWND hWnd, UINT uID )
{
   HWND hListbox;
   EVENTLOGRECORD *pEventRec;
   BYTE bBuffer[ 256 ];
   DWORD dwRead, dwNext, dwRecord;
   char str[ 512 ];
   HANDLE hLocalEvent;

   hListbox = GetDlgItem( hWnd, 1000 );            //  Get handle of listbox on
   if( ! hListbox )                             //  the main window for display
      return( FALSE );

   if( uID == IDM_VIEWAPPS )                    //  Open the event log, type is
      hLocalEvent = OpenEventLog( NULL, "Application" );   // determined by the
                                                         //  menu item selected
   if( uID == IDM_VIEWSYSTEM )
      hLocalEvent = OpenEventLog( NULL, "System" );

   if( uID == IDM_VIEWSECURITY )
      hLocalEvent = OpenEventLog( NULL, "Security" );

   if( ! hLocalEvent )            //  If unable to open log, tell user and exit
   {
      wsprintf( str, "Error %d opening Event Log!", GetLastError() );
      MessageBox( hWnd, str, TITLE, MB_OK | MB_ICONEXCLAMATION );
      return( FALSE );
   }
                                                          //  Reset the listbox
   SendMessage( hListbox, LB_RESETCONTENT, 0, 0 );
   pEventRec = (EVENTLOGRECORD *)bBuffer;  //  Make a structure from the buffer
   dwRecord = 0;                                       //  No records quite yet

                                                          //  Start a read loop
   while( ReadEventLog( hLocalEvent, EVENTLOG_FORWARDS_READ |
                        EVENTLOG_SEQUENTIAL_READ, 0, pEventRec, 256,
                        &dwRead, &dwNext ) )
   {
               //  Multiple records may be read at a time, so we need a subloop

      while( dwRead > 0 )                                    //  And here it is
      {
         wsprintf( str, "Rec #%d\tEvent ID #%d", ++dwRecord,
                   pEventRec->EventID );

         switch( pEventRec->EventType )               //  All this constructs a
         {                                       //  somewhat reasonable string
            case EVENTLOG_ERROR_TYPE:                 //  to add to the listbox
               lstrcat( str, "\tError\t " );
               break;

            case EVENTLOG_WARNING_TYPE:
               lstrcat( str, "\tWarning\t " );
               break;

            case EVENTLOG_INFORMATION_TYPE:
               lstrcat( str, "\tInformation\t " );
               break;

            case EVENTLOG_AUDIT_SUCCESS:
               lstrcat( str, "\tAudit Success\t " );
               break;

            case EVENTLOG_AUDIT_FAILURE:
               lstrcat( str, "\tAudit Failure\t " );
               break;

            default:
               wsprintf( str + lstrlen( str ), "\tType %.4x\t ",
                         pEventRec->EventType );
					break;
         }
                                      //  If there's any string data, append it
         if( pEventRec->NumStrings )
            lstrcat( str, (LPSTR)(LPBYTE)pEventRec + pEventRec->StringOffset );

                                                  //  Add string to the listbox
         SendMessage( hListbox, LB_ADDSTRING, 0, (LPARAM)(LPSTR)str );

                              //  Subject record size from total read last time
         dwRead -= pEventRec->Length;
                                  //  Make a new structure based on the old one
         pEventRec = (EVENTLOGRECORD *)( (LPBYTE)pEventRec +
							pEventRec->Length );
      }
                              //  When inner loop ends, reset structure pointer
      pEventRec = (EVENTLOGRECORD *)bBuffer;
   }
                              //  When all records are read, check for an error
   if( ! dwRecord && GetLastError() != 38 )     //  Error 38 is EOF...no report
   {
      wsprintf( str, "Error %d reading Event Log!", GetLastError() );
      MessageBox( hWnd, str, TITLE, MB_OK | MB_ICONEXCLAMATION );
   }

   CloseEventLog( hLocalEvent );         //  Close the log and we're outta here
   return( TRUE );
}



