/*
 * ExtApi.c - demonstrates how to use Windows 95 and NT 4 phonebook APIs.
 */

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include <ras.h>
#ifndef RASENTRY		// If ras.h does not have phonebook APIs...
#include "rnaph.h"
#endif

#include "extapi.h"
#include "pb95.h"
#include "pbnt4.h"

//
// Define prototypes for function pointers, with types included.
//

DWORD (WINAPI *fnRasValidateEntryName) (LPSTR lpszPhonebook, LPSTR szEntry);

DWORD (WINAPI *fnRasRenameEntry) (LPSTR lpszPhonebook, LPSTR szEntryOld, LPSTR szEntryNew);
   
DWORD (WINAPI *fnRasDeleteEntry) (LPSTR lpszPhonebook, LPSTR szEntry);

DWORD (WINAPI *fnRasGetEntryProperties) (LPSTR lpszPhonebook, LPSTR szEntry, 
		LPBYTE lpbEntry, LPDWORD lpdwEntrySize, LPBYTE lpb, LPDWORD lpdwSize);
    
DWORD (WINAPI *fnRasSetEntryProperties) (LPSTR lpszPhonebook, LPSTR szEntry,
		LPBYTE lpbEntryInfo, DWORD dwEntryInfoSize, LPBYTE lpbDevInfo, DWORD dwDevInfoSize);

DWORD (WINAPI *fnRasGetCountryInfo) (LPRASCTRYINFO lpCtryInfo, LPDWORD lpdwSize);
   
DWORD (WINAPI *fnRasEnumDevices) (LPRASDEVINFO lpRasDevInfo, LPDWORD lpcbSize, LPDWORD lpcDevices);

//
// Global handle to lib
//

HANDLE g_hRnaInst;



//
// GENERAL NOTES:
// This sample will operate correctly on both Windows 95 and NT 4 beta 2.
// Based on the current operating system version, the sample will use either
// call functions in ph95.c (for Windows 95) or phnt4.c (for Windows NT 4.0).
// See those source files for platform-specific notes.
//

//
// Prototypes private to this file, but common to either Win95 or NT 4
//

void Validate (char *szPhonebook, char *szEntry);
void Rename (char *szPhonebook, char *szOldEntry, char *szNewEntry);
void Delete (char *szPhonebook, char *szEntry);
void GetCountry (int nCountry);

//
// Get pointers to functions (OS version-dependent)
//

BOOL g_bUseNT;

BOOL InitFunctions (void)
	{
	OSVERSIONINFO ver;

	ver.dwOSVersionInfoSize = sizeof (ver);
	if (!GetVersionEx (&ver))
		{
		printf ("GetVersionEx failed!\n");
		return FALSE;
		}

	if (ver.dwMajorVersion < 4)
		{
		if (ver.dwPlatformId == VER_PLATFORM_WIN32_NT)
			printf ("Windows NT %i.%i does not support phonebook APIs.\n", 
					ver.dwMajorVersion, ver.dwMinorVersion);
		else
			printf ("This operating system is not supported by extapi.\n");

		return FALSE;
		}

	if (ver.dwPlatformId == VER_PLATFORM_WIN32_NT)
		{
		g_bUseNT = TRUE;
		printf ("Using Windows NT 4\n");
		return InitFunctionsNT4();
		}
	else if (ver.dwPlatformId != VER_PLATFORM_WIN32_WINDOWS)
		{
		printf ("Only Windows 95 and Windows NT 4.0 are supported by this sample.\n");
		return FALSE;
		}

	g_bUseNT = FALSE;
	printf ("Using Windows 95\n");
	return InitFunctions95();
	}


void HelpAndExit (void)
	{
	printf ("Command line options:\n\n"
		"Operation: (must include one)\n"
		"  -c    Create new entry (entry structure is hardcoded in sample code)\n"
		"  -v    Validate entry name text (checks for invalid symbols or duplicates)\n"
		"  -s    Set entry properties (changes are hardcoded in sample code)\n"
		"  -g    Get entry properties\n"
		"  -r    Rename entry\n"
		"  -d    Delete entry\n"
		"  -i    Get country info\n"
		"  -e    Enumerate devices\n\n"
		"Parameters: (optional)\n"
		"  -p:<phonebook>  Phonebook name, default=NULL\n"
		"  -n:<entryname>  Entry name, both -n and -o default to My Connection\n"
		"  -o:<oldname>    (-r & -s ops) -o specifies old name, -n specifies new name\n"
		"  -#:<code>       Country code for -i op; default is 1; specify 0 to enumerate.\n");
		
	exit (-1);
	}


#define OP_VALIDATE			1
#define OP_SET_PROPERTIES	2
#define OP_GET_PROPERTIES	3
#define OP_RENAME			4
#define OP_DELETE			5
#define OP_GET_COUNTRY		6
#define OP_ENUM_DEVICES		7
#define OP_CREATE			8

//
// Our program's entry point
//

void main (int argc, char *argv[])
	{
	int i;
	int nOperation = 0;
	char *szPhonebook = NULL;
	char *szEntry = "My Connection";
	char *szOldEntry = "My Connection";
	char *szCountryCode = "1";

	// Parse command line
	for (i = 1 ; i < argc ; i++)
		{
		if (argv[i][0] == '-' || argv[i][0] == '/')
			{
			switch (tolower (argv[i][1]))
				{
				case 'c':
					nOperation = OP_CREATE;
					break;

				case 'v':
					nOperation = OP_VALIDATE;
					break;

				case 's':
					nOperation = OP_SET_PROPERTIES;
					break;

				case 'g':
					nOperation = OP_GET_PROPERTIES;
					break;

				case 'r':
					nOperation = OP_RENAME;
					break;

				case 'd':
					nOperation = OP_DELETE;
					break;

				case 'i':
					nOperation = OP_GET_COUNTRY;
					break;

				case 'e':
					nOperation = OP_ENUM_DEVICES;
					break;

				case 'p':
					if (argv[i][2] == ':')
						szPhonebook = &argv[i][3];
					else if (!argv[i][2] && (i + 1) < argc)
						szPhonebook = argv[++i];
					else
						HelpAndExit();
					break;

				case 'n':
					if (argv[i][2] == ':')
						szEntry = &argv[i][3];
					else if (!argv[i][2] && (i + 1) < argc)
						szEntry = argv[++i];
					else
						HelpAndExit();
					break;

				case 'o':
					if (argv[i][2] == ':')
						szOldEntry = &argv[i][3];
					else if (!argv[i][2] && (i + 1) < argc)
						szOldEntry = argv[++i];
					else
						HelpAndExit();
					break;

				case '#':
					if (argv[i][2] == ':')
						szCountryCode = &argv[i][3];
					else if (!argv[i][2] && (i + 1) < argc)
						szCountryCode = argv[++i];
					else
						HelpAndExit();
					break;

				default:
					HelpAndExit();
				}
			}
		else
			HelpAndExit();
		}

	if (!nOperation)
		HelpAndExit();

	//
	// Initialize: LoadLibrary and GetProcAddress for all RAS APIs
	//

	if (!InitFunctions())
		exit (-1);


	//
	// Perform operation based on command line
	//

	switch (nOperation)
		{
		case OP_VALIDATE:
			Validate (szPhonebook, szEntry);
			break;

		case OP_SET_PROPERTIES:
			if (g_bUseNT)
				SetPropertiesNT4 (szPhonebook, szOldEntry, szEntry);
			else
				SetProperties95 (szPhonebook, szOldEntry, szEntry);
			break;

		case OP_GET_PROPERTIES:
			if (g_bUseNT)
				GetPropertiesNT4 (szPhonebook, szEntry);
			else
				GetProperties95 (szPhonebook, szEntry);
			break;

		case OP_RENAME:
			Rename (szPhonebook, szOldEntry, szEntry);
			break;

		case OP_DELETE:
			Delete (szPhonebook, szEntry);
			break;

		case OP_GET_COUNTRY:
			GetCountry (atoi (szCountryCode));
			break;

		case OP_ENUM_DEVICES:
			if (g_bUseNT)
				EnumDevicesNT4 (NULL, NULL);	// Call EnumDevices but don't request default modem
			else
				EnumDevices95 (NULL, NULL);		
			break;

		case OP_CREATE:
			if (g_bUseNT)
				NewEntryNT4 (szPhonebook, szEntry);
			else
				NewEntry95 (szPhonebook, szEntry);
			break;
		}

	// Clean up and exit
	FreeLibrary (g_hRnaInst);
	}


//
// Validate name and phonebook
//

void Validate (char *szPhonebook, char *szEntry)
	{
	DWORD rc;

	rc = fnRasValidateEntryName (szPhonebook, szEntry);
	printf ("RasValidateEntryName (%s, %s) rc=%i\n", szPhonebook, szEntry, rc);
	}


//
// Rename operation
//

void Rename (char *szPhonebook, char *szOldEntry, char *szNewEntry)
	{
	DWORD rc;

	//
	// WINDOWS 95:
	// If renaming a phonebook entry that no longer has a device installed for
	// it, error 625 is returned.  A user can create an entry, then remove the
	// device which the entry needs.
	//
	// WINDOWS NT:
	// When a device associated with a phonebook entry is removed, RAS continues
	// to maintain information for it until the device is actually deleted.
	// All APIs continue to work as if the device still exists.
	//
	
	rc = fnRasRenameEntry (szPhonebook, szOldEntry, szNewEntry);

	printf ("RasRenameEntry rc=%u\n", rc);
	}


//
// Entry delete operation
//

void Delete (char *szPhonebook, char *szEntry)
	{
	DWORD rc;
	
	rc = fnRasDeleteEntry (szPhonebook, szEntry);

	printf ("RasDeleteEntry rc=%u\n", rc);
	}


//
// Get country info
//

void GetCountry (int nCountry)
	{
	LPRASCTRYINFO lpInfo;
	DWORD dwSize;
	DWORD rc;

	dwSize = 512;								// set for sizing
	lpInfo = (LPRASCTRYINFO) malloc (dwSize);
	lpInfo->dwSize = sizeof (RASCTRYINFO);		// set for versioning, not for sizing

	if (nCountry)
		{
		lpInfo->dwCountryID = nCountry;

		rc = fnRasGetCountryInfo (lpInfo, &dwSize);

		printf ("RasGetCountryInfo (country code %i) rc=%i\n", nCountry, rc);

		if (!rc)
			{
			printf ("\nCountry ID:          %u\n", lpInfo->dwCountryID);
			printf ("Next Country ID:     %u\n", lpInfo->dwNextCountryID);
			printf ("Country Code:        %u\n", lpInfo->dwCountryCode);
			printf ("Country Name Offset: %u\n", lpInfo->dwCountryNameOffset);
			printf ("Country Name:        %s\n", (LPBYTE) lpInfo + (lpInfo->dwCountryNameOffset));
			}
		}
	else
		{
		lpInfo->dwCountryID = 1;

		printf (" ID           Name\n");
		do	{
			// Must set before each call to RasGetCountryInfo
			dwSize = 512;								// set for sizing
			lpInfo->dwSize = sizeof (RASCTRYINFO);		// set for versioning, not for sizing

			rc = fnRasGetCountryInfo (lpInfo, &dwSize);

			if (!rc)
				{
				printf ("%3i %s\n", lpInfo->dwCountryID, (LPBYTE) lpInfo + (lpInfo->dwCountryNameOffset));
				lpInfo->dwCountryID = lpInfo->dwNextCountryID;
				}
			else
				printf ("RasGetCountryInfo (country code %i) rc=%i\n", nCountry, rc);
			} while (!rc && lpInfo->dwCountryID);
		}

	free (lpInfo);
	}


//
// Display helper
//

void PrintIpAddr (char *szLabel, RASIPADDR ipaddr)
	{
	printf ("%s%i.%i.%i.%i\n", szLabel, ipaddr.a, ipaddr.b, ipaddr.c, ipaddr.d);
	}

