/*
 * pb95.c - Implements Win95 version of phonebook functions
 */


#define WIN32_LEAN_AND_MEAN
#define WINVER 0x400			// important!

#include <windows.h>

#include <ras.h>
#include <raserror.h>

#include <stdio.h>
#include <stdlib.h>

#ifndef RASENTRY

// Test for outdated ras.h (for example, a version included with VC 2)
#if RAS_MaxDeviceName != 128
	#error You have an outdated ras.h.  Make sure include path points to most recent Win32 SDK first.
#endif

#include "rnaph.h"

#else

//
// NT 4 beta header file does not currently define RASNP_Netbeui constant
//

#ifndef RASNP_Netbeui
#define RASNP_Netbeui 0x00000001
#endif

#endif // ifndef RASENTRY

#include "extapi.h"
#include "pb95.h"


//
// WINDOWS 95 GENERAL NOTES:
//
// Dialup Networking folder does not update.  It must be closed and restarted 
// to display new phone book entries.  A refresh is not sufficient.
//
// The functions in rnaph.h will be merged into ras.h, starting with the
// shell update release (SUR) Win32 SDK.  However, Windows 95 does not ship with
// these APIs, but you can include RNAPH.DLL with your application.
//
// Include ras.h in your project first, and use rnaph.h only when necessary.
// This sample demonstrates one technique to do this above.
//

BOOL InitFunctions95 (void)
	{
	// First try RASAPI32.DLL
	g_hRnaInst = LoadLibrary ("rasapi32.dll");
	fnRasGetEntryProperties = (void *) GetProcAddress (g_hRnaInst, "RasGetEntryProperties");
	
	// Next try RNAPH.DLL
	if (!fnRasGetEntryProperties)
		{
		FreeLibrary (g_hRnaInst);

		g_hRnaInst = LoadLibrary ("rnaph.dll");
		fnRasGetEntryProperties = (void *) GetProcAddress (g_hRnaInst, "RasGetEntryProperties");
		}

	if (!fnRasGetEntryProperties)
		{
		printf ("Can't find Phone Book APIs\n");
		return FALSE;
		}

	// Get the rest of the needed phone book functions
	fnRasSetEntryProperties = (void *) GetProcAddress (g_hRnaInst, "RasSetEntryProperties");
	fnRasValidateEntryName =  (void *) GetProcAddress (g_hRnaInst, "RasValidateEntryName");
	fnRasRenameEntry =        (void *) GetProcAddress (g_hRnaInst, "RasRenameEntry");
	fnRasDeleteEntry =        (void *) GetProcAddress (g_hRnaInst, "RasDeleteEntry");
	fnRasGetCountryInfo =     (void *) GetProcAddress (g_hRnaInst, "RasGetCountryInfo");
	fnRasEnumDevices =        (void *) GetProcAddress (g_hRnaInst, "RasEnumDevices");

	return TRUE;
	}


//
// Set properties of an existing entry
//

void SetProperties95 (char *szPhonebook, char *szOldEntry, char *szNewEntry)
	{
	DWORD rc;
	DWORD dwEntryInfoSize, dwDeviceInfoSize;
	LPRASENTRY lpEntryInfo;
	LPRASDEVINFO lpDeviceInfo;

	// Obtain sizes
	dwEntryInfoSize = 0;
	dwDeviceInfoSize = 0;

	rc = fnRasGetEntryProperties (szPhonebook, szOldEntry, 
								  NULL, &dwEntryInfoSize,
								  NULL, &dwDeviceInfoSize);

	// Make the call
	if (rc == ERROR_BUFFER_TOO_SMALL)
		{
		// Allocate memory
		lpEntryInfo = (LPRASENTRY) malloc (dwEntryInfoSize);
		if (dwDeviceInfoSize)
			lpDeviceInfo = (LPRASDEVINFO) malloc (dwDeviceInfoSize);
		else
			lpDeviceInfo = NULL;

		lpEntryInfo->dwSize = dwEntryInfoSize;
		if (lpDeviceInfo)
			lpDeviceInfo->dwSize = dwDeviceInfoSize;

		//
		// NOTE: On Windows 95, error 14 (out of memory) is returned if 
		// RasGetEntryProperties is called on an existing entry that no
		// longer has a device installed for it.  The error code may
		// change in the future.
		//

		// Get the entry properties
		rc = fnRasGetEntryProperties (szPhonebook, szOldEntry,
									  (LPBYTE) lpEntryInfo, &dwEntryInfoSize,
									  (LPBYTE) lpDeviceInfo, &dwDeviceInfoSize);

		if (!rc)
			{
			// Modify the properties here (this sample rotates the number)
			char szBuf[RAS_MaxPhoneNumber + 1];

			lstrcpy (szBuf, lpEntryInfo->szLocalPhoneNumber);
			lstrcpy (lpEntryInfo->szLocalPhoneNumber, &szBuf[1]);
			szBuf[1] = 0;
			lstrcat (lpEntryInfo->szLocalPhoneNumber, &szBuf[0]);

			//
			// REQUIRED:
			//  Make sure area code is full. It is safe to put a space when the entry
			//  is blank.  You might also use TAPI APIs to get a default area code if
			//  necessary, or throw out a dialog box and get it from the user.  This
			//  is irrespective of the RASEO_UseCountryAndAreaCodes flag.
			//

			if (!lpEntryInfo->szAreaCode[0])
				lstrcpy (lpEntryInfo->szAreaCode, " ");

			//
			// NOTES: To set SLIP, make sure RASEO_IpHeaderCompression is disabled.
			//		  Currently there is no way to set CSLIP or NetWare IPX 
			//		  dial-in support with an API.
			//
			//		  RasSetEntryProperties also reverses the RASEO_TerminalBeforeDial 
			//		  and RASEO_TerminalAfterDial flags.  RasGetEntryProperties returns 
			//		  the correct flags, RasSetEntryProperties reverses them.
			//
			//		  Possible workaround until this is fixed:  After setting the entry
			//		  properties once, call RasGetEntryProperties and call
			//		  RasSetEntryProperties again.  This double-reverse will fix the
			//		  flags, and the double-reverse will be harmless once this bug is 
			//		  fixed.
			//

			// Set the properties
			rc = fnRasSetEntryProperties (szPhonebook, szNewEntry,
										  (LPBYTE) lpEntryInfo, dwEntryInfoSize,
										  (LPBYTE) lpDeviceInfo, dwDeviceInfoSize);

			printf ("RasSetEntryProperties rc=%i\n", rc);
			}
		else
			printf ("Second RasGetEntryProperties call failed, rc=%i\n", rc);

		// Cleanup
		if (lpDeviceInfo)
			free (lpDeviceInfo);
		free (lpEntryInfo);
		}
	else
		printf ("Sizing call (RasGetEntryProperties) failed, rc=%i\n", rc);
	}

//
// Display the properties for an entry
//

void GetProperties95 (char *szPhonebook, char *szEntry)
	{
	DWORD rc;
	DWORD dwEntryInfoSize, dwDeviceInfoSize;
	LPRASENTRY lpEntryInfo;
	LPRASDEVINFO lpDeviceInfo;

	// Obtain sizes
	dwEntryInfoSize = 0;
	dwDeviceInfoSize = 0;

	//
	// NOTE: On Windows 95, error 14 (out of memory) is returned if 
	// RasGetEntryProperties is called on an existing entry that no
	// longer has a device installed for it.  The error code may
	// change in the future.
	//

	rc = fnRasGetEntryProperties (szPhonebook, szEntry, 
								  NULL, &dwEntryInfoSize,
								  NULL, &dwDeviceInfoSize);

	// Make the call
	if (rc == ERROR_BUFFER_TOO_SMALL)
		{
		// Allocate memory
		lpEntryInfo = (LPRASENTRY) malloc (dwEntryInfoSize);
		if (dwDeviceInfoSize)
			lpDeviceInfo = (LPRASDEVINFO) malloc (dwDeviceInfoSize);
		else
			lpDeviceInfo = NULL;

		lpEntryInfo->dwSize = dwEntryInfoSize;
		if (lpDeviceInfo)
			lpDeviceInfo->dwSize = dwDeviceInfoSize;

		// Call RAS
		rc = fnRasGetEntryProperties (szPhonebook, szEntry,
									  (LPBYTE) lpEntryInfo, &dwEntryInfoSize,
									  (LPBYTE) lpDeviceInfo, &dwDeviceInfoSize);

		// Print the results
		if (!rc)
			{
			printf ("Members of the RASENTRY struct:\n\n");

			printf ("  dwSize            %u\n", lpEntryInfo->dwSize);

			printf ("  dwfOptions        %08Xh\n", lpEntryInfo->dwfOptions);
			if (lpEntryInfo->dwfOptions & RASEO_UseCountryAndAreaCodes)
				printf ("    RASEO_UseCountryAndAreaCodes\n");
			if (lpEntryInfo->dwfOptions & RASEO_SpecificIpAddr)
				printf ("    RASEO_SpecificIpAddr\n");
			if (lpEntryInfo->dwfOptions & RASEO_SpecificNameServers)
				printf ("    RASEO_SpecificNameServers\n");
			if (lpEntryInfo->dwfOptions & RASEO_IpHeaderCompression)
				printf ("    RASEO_IpHeaderCompression\n");
			if (lpEntryInfo->dwfOptions & RASEO_RemoteDefaultGateway)
				printf ("    RASEO_RemoteDefaultGateway\n");
			if (lpEntryInfo->dwfOptions & RASEO_DisableLcpExtensions)
				printf ("    RASEO_DisableLcpExtensions\n");
			if (lpEntryInfo->dwfOptions & RASEO_TerminalBeforeDial)
				printf ("    RASEO_TerminalBeforeDial\n");
			if (lpEntryInfo->dwfOptions & RASEO_TerminalAfterDial)
				printf ("    RASEO_TerminalAfterDial\n");
			if (lpEntryInfo->dwfOptions & RASEO_ModemLights)
				printf ("    RASEO_ModemLights\n");
			if (lpEntryInfo->dwfOptions & RASEO_SwCompression)
				printf ("    RASEO_SwCompression\n");
			if (lpEntryInfo->dwfOptions & RASEO_RequireEncryptedPw)
				printf ("    RASEO_RequireEncryptedPw\n");
			if (lpEntryInfo->dwfOptions & RASEO_RequireMsEncryptedPw)
				printf ("    RASEO_RequireMsEncryptedPw\n");
			if (lpEntryInfo->dwfOptions & RASEO_RequireDataEncryption)
				printf ("    RASEO_RequireDataEncryption\n");
			if (lpEntryInfo->dwfOptions & RASEO_NetworkLogon)
				printf ("    RASEO_NetworkLogon\n");
			if (lpEntryInfo->dwfOptions & RASEO_UseLogonCredentials)
				printf ("    RASEO_UseLogonCredentials\n");
			if (lpEntryInfo->dwfOptions & RASEO_PromoteAlternates)
				printf ("    RASEO_PromoteAlternates\n");

			printf ("  dwCountryID        %u\n", lpEntryInfo->dwCountryID);
			printf ("  dwCountryCode      %u\n", lpEntryInfo->dwCountryCode);
			printf ("  szAreaCode         %s\n", lpEntryInfo->szAreaCode);
			printf ("  szLocalPhoneNumber %s\n", lpEntryInfo->szLocalPhoneNumber);

#ifndef _RNAPH_H_
			printf ("  dwAlternateOffset  %u\n", lpEntryInfo->dwAlternateOffset);	// newer
#else
			printf ("  dwAlternatesOffset %u\n", lpEntryInfo->dwAlternatesOffset);	// older
#endif

			PrintIpAddr ("  ipaddr             ", lpEntryInfo->ipaddr);
			PrintIpAddr ("  ipaddrDns          ", lpEntryInfo->ipaddrDns);
			PrintIpAddr ("  ipaddrDnsAlt       ", lpEntryInfo->ipaddrDnsAlt);
			PrintIpAddr ("  ipaddrWins         ", lpEntryInfo->ipaddrWins);
			PrintIpAddr ("  ipaddrWinsAlt      ", lpEntryInfo->ipaddrWinsAlt);

			printf ("  dwFrameSize:       %u\n", lpEntryInfo->dwFrameSize);

			printf ("  dwfNetProtocols    %08Xh\n", lpEntryInfo->dwfNetProtocols);
			if (lpEntryInfo->dwfNetProtocols & RASNP_Netbeui)
				printf ("    RASNP_Netbeui\n");
			if (lpEntryInfo->dwfNetProtocols & RASNP_Ip)
				printf ("    RASNP_Ip\n");
			if (lpEntryInfo->dwfNetProtocols & RASNP_Ipx)
				printf ("    RASNP_Ipx\n");

			printf ("  dwFramingProtocol  %08Xh\n", lpEntryInfo->dwFramingProtocol);
			if (lpEntryInfo->dwFramingProtocol & RASFP_Ppp)
				printf ("    RASFP_Ppp\n");
			if (lpEntryInfo->dwFramingProtocol & RASFP_Slip)
				printf ("    RASFP_Slip\n");
			if (lpEntryInfo->dwFramingProtocol & RASFP_Ras)
				printf ("    RASFP_Ras (NT 3.1 and WFW)\n");

			printf ("  szScript           %s\n", lpEntryInfo->szScript);
			printf ("  szAutodialDll      %s\n", lpEntryInfo->szAutodialDll);
			printf ("  szAutodialFunc     %s\n", lpEntryInfo->szAutodialFunc);
			printf ("  szDeviceType       %s\n", lpEntryInfo->szDeviceType);
			printf ("  szDeviceName       %s\n", lpEntryInfo->szDeviceName);
            printf ("  szX25PadType       %s\n", lpEntryInfo->szX25PadType);
			printf ("  szX25Address       %s\n", lpEntryInfo->szX25Address);
			printf ("  szX25Facilities    %s\n", lpEntryInfo->szX25Facilities);
			printf ("  szX25UserData      %s\n", lpEntryInfo->szX25UserData);
            printf ("  dwChannels         %u\n", lpEntryInfo->dwChannels);
            printf ("  dwReserved1        %u\n", lpEntryInfo->dwReserved1);
            printf ("  dwReserved2        %u\n", lpEntryInfo->dwReserved2);
			}
		else
			printf ("Second call to RasGetEntryProperties failed with rc=%i\n", rc);

		// Cleanup
		if (lpDeviceInfo)
			free (lpDeviceInfo);
		free (lpEntryInfo);
		}
	else
		{
		printf ("Sizing call (RasGetEntryProperties) failed, rc=%i\n", rc);
		if (!rc)
			{
			printf ("Expected ERROR_BUFFER_TOO_SMALL\n");
			printf ("dwEntryInfoSize=%u\ndwDeviceInfoSize=%u\n", dwEntryInfoSize, dwDeviceInfoSize);
			}
		}
	}


//
// EnumDevices enumerates all the RAS devices & returns default device if necessary
//

BOOL EnumDevices95 (LPSTR szDeviceNameBuf, LPSTR szDeviceTypeBuf)
	{
	LPRASDEVINFO lpRasDevInfo;
	DWORD dwBufSize;
	DWORD dwDevCount;
	DWORD rc;
	DWORD i;

	//
	// EnumDevices serves two purposes.  When szDeviceNameBuf is NULL,
	// the function prints out all devices and returns.  When szDeviceNameBuf
	// is non-NULL, it does not print device names, but instead copies the
	// first available device info to a buffer.
	//
	// lpRasDevInfo is an undocumented device specific structure in RAS, but 
	// if the device is a modem, it is documented in TAPI under comm/datamodem.
	// Make sure RasEntry.szDeviceType == "modem" before using this info.  Also,
	// this is undocumented as far as RAS is concerned and may change.
	//

	// Determine buffer size
	dwBufSize = 0;
	rc = fnRasEnumDevices (NULL, &dwBufSize, &dwDevCount);

	if (!rc)
		{
		printf ("No devices present.\n");
		return FALSE;
		}

	// Allocate a buffer, then get the devices
	lpRasDevInfo = (LPRASDEVINFO) malloc (dwBufSize);
	lpRasDevInfo->dwSize = sizeof (RASDEVINFO);

	rc = fnRasEnumDevices (lpRasDevInfo, &dwBufSize, &dwDevCount);

	// if either name buffer is NULL, print results and return
	if (!szDeviceNameBuf || !szDeviceTypeBuf)
		{
		if (!rc)
			{
			for (i = 0 ; i < dwDevCount ; i++)
				{
				printf ("%2i. %s (%s)\n", i + 1, 
						lpRasDevInfo[i].szDeviceName,
						lpRasDevInfo[i].szDeviceType);
				}
			}
		else
			{
			// Display error, then force a NULL return val
			printf ("RasEnumDevices rc=%i\n", rc);
			}

		free (lpRasDevInfo);
		return FALSE;
		}

	// Otherwise return default info
	lstrcpy (szDeviceNameBuf, lpRasDevInfo[0].szDeviceName);
	lstrcpy (szDeviceTypeBuf, lpRasDevInfo[0].szDeviceType);

	free (lpRasDevInfo);
	return TRUE;
	}


//
// Create a new entry using RasSetEntryProperties, but without getting
// existing settings first.
//

void NewEntry95 (char *szPhonebook, char *szEntry)
	{
	DWORD rc;

	#define RAS_OPTIONS  \
	   RASEO_UseCountryAndAreaCodes+\
	   RASEO_RemoteDefaultGateway+\
	   RASEO_ModemLights+\
	   RASEO_TerminalAfterDial

	RASENTRY reNew = 
		{
		sizeof(RASENTRY),	// dwSize
		RAS_OPTIONS,		// dwfOptions
		1,					// dwCountryID, 1 == United States
		0,					// dwCountryCode, 0 == Use default for Country ID
		"800",				// szAreaCode, could be a space but must be set
		"555-5555",			// szLocalPhoneNumber
		0,					// dwAlternateOffset (aka dwAlternatesOffset)
		{0,0,0,0},			// ipaddr
		{0,0,0,0},			// Dns
		{0,0,0,0},			// DnsAlt
		{0,0,0,0},			// addrWins
		{0,0,0,0},			// addrWinsAlt
		0,					// dwFrameSize
		RASNP_Ip,			// dwfNetProtocol
		RASFP_Ppp,			// dwFramingProtocol
		"",					// szScript
		"",					// szAutodialDll
		"",					// szAutodialFunc
		"",					// szDeviceType, this will be set below
		"",					// szDeviceName, this will be set below
		"", 				// szX25PadType
		"", 				// szX25Address
		"", 				// szX25Facilities
		"", 				// szX25UserData
		0,					// dwChannels
		0,					// dwReserved1
		0					// dwReserved2
		};

	//
	// Let's get a device name and type (it's required)
	//

	if (!EnumDevices95 (reNew.szDeviceName, reNew.szDeviceType))
		{
		printf ("A required device is not installed.  Cannot complete operation.\n");
		return;
		}

	//
	// Ready to add entry.  Call RasSetEntryProperties.
	//

	rc = fnRasSetEntryProperties (szPhonebook, szEntry,
								  (LPBYTE) &reNew, reNew.dwSize,
								  (LPBYTE) NULL, 0);

	printf ("RasSetEntryProperties rc=%i\n", rc);
	}

