//======================================================================
//
// FILEMON.c - main module for VxD FILEMON
//
// By Mark Russinovich and Bryce Cogswell
//
//======================================================================
#define   DEVICE_MAIN
#include  "ioctlcmd.h"
#include  "filemon.h"
#undef    DEVICE_MAIN

#if DEBUG
#define dprint(arg) dprintf arg
#else
#define dprint(arg)
#endif

//----------------------------------------------------------------------
//                     G L O B A L   D A T A 
//----------------------------------------------------------------------

// real service pointers with the hook thunks
ppIFSFileHookFunc       PrevIFSHookProc;

// hash table data 
PHASH_ENTRY		HashTable[NUMHASH];

// list of freed entries (to save some allocation calls)
PHASH_ENTRY		FreeEntries = NULL;

// buffer data
PSTORE_BUF		Store 		= NULL;
PSTORE_BUF		StoreExtra 	= NULL;
ULONG			Sequence 	= 0;

// maximum amount of buffers we will grab for buffered unread data
ULONG			NumStore 	= 0;
ULONG			MaxStore 	= 5;

// semaphore for critical sections
SEMHANDLE               StoreMutex, HashMutex;

// unknown error string
CHAR                    errstring[32];


//----------------------------------------------------------------------
//                   V X D  C O N T R O L
//----------------------------------------------------------------------

// device declaration
Declare_Virtual_Device(FILEMON)

// message handlers - we only care about dynamic loading and unloading
DefineControlHandler(SYS_DYNAMIC_DEVICE_INIT, OnSysDynamicDeviceInit);
DefineControlHandler(SYS_DYNAMIC_DEVICE_EXIT, OnSysDynamicDeviceExit);
DefineControlHandler(W32_DEVICEIOCONTROL, OnW32Deviceiocontrol);

//----------------------------------------------------------------------
// 
// ControlDispatcher
//
// Multiplexes incoming VxD messages from Windows to their handlers.
//
//----------------------------------------------------------------------
BOOL __cdecl ControlDispatcher(
	DWORD dwControlMessage,
	DWORD EBX,
	DWORD EDX,
	DWORD ESI,
	DWORD EDI,
	DWORD ECX)
{
  START_CONTROL_DISPATCH

    ON_SYS_DYNAMIC_DEVICE_INIT(OnSysDynamicDeviceInit);
    ON_SYS_DYNAMIC_DEVICE_EXIT(OnSysDynamicDeviceExit);
    ON_W32_DEVICEIOCONTROL(OnW32Deviceiocontrol);

  END_CONTROL_DISPATCH

  return TRUE;
}

//----------------------------------------------------------------------
// B U F F E R  M A N A G E M E N T  A N D  P R I N T  R O U T I N E S
//----------------------------------------------------------------------


//----------------------------------------------------------------------
//
// FilemonFreeStore
//
// Frees all the data output buffers that we have currently allocated.
//
//----------------------------------------------------------------------
VOID FilemonFreeStore()
{
    PSTORE_BUF 	prev;
    
    PageFree( (MEMHANDLE) StoreExtra, 0 );
    while( Store ) {
	prev = Store->Next;
	PageFree( (MEMHANDLE) Store, 0 );
	Store = prev;
    }
}	


//----------------------------------------------------------------------
//
// FilemonNewStore
//
// Called when the current buffer has filled up. This moves us to the
// pre-allocated buffer and then allocates another buffer.
//
//----------------------------------------------------------------------
void FilemonNewStore( void )
{
    PSTORE_BUF prev = Store;

    // if we have maxed out or haven't accessed the current store
    // just return
    if( MaxStore == NumStore ) {
	Store->Len = 0;
	return;	
    }

    // see if we can re-use a store
    if( !Store->Len ) 
	return;

    // move to the next buffer and allocate another one
    Store 	= StoreExtra;
    Store->Len  = 0;
    Store->Next = prev;
    StoreExtra = NULL;

    // have to release the lock because this can cause a re-entrance
dprintf("    New Release Storelock\n" );
    MUTEX_V( StoreMutex );
    PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
                  (PMEMHANDLE) &StoreExtra, (PVOID) &StoreExtra );
dprintf("    New Get Storelock\n" );
    MUTEX_P( StoreMutex );
dprintf("    New Got Storelock\n" );
    NumStore++;
}


//----------------------------------------------------------------------
//
// FilemonOldestStore
//
// Goes through the linked list of storage buffers and returns the 
// oldest one.
//
//----------------------------------------------------------------------
PSTORE_BUF FilemonOldestStore( void )
{
    PSTORE_BUF  ptr = Store, prev = NULL;

    while ( ptr->Next )
	ptr = (prev = ptr)->Next;
    if ( prev )
	prev->Next = NULL;    
    return ptr;
}


//----------------------------------------------------------------------
//
// FilemonResetStore
//
// When a GUI is no longer communicating with us, but we can't unload,
// we reset the storage buffers.
//
//----------------------------------------------------------------------
VOID FilemonResetStore()
{
   PSTORE_BUF  current, next;

   current = Store->Next;
   while( current ) {
        next = current->Next;
	PageFree( (MEMHANDLE) current, 0 );
        current = next;
   }
   Store->Len = 0;
   Store->Next = NULL;
}


//----------------------------------------------------------------------
//
// UpdateStore
//
// Add a new string to Store, if it fits.
//
//----------------------------------------------------------------------
void UpdateStore( ULONG seq, const char * format, ... )
{	
    PENTRY		Entry;
    int			len;
    va_list		arg_ptr;

dprintf("  Get Storelock\n" );
    MUTEX_P( StoreMutex );
dprintf("  Got Storelock\n" );

    // See if its time to switch to extra buffer
    if ( Store->Len > MAX_STORE-350  &&  StoreExtra )  
	FilemonNewStore();

    if ( Store->Len <= MAX_STORE-350 )  { 

	/* Set our sequence number */
	Entry = (void *)(Store->Data+Store->Len);
	Entry->seq = seq;

	va_start( arg_ptr, format );
	len = vsprintf( Entry->text, format, arg_ptr );
	va_end( arg_ptr );

	Store->Len += sizeof(Entry->seq)+len+1;		// +1 for null char
    }
    MUTEX_V( StoreMutex );
dprintf("  Release Storelock\n" );
}


//----------------------------------------------------------------------
//
// FilemonHashCleanup
//
// Called when we are unloading to free any memory that we have 
// in our possession.
//
//----------------------------------------------------------------------
VOID FilemonHashCleanup()
{
   PHASH_ENTRY		hashEntry, nextEntry;
   ULONG		i;
  
   // first free the hash table entries
   for( i = 0; i < NUMHASH; i++ ) {
	hashEntry = HashTable[i];
	while( hashEntry ) {
		nextEntry = hashEntry->Next;
		HeapFree( hashEntry->FullName, 0 );
		HeapFree( hashEntry, 0 );
		hashEntry = nextEntry;
	}
   }

   // now, free structures on our free list
   while( FreeEntries ) {
	nextEntry = FreeEntries->Next;
	HeapFree( FreeEntries, 0 );
	FreeEntries = nextEntry;
   }
}

//----------------------------------------------------------------------
//
// FilemonStoreHash
//
// Stores the key and associated fullpath in the hash table.
//
//----------------------------------------------------------------------
VOID FilemonStoreHash( sfn_t filenumber, PCHAR fullname )
{
  PHASH_ENTRY     newEntry;

dprintf("     store get mutex\n");
  MUTEX_P( HashMutex );
dprintf("     store got mutex\n");
  if( FreeEntries ) {
    newEntry    = FreeEntries;
    FreeEntries = FreeEntries->Next;
    MUTEX_V( HashMutex );
  } else {
    // have to let go, because memory allocation can 
    // cause re-entrancy	
    MUTEX_V( HashMutex );
    newEntry = HeapAllocate( sizeof(HASH_ENTRY), 0 );
  }

  newEntry->filenumber 	        = filenumber;
  newEntry->FullName   	        = HeapAllocate( strlen(fullname)+1, 0);
  strcpy( newEntry->FullName, fullname );

  MUTEX_P( HashMutex );
  newEntry->Next		= HashTable[ HASHOBJECT(filenumber) ];
  HashTable[ HASHOBJECT(filenumber) ] = newEntry;	
  MUTEX_V( HashMutex );
dprintf("     store release mutex\n");
}

//----------------------------------------------------------------------
//
// FilemonFreeHashEntry
//
// When we see a file close, we can free the string we had associated
// with the fileobject being closed since we know it won't be used
// again.
//
//----------------------------------------------------------------------
VOID FilemonFreeHashEntry( sfn_t filenumber )
{
   PHASH_ENTRY		hashEntry, prevEntry;

dprintf("     free get mutex\n");
   MUTEX_P( HashMutex );
dprintf("     free got mutex\n");

   // look-up the entry
   hashEntry = HashTable[ HASHOBJECT( filenumber ) ];
   prevEntry = NULL;
dprintf("         scan hash table\n");
   while( hashEntry && hashEntry->filenumber != filenumber ) {
	prevEntry = hashEntry;
	hashEntry = hashEntry->Next;
   }
dprintf("         scan done\n");  
   // if we fall off, just return
   if( !hashEntry ) {
	MUTEX_V( HashMutex );
dprintf("     free release mutex (not found)\n");
	return;
   }

   // remove it from the hash list 
   if( prevEntry ) 
	prevEntry->Next = hashEntry->Next;
   else 
	HashTable[ HASHOBJECT( filenumber )] = hashEntry->Next;

   // put it on our free list
   hashEntry->Next 	= FreeEntries;
   FreeEntries 		= hashEntry;

   MUTEX_V( HashMutex );

   // free the memory associated with it
   HeapFree( hashEntry->FullName, 0 );
dprintf("     free release mutex\n");
}

//----------------------------------------------------------------------
//
// ErrorString
//
// Returns the string form of an error code.
//
//----------------------------------------------------------------------
PCHAR ErrorString( DWORD retval )
{
  switch( retval ) {
  case ERROR_SUCCESS:
    return "SUCCESS";
  case ERROR_OUTOFMEMORY:
    return "OUTOFMEM";
  case ERROR_ACCESS_DENIED:
    return "ACCDENIED";
  case ERROR_PATH_NOT_FOUND:
    return "NOTFOUND";
  case ERROR_TOO_MANY_OPEN_FILES:
    return "TOOMANYOPEN";
  case ERROR_FILE_NOT_FOUND:
    return "NOTFOUND";
  case ERROR_NO_MORE_ITEMS:
    return "NOMORE";
  case ERROR_MORE_DATA:
    return "MOREDATA";
  case ERROR_INVALID_DRIVE:
    return "INVALIDDRIVE";
  case ERROR_NOT_SAME_DEVICE:
    return "DIFFERENTDEVICE";
  case ERROR_WRITE_PROTECT:
    return "WRITEPROTECTED";
  case ERROR_BAD_UNIT:
    return "BADUNIT";
  case ERROR_NOT_READY:
    return "NOTREADY";
  case ERROR_NO_MORE_FILES:
    return "NOMORE";
  default:
    sprintf(errstring, "0x%x", retval );
    return errstring;
  }
}

//----------------------------------------------------------------------
//
// wstrlen
//
//----------------------------------------------------------------------
int wstrlen( unsigned short *unistring )
{
  int i = 0;
  int len = 0;
  
  while( unistring[i++] != 0 ) len+=2;
  return len;
}


//----------------------------------------------------------------------
//
// FilmonGetProcess
//
// Retrieves the process name.
//
//----------------------------------------------------------------------
PCHAR FilemonGetProcess( PCHAR ProcessName )
{
  PVOID       CurProc;
  PVOID       ring3proc;
  char        *name;

  // get the ring0 process pointer
  CurProc = VWIN32_GetCurrentProcessHandle();
  
  // now, map the ring3 PCB 
  ring3proc = (PVOID) SelectorMapFlat( Get_Sys_VM_Handle(), 
			       (DWORD) (*(PDWORD) ((char *) CurProc + 0x38)) | 0x7, 0 );

  if( ring3proc == (PVOID) -1 ) {
    strcpy( ProcessName, "???");
    return ProcessName;
  }

  // copy out the process name (max 8 characters)
  name = ((char *)ring3proc) + 0xF2;
  if( name[0] >= 'A' && name[0] < 'z' ) {
    strcpy( ProcessName, name );
    ProcessName[8] = 0;
  } else
    strcpy( ProcessName, "???" );
  return ProcessName;
}


//----------------------------------------------------------------------
//
// FilemonGetFullPath
//
// Returns the full pathname of a file, if we can obtain one, else
// returns a handle.
//
//----------------------------------------------------------------------
PCHAR FilemonGetFullPath(  sfn_t filenumber, PCHAR fullname )
{
  PHASH_ENTRY		hashEntry;

  // see if we find the key in the hash table
  fullname[0] = 0;
  MUTEX_P( HashMutex );
  hashEntry = HashTable[ HASHOBJECT( filenumber ) ];
  while( hashEntry && hashEntry->filenumber != filenumber )
    hashEntry = hashEntry->Next;
  MUTEX_V( HashMutex );
  if( hashEntry ) {
    strcpy( fullname, hashEntry->FullName );
  } else {
    sprintf( fullname, "0x%X", filenumber );
  }
  return fullname;
}


//----------------------------------------------------------------------
//
// FilemonConvertPath
//
// Converts a unicode path name to ANSI.
//
//----------------------------------------------------------------------
PCHAR FilemonConvertPath( int drive, path_t ppath, PCHAR fullpathname )
{
  int  i = 0;
  _QWORD result;

  if( drive != 0xFF ) {
    fullpathname[0] = drive+'A'-1;
    fullpathname[1] = ':';
    i = 2;
  } 
  UniToBCSPath( &fullpathname[i], ppath->pp_elements, 256, BCS_WANSI, &result );
  return( fullpathname );
}


//----------------------------------------------------------------------
//
// FilemonHookProc
//
// All (most) IFS functions come through this routine for us to look
// at.
//
//----------------------------------------------------------------------
int _cdecl FilemonHookProc(pIFSFunc pfn, int fn, int Drive, int ResType,
			   int CodePage, pioreq pir)
{
  int                retval;
  char               fullpathname[256];
  char               processname[256];
  char               data[256];
  char               drivestring[4];
  ioreq              origir;
  _WIN32_FIND_DATA   *finddata;
  struct srch_entry  *search;
  _QWORD             result;
  int                i, j;

dprintf("%d\n", fn );
  // inititlize default data
  data[0] = 0;

  // save original iorequest because some entries get modified
  origir = *pir;

  // get the current process name
  FilemonGetProcess( processname );

  // call the previous hooker first, to get the return code
  retval = (*PrevIFSHookProc)(pfn, fn, Drive, ResType, CodePage, pir);

  // now extract parameters based on the function type
  switch( fn ) {

  case IFSFN_OPEN:
    sprintf(data,"");
    if( origir.ir_options & ACTION_CREATENEW ) strcat(data,"CREATENEW ");
    if( origir.ir_options & ACTION_CREATEALWAYS ) strcat(data,"CREATEALWAYS ");
    if( origir.ir_options & ACTION_OPENEXISTING ) strcat(data,"OPENEXISTING ");
    if( origir.ir_options & ACTION_OPENALWAYS ) strcat(data,"OPENALWAYS ");
    if( origir.ir_options & ACTION_REPLACEEXISTING ) strcat(data,"REPLACEEXISTING ");

    UpdateStore( Sequence++, "%s\tOpen\t%s\t%s\t%s", 
		 processname, FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ),
		 data, ErrorString( retval ));
    FilemonStoreHash( pir->ir_sfn, fullpathname );
    break;

  case IFSFN_READ:
  case IFSFN_WRITE:
    sprintf( data, "Offset: %ld Length: %ld", origir.ir_pos, origir.ir_length );
    UpdateStore( Sequence++, "%s\t%s\t%s\t%s\t%s", 
		 processname, fn == IFSFN_READ? "Read" : "Write", 
		 FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_CLOSE:
    UpdateStore( Sequence++, "%s\tClose\t%s\t\t%s", 
		 processname, FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 ErrorString( retval ));
    FilemonFreeHashEntry( pir->ir_sfn );
    break;

  case IFSFN_DIR:
    switch( pir->ir_flags ) {
    case CREATE_DIR:
      sprintf(data, "CREATE");
      break;
    case DELETE_DIR:
      sprintf(data,"DELETE");
      break;
    case CHECK_DIR:
      sprintf(data,"CHECK");
      break;
    default:
      sprintf(data,"QUERY");
      break;
    }
    UpdateStore( Sequence++, "%s\tDirectory\t%s\t%s\t%s", 
		 processname, 
		 FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_SEEK:
    sprintf(data, "%s Offset: %ld",
	    pir->ir_flags == FILE_BEGIN ? "Beginning" : "End",
	    origir.ir_pos );
    UpdateStore( Sequence++, "%s\tSeek\t%s\t%s\t%s", 
		 processname, FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_COMMIT:
    sprintf(data, "%s", pir->ir_options == FILE_COMMIT_ASYNC ? 
	    "ASYNC" : "NOACCESSUPDATE" );
    UpdateStore( Sequence++, "%s\tCommit\t%s\t%s\t%s", 
		 processname, FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_FILELOCKS:
    sprintf(data, "Offset: %ld Length:%ld", origir.ir_pos, origir.ir_locklen );
    UpdateStore( Sequence++, "%s\t%s\t%s\t%s\t%s", 
		 processname, origir.ir_flags == LOCK_REGION ? "Lock" : "Unlock",
		 FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_FINDOPEN:
    if( !retval ) {
      finddata = (_WIN32_FIND_DATA *) pir->ir_data;
      UniToBCS( data, finddata->cFileName, 256, wstrlen(finddata->cFileName),BCS_WANSI, &result );
      data[wstrlen(finddata->cFileName)/2] = 0;
    }
    UpdateStore( Sequence++, "%s\tFindOpen\t%s\t%s\t%s", 
		 processname, FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ),
		 data, ErrorString( retval ));
    FilemonStoreHash( pir->ir_sfn, fullpathname );
    break;

  case IFSFN_FINDNEXT:
    if( !retval ) {
      finddata = (_WIN32_FIND_DATA *) pir->ir_data;
      UniToBCS( data, finddata->cFileName, 256, wstrlen(finddata->cFileName),BCS_WANSI, &result );
      data[wstrlen(finddata->cFileName)/2] = 0;
    }
    UpdateStore( Sequence++, "%s\tFindNext\t%s\t%s\t%s", 
		 processname, FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_FINDCLOSE:
    UpdateStore( Sequence++, "%s\tFindClose\t%s\t\t%s", 
		 processname, FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 ErrorString( retval ));
    FilemonFreeHashEntry( pir->ir_sfn );
    break;

  case IFSFN_FILEATTRIB:
    switch(origir.ir_flags ) {
    case GET_ATTRIBUTES:
      sprintf(data,"GetAttributes");
      break;
    case SET_ATTRIBUTES:
      sprintf(data, "SetAttributes" );
      break;
    case GET_ATTRIB_COMP_FILESIZE:
      sprintf(data, "GET_ATTRIB_COMP_FILESIZE" );
      break;
    case SET_ATTRIB_MODIFY_DATETIME:
      sprintf(data, "SET_ATTRIB_MODIFY_DATETIME");
      break;
    case SET_ATTRIB_LAST_ACCESS_DATETIME:
      sprintf(data, "SET_ATTRIB_LAST_ACCESS_DATETIME");
      break;
    case GET_ATTRIB_LAST_ACCESS_DATETIME:
      sprintf(data, "GET_ATTRIB_LAST_ACCESS_DATETIME");
      break;
    case SET_ATTRIB_CREATION_DATETIME:
      sprintf(data, "SET_ATTRIB_CREATION_DATETIME");
      break;
    case GET_ATTRIB_CREATION_DATETIME:
      sprintf(data, "GET_ATTRIB_CREATION_DATETIME");
      break;
    }
    UpdateStore( Sequence++, "%s\tAttributes\t%s\t%s\t%s", 
		 processname, 
		 FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_FILETIMES:
    switch( pir->ir_flags ) {
    case GET_MODIFY_DATETIME:
      sprintf(data, "Get Modify");
      break;
    case SET_MODIFY_DATETIME:
      sprintf(data, "Set Modify");
      break;
    case GET_LAST_ACCESS_DATETIME:
      sprintf(data, "Get Access");
      break;
    case SET_LAST_ACCESS_DATETIME:
      sprintf(data, "Set Access");
      break;
    case GET_CREATION_DATETIME:
      sprintf(data, "Get Creation");
      break;
    case SET_CREATION_DATETIME:
      sprintf(data, "Set Creation");
      break;
    }
    UpdateStore( Sequence++, "%s\tAttributes\t%s\t%s\t%s", 
		 processname, 
		 FilemonGetFullPath( pir->ir_sfn, fullpathname ),
		 data, ErrorString( retval ));
    break;

  case IFSFN_FLUSH:
    UpdateStore( Sequence++, "%s\tFlushVolume\t\t\t%s",
		 processname, ErrorString( retval ));
    break;

  case IFSFN_DELETE:
    UpdateStore( Sequence++, "%s\tDelete\t%s\t\t%s", 
		 processname, 
		 FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ),
		 ErrorString( retval ));
    FilemonFreeHashEntry( pir->ir_sfn );
    break;

  case IFSFN_SEARCH:
    if( pir->ir_flags == SEARCH_FIRST ) 
      FilemonConvertPath( Drive, pir->ir_ppath, fullpathname );
    else
      sprintf(fullpathname, "SearchNext" );
    if( !retval ) {
      j = 0;
      if( origir.ir_attr & FILE_ATTRIBUTE_LABEL ) {
	sprintf(data, "VolumeLabel: " );
	j = strlen( data );
      }
      search = (struct srch_entry *) origir.ir_data;
      for( i = 0; i < 13; i++ ) 
	if( search->se_name[i] != ' ' ) data[j++] = search->se_name[i];
      data[j] = 0;
    }
    UpdateStore( Sequence++, "%s\tSearch\t%s\t%s\t%s", 
		 processname, fullpathname, data, ErrorString( retval ));    
    break;
    
  case IFSFN_GETDISKINFO:

    if( !retval ) sprintf(data, "Free Space: %ld bytes", 
			  pir->ir_length * pir->ir_numfree );
    drivestring[0] = Drive+'A'-1;
    drivestring[1] = ':';
    drivestring[2] = 0;
    UpdateStore( Sequence++, "%s\tGetDiskInfo\t%s\t%s\t%s",
		 processname, drivestring, data, ErrorString( retval ));
    break;

  case IFSFN_RENAME:
    UpdateStore( Sequence++, "%s\tRename\t%s\t%s\t%s",
		 processname, FilemonConvertPath( Drive, pir->ir_ppath, fullpathname ),
		 FilemonConvertPath( Drive, pir->ir_ppath2, data ),
		 ErrorString( retval ));		 
    break;
  }
dprintf("==>%d\n", fn );
  return retval;
}


//----------------------------------------------------------------------
//
// OnSysDynamicDeviceInit
//
// Dynamic init. Hook all registry related VxD APIs.
//
//----------------------------------------------------------------------
BOOL OnSysDynamicDeviceInit()
{
  int i;

  // initialize mutex
  MUTEX_INIT( StoreMutex );
  MUTEX_INIT( HashMutex );

  // zero hash table
  for(i = 0; i < NUMHASH; i++ ) HashTable[i] = NULL;

  // allocate two buffers
  PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
	       (PMEMHANDLE) &Store, (PVOID) &Store );
  PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
	       (PMEMHANDLE) &StoreExtra, (PVOID) &StoreExtra );
  Store->Len = StoreExtra->Len   = 0;
  Store->Next = StoreExtra->Next = NULL;
  NumStore = 2;

  // hook IFS functions
  PrevIFSHookProc = IFSMgr_InstallFileSystemApiHook(FilemonHookProc);

  return TRUE;
}

//----------------------------------------------------------------------
//
// OnSysDynamicDeviceExit
//
// Dynamic exit. Unhook everything.
//
//----------------------------------------------------------------------
BOOL OnSysDynamicDeviceExit()
{
  // unhook IFS functions
  IFSMgr_RemoveFileSystemApiHook(FilemonHookProc);

  // free all memory
  FilemonHashCleanup();
  FilemonFreeStore();
  return TRUE;
}

//----------------------------------------------------------------------
//
// OnW32Deviceiocontrol
//
// Interface with the GUI.
//
//----------------------------------------------------------------------
DWORD OnW32Deviceiocontrol(PIOCTLPARAMS p)
{
  PSTORE_BUF      old;

  switch( p->dioc_IOCtlCode ) {
  case 0:
    return 0;
  case FILEMON_zerostats:
    MUTEX_P( StoreMutex );
    while ( Store->Next )  {
      // release next
      old = Store->Next;
      Store->Next = old->Next;
      MUTEX_P( StoreMutex );
      PageFree( (MEMHANDLE) old, 0 );
      MUTEX_V( StoreMutex );
      NumStore--;
    }
    Store->Len = 0;
    MUTEX_V( StoreMutex );
    return 0;

  case FILEMON_getstats:
    // Copy buffer into user space.
    MUTEX_P( StoreMutex );
    if ( MAX_STORE > p->dioc_cbOutBuf ) {
      // Don't do this; its not worth implementing the fix.
      MUTEX_V( StoreMutex );
      return 1;
    } else if ( Store->Len  ||  Store->Next ) {

      // Switch to a new store
      FilemonNewStore();

      // Fetch the oldest to give to user
      old = FilemonOldestStore();
      MUTEX_V( StoreMutex );

      // Copy it
      memcpy( p->dioc_OutBuf, old->Data, old->Len );

      // Return length of copied info
      *p->dioc_bytesret = old->Len;

      // Deallocate buffer
      PageFree( (MEMHANDLE) old, 0 );
      NumStore--;
    } else {
      // no unread data
      MUTEX_V( StoreMutex );
      *p->dioc_bytesret = 0;
    }
    return 0;

  default:
    return 1;
  }
}
