/******************************************************************************
*                                                                             *
*          IEList32: "Inline Edit" Listbox by Mark Gamber - 12/23/94          *
*                                                                             *
*  This is a custom control which creates an ownerdraw listbox and handles    *
*  double-clicks and an edit message from the parent to allow the user to     *
*  edit the listbox item "inline".                                            *
*                                                                             *
*  By using the program or any included part, the user assumes full           *
*  responsibility for it's use and may not hold the author liable for any     *
*  loss or damage. If unable to accept this condition, the program and all    *
*  included parts must be destroyed immediately. The program and all included *
*  parts are public domain, free and unsupported.                             *
*                                                                             *
******************************************************************************/

#include "windows.h"

// ============================================================================

#define IEL_EDITITEM  (WM_USER+1024)              //  Tell listbox to edit item

// ============================================================================
// === Function Prototypes ====================================================
// ============================================================================

LRESULT WINAPI MainWndProc( HWND, UINT, WPARAM, LPARAM );
                                      //  The following are part of the control
BOOL InitializeIEList( HINSTANCE );      //  Call this before creating controls
LRESULT WINAPI IEListboxWndProc( HWND, UINT, WPARAM, LPARAM );
LRESULT WINAPI IENewListboxProc( HWND, UINT, WPARAM, LPARAM );
LRESULT WINAPI IENewEditProc( HWND, UINT, WPARAM, LPARAM );
BOOL AddIEListText( HWND, HWND );
BOOL DrawIEListbox( HWND, LPDRAWITEMSTRUCT );
BOOL StartIEdit( HWND );
BOOL IELDrawButton( LPDRAWITEMSTRUCT );

// ============================================================================
// === Global Variables =======================================================
// ============================================================================

HINSTANCE hInst;
HWND hMainWnd;

// ============================================================================
// === Application Entry Point ================================================
// ============================================================================

int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd,
                      int nShow )
{
   WNDCLASS wc;
   MSG msg;

   wc.style = 0;                           //  Register main window class first
   wc.hInstance = hInstance;
   wc.lpfnWndProc = MainWndProc;
   wc.lpszClassName = "IELTEST32";
   wc.lpszMenuName = "MainMenu";
   wc.hCursor = LoadCursor( NULL, IDC_ARROW );
   wc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
   wc.hbrBackground = (HBRUSH)( COLOR_WINDOW + 1 );
   wc.cbClsExtra = wc.cbWndExtra = 0;

   if( ! RegisterClass( &wc ) )
      return( FALSE );

   if( ! InitializeIEList( hInstance ) )         //  Initialize control or exit
      return( FALSE );                             //  (it registers the class)

   hInst = hInstance;                    //  Save the instance handle for later
                                                     //  Create the main window
   hMainWnd = CreateWindow( "IELTEST32", "IEListbox", WS_OVERLAPPEDWINDOW,
                            CW_USEDEFAULT, CW_USEDEFAULT, 308, 200,
                            NULL, NULL, hInstance, NULL );
   if( ! hMainWnd )
      return( FALSE );

   ShowWindow( hMainWnd, nShow );                      //  Show the main window
   UpdateWindow( hMainWnd );

   while( GetMessage( &msg, NULL, 0, 0 ) )        //  Loop here until destroyed
   {
      TranslateMessage( &msg );
      DispatchMessage( &msg );
   }

   return( FALSE );
}

// ============================================================================
// === Main Window used to create IEList32 ====================================
// ============================================================================
   
LRESULT WINAPI MainWndProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_CREATE:
      {
         int i, iOffset;
         char szStr[ 1024 ];
         RECT Rect;
         
         
         GetClientRect( hWnd, &Rect );       //  Strings can be added the usual
         iOffset = 0;                        //  listbox way using LB_ADDSTRING
                                 //  But this control also takes an init string
         for( i = 0; i < 64; i++ )                           //  because it can
         {
            if( i == 0 )                    //  Build up an init string made of
               lstrcpy( szStr + iOffset, "Inplace Edit Listbox" );
            else                //  a number of smaller strings (max 128 chars)
               if( i == 1 )            //  and a CR (\n) separating each string
                  lstrcpy( szStr + iOffset, "12/23/94 by Mark Gamber" );
               else                   //  This init string is parsed into items
                  wsprintf( szStr + iOffset, "Item #%d", i + 1 );    //  by the
                               //  control and added to the listbox on creation
            iOffset = iOffset + lstrlen( szStr + iOffset );
            szStr[ iOffset ] = '\n';
            ++iOffset;                //  The whole thing ends with a NULL (\0)
         }
         szStr[ iOffset + 1 ] = '\0';
                                           //  Create the control or die trying
         if( ! CreateWindow( "IEList32", szStr,
                             WS_CHILD | WS_VISIBLE,
                             2, 18, Rect.right - 3, Rect.bottom - 20,
                             hWnd, (HMENU)100, hInst, NULL ) )
            return( -1 );
                                     //  Set a selection and give control focus
         SendMessage( GetDlgItem( hWnd, 100 ), LB_SETCURSEL, 0, 0 );
         SetFocus( GetDlgItem( hWnd, 100 ) );
         break;
      }


      case WM_PAINT:           //  Put up "double-click" message on main window
      {
         PAINTSTRUCT ps;
         HDC hDC;
         char szStr[ 64 ];

         hDC = BeginPaint( hWnd, &ps );
         lstrcpy( szStr, "Double-click to edit an item" );
         TextOut( hDC, 1, 1, szStr, lstrlen( szStr ) );
         EndPaint( hWnd, &ps );
         break;
      }


      case WM_SIZE:                       //  Resize control to fit main window
      {
         RECT Rect;

         GetClientRect( hWnd, &Rect );
         SetWindowPos( GetDlgItem( hWnd, 100 ), NULL, 0, 0, Rect.right - 4,
                       Rect.bottom - 20, SWP_NOZORDER | SWP_NOMOVE );
         break;
      }


      case WM_DESTROY:               //  Kill app when main window is destroyed
         PostQuitMessage( 0 );
         break;


      case WM_COMMAND:                                           //  Menu items
      {
         if( wParam == 101 )                                      //  Edit item
         {
            int iItem;
                                              //  Get current control selection
            iItem = SendDlgItemMessage( hWnd, 100, LB_GETCURSEL, 0, 0 );
            if( iItem != LB_ERR )        //  If valid, set control to edit mode
               SendDlgItemMessage( hWnd, 100, WM_COMMAND, IEL_EDITITEM, 0 );

            break;
         }
         if( wParam == 102 )
         {
            int iCount, i, dwColor;
            RECT Rect;

                                         //  Get number of items in the listbox
            iCount = SendDlgItemMessage( hWnd, 100, LB_GETCOUNT, 0, 0 );
            for( i = 0; i < iCount; i++ )      //  Reset the color of each item
            {                                      //  Get the item color first
               dwColor = SendDlgItemMessage( hWnd, 100, LB_GETITEMDATA,
                                             i, 0 );
               if( dwColor != 0 )                  //  If it's not zero (black)
               {                                      //  Reset it to black and
                  SendDlgItemMessage( hWnd, 100, LB_SETITEMDATA, i, 0 );
                  SendDlgItemMessage( hWnd, 100, LB_GETITEMRECT, i,
                                      (LPARAM)&Rect ); // force an item repaint
                  InvalidateRect( GetDlgItem( hWnd, 100 ), &Rect, TRUE );
               }
            }
            break;
         }
         if( wParam == 103 )
         {
            char szStr[ 128 ];

            LoadString( hInst, 100, szStr, 128 );
            MessageBox( hWnd, szStr, "IEListbox",
                        MB_OK | MB_ICONINFORMATION );
            SetFocus( GetDlgItem( hWnd, 100 ) );
            break;
         }
         if( wParam == 104 )                               //  Kill the program
         {
            DestroyWindow( hWnd );
            break;
         }
         break;
      }


      default:
         return( DefWindowProc( hWnd, msg, wParam, lParam ) );
   }
   return( FALSE );
}                                                      //  End of MainWndProc()

// ============================================================================
//    Everything below this point can be cut-n-pasted to add the control to
//    another application. Call InitializeIEList() or incorporate the code
//    to register the class elsewhere in the code before attempting to create
//    a window from the classname.
// ============================================================================

// ============================================================================
// === Register Custom Control Class ==========================================
// ============================================================================

BOOL InitializeIEList( HINSTANCE hInstance )
{
   WNDCLASS wc;

   wc.style = 0;
   wc.hInstance = hInstance;
   wc.lpfnWndProc = IEListboxWndProc;
   wc.lpszClassName = "IELIST32";
   wc.lpszMenuName = NULL;
   wc.hCursor = LoadCursor( NULL, IDC_ARROW );
   wc.hIcon = (HICON)NULL;
   wc.hbrBackground = (HBRUSH)( COLOR_WINDOW + 1 );
   wc.cbClsExtra = wc.cbWndExtra = 0;

   if( ! RegisterClass( &wc ) )                     //  Fail if this call fails
      return( FALSE );

   return( TRUE );
}

// ============================================================================
// === Custom Control Window Procedure ========================================
// ============================================================================

LRESULT WINAPI IEListboxWndProc( HWND hWnd, UINT msg, WPARAM wParam,
                                 LPARAM lParam )
{
   if( msg >= LB_ADDSTRING && msg < LB_MSGMAX )       //  Pass listbox messages
      return( SendDlgItemMessage( hWnd, 100, msg,   //  to something that cares
                                  wParam, lParam ) );        //  like a listbox


   switch( msg )
   {
      case WM_CREATE:                           //  When this window is created
      {
         RECT Rect;
         HWND hListbox;
         HINSTANCE hUseInst;


         GetClientRect( hWnd, &Rect );               //  Get this window's size
         hUseInst = (HINSTANCE)GetWindowLong( hWnd, GWL_HINSTANCE ); 
                                       //  Create ownerdraw listbox with notify
         hListbox = CreateWindow( "listbox", NULL, LBS_OWNERDRAWFIXED | 
                                  LBS_NOTIFY | WS_CHILD | WS_VISIBLE | 
                                  WS_VSCROLL | LBS_HASSTRINGS | 
                                  LBS_NOINTEGRALHEIGHT | WS_BORDER,
                                  1, 1, Rect.right - 1, Rect.bottom - 1,
                                  hWnd, (HMENU)100,    //  Child of this window
                                  hUseInst, NULL );
         if( ! hListbox )
            return( -1 );                        //  Die here if listbox failed
                                         //  Create a font and save it's handle
                       //  It's ownerdraw so text can be centered vertically to
                               //  prevent "jumping" when the edit is displayed
         if( GetWindowTextLength( hWnd ) )
            AddIEListText( hWnd, hListbox );

         SetProp( hWnd, "HFONT", CreateFont( -8, 0, 0, 0, FW_BOLD,
                  0, 0, 0, 0, 0, 0, 0, 0, "MS Sans Serif" ) );

         SendMessage( hListbox, WM_SETFONT,              //  Apply the new font
                      (WPARAM)GetProp( hWnd, "HFONT" ), 0 );
                                //  Get height of a listbox item and enlarge it
         Rect.top = SendMessage( hListbox, LB_GETITEMHEIGHT, 0, 0 );
                                                 //  a bit so the edit looks ok
         SendMessage( hListbox, LB_SETITEMHEIGHT, 0, (LPARAM)Rect.top + 2 );

         SetProp( hListbox, "OLDPROC",                 //  Subclass the listbox
                  (HANDLE)GetWindowLong( hListbox, GWL_WNDPROC ) );
         SetWindowLong( hListbox, GWL_WNDPROC, (DWORD)IENewListboxProc );

         SetProp( hWnd, "InEdit", (HANDLE)0 );         //  Not in edit mode yet
         break;
      }                                              //  Now officially created


      case WM_DESTROY:                       //  Clean up before going to sleep
         DeleteObject( GetProp( hWnd, "HFONT" ) );
         RemoveProp( hWnd, "HFONT" );
         RemoveProp( hWnd, "InEdit" );
         break;


      case WM_SETFONT:                //  Save font handle where we can find it
         SendDlgItemMessage( hWnd, 100, msg, wParam, lParam );
         SetProp( hWnd, "HFONT", (HFONT)wParam );
         return( TRUE );

      
      case WM_COMMAND:
         if( HIWORD( wParam ) == LBN_DBLCLK ||      //  Something wants to edit
             wParam == IEL_EDITITEM )                     //   the current item
         {
            if( wParam == IEL_EDITITEM &&                //  Can't edit nothing
                SendMessage( hWnd, LB_GETCURSEL, 0, 0 ) == LB_ERR )
               break;

            if( ! GetProp( hWnd, "InEdit" ) )       //  If not in edit mode now
               if( StartIEdit( hWnd ) )        //  and the controls are created
                  SetProp( hWnd, "InEdit", (HANDLE)1 );    //  In edit mode now

            break;
         }
         if( HIWORD( wParam ) == LBN_SELCHANGE && //  If listbox item selection
             GetProp( hWnd, "InEdit" ) )                        //  is changing
         {                                        //  Kill everything quick and
            DestroyWindow( GetDlgItem( GetDlgItem( hWnd, 100 ), 100 ) );
            DestroyWindow( GetDlgItem( GetDlgItem( hWnd, 100 ), 1 ) );
            DestroyWindow( GetDlgItem( GetDlgItem( hWnd, 100 ), 2 ) );
            SetProp( hWnd, "InEdit", (HANDLE)0 );    //  no longer in edit mode
            break;
         }
         SendMessage( GetParent( hWnd ), msg,    //  Pass LB messages to parent
                      MAKELONG( (WORD)GetWindowLong( hWnd, GWL_ID ),
                                HIWORD( wParam ) ), (LPARAM)hWnd );
         break;


      case WM_DRAWITEM:                         //  Paint the ownerdraw listbox
         return( DrawIEListbox( hWnd, (LPDRAWITEMSTRUCT)lParam ) );


      case WM_SIZE:              //  If control is resizing, resize listbox too
      {
         RECT Rect;

         if( GetProp( hWnd, "InEdit" ) )           //  If in edit mode, kill it
         {
            DestroyWindow( GetDlgItem( GetDlgItem( hWnd, 100 ), 100 ) );
            DestroyWindow( GetDlgItem( GetDlgItem( hWnd, 100 ), 1 ) );
            DestroyWindow( GetDlgItem( GetDlgItem( hWnd, 100 ), 2 ) );
            SetProp( hWnd, "InEdit", (HANDLE)0 );
         }

         GetClientRect( hWnd, &Rect );     //  Then size listbox to this window
         SetWindowPos( GetDlgItem( hWnd, 100 ), NULL, 0, 0, Rect.right,
                       Rect.bottom, SWP_NOZORDER | SWP_NOMOVE );
         break;
      }


      case WM_SETFOCUS:   //  If this window gets focus, pass it to the listbox
         SetFocus( GetDlgItem( hWnd, 100 ) );       //  since this does nothing
         break;


      default:
         return( DefWindowProc( hWnd, msg, wParam, lParam ) );
   }
   return( FALSE );
}                                                 //  End of IEListboxWndProc()

// ============================================================================
// === Adds items passed in WM_CREATE of control to listbox ===================
// ============================================================================

BOOL AddIEListText( HWND hWnd, HWND hListbox )
{
   HANDLE hMem;
   char *lpStr, *pSep;
   int iItem;

                                            //  Allocate enough buffer for text
   hMem = GlobalAlloc( GHND, GetWindowTextLength( hWnd ) + 8 );
   if( hMem )
   {
      lpStr = GlobalLock( hMem );       //  Get text passed thru CreateWindow()
      GetWindowText( hWnd, lpStr, GetWindowTextLength( hWnd ) );
      pSep = strchr( lpStr, '\n' );             //  CR used for item separation
      while( pSep )                        //  As long as there's a next one...
      {
         *pSep = '\0';                              //  Cut off the current one
                                              //  Add current string to listbox
         iItem = SendMessage( hListbox, LB_ADDSTRING, 0, (LPARAM)lpStr );
         if( iItem != LB_ERR )
            SendMessage( hListbox, LB_SETITEMDATA, iItem,    //  Initialize the
                         GetSysColor( COLOR_WINDOWTEXT ) );    //  item's color

         lpStr = pSep + 1;                //  Set current string to next string
         pSep = strchr( lpStr, '\n' );              //  and search for a new CR
      }
      if( pSep )
         if( lstrlen( pSep ) )
               
      GlobalUnlock( hMem );
      GlobalFree( hMem );
   }
   return( TRUE );
}

// ============================================================================
// === Handles WM_DRAWITEM from Ownerdraw Listbox =============================
// ============================================================================

BOOL DrawIEListbox( HWND hWnd, LPDRAWITEMSTRUCT lpDs )
{
   RECT Rect;
   DWORD dwColor;
   char szText[ 128 ];


   if( lpDs->itemID < 0 )                          //  If nothing to draw, exit
      return( FALSE );

   szText[ 0 ] = '\0';                     //  Initialize string (just in case)

   GetClientRect( lpDs->hwndItem, &Rect );       //  Get size of entire listbox
   Rect.top = lpDs->rcItem.top;                //  Use item Y position and size
   Rect.bottom = lpDs->rcItem.bottom;
         
         //  Fill item RECT with brush returned from CTLCOLOR message to parent
   FillRect( lpDs->hDC, &Rect, (HBRUSH)SendMessage( GetParent( hWnd ),
             WM_CTLCOLORLISTBOX, (WPARAM)lpDs->hDC, (LPARAM)hWnd ) );
                                       //  Get current text color for this item
   dwColor = SendMessage( lpDs->hwndItem, LB_GETITEMDATA,
                          lpDs->itemID, 0 );
   if( SendMessage( lpDs->hwndItem, LB_GETTEXT, lpDs->itemID,
                    (LPARAM)szText ) )
   {
      SetTextColor( lpDs->hDC, dwColor );
      SetBkMode( lpDs->hDC, TRANSPARENT );         //  No text background color
      SelectObject( lpDs->hDC, GetProp( hWnd, "HFONT" ) );

      Rect.left += 2;                     //  Move text left a couple pixels so
      DrawText( lpDs->hDC, szText, lstrlen( szText ), &Rect,     //  it doesn't
                DT_SINGLELINE | DT_LEFT | DT_VCENTER ); // mesh with the border
      Rect.left -= 2;                     //  Move it back after text operation
   }

   if( lpDs->itemState & ODS_SELECTED )    //  Invert item if selected and exit
      InvertRect( lpDs->hDC, &Rect );

   return( TRUE );
}                                                    //  End of DrawIEListbox()

// ============================================================================
// === Listbox Subclass Procedure =============================================
// ============================================================================

LRESULT WINAPI IENewListboxProc( HWND hWnd, UINT msg, WPARAM wParam,
                                 LPARAM lParam )
{
   LRESULT ( WINAPI *lpfnOldProc )( HWND, UINT, WPARAM, LPARAM );

                                           //  ALWAYS need the old proc address
   (DWORD)lpfnOldProc = (DWORD)GetProp( hWnd, "OLDPROC" );

   if( msg == WM_COMMAND )      //  If something is coming from edit or buttons
   {
      if( wParam == IDOK || wParam == IDCANCEL )     //  Them's button messages
      {
         int iItem;
         char szStr[ 128 ];

         if( wParam == IDOK )             //  If OK was pressed (Enter pressed)
         {                                            //  Get the selected item
            iItem = (int)GetProp( GetDlgItem( hWnd, 100 ), "Item" );
            if( iItem != LB_ERR )                      //  If it's a valid item
            {                                              //  Get any text and
               GetWindowText( GetDlgItem( hWnd, 100 ), szStr, 128 ); 
               SendMessage( hWnd, LB_DELETESTRING, iItem, 0 );  //  delete item
               iItem = SendMessage( hWnd, LB_INSERTSTRING, iItem,// insert item
                                    (LPARAM)szStr );        //  to replace item
               if( iItem != LB_ERR )              //  Select item if it went ok
               {
                  SendMessage( hWnd, LB_SETCURSEL, iItem, 0 );
                  SendMessage( hWnd, LB_SETITEMDATA, iItem,  //  Set item color
                               RGB( 128, 0, 0 ) ); // to show it's been changed
               }
            }
         }
                                            //  Kill off all this editting junk
         RemoveProp( GetDlgItem( hWnd, 100 ), "Item" );
         DestroyWindow( GetDlgItem( hWnd, 100 ) );
         DestroyWindow( GetDlgItem( hWnd, 1 ) );
         DestroyWindow( GetDlgItem( hWnd, 2 ) );
         SetProp( GetParent( hWnd ), "InEdit", (HANDLE)0 );    //  Not editting
         return( TRUE );
      }
   }


   if( msg == WM_DESTROY )              //  If this listbox is about to bite it
   {                                        //  Replace orginal proc address to
      SetWindowLong( hWnd, GWL_WNDPROC, (DWORD)lpfnOldProc );
      RemoveProp( hWnd, "OLDPROC" );                   //  kill the PROP safely
   }                                      //  Fall through to normal LB destroy


   if( msg == WM_VSCROLL )                         //  If the scrollbar is used
   {
      if( GetProp( GetParent( hWnd ), "InEdit" ) )    //  and it's in edit mode
      {
         RemoveProp( GetDlgItem( hWnd, 100 ), "Item" );    //  it's not in edit
         DestroyWindow( GetDlgItem( hWnd, 100 ) );             //  mode anymore
         DestroyWindow( GetDlgItem( hWnd, 1 ) );    //  There's no problem with
         DestroyWindow( GetDlgItem( hWnd, 2 ) );       //  scrolling but I felt
         SetProp( GetParent( hWnd ), "InEdit", (HANDLE)0 ); // like adding this
      }
   }


   if( msg == WM_DRAWITEM )                  //  Possible buttons are ownerdraw
      return( IELDrawButton( (LPDRAWITEMSTRUCT)lParam ) );     //  So draw them

                                            //  Let normal listbox stuff happen
   return( CallWindowProc( lpfnOldProc, hWnd, msg, wParam, lParam ) );
}                                                 //  End of IENewListboxProc()

// ============================================================================
// === Start Edit and Buttons as children of Listbox ==========================
// ============================================================================

BOOL StartIEdit( HWND hWnd )
{
   HWND hListbox, hEdit, hOK, hCancel;
   int iItem, iYsize, iYpos;
   char szStr[ 128 ];
   RECT Rect;
   HFONT hFont;
   HINSTANCE hUseInst;


   hListbox = GetDlgItem( hWnd, 100 );                    //  Do this for speed
   hUseInst = (HINSTANCE)GetWindowLong( hWnd, GWL_HINSTANCE );

   iItem = SendMessage( hListbox, LB_GETCURSEL, 0, 0 );       //  Get selection
   if( iItem == LB_ERR )                                       //  from listbox
      return( FALSE );
                                                 //  Get text for selected item
   SendMessage( hListbox, LB_GETTEXT, iItem, (LPARAM)szStr );
                                     //  Get size and position of selected item
   SendMessage( hListbox, LB_GETITEMRECT, iItem, (LPARAM)&Rect );
   iYpos = Rect.top - 1;                 //  Make the edit control a tad larger
   iYsize = Rect.bottom + 1 - iYpos;     //  then a listbox item so it looks ok
   if( iYpos < 0 )
      iYpos = 0;                      //  But if it's topmost, expand down only
               //  Since the above subtracted a negative number, the size is OK
   hEdit = CreateWindow( "edit", szStr, WS_CHILD | WS_VISIBLE | WS_BORDER |
                         ES_AUTOHSCROLL,
                         Rect.left, iYpos,          //  Create the edit control
                         Rect.right - Rect.left - ( iYsize * 2 ), iYsize,
                         hListbox, (HMENU)100,         //  Child of the listbox
                         hUseInst, NULL );
   if( ! hEdit )
      return( FALSE );

   hFont = GetProp( hWnd, "HFONT" );                //  Get the font to be used
   SendMessage( hEdit, WM_SETFONT, (WPARAM)hFont, 1 );    //  Apply to the edit
   SetProp( hEdit, "Item", (HANDLE)iItem );      //  Save the LB item with edit
                                                  //  Subclass the edit control
   SetProp( hEdit, "OLDPROC", (HANDLE)GetWindowLong( hEdit, GWL_WNDPROC ) );
   SetWindowLong( hEdit, GWL_WNDPROC, (DWORD)IENewEditProc );
                                                     //  Create the two buttons
   hOK = CreateWindow( "button", NULL, WS_CHILD | WS_VISIBLE |
                       BS_OWNERDRAW,                   //  Position and size so
                       Rect.right - ( iYsize * 2 ), iYpos,   //  they're square
                       iYsize, iYsize,
                       hListbox, (HMENU)1, hUseInst, NULL );
   if( ! hOK )
   {
      DestroyWindow( hEdit );
      return( FALSE );
   }
                                          //  This here being the Cancel button
   hCancel = CreateWindow( "button", NULL, WS_CHILD | WS_VISIBLE |
                           BS_OWNERDRAW,
                           Rect.right - iYsize, iYpos,
                           iYsize, iYsize,
                           hListbox, (HMENU)2, hUseInst, NULL );
   if( ! hCancel )
   {
      DestroyWindow( hOK );        //  Kill everything, let Windows sort it out
      DestroyWindow( hEdit );
      return( FALSE );
   }

   SetFocus( hEdit );                 //  Give the edit the focus and we're set
   return( TRUE );
}                                                       //  End of StartIEdit()

// ============================================================================
// === Draw the child buttons of the Listbox ==================================
// ============================================================================

BOOL IELDrawButton( LPDRAWITEMSTRUCT lpDs )
{
   HBRUSH hBrush;
   HPEN hPen;
   int iOffset;

                            //  First, fill the background with a buttony color
   hBrush = CreateSolidBrush( GetSysColor( COLOR_BTNFACE ) );
   hBrush = SelectObject( lpDs->hDC, hBrush );
   Rectangle( lpDs->hDC, lpDs->rcItem.left, lpDs->rcItem.top,
              lpDs->rcItem.right - lpDs->rcItem.left,
              lpDs->rcItem.bottom - lpDs->rcItem.top );
   DeleteObject( SelectObject( lpDs->hDC, hBrush ) );
                                            //  Now draw the left and top sides
   if( lpDs->itemState & ODS_SELECTED )        //  The color is based on if the
      hPen = SelectObject( lpDs->hDC, CreatePen( PS_SOLID, 1,     //  button is
                           GetSysColor( COLOR_BTNSHADOW ) ) );   //  pressed or
   else                                                         //  not pressed
      hPen = SelectObject( lpDs->hDC, CreatePen( PS_SOLID, 1,
                           GetSysColor( COLOR_BTNHIGHLIGHT ) ) );
                                                            //  Draw some stuff
   MoveToEx( lpDs->hDC, lpDs->rcItem.left, lpDs->rcItem.bottom - 2, NULL );
   LineTo( lpDs->hDC, lpDs->rcItem.left, lpDs->rcItem.top + 1 );
   LineTo( lpDs->hDC, lpDs->rcItem.right - 1, lpDs->rcItem.top + 1 );

   if( ! ( lpDs->itemState & ODS_SELECTED ) )       //  Time to draw the bottom
      hPen = SelectObject( lpDs->hDC, CreatePen( PS_SOLID, 1,     //  and right
                           GetSysColor( COLOR_BTNSHADOW ) ) );   //  sides. The
   else                                                //  color is based on if
      hPen = SelectObject( lpDs->hDC, CreatePen( PS_SOLID, 1,    //  the button
                           GetSysColor( COLOR_BTNFACE ) ) );     //  is pressed
                                                                   //  Draw em!
   MoveToEx( lpDs->hDC, lpDs->rcItem.left + 1, lpDs->rcItem.bottom - 2, NULL );
   LineTo( lpDs->hDC, lpDs->rcItem.right - 2, lpDs->rcItem.bottom - 2 );
   LineTo( lpDs->hDC, lpDs->rcItem.right - 2, lpDs->rcItem.top + 1 );
                                      //  Create a double-wide pen for interior
                                                //  Offset one pixel if pressed
   iOffset = ( lpDs->itemState & ODS_SELECTED ) != 0;
   if( lpDs->CtlID == IDCANCEL )
   {                                                 //  If Cancel, make an "X"
      DeleteObject( SelectObject( lpDs->hDC, 
                    CreatePen( PS_SOLID, 2, RGB( 128, 0, 0 ) ) ) );

      MoveToEx( lpDs->hDC, lpDs->rcItem.left + 2 + iOffset, 
                lpDs->rcItem.bottom - 4 + iOffset, NULL );
      LineTo( lpDs->hDC, lpDs->rcItem.right - 5 + iOffset,
              lpDs->rcItem.top + 3 + iOffset );
      MoveToEx( lpDs->hDC, lpDs->rcItem.left + 2 + iOffset,
                lpDs->rcItem.top + 3 + iOffset, NULL );
      LineTo( lpDs->hDC, lpDs->rcItem.right - 5 + iOffset,
              lpDs->rcItem.bottom - 4 + iOffset );
   }
   else
   {                                                    //  OK gets a checkmark
      DeleteObject( SelectObject( lpDs->hDC, 
                    CreatePen( PS_SOLID, 2, RGB( 0, 128, 128 ) ) ) );

      MoveToEx( lpDs->hDC, lpDs->rcItem.left + 3 + iOffset,
                ( ( lpDs->rcItem.bottom - 4 ) / 2 ) + iOffset, NULL );
      LineTo( lpDs->hDC, ( ( lpDs->rcItem.right - 5 ) / 2 ) + iOffset,
              lpDs->rcItem.bottom - 4 + iOffset );
      LineTo( lpDs->hDC, lpDs->rcItem.right - 5 + iOffset,
              lpDs->rcItem.top + 3 + iOffset );
   }

   DeleteObject( SelectObject( lpDs->hDC, hPen ) );   //  All done with buttons
   return( TRUE );
}                                                    //  End of IELDrawButton()

// ============================================================================
// === Edit subclass procedure ================================================
// ============================================================================

LRESULT WINAPI IENewEditProc( HWND hWnd, UINT msg, WPARAM wParam,
                              LPARAM lParam )
{
   LRESULT ( WINAPI *lpfnOldProc )( HWND, UINT, WPARAM, LPARAM );

                                          //  ALWAYS need the old proc address!
   (DWORD)lpfnOldProc = (DWORD)GetProp( hWnd, "OLDPROC" );
   if( msg == WM_DESTROY )                 //  If this window is getting killed
   {                                      //  Reset it's old proc address so...
      SetWindowLong( hWnd, GWL_WNDPROC, (DWORD)lpfnOldProc );
      RemoveProp( hWnd, "OLDPROC" );      //  ...the PROP can be removed safely
   }                                  //  Let it fall through to normal destroy


   if( msg == WM_CHAR )                       //  Looking for Return and Escape
   {
      if( LOWORD( wParam ) == VK_ESCAPE ||                //  and if one or the
          LOWORD( wParam ) == VK_RETURN )                  //  other is pressed
      {
         if( LOWORD( wParam ) == VK_RETURN )       //  Return a message to it's
            PostMessage( GetParent( hWnd ), WM_COMMAND, IDOK, 0 );   //  parent
         else                                         //  based on the keypress
            PostMessage( GetParent( hWnd ), WM_COMMAND, IDCANCEL, 0 );
      }
   }
                                          //  Let normal edit stuff happen, too
   return( CallWindowProc( lpfnOldProc, hWnd, msg, wParam, lParam ) );
}                                                    //  End of IENewEditProc()

// ============================================================================
// ============================================================================

