/******************************************************************************
*                                                                             *
*              Tricky Edit Controls in a Listbox by Mark Gamber               *
*                                                                             *
*  This program is public domain. In using this program or any part thereof,  *
*     the user agrees to indemnify and hold harmless the author and software  *
*     from any damages that may occur. If unable to comply, the software must *
*     be immediately destroyed.                                               *
*                                                                             *
*******************************************************************************
*                                                                             *
*  This is an example of using an ownerdraw listbox to hold an array of other *
*  controls, in this case, edit controls.                                     *
*                                                                             *
******************************************************************************/

#include "windows.h"

// === Function Prototypes ====================================================

BOOL WINAPI MainDlgProc( HWND, UINT, WPARAM, LPARAM );
LRESULT WINAPI LbSubclassProc( HWND, UINT, WPARAM, LPARAM );

// === Various Global Variables ===============================================

HINSTANCE hInst;
FARPROC lpfnOldLbProc;
HBRUSH hYellowBrush;
HBRUSH hBlueBrush;

// ==== Program Entry Point ===================================================

int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd, int nShow )
{
   hInst = hInstance;
                                            //  Create brushes we'll need later
   hYellowBrush = CreateSolidBrush( RGB( 255, 255, 0 ) );
   hBlueBrush = CreateSolidBrush( RGB( 0, 255, 255 ) );

   DialogBox( hInstance, MAKEINTRESOURCE( 10000 ), NULL, MainDlgProc );

   DeleteObject( hYellowBrush );              //  Delete brushes on our way out
   DeleteObject( hBlueBrush );

   return( FALSE );
}

// === Main Dialog box Proc ===================================================

BOOL WINAPI MainDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:                          //  The dialog box is opened
      {
         int i, index;
         char str[ 20 ];
         RECT Rect;
         HWND hWnd;
                                                           //  Subclass listbox
         lpfnOldLbProc = (FARPROC)SetWindowLong( GetDlgItem( hDlg, 100 ),
                                    GWL_WNDPROC, (DWORD)LbSubclassProc );

                              //  Get listbox width and subtract the amount for
         GetClientRect( GetDlgItem( hDlg, 100 ), &Rect );     //  the scrollbar
         Rect.right = Rect.right - GetSystemMetrics( SM_CXVSCROLL );

         for( i = 0; i < 20; i++ )
         {
            wsprintf( str, "Edit Control #%d", i + 1 );

            /******************************************************************
            *  Here we add strings to the listbox to force scrolling. If the  *
            *  string takes, and it should, we then create an edit control at *
            *  a position relative to the listbox given the item size (24)    *
            *  current edit control number (0-19). The edit is a child of the *
            *  listbox and the IDs start at 1000. If the edit is created the  *
            *  string added to the listbox is also added to the edit control. *
            ******************************************************************/

            if( SendMessage( GetDlgItem( hDlg, 100 ), LB_ADDSTRING, 0,
                             (LPARAM)(LPSTR)str ) != LB_ERR )
            {
               hWnd = CreateWindow( "edit", str,
                                    WS_CHILD | WS_VISIBLE | ES_AUTOHSCROLL |
                                    WS_BORDER | ES_WANTRETURN,
                                    0, i * 24, Rect.right, 24,
                                    GetDlgItem( hDlg, 100 ),
                                    (HMENU)( i + 1000 ), hInst, NULL );
               if( hWnd )
                  SetWindowText( hWnd, str );
            }
         }
         return( TRUE );
      }

      case WM_MEASUREITEM:          //  Windows needs to know listbox item size
      {
         LPMEASUREITEMSTRUCT lpMs = (LPMEASUREITEMSTRUCT)lParam;
         RECT Rect;

         GetClientRect( hDlg, &Rect );

         lpMs->itemWidth = Rect.right - 4;        //  Give it any old junk here
         lpMs->itemHeight = 24;              //  But we insist on 24 units high
         return( TRUE );
      }

      case WM_DRAWITEM:
         return( TRUE );                 //  Don't draw anything in the listbox

      case WM_COMMAND:
      {
         if( wParam == IDCANCEL )
         {
            EndDialog( hDlg, TRUE );
            return( TRUE );
         }
         break;
      }
   }
   return( FALSE );
}

// === Listbox subclass proc ==================================================


LRESULT WINAPI LbSubclassProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
   /***************************************************************************
   *  This little segment reposts any commands from the edits back to the     *
   *  listbox's parent, in this case the dialog box.                          *
   ***************************************************************************/

   if( msg == WM_COMMAND && wParam > 999 )
      PostMessage( GetParent( hWnd ), WM_COMMAND, wParam, lParam );

   /***************************************************************************
   *  Here, an edit control needs it's colors. Set the edit DC to yellow or   *
   *  blue background based on it's ID. Set the text to black.                *
   ***************************************************************************/

   if( msg == WM_CTLCOLOR &&
       ( HIWORD( lParam ) == CTLCOLOR_EDIT ||
         HIWORD( lParam ) == CTLCOLOR_MSGBOX ) )
   {
      int id;

      id = GetDlgCtrlID( (HWND)LOWORD( lParam ) );
      SetTextColor( (HDC)wParam, RGB( 0, 0, 0 ) );

      if( ! ( id % 2 ) )                     //  If even number edit control ID
      {                                          //  Set to a yellow background
         SetBkColor( (HDC)wParam, RGB( 255, 255, 0 ) );
         UnrealizeObject( hYellowBrush );
         return( (LRESULT)hYellowBrush );
      }
      else
      {                                           //  Otherwise, set it to blue
         SetBkColor( (HDC)wParam, RGB( 0, 255, 255 ) );
         UnrealizeObject( hBlueBrush );
         return( (LRESULT)hBlueBrush );
      }
   }

                                    //  Finish with a call to the original proc
   return( CallWindowProc( lpfnOldLbProc, hWnd, msg, wParam, lParam ) );
}



