//======================================================================
// 
// Filemon.c
//
// Copyright (C) 1996 Mark Russinovich and Bryce Cogswell
//
// Implements the equivalent of Windows 95 IFSMgr_FileSystemApiHook.
//
//======================================================================
#include "ntddk.h"
#include "stdarg.h"
#include "stdio.h"
#include "ioctlcmd.h"
#include "filemon.h"


//----------------------------------------------------------------------
//                           DEFINES
//----------------------------------------------------------------------

// Set this to 0 to make a processor-independent version that does
// not print out the process name
#define GETPROCESS 1

// print macro that only turns on when debugging is on
#if DBG
#define DbgPrint(arg) DbgPrint arg
#else
#define DbgPrint(arg) 
#endif

// convenient mutex macros
# define MUTEX_TYPE	KSPIN_LOCK
# define MUTEX_INIT(v)	KeInitializeSpinLock( &v )
# define MUTEX_P(v, o)	KeAcquireSpinLock( &v, o );
# define MUTEX_V(v, n)	KeReleaseSpinLock( &v, n )


//----------------------------------------------------------------------
//                         FORWARD DEFINES
//---------------------------------------------------------------------- 
NTSTATUS FilemonDispatch( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp );
VOID 	 FilemonUnload( IN PDRIVER_OBJECT DriverObject );
BOOLEAN	 FilemonFastIoCheckifPossible( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
		IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN BOOLEAN CheckForReadOperation,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoRead( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
		IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, OUT PVOID Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoWrite( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
		IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN PVOID Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoQueryBasicInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
		OUT PFILE_BASIC_INFORMATION Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoQueryStandardInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
		OUT PFILE_STANDARD_INFORMATION Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoLock( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
		IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
		BOOLEAN FailImmediately, BOOLEAN ExclusiveLock,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoUnlockSingle( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
		IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoUnlockAll( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN	 FilemonFastIoUnlockAllByKey( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId, ULONG Key,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoDeviceControl( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait,
		IN PVOID InputBuffer, IN ULONG InputBufferLength, 
		OUT PVOID OutbufBuffer, IN ULONG OutputBufferLength, IN ULONG IoControlCode,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject );
VOID	 FilemonFastIoAcquireFile( PFILE_OBJECT FileObject );
VOID	 FilemonFastIoReleaseFile( PFILE_OBJECT FileObject );
VOID	 FilemonFastIoDetachDevice( PDEVICE_OBJECT SourceDevice, PDEVICE_OBJECT TargetDevic );
// new NT 4.0 defined Fast I/O entry calls
BOOLEAN  FilemonFastIoQueryNetworkOpenInfo(IN PFILE_OBJECT FileObject,
    		IN BOOLEAN Wait, OUT struct _FILE_NETWORK_OPEN_INFORMATION *Buffer,
    		OUT struct _IO_STATUS_BLOCK *IoStatus, IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoAcquireForModWrite( IN PFILE_OBJECT FileObject,
    		IN PLARGE_INTEGER EndingOffset, OUT struct _ERESOURCE **ResourceToRelease,
    		IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlRead( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
		IN ULONG LockKey, OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlReadComplete( IN PFILE_OBJECT FileObject,
		IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoPrepareMdlWrite( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length, IN ULONG LockKey,
		OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlWriteComplete( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain,
    		IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoReadCompressed( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
    		IN ULONG LockKey, OUT PVOID Buffer, OUT PMDL *MdlChain,
    		OUT PIO_STATUS_BLOCK IoStatus,
		OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
    		IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoWriteCompressed( IN PFILE_OBJECT FileObject,
    		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
    		IN ULONG LockKey, IN PVOID Buffer, OUT PMDL *MdlChain,
    		OUT PIO_STATUS_BLOCK IoStatus,
    		IN struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
    		IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlReadCompleteCompressed( IN PFILE_OBJECT FileObject,
    		IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoMdlWriteCompleteCompressed( IN PFILE_OBJECT FileObject,
    		IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain,
    		IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  FilemonFastIoQueryOpen( IN struct _IRP *Irp,
    		OUT PFILE_NETWORK_OPEN_INFORMATION NetworkInformation,
    		IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoReleaseForModWrite( IN PFILE_OBJECT FileObject,
		IN struct _ERESOURCE *ResourceToRelease, IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoAcquireForCcFlush( IN PFILE_OBJECT FileObject,
    		IN PDEVICE_OBJECT DeviceObject );
NTSTATUS FilemonFastIoReleaseForCcFlush( IN PFILE_OBJECT FileObject,
    		IN PDEVICE_OBJECT DeviceObject );

//----------------------------------------------------------------------
//                         GLOBALS
//---------------------------------------------------------------------- 
// our user-inteface device object
PDEVICE_OBJECT 		GUIDevice;

// NT version number 
ULONG			NTBuild;

// is a GUI talking to us?
BOOLEAN			GUIActive = FALSE;

// are we unloading?
BOOLEAN			UnloadInProgress = FALSE;

// keep track of outstanding IRPs
ULONG			OutstandingIRPCount = 0;
MUTEX_TYPE		CountMutex;

// table of our hook devices for each drive letter
PDEVICE_OBJECT		LDriveDevices[26];

// this table keeps track of which drives are controlled by common
// network redirectors (unhooking one results in all of the group
// being unhooked)
int			LDriveMap[26];
int			LDriveGroup = 0;

// hash table for keeping names around. This is necessary because 
// at any time the name information in the fileobjects that we
// see can be deallocated and reused. If we want to print accurate
// names, we need to keep them around ourselves.
PHASH_ENTRY		HashTable[NUMHASH];

// list of freed entries (to save some allocation calls)
PHASH_ENTRY		FreeEntries = NULL;

// mutex for hash table accesses
MUTEX_TYPE		HashMutex;

// ANSI string direct file name
ANSI_STRING		DirectFileName;

// Data structure for storing messages we generate
PSTORE_BUF		Store 		= NULL;
ULONG			Sequence 	= 0;
MUTEX_TYPE		StoreMutex;

// maximum amount of data we will grab for buffered unread data
ULONG			NumStore 	= 0;
ULONG			MaxStore 	= 1000000/MAX_STORE;

// names
CHAR	*FileInformation[] = {
    "",
    "FileDirectoryInformation",
    "FileFullDirectoryInformation",
    "FileBothDirectoryInformation",
    "FileBasicInformation",
    "FileStandardInformation",
    "FileInternalInformation",
    "FileEaInformation",
    "FileAccessInformation",
    "FileNameInformation",
    "FileRenameInformation",
    "FileLinkInformation",
    "FileNamesInformation",
    "FileDispositionInformation",
    "FilePositionInformation",
    "FileFullEaInformation",
    "FileModeInformation",
    "FileAlignmentInformation",
    "FileAllInformation",
    "FileAllocationInformation",
    "FileEndOfFileInformation",
    "FileAlternateNameInformation",
    "FileStreamInformation",
    "FilePipeInformation",
    "FilePipeLocalInformation",
    "FilePipeRemoteInformation",
    "FileMailslotQueryInformation",
    "FileMailslotSetInformation",
    "FileCompressionInformation",
    "FileCopyOnWriteInformation",
    "FileCompletionInformation",
    "FileMoveClusterInformation",
    "FileOleClassIdInformation",
    "FileOleStateBitsInformation",
    "FileApplicationExplorableInformation",
    "FileApplicationExplorableChildrenInformation",
    "FileObjectIdInformation",
    "FileOleAllInformation",
    "FileOleDirectoryInformation",
    "FileTransactionCommitInformation",
    "FileContentIndexInformation",
    "FileInheritContentIndexInformation",
    "FileOleInformation",
    "FileMaximumInformation",
};

CHAR *VolumeInformation[] = {
    "",
    "FileFsVolumeInformation",
    "FileFsLabelInformation",
    "FileFsSizeInformation",
    "FileFsDeviceInformation",
    "FileFsAttributeInformation",
    "FileFsQuotaQueryInformation",
    "FileFsQuotaSetInformation",
    "FileFsControlQueryInformation",
    "FileFsControlSetInformation",
    "FileFsMaximumInformation",
};
	

// our own fastio structure for hooking fast I/O calls
FAST_IO_DISPATCH	FastIOHook = {
	sizeof(FAST_IO_DISPATCH), 
	FilemonFastIoCheckifPossible,
	FilemonFastIoRead,
	FilemonFastIoWrite,
	FilemonFastIoQueryBasicInfo,
	FilemonFastIoQueryStandardInfo,
	FilemonFastIoLock,
	FilemonFastIoUnlockSingle,
	FilemonFastIoUnlockAll,
	FilemonFastIoUnlockAllByKey,
	FilemonFastIoDeviceControl,
	FilemonFastIoAcquireFile,
	FilemonFastIoReleaseFile,
	FilemonFastIoDetachDevice,

	// new for NT 4.0
	FilemonFastIoQueryNetworkOpenInfo,
	FilemonFastIoAcquireForModWrite,
	FilemonFastIoMdlRead,
	FilemonFastIoMdlReadComplete,
	FilemonFastIoPrepareMdlWrite,
	FilemonFastIoMdlWriteComplete,
	FilemonFastIoReadCompressed,
	FilemonFastIoWriteCompressed,
	FilemonFastIoMdlReadCompleteCompressed,
	FilemonFastIoMdlWriteCompleteCompressed,
	FilemonFastIoQueryOpen,
	FilemonFastIoReleaseForModWrite,
	FilemonFastIoAcquireForCcFlush,
	FilemonFastIoReleaseForCcFlush
};


//----------------------------------------------------------------------
//
// FilemonFreeStore
//
// Frees all the data output buffers that we have currently allocated.
//
//----------------------------------------------------------------------
VOID FilemonFreeStore()
{
    PSTORE_BUF 	next;
    
    while( Store ) {
	next = Store->Next;
	ExFreePool( Store );
	Store = next;
    }
}	


//----------------------------------------------------------------------
//
// FilemonNewStore
//
// Called when the current buffer has filled up. This moves us to the
// pre-allocated buffer and then allocates another buffer.
//
//----------------------------------------------------------------------
void FilemonNewStore( void )
{
    PSTORE_BUF prev = Store, newstore;

    // if we have maxed out or haven't accessed the current store
    // just return
    if( MaxStore == NumStore ) {
	Store->Len = 0;
	return;	
    }

    // see if we can re-use a store
    if( !Store->Len ) 
	return;

    // move to the next buffer and allocate another one
    newstore = ExAllocatePool( NonPagedPool, sizeof(*Store) );
    if( newstore ) { 
    	Store 	= newstore;
    	Store->Len  = 0;
    	Store->Next = prev;
	NumStore++;
    } else 
	Store->Len = 0;
}


//----------------------------------------------------------------------
//
// FilemonOldestStore
//
// Goes through the linked list of storage buffers and returns the 
// oldest one.
//
//----------------------------------------------------------------------
PSTORE_BUF FilemonOldestStore( void )
{
    PSTORE_BUF  ptr = Store, prev = NULL;

    while ( ptr->Next )
	ptr = (prev = ptr)->Next;
    if ( prev )
	prev->Next = NULL;    
    return ptr;
}


//----------------------------------------------------------------------
//
// FilemonResetStore
//
// When a GUI is no longer communicating with us, but we can't unload,
// we reset the storage buffers, but keep one store around.
//
//----------------------------------------------------------------------
VOID FilemonResetStore()
{
   PSTORE_BUF  nextstore, curstore;

   curstore = Store->Next;
   while( curstore ) {
        nextstore = curstore->Next;
	ExFreePool( curstore );
	curstore = nextstore;
   }
   Store->Len = 0;
   Store->Next = NULL;
}


//----------------------------------------------------------------------
//
// UpdateStore
//
// Add a new string to Store, if it fits.
//
//----------------------------------------------------------------------
void UpdateStore( ULONG seq, const CHAR * format, ... )
{	
    PENTRY		Entry;
    int			len;
    va_list		arg_ptr;
    KIRQL               oldirql;

    // only do this if a GUI is active
    if( !GUIActive ) return;

    // Send text out as debug output  This is x86 specific.
#define A (&format)
    DbgPrint(( (char *)format, A[1], A[2], A[3], A[4], A[5], A[6] ));
    DbgPrint(( "\n" ));
#undef A

    MUTEX_P( StoreMutex, &oldirql );
    
    // See if its time to switch to new buffer
    if ( Store->Len > MAX_STORE-350 )  
	FilemonNewStore();

    if ( Store->Len <= MAX_STORE-350 )  { 

	/* Set our sequence number */
	Entry = (void *)(Store->Data+Store->Len);
	Entry->seq = seq;

	va_start( arg_ptr, format );
	len = vsprintf( Entry->text, format, arg_ptr );
	va_end( arg_ptr );

	Store->Len += sizeof(Entry->seq)+len+1;		// +1 for null char
    }
    MUTEX_V( StoreMutex, oldirql );
}


//----------------------------------------------------------------------
// 
// strncatZ
//
// Appends a string to another and attaches a null.
//
//----------------------------------------------------------------------
PCHAR strncatZ( PCHAR dest, PCHAR source, int length )
{	
   int 	origlen = strlen(dest);

   strncpy( dest+origlen, source, length );
   dest[ origlen+length ] = 0;
   return(dest);
}


//----------------------------------------------------------------------
//
// FilemonHashCleanup
//
// Called when we are unloading to free any memory that we have 
// in our possession.
//
//----------------------------------------------------------------------
VOID FilemonHashCleanup()
{
   PHASH_ENTRY		hashEntry, nextEntry;
   ULONG		i;
  
   // first free the hash table entries
   for( i = 0; i < NUMHASH; i++ ) {
	hashEntry = HashTable[i];
	while( hashEntry ) {
		nextEntry = hashEntry->Next;
		ExFreePool( hashEntry->FullPathName );
		ExFreePool( hashEntry );
		hashEntry = nextEntry;
	}
   }

   // now, free structures on our free list
   while( FreeEntries ) {
	nextEntry = FreeEntries->Next;
	ExFreePool( FreeEntries );
	FreeEntries = nextEntry;
   }
}


//----------------------------------------------------------------------
//
// FilemonFreeHashEntry
//
// When we see a file close, we can free the string we had associated
// with the fileobject being closed since we know it won't be used
// again.
//
//----------------------------------------------------------------------
VOID FilemonFreeHashEntry( PFILE_OBJECT fileObject )
{
   PHASH_ENTRY		hashEntry, prevEntry;
   KIRQL                oldirql;

   MUTEX_P( HashMutex, &oldirql );

   // look-up the entry
   hashEntry = HashTable[ HASHOBJECT( fileObject ) ];
   prevEntry = NULL;
   while( hashEntry && hashEntry->FileObject != fileObject ) {
	prevEntry = hashEntry;
	hashEntry = hashEntry->Next;
   }
  
   // if we fall off, just return
   if( !hashEntry ) {
	MUTEX_V( HashMutex, oldirql );
	return;
   }

   // remove it from the hash list 
   if( prevEntry ) 
	prevEntry->Next = hashEntry->Next;
   else 
	HashTable[ HASHOBJECT( fileObject )] = hashEntry->Next;

   // free the memory associated with it
   ExFreePool( hashEntry->FullPathName );

   // put it on our free list
   hashEntry->Next 	= FreeEntries;
   FreeEntries 		= hashEntry;

   MUTEX_V( HashMutex, oldirql );
}


//----------------------------------------------------------------------
//
// FilemonGetFullPath
//
// Takes a fileobject and filename and returns a canonical path,
// nicely formatted, in fullpathname.
//
//----------------------------------------------------------------------
VOID FilemonGetFullPath( PFILE_OBJECT fileObject, PHOOK_EXTENSION hookExt, 
			PCHAR fullPathName )
{
    PFILE_OBJECT	relFileObject;
    ANSI_STRING		pathName;
    PHASH_ENTRY		hashEntry, newEntry;
    ANSI_STRING         fileName;
    BOOLEAN		directio = FALSE;
    KIRQL               oldirql;

    // only do this if a GUI is active
    if( !GUIActive ) return;

    // first, do a lookup in the hash table to see if we already have it
    MUTEX_P( HashMutex, &oldirql );
    hashEntry = HashTable[ HASHOBJECT( fileObject ) ];
    while( hashEntry && hashEntry->FileObject != fileObject ) 
	hashEntry = hashEntry->Next;

    // did we find it?
    if( hashEntry ) {
        strcpy( fullPathName, hashEntry->FullPathName );
	MUTEX_V( HashMutex, oldirql );
	return;
    }
    MUTEX_V( HashMutex, oldirql );

    // get the relative name (if there is one)
    if( fileObject->FileName.Buffer && !(fileObject->Flags & FO_DIRECT_DEVICE_OPEN) ) 
   	RtlUnicodeStringToAnsiString( &fileName, &fileObject->FileName, TRUE );
    else {
        // no filename - direct access
	directio = TRUE;
	fileName = DirectFileName;
    }

    // nope, so first create the name 
    // see if its a relative path name and if so, create the full path   
    sprintf( fullPathName, "%c:", hookExt->LogicalDrive );
    if( fileName.Buffer[0] != '\\' ) {
	relFileObject = fileObject->RelatedFileObject;	
	// make sure we're dealing with a real file object
	if( relFileObject && relFileObject->Type == IO_TYPE_FILE && 
		relFileObject->FileName.Buffer ) {
		RtlUnicodeStringToAnsiString( &pathName, 
			&relFileObject->FileName, TRUE );
		// if its a network path, chop the primary drive letter
		if( pathName.Buffer[2] == L':' ) 
			strncatZ( fullPathName, &pathName.Buffer[3], 
					pathName.Length - 3);
		else 
			strncatZ( fullPathName, pathName.Buffer, pathName.Length );
		if( pathName.Length != 1 )
			strncatZ( fullPathName, "\\", 1);
		strncatZ( fullPathName, fileName.Buffer, fileName.Length );
		RtlFreeAnsiString( &pathName );
	} else 
		// no full path - oh, well, do the best we can
		strncatZ( fullPathName, fileName.Buffer, fileName.Length );
    } else {
  	// if its a network pathname then chop the primary drive letter
	if( fileName.Buffer[2] == ':' ) {
		fullPathName[0] = 0;
		strncatZ( fullPathName, &fileName.Buffer[1], fileName.Length-1 );
	} else 
		strncatZ( fullPathName, fileName.Buffer, fileName.Length );
    }

    // free name
    if( !directio )
	RtlFreeAnsiString( &fileName );

    // got a name, now create a copy and put it in the hash table
    MUTEX_P( HashMutex, &oldirql );
    if( FreeEntries ) {
	newEntry    = FreeEntries;
	FreeEntries = FreeEntries->Next;
    } else 
	newEntry = ExAllocatePool( NonPagedPool, sizeof(HASH_ENTRY) );

    newEntry->FileObject 	= fileObject;
    newEntry->FullPathName	= ExAllocatePool( NonPagedPool, strlen(fullPathName)+1);
    newEntry->Next		= HashTable[ HASHOBJECT(fileObject) ];
    HashTable[ HASHOBJECT(fileObject) ] = newEntry;	
    MUTEX_V( HashMutex, oldirql );
    strcpy( newEntry->FullPathName, fullPathName );
}


//----------------------------------------------------------------------
//
// FilemonGetProcess
//
// Uses undocumented data structure offsets to obtain the name of the
// currently executing process.
//
//----------------------------------------------------------------------
PCHAR FilemonGetProcess( PCHAR Name )
{
#if GETPROCESS
    PEPROCESS	  	curproc;
    char                *nameptr;

    curproc = PsGetCurrentProcess();
    nameptr   = (char *) ((ULONG)curproc + NAMEOFFSET);
    strncpy( Name, nameptr, 16 );
    return Name;
#else
    strcpy( Name, "??" );
    return Name;
#endif
}


//----------------------------------------------------------------------
//
// DriverEntry
//
// Installable driver initialization. Here we just set ourselves up.
//
//----------------------------------------------------------------------
NTSTATUS DriverEntry(IN PDRIVER_OBJECT DriverObject, IN PUNICODE_STRING RegistryPath )
{
    UNICODE_STRING	   directFileUnicodeString;
    WCHAR		   directFileString[] = L" DIRECT I/O";
    NTSTATUS               ntStatus;
    WCHAR                  deviceNameBuffer[]  = L"\\Device\\Filemon";
    UNICODE_STRING         deviceNameUnicodeString;
    WCHAR                  deviceLinkBuffer[]  = L"\\DosDevices\\Filemon";
    UNICODE_STRING         deviceLinkUnicodeString;
    ULONG		   i;

    DbgPrint (("Filemon.SYS: entering DriverEntry\n"));

    // if not NT 4.0 Final Release, fix FastIO table so that we work under NT 4.0 RCs and Betas
    if( *NtBuildNumber < NT4FINAL ) 
	FastIOHook.SizeOfFastIoDispatch = (ULONG) &FastIOHook.FastIoQueryNetworkOpenInfo - 
			(ULONG) &FastIOHook;

    // setup our name
    RtlInitUnicodeString (&deviceNameUnicodeString,
                          deviceNameBuffer );

    // set up the device used for GUI communications
    ntStatus = IoCreateDevice ( DriverObject,
                               sizeof (DEVICE_EXTENSION),
                               &deviceNameUnicodeString,
                               FILE_DEVICE_FILEMON,
                               0,
                               TRUE,
                               &GUIDevice );
    if (NT_SUCCESS(ntStatus)) {
        // Create a symbolic link that the GUI can specify to gain access
        // to this driver/device
        RtlInitUnicodeString (&deviceLinkUnicodeString,
                              deviceLinkBuffer );
        ntStatus = IoCreateSymbolicLink (&deviceLinkUnicodeString,
                                         &deviceNameUnicodeString );
        if (!NT_SUCCESS(ntStatus))
            DbgPrint (("Filemon.SYS: IoCreateSymbolicLink failed\n"));

        // Create dispatch points for all routines that must be handled
	DriverObject->MajorFunction[IRP_MJ_READ]	   	= 
	DriverObject->MajorFunction[IRP_MJ_CLOSE]	   	=
	DriverObject->MajorFunction[IRP_MJ_WRITE]	   	=
	DriverObject->MajorFunction[IRP_MJ_CREATE]	   	=
	DriverObject->MajorFunction[IRP_MJ_FLUSH_BUFFERS]	=
	DriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION]	=
	DriverObject->MajorFunction[IRP_MJ_SET_INFORMATION]	=
	DriverObject->MajorFunction[IRP_MJ_QUERY_EA]		=
	DriverObject->MajorFunction[IRP_MJ_SET_EA]		=
	DriverObject->MajorFunction[IRP_MJ_QUERY_VOLUME_INFORMATION] = 
	DriverObject->MajorFunction[IRP_MJ_SET_VOLUME_INFORMATION] =
	DriverObject->MajorFunction[IRP_MJ_DIRECTORY_CONTROL] 	=
	DriverObject->MajorFunction[IRP_MJ_FILE_SYSTEM_CONTROL]	=
	DriverObject->MajorFunction[IRP_MJ_SHUTDOWN]	   	=
	DriverObject->MajorFunction[IRP_MJ_QUERY_SECURITY]	=
	DriverObject->MajorFunction[IRP_MJ_SET_SECURITY]        =
	DriverObject->MajorFunction[IRP_MJ_LOCK_CONTROL]   	=
	DriverObject->MajorFunction[IRP_MJ_CLEANUP]        	=
        DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] 	= FilemonDispatch;
        DriverObject->DriverUnload                         	= FilemonUnload;

	// for fast I/O hooking
	DriverObject->FastIoDispatch = &FastIOHook;
    }
    if (!NT_SUCCESS(ntStatus)) {
	DbgPrint(("Filemon: Failed to create our device!\n"));
        // Something went wrong, so clean up (free resources etc)
        if( GUIDevice )
            IoDeleteDevice( GUIDevice );
	return ntStatus;
    }

    // zero structures
    for(i = 0; i < 26; i++ ) 		LDriveMap[i] = 0;
    for(i = 0; i < NUMHASH; i++ ) 	HashTable[i] = NULL;

    // initialize our semaphores
    MUTEX_INIT( StoreMutex );
    MUTEX_INIT( HashMutex );
    MUTEX_INIT( CountMutex );

    // initialize the direct I/O string name
    RtlInitUnicodeString( &directFileUnicodeString, directFileString );
    RtlUnicodeStringToAnsiString( &DirectFileName, &directFileUnicodeString, TRUE );

    // allocate our trace storage buffer
    Store	= ExAllocatePool( NonPagedPool, sizeof(*Store) );
    if ( !Store ) return STATUS_INSUFFICIENT_RESOURCES;
    Store->Len  = 0;
    Store->Next = NULL;
    NumStore = 1;

    return ntStatus;
}


//----------------------------------------------------------------------
//
// HookDrive
//
// Hook the Fastfat device by determining which device we must layer
// on top of the same way NT determines which device to send filesytem
// requests to.
//
//----------------------------------------------------------------------
BOOLEAN HookDrive( IN char Drive, IN PDRIVER_OBJECT DriverObject )
{
   IO_STATUS_BLOCK	ioStatus;
   HANDLE		ntFileHandle;   
   OBJECT_ATTRIBUTES	objectAttributes;
   PDEVICE_OBJECT	fileSysDevice;
   PDEVICE_OBJECT	hookDevice;
   UNICODE_STRING	fileNameUnicodeString;
   WCHAR 		filename[] = L"\\DosDevices\\A:\\";
   NTSTATUS             ntStatus;
   ULONG		i;
   PFILE_OBJECT		fileObject;
   PHOOK_EXTENSION 	hookExtension;

    if ( Drive >= 'a' && Drive <= 'z' )	Drive -= 'a';
    else				Drive -= 'A';
    if ( (unsigned char)Drive >= 26 )	return FALSE;

    // have we already hooked this drive?
    if ( LDriveDevices[Drive] == NULL )  {

	// point to the current logical drive
	filename[12] = 'A'+Drive;

	// have to figure out what device to hook - first open the root directory
   	RtlInitUnicodeString( &fileNameUnicodeString, filename );
   	InitializeObjectAttributes( &objectAttributes, &fileNameUnicodeString, 
			OBJ_CASE_INSENSITIVE, NULL, NULL );
   	ntStatus = ZwCreateFile( &ntFileHandle, SYNCHRONIZE|FILE_READ_ACCESS, 
			&objectAttributes, &ioStatus, NULL, 0, FILE_SHARE_READ|FILE_SHARE_WRITE, 
			FILE_OPEN, 
			FILE_SYNCHRONOUS_IO_NONALERT|
			FILE_OPEN_FOR_BACKUP_INTENT|FILE_DIRECTORY_FILE, 
			NULL, 0 );
   	if( !NT_SUCCESS( ntStatus ) ) {
		DbgPrint(("Filemon: Could not open drive %c: %x\n", 'A'+Drive, ntStatus ));
		return FALSE;
   	}
   	DbgPrint(("Filemon:  opened the root directory!!! handle: %x\n", ntFileHandle));   

   	// got the file handle, so now look-up the file-object
   	ntStatus = ObReferenceObjectByHandle( ntFileHandle, FILE_READ_DATA, 
						NULL, KernelMode, &fileObject, NULL );
   	if( !NT_SUCCESS( ntStatus )) {
		DbgPrint(("Filemon: Could not get fileobject from handle: %c\n", 'A'+Drive ));
		return FALSE;
   	}
  
   	// now, find out what device is associated with the object
   	fileSysDevice = IoGetRelatedDeviceObject( fileObject );
   	if ( ! fileSysDevice ) {
		DbgPrint(("Filemon: Could not get related device object: %c\n", 'A'+Drive ));
  		ZwClose( ntFileHandle );
		return FALSE;
   	}

	// make sure we haven't already attached to this device (which can happen for directory
	// mounted drives for networks)
  	for( i = 0; i < 26; i++ ) {
		if( LDriveDevices[i] == fileSysDevice ) {
			// yep, already watching it so add it to a group
			ObDereferenceObject( fileObject );
		   	ZwClose( ntFileHandle );
			LDriveMap[ Drive ]     = LDriveMap[i];
			LDriveDevices[ Drive ] = fileSysDevice;
			return TRUE;
		}
	}

   	// create a device to attach to the chain
   	ntStatus = IoCreateDevice( DriverObject,
				    sizeof(HOOK_EXTENSION),
				    NULL,
				    fileSysDevice->DeviceType,
				    0,
			            FALSE,
				    &hookDevice );
   	if ( !NT_SUCCESS(ntStatus) ) {
		DbgPrint(("Filemon: failed to create associated device: %c\n", 'A'+Drive ));   
		return FALSE;
 	}

	// clear our init flag as per NT DDK KB article on creating device objects from 
	// a dispatch routine
	hookDevice->Flags &= ~DO_DEVICE_INITIALIZING;

	// set-up the device extensions
	hookExtension = hookDevice->DeviceExtension;
	hookExtension->LogicalDrive = 'A'+Drive;
	hookExtension->FileSystem   = fileSysDevice;

   	// now, attach ourselves to the device
   	ntStatus = IoAttachDeviceByPointer( hookDevice, fileSysDevice );
   	if ( !NT_SUCCESS(ntStatus) )  {
		// if we couldn't attach
   		DbgPrint(("Filemon: Connect with Filesystem failed: %c (%x) =>%x\n", 
			'A'+Drive, fileSysDevice, ntStatus ));
		// cleanup
		ObDereferenceObject( fileObject );
	   	ZwClose( ntFileHandle );
		return FALSE;
	} else {
		// assign this drive a drive group if it doesn't have one
		DbgPrint(("Filemon: Successfully connected to Filesystem device %c\n", 'A'+Drive ));
		if( !LDriveMap[ Drive ] )
			LDriveMap[ Drive ] = ++LDriveGroup;
	}
	
	// close the file and update our list
	ObDereferenceObject( fileObject );
	ZwClose( ntFileHandle );
	LDriveDevices[Drive] = hookDevice;
    }

    return TRUE;
}


//----------------------------------------------------------------------
//
// UnhookDrive
//
// Unhook a previously hooked drive
//
//----------------------------------------------------------------------
BOOLEAN UnhookDrive( IN char Drive )
{
    if ( Drive >= 'a' && Drive <= 'z' )	Drive -= 'a';
    else				Drive -= 'A';
    if ( (unsigned char)Drive >= 26 )
	return FALSE;

    if ( LDriveDevices[Drive] )  {
	PHOOK_EXTENSION hookExt = LDriveDevices[Drive]->DeviceExtension;
	IoDetachDevice( hookExt->FileSystem );
	IoDeleteDevice( LDriveDevices[Drive] );
	LDriveDevices[Drive] = NULL;
    }
    return TRUE;
}


//----------------------------------------------------------------------
//
// HookDriveSet
//
// Hook/Unhook a set of drives specified by user.  
// Return the set successfully hooked.
//
//----------------------------------------------------------------------
ULONG HookDriveSet( IN ULONG DriveSet, IN PDRIVER_OBJECT DriverObject )
{
    ULONG drive, i;

    for ( drive = 0; drive < 26; ++drive )  {
	ULONG bit = 1 << drive;
	if ( bit & DriveSet )  {
	    // Try to hook drive 
	    if ( ! HookDrive( (char)('A'+drive), DriverObject ) ) 
		// Remove from drive set if can't be hooked
		DriveSet &= ~bit;
	    else {
		// hook drives in same drive group		
		for( i = 0; i < 26; i++ ) {
			if( LDriveMap[i] == LDriveMap[ drive ] &&
				!LDriveDevices[i] ) {
				DriveSet |= ( 1<<i );
				LDriveDevices[i] = LDriveDevices[drive];
			}
		}
	    }
	} else {
	   // Try to unhook drive 
	   if ( ! UnhookDrive( (char)('A'+drive) ) ) 
		DriveSet |= bit;	
	   else {
		// mark drives in same group as unhooked
		for( i = 0; i< 26; i++ ) {
			if( LDriveMap[i] == LDriveMap[ drive ] && 
				LDriveDevices[i] ) {
				DriveSet &= ~(1 << i); 
				LDriveDevices[i] = NULL;
			}
		}
	   }
	}
    }
    // Return set of drives currently hooked
    return DriveSet;
}


//----------------------------------------------------------------------
//
// ErrorString
//
// Returns string representing error condition.
//
//----------------------------------------------------------------------
PCHAR ErrorString( NTSTATUS RetStat, PCHAR Buffer ) {

  switch( RetStat ) {

  case STATUS_SUCCESS:
	strcpy( Buffer, "SUCCESS" );
	break;
  case STATUS_NOT_SUPPORTED:
	strcpy( Buffer, "NOT SUPPORTED" );
	break;
  case STATUS_NO_MORE_FILES:
	strcpy( Buffer, "NO MORE FILES" );
	break;
  case STATUS_OBJECT_NAME_INVALID:
	strcpy( Buffer, "NAME INVALID" );
	break;
  case STATUS_OBJECT_NAME_NOT_FOUND:
	strcpy( Buffer, "FILE NOT FOUND" );
	break;
  case STATUS_NO_SUCH_FILE:
	strcpy( Buffer, "NO SUCH FILE" );
	break;
  case STATUS_OBJECT_NAME_COLLISION:
	strcpy( Buffer, "NAME COLLISION" );
	break;
  case STATUS_NONEXISTENT_SECTOR:
	strcpy( Buffer, "NONEXISTENT SECTOR" );
	break;
  case STATUS_BAD_NETWORK_PATH:
	strcpy( Buffer, "BAD NETWORK PATH" );
	break;
  case STATUS_OBJECT_PATH_NOT_FOUND:
	strcpy( Buffer, "PATH NOT FOUND" );
	break;
  case STATUS_NO_SUCH_DEVICE:
	strcpy( Buffer, "INVALID PARAMETER" );
	break;
  case STATUS_END_OF_FILE:
	strcpy( Buffer, "END OF FILE" );
	break;
  case STATUS_NOTIFY_CLEANUP:
	strcpy( Buffer, "NOTIFY CLEANUP" );
	break;
  case STATUS_BUFFER_OVERFLOW:
	strcpy( Buffer, "BUFFER OVERFLOW" );
	break;
  case STATUS_NO_MORE_ENTRIES:
	strcpy( Buffer, "NO MORE ENTRIES" );
	break;
  case STATUS_ACCESS_DENIED:
	strcpy( Buffer, "ACCESS DENIED" );
	break;
  case STATUS_SHARING_VIOLATION:
	strcpy( Buffer, "SHARING VIOLATION" );
	break;       
  case STATUS_INVALID_PARAMETER:
	strcpy( Buffer, "INVALID PARAMETER" );
	break;       
  case STATUS_OPLOCK_BREAK_IN_PROGRESS:
	strcpy( Buffer, "OPLOCK BREAK" );
	break;       
  case STATUS_PENDING:
	strcpy( Buffer, "PENDING" );
	break;       
  case STATUS_REPARSE:
	strcpy( Buffer, "REPARSE" );
	break;       
  case STATUS_MORE_ENTRIES:
	strcpy( Buffer, "MORE" );
	break;       
  case STATUS_DELETE_PENDING:
	strcpy( Buffer, "DELETE PEND" );
	break;       
  case STATUS_LOCK_NOT_GRANTED:
	strcpy( Buffer, "NOT GRANTED" );
	break;       
  case STATUS_FILE_IS_A_DIRECTORY:
        strcpy( Buffer, "IS DIRECTORY" );
	break;
  case STATUS_CANCELLED:
        strcpy( Buffer, "CANCELLED" );
	break;
  default:
	sprintf( Buffer, "* 0x%X", RetStat );
	break;
  }
  return Buffer;
}


//----------------------------------------------------------------------
//
// FilemonFastIoCheckIfPossible
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoCheckifPossible( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
		IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN BOOLEAN CheckForReadOperation,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR		        fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoCheckIfPossible ) {
	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoCheckIfPossible( 
		FileObject, FileOffset, Length,
		Wait, LockKey, CheckForReadOperation, IoStatus, hookExt->FileSystem );
	UpdateStore( Sequence++, "%s\tFASTIO_CHECK_IF_POSSIBLE\t%s\t%s Offset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, CheckForReadOperation ? "Read:" : "Write:",
		FileOffset->LowPart, Length, 
		retval?"SUCCESS":"FAILURE" ); 
   }
   return retval;
}


//----------------------------------------------------------------------
// 
// FilemonFastIoRead
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoRead( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
		IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, OUT PVOID Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoRead ) {
	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoRead( 
		FileObject, FileOffset, Length,
		Wait, LockKey, Buffer, IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_READ\t%s\tOffset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, Length, 
		retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoWrite
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoWrite( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset, 
		IN ULONG Length, IN BOOLEAN Wait, IN ULONG LockKey, IN PVOID Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoWrite ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoWrite( 
		FileObject, FileOffset, Length,	Wait, LockKey, 
		Buffer, IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_WRITE\t%s\tOffset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, Length, 
		retval?"SUCCESS":"FAILURE" ); 
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryBasicinfo
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoQueryBasicInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
		OUT PFILE_BASIC_INFORMATION Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryBasicInfo ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryBasicInfo( 
		FileObject, Wait, Buffer, IoStatus, hookExt->FileSystem );
	if( retval ) 
   		UpdateStore( Sequence++, "%s\tFASTIO_QUERY_BASIC_INFO\t%s\tAttributes:%04x\t%s", 
			FilemonGetProcess( name ),
			fullPathName, ((PFILE_BASIC_INFORMATION) Buffer)->FileAttributes, 
			retval?"SUCCESS":"FAILURE" );
	else 
   		UpdateStore( Sequence++, "%s\tFASTIO_QUERY_BASIC_INFO\t%s\t\t%s", 
			FilemonGetProcess( name ),
			fullPathName, retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryStandardInfo
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoQueryStandardInfo( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait, 
		OUT PFILE_STANDARD_INFORMATION Buffer,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on 
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryStandardInfo ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryStandardInfo( 
		FileObject, Wait, Buffer, IoStatus, hookExt->FileSystem );
	if( retval ) 
   		UpdateStore( Sequence++, "%s\tFASTIO_QUERY_STANDARD_INFO\t%s\tSize: %ld Delete: %s\t%s", 
			FilemonGetProcess( name ), fullPathName, 
			((PFILE_STANDARD_INFORMATION) Buffer)->EndOfFile.LowPart, 
			((PFILE_STANDARD_INFORMATION) Buffer)->DeletePending ? "Yes" : "No", 
			retval?"SUCCESS":"FAILURE" );
	else
   		UpdateStore( Sequence++, "%s\tFASTIO_QUERY_STANDARD_INFO\t%s\t\t%s", 
			FilemonGetProcess( name ), fullPathName, 
			retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoLock
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoLock( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
		IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
		BOOLEAN FailImmediately, BOOLEAN ExclusiveLock,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;   
   CHAR 		fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoLock ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoLock( 
		FileObject, FileOffset, Length, ProcessId, Key, FailImmediately, 
		ExclusiveLock, IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_LOCK\t%s\tExclusive: %s\t%s", 
		FilemonGetProcess( name ), fullPathName, 
		ExclusiveLock ? "Yes":"No",
		retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoUnlockSingle
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoUnlockSingle( IN PFILE_OBJECT FileObject, IN PLARGE_INTEGER FileOffset,
		IN PLARGE_INTEGER Length, PEPROCESS ProcessId, ULONG Key,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR 		fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;   
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockSingle ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockSingle(
		FileObject, FileOffset, Length, ProcessId, Key, 
		IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_UNLOCK\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoUnlockAll
//
//----------------------------------------------------------------------
BOOLEAN	 FilemonFastIoUnlockAll( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if ( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockAll ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockAll( 
		FileObject, ProcessId, IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_UNLOCK_ALL\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoUnlockAllByKey
//
//----------------------------------------------------------------------	
BOOLEAN	 FilemonFastIoUnlockAllByKey( IN PFILE_OBJECT FileObject, PEPROCESS ProcessId, ULONG Key,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockAllByKey ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoUnlockAllByKey( 
		FileObject, ProcessId, Key, IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_UNLOCK_ALL_BY_KEY\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, retval?"SUCCESS":"FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryNetworkOpenInfo
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoQueryNetworkOpenInfo(IN PFILE_OBJECT FileObject,
    		IN BOOLEAN Wait,
    		OUT struct _FILE_NETWORK_OPEN_INFORMATION *Buffer,
    		OUT PIO_STATUS_BLOCK IoStatus,
    		IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryNetworkOpenInfo ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryNetworkOpenInfo( 
			FileObject, Wait, Buffer, IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_QUERY_NETWORK_OPEN_INFO\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, retval ? "SUCCESS" : "FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoAcquireForModWrite
//
//----------------------------------------------------------------------	
NTSTATUS FilemonFastIoAcquireForModWrite( IN PFILE_OBJECT FileObject,
    		IN PLARGE_INTEGER EndingOffset,
    		OUT struct _ERESOURCE **ResourceToRelease,
    		IN PDEVICE_OBJECT DeviceObject )
{
   NTSTATUS   		retval = STATUS_NOT_SUPPORTED;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], errval[32], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireForModWrite ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireForModWrite( 
		FileObject, EndingOffset, ResourceToRelease, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_ACQUIRE_FOR_MOD_WRITE\t%s\tEndOffset: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, EndingOffset, 
		ErrorString( retval, errval ) );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlRead
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoMdlRead( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
		IN ULONG LockKey, OUT PMDL *MdlChain,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->MdlRead ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlRead( 
			FileObject, FileOffset, Length, LockKey, MdlChain, 
			IoStatus, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_MDL_READ\t%s\tOffset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, Length, 
		retval ? "SUCCESS" : "FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlReadComplete
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoMdlReadComplete( IN PFILE_OBJECT FileObject,
		IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->MdlReadComplete ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
   	retval = (BOOLEAN) hookExt->FileSystem->DriverObject->FastIoDispatch->MdlReadComplete( FileObject, 
		MdlChain, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_MDL_READ_COMPLETE\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, "OK" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoPrepareMdlWrite
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoPrepareMdlWrite( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
		IN ULONG LockKey, OUT PMDL *MdlChain,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   IoStatus->Status      = STATUS_NOT_SUPPORTED;
   IoStatus->Information = 0;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->PrepareMdlWrite ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->PrepareMdlWrite( 
			FileObject, FileOffset, Length, LockKey, MdlChain, IoStatus, 
			hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_PREPARE_MDL_WRITE\t%s\tOffset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, Length, 
		retval ? "SUCCESS" : "FAILURE" );
   } 
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlWriteComplete
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoMdlWriteComplete( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain, 
		IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->MdlWriteComplete ) { 
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
   	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlWriteComplete( 
			FileObject, FileOffset, MdlChain, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_MDL_WRITE_COMPLETE\t%s\tOffset: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, "OK" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoReadCompressed
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoReadCompressed( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
    		IN ULONG LockKey, OUT PVOID Buffer,
    		OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
    		OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
    		IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoReadCompressed ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoReadCompressed( 
			FileObject, FileOffset, Length, LockKey, Buffer, MdlChain, IoStatus,
			CompressedDataInfo, CompressedDataInfoLength, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_READ_COMPRESSED\t%s\tOffset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, 
		Length, retval ? "SUCCESS" : "FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoWriteCompressed
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoWriteCompressed( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN ULONG Length,
    		IN ULONG LockKey, OUT PVOID Buffer,
    		OUT PMDL *MdlChain, OUT PIO_STATUS_BLOCK IoStatus,
    		OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
    		IN ULONG CompressedDataInfoLength, IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoWriteCompressed ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoWriteCompressed( 
			FileObject, FileOffset, Length, LockKey, Buffer, MdlChain, IoStatus,
			CompressedDataInfo, CompressedDataInfoLength, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_WRITE_COMPRESSED\t%s\tOffset: %ld Length: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, Length, 
		retval ? "SUCCESS" : "FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlReadCompleteCompressed
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoMdlReadCompleteCompressed( IN PFILE_OBJECT FileObject,
		IN PMDL MdlChain, IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->MdlReadCompleteCompressed ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
   	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlReadCompleteCompressed( 
			FileObject, MdlChain, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_MDL_READ_COMPLETE_COMPRESSED\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, "OK" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoMdlWriteCompleteCompressed
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoMdlWriteCompleteCompressed( IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset, IN PMDL MdlChain, 
		IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension; 
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->MdlWriteCompleteCompressed ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
   	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->MdlWriteCompleteCompressed( 
			FileObject, FileOffset, MdlChain, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_MDL_WRITE_COMPLETE_COMPRESSED\t%s\tOffset: %ld\t%s", 
		FilemonGetProcess( name ), fullPathName, FileOffset->LowPart, "OK" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoQueryOpen
//
// This call actually passes an IRP! (what the heck is it doing as
// a fastio call?) Anyway, we haven't figured out how to filter this
// without blue-screening.
//
//----------------------------------------------------------------------	
BOOLEAN FilemonFastIoQueryOpen( IN PIRP Irp,
    		OUT PFILE_NETWORK_OPEN_INFORMATION NetworkInformation,
    		IN PDEVICE_OBJECT DeviceObject )
{
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   PFILE_OBJECT		fileobject;
   CHAR			fullPathName[256], name[20];
   PIO_STACK_LOCATION 	currentIrpStack;
   PIO_STACK_LOCATION   nextIrpStack;

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryOpen ) {
	currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
	nextIrpStack    = IoGetNextIrpStackLocation(Irp);
	fileobject = currentIrpStack->FileObject;
	FilemonGetFullPath( fileobject, hookExt, fullPathName );
	// copy parameters down to next level in the stack
	*nextIrpStack = *currentIrpStack;
	nextIrpStack->DeviceObject = hookExt->FileSystem;
	IoSetNextIrpStackLocation( Irp );

	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoQueryOpen( 
			Irp, NetworkInformation, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_QUERY_OPEN\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, retval ? "SUCCESS" : "FAILURE" );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoReleaseForModWrite
//
//----------------------------------------------------------------------	
NTSTATUS FilemonFastIoReleaseForModWrite( IN PFILE_OBJECT FileObject,
    		IN struct _ERESOURCE *ResourceToRelease,
    		IN PDEVICE_OBJECT DeviceObject )
{
   NTSTATUS   		retval = STATUS_NOT_SUPPORTED;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], errval[32], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseForModWrite ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseForModWrite( 
		FileObject,  ResourceToRelease, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_RELEASE_FOR_MOD_WRITE\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, ErrorString( retval, errval ) );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoAcquireForCcFlush
//
//----------------------------------------------------------------------	
NTSTATUS FilemonFastIoAcquireForCcFlush( IN PFILE_OBJECT FileObject,
    		IN PDEVICE_OBJECT DeviceObject )
{
   NTSTATUS   		retval = STATUS_NOT_SUPPORTED;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], errval[32], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireForCcFlush ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireForCcFlush( 
		FileObject, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_ACQUIRE_FOR_CC_FLUSH\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, ErrorString( retval, errval ) );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoReleaseForCcFlush
//
//----------------------------------------------------------------------	
NTSTATUS FilemonFastIoReleaseForCcFlush( IN PFILE_OBJECT FileObject,
    		IN PDEVICE_OBJECT DeviceObject )
{
   NTSTATUS   		retval = STATUS_NOT_SUPPORTED;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], errval[32], name[20];

   // pass it on
   hookExt = DeviceObject->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseForCcFlush ) {
   	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	retval = hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseForCcFlush( 
		FileObject, hookExt->FileSystem );
   	UpdateStore( Sequence++, "%s\tFASTIO_RELEASE_FOR_CC_FLUSH\t%s\t\t%s", 
		FilemonGetProcess( name ), fullPathName, ErrorString( retval, errval ) );
   }
   return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoDeviceControl
//
//----------------------------------------------------------------------
BOOLEAN  FilemonFastIoDeviceControl( IN PFILE_OBJECT FileObject, IN BOOLEAN Wait,
		IN PVOID InputBuffer, IN ULONG InputBufferLength, 
		OUT PVOID OutputBuffer, IN ULONG OutputBufferLength, IN ULONG IoControlCode,
		OUT PIO_STATUS_BLOCK IoStatus, IN PDEVICE_OBJECT DeviceObject ) {
   BOOLEAN   		retval = FALSE;
   PHOOK_EXTENSION	hookExt;
   PSTORE_BUF		old;
   CHAR			fullPathName[256], name[20];
   KIRQL                oldirql;

   if ( DeviceObject == GUIDevice )  {
	// Its a message from our GUI!
	IoStatus->Status      = STATUS_SUCCESS;	// Assume success
	IoStatus->Information = 0;		// Assume nothing returned
        switch ( IoControlCode ) {
	case FILEMON_setdrives:
 	    DbgPrint (("Filemon: set drives\n"));
	    if ( InputBufferLength != sizeof(ULONG) ||
		 OutputBufferLength < sizeof(ULONG) ||
		 InputBuffer == NULL ||
		 OutputBuffer == NULL ) {
		IoStatus->Status = 1;	// blech
		break;
	    }
	    *(ULONG *)OutputBuffer = HookDriveSet( *(ULONG *)InputBuffer, DeviceObject->DriverObject );
	    IoStatus->Information = sizeof(ULONG);
	    break;
	case FILEMON_unloadquery:
	    // is it possible to unload?
	    MUTEX_P( CountMutex, &oldirql );
	    IoStatus->Information = OutstandingIRPCount;
	    if( !OutstandingIRPCount ) {
		UnloadInProgress = TRUE;

		// disconnect from filesystems
		HookDriveSet( 0, DeviceObject->DriverObject );
	    }
	    MUTEX_V( CountMutex, oldirql );
	    break;
	case FILEMON_zerostats:
	    // Zero contents of buffer
	    DbgPrint (("Filemon: zero stats\n"));
	    MUTEX_P( StoreMutex, &oldirql );
	    while ( Store->Next )  {
		// release next
		old = Store->Next;
		Store->Next = old->Next;
		ExFreePool( old );
		NumStore--;
	    }
	    Store->Len = 0;
	    MUTEX_V( StoreMutex, oldirql );
	    break;
	case FILEMON_getstats:
    	    // Copy buffer into user space.
    	    DbgPrint (("Filemon: get stats\n"));
	    MUTEX_P( StoreMutex, &oldirql );
	    if ( MAX_STORE > OutputBufferLength )  {
		// Don't do this; its not worth implementing the fix.
		MUTEX_V( StoreMutex, oldirql );
		return FALSE;
	    } else if ( Store->Len  ||  Store->Next ) {

		// Switch to a new store
		FilemonNewStore();

		// Fetch the oldest to give to user
		old = FilemonOldestStore();
		MUTEX_V( StoreMutex, oldirql );

		// Copy it
		memcpy( OutputBuffer, old->Data, old->Len );

    		// Return length of copied info
		IoStatus->Information = old->Len;

		// Deallocate buffer
		ExFreePool( old );
		NumStore--;
	    } else {
		// no unread data
		MUTEX_V( StoreMutex, oldirql );
		IoStatus->Information = 0;
	    }
	    break;
 
        default:
            DbgPrint (("Filemon: unknown IRP_MJ_DEVICE_CONTROL\n"));
 	    IoStatus->Status = 2;	// Bad request
            break;
        }
	retval = TRUE;
   } else {
	// pass it on
	hookExt = DeviceObject->DeviceExtension;
	FilemonGetFullPath( FileObject, hookExt, fullPathName );
	if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoDeviceControl ) 
		retval = hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoDeviceControl( 
			FileObject, Wait, InputBuffer, InputBufferLength, OutputBuffer, 
			OutputBufferLength, IoControlCode, IoStatus, DeviceObject );
	UpdateStore( Sequence++, "%s\tFASTIO_DEVICE_CONTROL\t%s\tIoControl: %08X\t%s", 
		FilemonGetProcess( name ), fullPathName, IoControlCode, 
		retval ? "SUCCESS" : "FAILURE" );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// FilemonFastIoAcquireFile
//
//----------------------------------------------------------------------
VOID FilemonFastIoAcquireFile( PFILE_OBJECT FileObject ) {
   PDEVICE_OBJECT	deviceObject;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   deviceObject = IoGetRelatedDeviceObject( FileObject );
   hookExt = deviceObject->DeviceExtension;
   FilemonGetFullPath( FileObject, hookExt, fullPathName );
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireFileForNtCreateSection ) 
	hookExt->FileSystem->DriverObject->FastIoDispatch->AcquireFileForNtCreateSection( 
		FileObject );
   UpdateStore( Sequence++, "%s\tFASTIO_ACQUIRE_FILE\t%s\t\tOK", FilemonGetProcess( name ), 
	fullPathName );
}


//----------------------------------------------------------------------
//
// FilemonFastIoReleaseFile
//
//----------------------------------------------------------------------
VOID FilemonFastIoReleaseFile( PFILE_OBJECT FileObject ) {
   PDEVICE_OBJECT	deviceObject;
   PHOOK_EXTENSION	hookExt;
   CHAR			fullPathName[256], name[20];

   // pass it on
   deviceObject = IoGetRelatedDeviceObject( FileObject );
   hookExt = deviceObject->DeviceExtension;
   FilemonGetFullPath( FileObject, hookExt, fullPathName );
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseFileForNtCreateSection ) 
	hookExt->FileSystem->DriverObject->FastIoDispatch->ReleaseFileForNtCreateSection( FileObject );
   UpdateStore( Sequence++, "%s\tFASTIO_RELEASE_FILE\t%s\t\tOK", FilemonGetProcess( name ),
	fullPathName );
}


//----------------------------------------------------------------------
//
// FilemonFastIoDetachDevice
//
// We get this call when a device that we have hooked is being deleted.
// This happens when, for example, a floppy is formatted. We have
// to detach from it and delete our device. We should notify the GUI
// that the hook state has changed, but its not worth the trouble.
//
//----------------------------------------------------------------------
VOID FilemonFastIoDetachDevice( PDEVICE_OBJECT SourceDevice, PDEVICE_OBJECT TargetDevice ) {
   PHOOK_EXTENSION	hookExt;
   ULONG		i;
   CHAR			name[20];

   // see if a device (like a floppy) is being removed out from under us
   for( i = 0; i < 26; i++ ) {
	if( SourceDevice == LDriveDevices[i] ) {
		// yep, got to close up shop
		hookExt = SourceDevice->DeviceExtension;
		DbgPrint(("Filemon: Detaching from drive: %c\n", 
			hookExt->LogicalDrive ));
		IoDetachDevice( TargetDevice );
		IoDeleteDevice( SourceDevice );
		LDriveDevices[i] = NULL;
		LDriveMap[i] = 0;
		return;
	}
   }

   // not for us, pass it on
   hookExt = SourceDevice->DeviceExtension;
   if( hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoDetachDevice ) {
	hookExt->FileSystem->DriverObject->FastIoDispatch->FastIoDetachDevice( 
		SourceDevice, TargetDevice );
	UpdateStore( Sequence++, "%s\tFASTIO_DETACH_DEVICE\t\t\tOK", 
		FilemonGetProcess( name ) );
   }
}


//----------------------------------------------------------------------
// 
// FilemonHookDone
//
// Gets control after a filesystem operation has completed so that
// we can get timing information about it.
//
//----------------------------------------------------------------------
NTSTATUS FilemonHookDone( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp,
				IN PVOID Context )
{
  PIO_STACK_LOCATION	IrpSp;
  int			seq = (int)Context;
  CHAR			errstring[256], errval[32];
  KIRQL                 oldirql;

  // request completed - look at the result 
  IrpSp = IoGetCurrentIrpStackLocation( Irp );	

  // store info in buffer
  if( GUIActive ) {
      sprintf( errstring, "\t\t\t\t%s", ErrorString( Irp->IoStatus.Status, errval ));
      UpdateStore( seq, errstring );
  }
  
  // decrement oustanding IRP count
  MUTEX_P( CountMutex, &oldirql );
  OutstandingIRPCount--;
  MUTEX_V( CountMutex, oldirql );

  // have to mark Irp as pending if necessary
  if( Irp->PendingReturned )
	IoMarkIrpPending( Irp );
  return Irp->IoStatus.Status;
}


//----------------------------------------------------------------------
//
// FilemonHookRoutine
//
// This routine is the main hook routine where we figure out what
// calls are being sent to the file system.
//
//----------------------------------------------------------------------
NTSTATUS FilemonHookRoutine( PDEVICE_OBJECT HookDevice, IN PIRP Irp )
{
    PIO_STACK_LOCATION currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
    PIO_STACK_LOCATION nextIrpStack    = IoGetNextIrpStackLocation(Irp);
    PFILE_OBJECT       fileObject;
    PHOOK_EXTENSION    hookExt;
    CHAR	       fullPathName[256], name[20];
    ULONG	       seq = Sequence++;
    KIRQL              oldirql;

    // get the filename of the file object
    fileObject 	  = currentIrpStack->FileObject;

    // get the target filesystem device information
    hookExt = HookDevice->DeviceExtension;

    // get the canonical pathname
    if( GUIActive ) FilemonGetFullPath( fileObject, hookExt, fullPathName );

    // Determine what function we're dealing with
    switch( currentIrpStack->MajorFunction ) {
    case IRP_MJ_CREATE:
	// clear any existing entry for this file object 
	FilemonFreeHashEntry( fileObject );
	UpdateStore( seq, "%s\tIRP_MJ_CREATE\t%s\tAttributes:%04x Options:%08x", 
			FilemonGetProcess( name ), fullPathName,
			currentIrpStack->Parameters.Create.FileAttributes,
			currentIrpStack->Parameters.Create.Options );
	break;
    case IRP_MJ_READ:
	UpdateStore( seq, "%s\tIRP_MJ_READ\t%s\tOffset: %ld Length: %ld", 
		FilemonGetProcess( name ), fullPathName, 
		currentIrpStack->Parameters.Read.ByteOffset.LowPart,
		currentIrpStack->Parameters.Read.Length );
	break;
    case IRP_MJ_WRITE:
	UpdateStore( seq, "%s\tIRP_MJ_WRITE\t%s\tOffset: %ld Length: %ld", 
			FilemonGetProcess( name ), fullPathName, 
			currentIrpStack->Parameters.Write.ByteOffset.LowPart,
			currentIrpStack->Parameters.Write.Length );
	break;
    case IRP_MJ_CLOSE:
	UpdateStore( seq, "%s\tIRP_MJ_CLOSE\t%s\t", FilemonGetProcess( name ),
		fullPathName );
	// we can clean up for this fileobject now
	FilemonFreeHashEntry( fileObject );
	break;
    case IRP_MJ_FLUSH_BUFFERS:
	UpdateStore( seq, "%s\tIRP_MJ_FLUSH\t%s\t", FilemonGetProcess( name ), 
		fullPathName );
	break;
    case IRP_MJ_QUERY_INFORMATION:
	UpdateStore( seq, "%s\tIRP_MJ_QUERY_INFORMATION\t%s\t%s", 
		FilemonGetProcess( name ), fullPathName, 
		FileInformation[currentIrpStack->Parameters.QueryFile.FileInformationClass] );
	break;
    case IRP_MJ_SET_INFORMATION:
	UpdateStore( seq, "%s\tIRP_MJ_SET_INFORMATION\t%s\t%s", 
		FilemonGetProcess( name ), fullPathName,
		FileInformation[currentIrpStack->Parameters.SetFile.FileInformationClass] );
	break;
    case IRP_MJ_QUERY_EA:
	UpdateStore( seq, "%s\tIRP_MJ_QUERY_EA\t%s\t", FilemonGetProcess( name ), 
		fullPathName );
	break;
    case IRP_MJ_SET_EA:
	UpdateStore( seq, "%s\tIRP_MJ_SET_EA\t%s\t", FilemonGetProcess( name ), fullPathName );
	break;
    case IRP_MJ_QUERY_VOLUME_INFORMATION:
	UpdateStore( seq, "%s\tIRP_MJ_QUERY_VOLUME_INFORMATION\t%s\t%s", 
		FilemonGetProcess( name ), fullPathName,
		VolumeInformation[currentIrpStack->Parameters.QueryVolume.FsInformationClass] );
	break;
    case IRP_MJ_SET_VOLUME_INFORMATION:
	UpdateStore( seq, "%s\tIRP_MJ_SET_VOLUME_INFORMATION\t%s\t", 
		FilemonGetProcess( name ), fullPathName );
	break;
    case IRP_MJ_DIRECTORY_CONTROL:
	UpdateStore( seq, "%s\tIRP_MJ_DIRECTORY_CONTROL\t%s\t", 
		FilemonGetProcess( name ), fullPathName );
	break;
    case IRP_MJ_FILE_SYSTEM_CONTROL:
	UpdateStore( seq, "%s\tIRP_MJ_FILE_SYSTEM_CONTROL\t%s\t", 
		FilemonGetProcess( name ), fullPathName );
	break;
    case IRP_MJ_SHUTDOWN:
	UpdateStore( seq, "%s\tIRP_MJ_SHUTDOWN\t\t", FilemonGetProcess( name ) );
	break;
    case IRP_MJ_LOCK_CONTROL:
	UpdateStore( seq, "%s\tIRP_MJ_LOCK_CONTROL\t%s\t", FilemonGetProcess( name ),
		fullPathName );
	break;
    case IRP_MJ_CLEANUP:
	UpdateStore( seq, "%s\tIRP_MJ_CLEANUP\t%s\t", FilemonGetProcess( name ),
		fullPathName );
	break;
    case IRP_MJ_DEVICE_CONTROL:
	UpdateStore( seq, "%s\tIRP_MJ_DEVICE_CONTROL\t%s\t", FilemonGetProcess( name ),
		fullPathName );
	break;
    case IRP_MJ_CREATE_MAILSLOT:
	UpdateStore( seq, "%s\tIRP_MJ_CREAT_MAILSLOT\t%s\t", 
		FilemonGetProcess( name ), fullPathName );
	break;
    case IRP_MJ_QUERY_SECURITY:
	UpdateStore( seq, "%s\tIRP_MJ_QUERY_SECURITY\t%s\t", 
		FilemonGetProcess( name ), fullPathName );
	break;
    case IRP_MJ_SET_SECURITY:
	UpdateStore( seq, "%s\tIRP_MJ_SET_SECURITY\t%s\t", 
		FilemonGetProcess( name ), fullPathName );
	break;
    default:
	UpdateStore( seq, "%s\t*UNKNOWN* 0x%X\t\t", FilemonGetProcess( name ),
		currentIrpStack->MajorFunction );
	break;
    }

    // copy parameters down to next level in the stack
    *nextIrpStack = *currentIrpStack;

    // set the completion callback if we are not unloading
    MUTEX_P( CountMutex, &oldirql );
    if( !UnloadInProgress ) {
        OutstandingIRPCount++;
	IoSetCompletionRoutine( Irp, FilemonHookDone, (void *)seq, TRUE, TRUE, TRUE );
    }
    MUTEX_V( CountMutex, oldirql );

    // return the results of the call to the caller
    return IoCallDriver( hookExt->FileSystem, Irp );
}


//----------------------------------------------------------------------
//
// FilemonDeviceRoutine
//
// In this routine we handle requests to our own device. The only 
// requests we care about handling explicitely are IOCTL commands that
// we will get from the GUI. We also expect to get Create and Close 
// commands when the GUI opens and closes communications with us.
//
//----------------------------------------------------------------------
NTSTATUS FilemonDeviceRoutine( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp )
{
    PIO_STACK_LOCATION  irpStack;
    PDEVICE_EXTENSION   deviceExtension;
    PVOID               inputBuffer;
    PVOID		outputBuffer;
    ULONG               inputBufferLength;
    ULONG               outputBufferLength;
    ULONG               ioControlCode;

    // go ahead and set the request up as successful
    Irp->IoStatus.Status      = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    // Get a pointer to the current location in the Irp. This is where
    //     the function codes and parameters are located.
    irpStack = IoGetCurrentIrpStackLocation (Irp);

    // Get a pointer to the device extension
    deviceExtension = DeviceObject->DeviceExtension;

    // Get the pointer to the input/output buffer and its length
    inputBuffer		= Irp->AssociatedIrp.SystemBuffer;
    inputBufferLength	= irpStack->Parameters.DeviceIoControl.InputBufferLength;
    outputBuffer	= Irp->AssociatedIrp.SystemBuffer;
    outputBufferLength	= irpStack->Parameters.DeviceIoControl.OutputBufferLength;
    ioControlCode	= irpStack->Parameters.DeviceIoControl.IoControlCode;

    switch (irpStack->MajorFunction) {
    case IRP_MJ_CREATE:
        DbgPrint(("Filemon: IRP_MJ_CREATE\n"));
	// fail multiple GUI connects at one time
	if( GUIActive )
		Irp->IoStatus.Status = 2;
	else
		Sequence = 0;
	GUIActive++;
	DbgPrint((" GUI Active: %d\n", GUIActive ));
        break;

    case IRP_MJ_CLOSE:
        DbgPrint(("Filemon: IRP_MJ_CLOSE\n"));
	GUIActive--;
	DbgPrint((" GUI closing: %d\n", GUIActive ));
	if( !GUIActive )
		FilemonResetStore();
        break;

    case IRP_MJ_DEVICE_CONTROL:
        DbgPrint (("Filemon: IRP_MJ_DEVICE_CONTROL\n"));

	// get output buffer if its there
	if( Irp->MdlAddress ) 
	    outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

	// Its a request from the GUI. Simply call our fast handler.
	FilemonFastIoDeviceControl(	irpStack->FileObject, TRUE,
					inputBuffer, inputBufferLength, 
					outputBuffer, outputBufferLength,
					ioControlCode, &Irp->IoStatus, DeviceObject );
        break;
    }
    IoCompleteRequest( Irp, IO_NO_INCREMENT );
    return STATUS_SUCCESS;   
}


//----------------------------------------------------------------------
//
// FilemonDispatch
//
// Based on which device the Irp is destined for we call either the
// filesystem filter function, or our own device handling routine.
//
//----------------------------------------------------------------------
NTSTATUS FilemonDispatch( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp )
{
    // is it a hook or a request for us?
    if ( GUIDevice != DeviceObject ) 
	return( FilemonHookRoutine( DeviceObject, Irp ));
    else
	return( FilemonDeviceRoutine( DeviceObject, Irp ));
}


//----------------------------------------------------------------------
//
// FilemonUnload
//
// Our job is done - time to leave.
//
//----------------------------------------------------------------------
VOID FilemonUnload( IN PDRIVER_OBJECT DriverObject )
{
   WCHAR                  deviceLinkBuffer[]  = L"\\DosDevices\\Filemon";
   UNICODE_STRING         deviceLinkUnicodeString;

   // delete the symbolic link for our device
   RtlInitUnicodeString( &deviceLinkUnicodeString, deviceLinkBuffer );
   IoDeleteSymbolicLink( &deviceLinkUnicodeString );

   DbgPrint(("Filemon.SYS: unloading\n"));

   // Delete the device object
   if ( GUIDevice == DriverObject->DeviceObject )  {
        IoDeleteDevice( DriverObject->DeviceObject );
   } else {
        IoDeleteDevice( DriverObject->DeviceObject );
 	IoDeleteDevice( GUIDevice ); 
   }

   DbgPrint(("Filemon.SYS: deleted devices\n"));

   // now we can free any memory we have outstanding
   RtlFreeAnsiString( &DirectFileName );
   FilemonHashCleanup();
   FilemonFreeStore();

   DbgPrint(("Filemon.SYS: freed memory\n"));
}
