//======================================================================
//
// REGMON.c - main module for VxD REGMON
//
// By Mark Russinovich and Bryce Cogswell
//
//======================================================================
#define   DEVICE_MAIN
#include  "ioctlcmd.h"
#include  "regmon.h"
#undef    DEVICE_MAIN

#if DEBUG
#define dprint(arg) dprintf arg
#else
#define dprint(arg)
#endif

//----------------------------------------------------------------------
//                     G L O B A L   D A T A 
//----------------------------------------------------------------------

// real service pointers with the hook thunks
HDSC_Thunk ROKThunk;
LONG (*RealRegOpenKey)(HKEY, PCHAR, PHKEY );
HDSC_Thunk RCKThunk;
LONG (*RealRegCloseKey)(HKEY );
HDSC_Thunk RFKThunk;
LONG (*RealRegFlushKey)(HKEY );
HDSC_Thunk RCRKThunk;
LONG (*RealRegCreateKey)(HKEY, PCHAR, PHKEY );
HDSC_Thunk RDKThunk;
LONG (*RealRegDeleteKey)(HKEY, PCHAR );
HDSC_Thunk RDVThunk;
LONG (*RealRegDeleteValue)(HKEY, PCHAR );
HDSC_Thunk REKThunk;
LONG (*RealRegEnumKey)(HKEY, DWORD, PCHAR, DWORD );
HDSC_Thunk REVThunk;
LONG (*RealRegEnumValue)(HKEY, DWORD, PCHAR, PDWORD, PDWORD, PDWORD,
			 PBYTE, PDWORD );
HDSC_Thunk RQIKThunk;
LONG (*RealRegQueryInfoKey)(HKEY, PCHAR, PDWORD, DWORD, PDWORD, PDWORD,
			    PDWORD, PDWORD, PDWORD, PDWORD, PDWORD, 
			    PFILETIME );
HDSC_Thunk RQVThunk;
LONG (*RealRegQueryValue)( HKEY, PCHAR, PCHAR, PLONG );
HDSC_Thunk RQVEThunk;
LONG (*RealRegQueryValueEx)(HKEY, PCHAR, PDWORD, PDWORD, PBYTE, PDWORD );
HDSC_Thunk RSVThunk;
LONG (*RealRegSetValue)( HKEY, PCHAR, DWORD, PCHAR, DWORD );
HDSC_Thunk RSVEThunk;
LONG (*RealRegSetValueEx)(HKEY, PCHAR, DWORD, DWORD, PBYTE, DWORD );
HDSC_Thunk RRPDKThunk;
LONG (*RealRegRemapPreDefKey)(HKEY, HKEY );
HDSC_Thunk RQMVThunk;
LONG (*RealRegQueryMultipleValues)(HKEY, PVALENT, DWORD, PCHAR, PDWORD );
HDSC_Thunk RCDKThunk;
LONG (*RealRegCreateDynKey)( PCHAR, PVOID, PVOID, PVOID, DWORD, PVMMHKEY);

// hash table data 
PHASH_ENTRY		HashTable[NUMHASH];

// list of freed entries (to save some allocation calls)
PHASH_ENTRY		FreeEntries = NULL;

// buffer data
PSTORE_BUF		Store 		= NULL;
PSTORE_BUF		StoreExtra 	= NULL;
ULONG			Sequence 	= 0;

// maximum amount of buffers we will grab for buffered unread data
ULONG			NumStore 	= 0;
ULONG			MaxStore 	= 5;

// semaphore for critical sections
SEMHANDLE               StoreMutex, HashMutex;

// unknown error string
CHAR                    errstring[32];


//----------------------------------------------------------------------
//                   V X D  C O N T R O L
//----------------------------------------------------------------------

// device declaration
Declare_Virtual_Device(REGMON)

// message handlers - we only care about dynamic loading and unloading
DefineControlHandler(SYS_DYNAMIC_DEVICE_INIT, OnSysDynamicDeviceInit);
DefineControlHandler(SYS_DYNAMIC_DEVICE_EXIT, OnSysDynamicDeviceExit);
DefineControlHandler(W32_DEVICEIOCONTROL, OnW32Deviceiocontrol);

//----------------------------------------------------------------------
// 
// ControlDispatcher
//
// Multiplexes incoming VxD messages from Windows to their handlers.
//
//----------------------------------------------------------------------
BOOL __cdecl ControlDispatcher(
	DWORD dwControlMessage,
	DWORD EBX,
	DWORD EDX,
	DWORD ESI,
	DWORD EDI,
	DWORD ECX)
{
  START_CONTROL_DISPATCH

    ON_SYS_DYNAMIC_DEVICE_INIT(OnSysDynamicDeviceInit);
    ON_SYS_DYNAMIC_DEVICE_EXIT(OnSysDynamicDeviceExit);
    ON_W32_DEVICEIOCONTROL(OnW32Deviceiocontrol);

  END_CONTROL_DISPATCH

  return TRUE;
}

//----------------------------------------------------------------------
// B U F F E R  M A N A G E M E N T  A N D  P R I N T  R O U T I N E S
//----------------------------------------------------------------------


//----------------------------------------------------------------------
//
// RegmonFreeStore
//
// Frees all the data output buffers that we have currently allocated.
//
//----------------------------------------------------------------------
VOID RegmonFreeStore()
{
    PSTORE_BUF 	prev;
    
    PageFree( (MEMHANDLE) StoreExtra, 0 );
    while( Store ) {
	prev = Store->Next;
	PageFree( (MEMHANDLE) Store, 0 );
	Store = prev;
    }
}	


//----------------------------------------------------------------------
//
// RegmonNewStore
//
// Called when the current buffer has filled up. This moves us to the
// pre-allocated buffer and then allocates another buffer.
//
//----------------------------------------------------------------------
void RegmonNewStore( void )
{
    PSTORE_BUF prev = Store;

    // if we have maxed out or haven't accessed the current store
    // just return
    if( MaxStore == NumStore ) {
	Store->Len = 0;
	return;	
    }

    // see if we can re-use a store
    if( !Store->Len ) 
	return;

    // move to the next buffer and allocate another one
    Store 	= StoreExtra;
    Store->Len  = 0;
    Store->Next = prev;
    PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
                  (PMEMHANDLE) &StoreExtra, (PVOID) &StoreExtra );
    NumStore++;
}


//----------------------------------------------------------------------
//
// RegmonOldestStore
//
// Goes through the linked list of storage buffers and returns the 
// oldest one.
//
//----------------------------------------------------------------------
PSTORE_BUF RegmonOldestStore( void )
{
    PSTORE_BUF  ptr = Store, prev = NULL;

    while ( ptr->Next )
	ptr = (prev = ptr)->Next;
    if ( prev )
	prev->Next = NULL;    
    return ptr;
}


//----------------------------------------------------------------------
//
// RegmonResetStore
//
// When a GUI is no longer communicating with us, but we can't unload,
// we reset the storage buffers.
//
//----------------------------------------------------------------------
VOID RegmonResetStore()
{
   PSTORE_BUF  current, next;

   current = Store->Next;
   while( current ) {
        next = current->Next;
	PageFree( (MEMHANDLE) current, 0 );
        current = next;
   }
   Store->Len = 0;
   Store->Next = NULL;
}


//----------------------------------------------------------------------
//
// UpdateStore
//
// Add a new string to Store, if it fits.
//
//----------------------------------------------------------------------
void UpdateStore( ULONG seq, const char * format, ... )
{	
    PENTRY		Entry;
    int			len;
    va_list		arg_ptr;

    MUTEX_P( StoreMutex );
    
    // See if its time to switch to extra buffer
    if ( Store->Len > MAX_STORE-350  &&  StoreExtra )  
	RegmonNewStore();

    if ( Store->Len <= MAX_STORE-350 )  { 

	/* Set our sequence number */
	Entry = (void *)(Store->Data+Store->Len);
	Entry->seq = seq;

	va_start( arg_ptr, format );
	len = vsprintf( Entry->text, format, arg_ptr );
	va_end( arg_ptr );

	Store->Len += sizeof(Entry->seq)+len+1;		// +1 for null char
    }
    MUTEX_V( StoreMutex );
}


//----------------------------------------------------------------------
//
// RegmonHashCleanup
//
// Called when we are unloading to free any memory that we have 
// in our possession.
//
//----------------------------------------------------------------------
VOID RegmonHashCleanup()
{
   PHASH_ENTRY		hashEntry, nextEntry;
   ULONG		i;
  
   // first free the hash table entries
   for( i = 0; i < NUMHASH; i++ ) {
	hashEntry = HashTable[i];
	while( hashEntry ) {
		nextEntry = hashEntry->Next;
		HeapFree( hashEntry->FullName, 0 );
		HeapFree( hashEntry, 0 );
		hashEntry = nextEntry;
	}
   }

   // now, free structures on our free list
   while( FreeEntries ) {
	nextEntry = FreeEntries->Next;
	HeapFree( FreeEntries, 0 );
	FreeEntries = nextEntry;
   }
}

//----------------------------------------------------------------------
//
// RegmonStoreHash
//
// Stores the key and associated fullpath in the hash table.
//
//----------------------------------------------------------------------
VOID RegmonStoreHash( HKEY hkey, PCHAR fullname )
{
  PHASH_ENTRY     newEntry;

  MUTEX_P( HashMutex );
  if( FreeEntries ) {
    newEntry    = FreeEntries;
    FreeEntries = FreeEntries->Next;
  } else 
    newEntry = HeapAllocate( sizeof(HASH_ENTRY), 0 );

  newEntry->hkey    	        = hkey;
  newEntry->FullName   	        = HeapAllocate( strlen(fullname)+1, 0);
  newEntry->Next		= HashTable[ HASHOBJECT(hkey) ];
  HashTable[ HASHOBJECT(hkey) ] = newEntry;	
  strcpy( newEntry->FullName, fullname );
  MUTEX_V( HashMutex );
}

//----------------------------------------------------------------------
//
// RegmonFreeHashEntry
//
// When we see a file close, we can free the string we had associated
// with the fileobject being closed since we know it won't be used
// again.
//
//----------------------------------------------------------------------
VOID RegmonFreeHashEntry( HKEY hkey )
{
   PHASH_ENTRY		hashEntry, prevEntry;

   MUTEX_P( HashMutex );

   // look-up the entry
   hashEntry = HashTable[ HASHOBJECT( hkey ) ];
   prevEntry = NULL;
   while( hashEntry && hashEntry->hkey != hkey ) {
	prevEntry = hashEntry;
	hashEntry = hashEntry->Next;
   }
  
   // if we fall off, just return
   if( !hashEntry ) {
	MUTEX_V( HashMutex );
	return;
   }

   // remove it from the hash list 
   if( prevEntry ) 
	prevEntry->Next = hashEntry->Next;
   else 
	HashTable[ HASHOBJECT( hkey )] = hashEntry->Next;

   // free the memory associated with it
   HeapFree( hashEntry->FullName, 0 );

   // put it on our free list
   hashEntry->Next 	= FreeEntries;
   FreeEntries 		= hashEntry;

   MUTEX_V( HashMutex );
}

//----------------------------------------------------------------------
//
// ErrorString
//
// Returns the string form of an error code.
//
//----------------------------------------------------------------------
PCHAR ErrorString( DWORD retval )
{
  switch( retval ) {
  case ERROR_SUCCESS:
    return "SUCCESS";
  case ERROR_KEY_DELETED:
    return "KEYDELETED";
  case ERROR_BADKEY:
    return "BADKEY";
  case ERROR_REGISTRY_IO_FAILED:
    return "IOFAILED";
  case ERROR_REGISTRY_CORRUPT:
    return "CORRUPT";
  case ERROR_BADDB:
    return "BADDB";
  case ERROR_OUTOFMEMORY:
    return "OUTOFMEM";
  case ERROR_ACCESS_DENIED:
    return "ACCDENIED";
  case ERROR_FILE_NOT_FOUND:
    return "NOTFOUND";
  case ERROR_NO_MORE_ITEMS:
    return "NOMORE";
  case ERROR_MORE_DATA:
    return "MOREDATA";
  default:
    sprintf(errstring, "%x\n", retval );
    return errstring;
  }
}


//----------------------------------------------------------------------
//
// GetProcess
//
// Retrieves the process name.
//
//----------------------------------------------------------------------
PCHAR GetProcess( PCHAR ProcessName )
{
  PVOID       CurProc;
  PVOID       ring3proc;
  char        *name;

  // get the ring0 process pointer
  CurProc = VWIN32_GetCurrentProcessHandle();
  
  // now, map the ring3 PCB 
  ring3proc = (PVOID) SelectorMapFlat( Get_Sys_VM_Handle(), 
			       (DWORD) (*(PDWORD) ((char *) CurProc + 0x38)) | 0x7, 0 );

  if( ring3proc == (PVOID) -1 ) {
    strcpy( ProcessName, "???");
    return ProcessName;
  }

  // copy out the process name (max 8 characters)
  name = ((char *)ring3proc) + 0xF2;
  if( name[0] >= 'A' && name[0] < 'z' ) {
    strcpy( ProcessName, name );
    ProcessName[8] = 0;
  } else
    strcpy( ProcessName, "???" );
  return ProcessName;
}


//----------------------------------------------------------------------
//
// GetFullName
//
// Returns the full pathname of a key, if we can obtain one, else
// returns a handle.
//
//----------------------------------------------------------------------
void GetFullName( HKEY hKey, PCHAR lpszSubKey, PCHAR lpszValue, 
		 PCHAR fullname )
{
  PHASH_ENTRY		hashEntry;
  CHAR                  tmpkey[16];

  // see if we find the key in the hash table
  fullname[0] = 0;
  MUTEX_P( HashMutex );
  hashEntry = HashTable[ HASHOBJECT( hKey ) ];
  while( hashEntry && hashEntry->hkey != hKey )
    hashEntry = hashEntry->Next;
  MUTEX_V( HashMutex );
  if( hashEntry ) {
    strcpy( fullname, hashEntry->FullName );
  } else {
    // okay, make a name
    switch( hKey ) {
    case HKEY_CLASSES_ROOT:
      strcat(fullname, "ROOT");
      break;
    case HKEY_CURRENT_USER:
      strcat(fullname, "CURRENT");
      break;
    case HKEY_LOCAL_MACHINE:
      strcat(fullname, "LOCAL");
      break;
    case HKEY_USERS:
      strcat(fullname, "USERS");
      break;
    case HKEY_CURRENT_CONFIG:
      strcat(fullname, "CONFIG");
      break;
    case HKEY_DYN_DATA:
      strcat(fullname, "DYNDATA");
      break;
    default:
      // we will only get here if key was created before we loaded
      sprintf( tmpkey, "0x%X", hKey );					
      strcat(fullname, tmpkey );
      break;
    }
  }

  // append subkey and value, if they are there
  if( lpszSubKey ) {
    if( lpszSubKey[0] ) {
      strcat( fullname, "\\" );
      strcat( fullname, lpszSubKey );
    }
  }
  if( lpszValue ) {
    if( lpszValue[0] ) {
      strcat( fullname, "\\" );
      strcat( fullname, lpszValue );
    }
  }
}

//----------------------------------------------------------------------
// REGISTRY HOOKS
//
// All these hooks do essentially the same thing: dump API-specific
// info into the data buffer, call the original handler, and then
// dump any return information.
//----------------------------------------------------------------------
LONG HookRegOpenKey( HKEY hkey, PCHAR lpszSubKey, PHKEY phkResult) {
  LONG     retval;
  CHAR     fullname[256], data[32], process[16];

  GetFullName( hkey, lpszSubKey, NULL, fullname );
  retval = RealRegOpenKey( hkey, lpszSubKey, phkResult );
  data[0] = 0;
  if( retval == ERROR_SUCCESS ) {
    RegmonFreeHashEntry( *phkResult );
    RegmonStoreHash( *phkResult, fullname );
    sprintf(data,"hKey: 0x%X", *phkResult );
  }  
  UpdateStore( Sequence++, "%s\tOpenKey\t%s\t%s\t%s", 
	       GetProcess(process), fullname, data,
	       ErrorString( retval ));
  return retval;
}

LONG HookRegCreateKey( HKEY hkey, PCHAR lpszSubKey, PHKEY phkResult) {
  LONG      retval;
  CHAR      fullname[256], data[32], process[16];

  GetFullName( hkey, lpszSubKey, NULL, fullname );
  retval = RealRegCreateKey( hkey, lpszSubKey, phkResult );
  data[0] = 0;
  if( retval == ERROR_SUCCESS ) {
    RegmonFreeHashEntry( *phkResult );
    RegmonStoreHash( *phkResult, fullname );
    sprintf(data,"hKey: 0x%X", *phkResult );
  }
  UpdateStore( Sequence++, "%s\tCreateKey\t%s\t%s\t%s", 
	       GetProcess( process), fullname, data,
	       ErrorString( retval ));
  return retval;
}

LONG HookRegDeleteKey( HKEY hkey, PCHAR lpszSubKey ) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hkey, lpszSubKey, NULL, fullname );  
  retval = RealRegDeleteKey( hkey, lpszSubKey );
  UpdateStore( Sequence++, "%s\tDeleteKey\t%s\t\t%s", 
	       GetProcess(process), fullname, 
	       ErrorString( retval ));
  return retval;
}

LONG HookRegDeleteValue( HKEY hkey, PCHAR lpszSubKey ) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hkey, lpszSubKey, NULL, fullname );  
  retval = RealRegDeleteValue( hkey, lpszSubKey );
  UpdateStore( Sequence++, "%s\tDeleteValue\t%s\t\t%s", 
	       GetProcess(process), fullname, 
	       ErrorString( retval ));
  return retval;
}

LONG HookRegCloseKey(HKEY hkey) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hkey, NULL, NULL, fullname );  
  retval = RealRegCloseKey( hkey );
  UpdateStore( Sequence++, "%s\tCloseKey\t%s\t\t%s", 
	       GetProcess(process), fullname, 
	       ErrorString( retval ));
  RegmonFreeHashEntry( hkey );
  return retval;
}

LONG HookRegFlushKey( HKEY hkey ) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hkey, NULL, NULL, fullname );  
  retval = RealRegFlushKey( hkey );
  UpdateStore( Sequence++, "%s\tFlushKey\t%s\t\t%s", 
	       GetProcess(process), fullname, 
	      ErrorString( retval ));
  return retval;
}

LONG HookRegEnumKey(HKEY hkey, DWORD iSubKey, PCHAR lpszName, DWORD cchName) {
  LONG      retval;
  CHAR      fullname[256], process[16];
  
  GetFullName( hkey, NULL, NULL, fullname );
  retval = RealRegEnumKey( hkey, iSubKey, lpszName, cchName );
  
  UpdateStore( Sequence++, "%s\tEnumKey\t%s\t%s\t%s", 
	       GetProcess(process), fullname, 
	       retval == ERROR_SUCCESS ? lpszName : "", 
	       ErrorString( retval ));  
  return retval;
}

LONG HookRegEnumValue(HKEY hkey, DWORD iValue, PCHAR lpszValue, 
		      PDWORD lpcchValue, PDWORD lpdwReserved, 
		      PDWORD lpdwType, PBYTE lpbData, PDWORD lpcbData) {
  LONG      retval;
  int       i, len;
  CHAR      fullname[256], data[256], tmp[16], process[16];

  GetFullName( hkey, NULL, NULL, fullname );
  retval = RealRegEnumValue( hkey, iValue, lpszValue, lpcchValue,
			     lpdwReserved, lpdwType, lpbData, lpcbData );
  data[0] = 0;
  if( retval == ERROR_SUCCESS && lpbData && *lpcbData ) {
    strcat( data, lpszValue );
    strcat( data, ": ");
    if( *lpdwType == REG_SZ ) {
      strcat( data, "\"");
      strcat( data+1, lpbData );
      strcat( data, "\"");
    } else if( *lpdwType == REG_BINARY ) {
      if( *lpcbData > 8 ) len = 8;
      else len = *lpcbData;
      for( i = 0; i < len; i++ ) {
	sprintf( tmp, "%X ", lpbData[i]);
	strcat( data, tmp );
      }
      if( *lpcbData > 8) strcat( data, "...");
    } else if( *lpdwType == REG_DWORD ) {
      sprintf( tmp, "0x%X", *(PDWORD) lpbData );
      strcat( data, tmp );
    }
  }
  UpdateStore( Sequence++, "%s\tEnumValue\t%s\t%s\t%s", 
	       GetProcess(process), fullname, 
	       data, ErrorString( retval ));  
  return retval;
}

LONG HookRegQueryInfoKey( HKEY hKey, PCHAR lpszClass, PDWORD lpcchClass,  
			 DWORD lpdwReserved, PDWORD lpcSubKeys, 
			 PDWORD lpcchMaxSubKey, PDWORD  pcchMaxClass, 
			 PDWORD lpcValues, PDWORD lpcchMaxValueName, 
			 PDWORD lpcbMaxValueData, 
			 PDWORD lpcbSecurityDescriptor, 
			 PFILETIME lpftLastWriteTime) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hKey, NULL, NULL, fullname );
  retval = RealRegQueryInfoKey( hKey, lpszClass, lpcchClass, lpdwReserved, 
				lpcSubKeys, lpcchMaxSubKey, pcchMaxClass,
				lpcValues, lpcchMaxValueName,
				lpcbMaxValueData,
				lpcbSecurityDescriptor,
				lpftLastWriteTime );
  UpdateStore( Sequence++, "%s\tQueryKey\t%s\t\t%s", 
	       GetProcess(process), fullname, 
	       ErrorString( retval ));  
  return retval;
}

LONG HookRegQueryValue( HKEY hkey, PCHAR lpszSubKey, PCHAR lpszValue, 
		       PLONG pcbValue ) {
  LONG      retval;
  CHAR      fullname[256], data[512], process[16];

  GetFullName( hkey, lpszSubKey, NULL, fullname );
  retval = RealRegQueryValue( hkey, lpszSubKey, lpszValue, pcbValue );
  data[0] = 0;
  if( retval == ERROR_SUCCESS && lpszValue && *pcbValue ) {
    strcpy( data, "\"");
    strncat( data, lpszValue, 256 );
    strcat( data, "\"");
  }
  UpdateStore( Sequence++, "%s\tQueryValue\t%s\t%s\t%s", 
	       GetProcess(process), fullname, data,
	      ErrorString( retval ));
  return retval;
}

LONG HookRegQueryValueEx( HKEY hkey, PCHAR lpszValueName, 
			 PDWORD lpdwReserved, PDWORD lpdwType, 
			 PBYTE lpbData, PDWORD lpcbData ) {
  LONG      retval;
  int       i, len;
  CHAR      fullname[256], data[512], tmp[16], process[16];
  
  GetFullName( hkey, NULL, lpszValueName, fullname );
  retval = RealRegQueryValueEx( hkey, lpszValueName, lpdwReserved, 
				lpdwType, lpbData, lpcbData );
  data[0] = 0;
  if( retval == ERROR_SUCCESS && lpbData ) {
    if( *lpdwType == REG_SZ ) {
      strcpy( data, "\"");
      strncat( data+1, lpbData, 256 );
      strcat( data, "\"");
    } else if( *lpdwType == REG_BINARY ) {
      if( *lpcbData > 8 ) len = 8;
      else len = *lpcbData;
      for( i = 0; i < len; i++ ) {
	sprintf( tmp, "%X ", lpbData[i]);
	strcat( data, tmp );
      }
      if( *lpcbData > 8) strcat( data, "...");
    } else if( *lpdwType == REG_DWORD )
      sprintf( data, "0x%X", *(PDWORD) lpbData );
  } 
  UpdateStore( Sequence++, "%s\tQueryValueEx\t%s\t%s\t%s", 
	       GetProcess(process), fullname, data,
	       ErrorString( retval ));  
  return retval;
}

LONG HookRegSetValue( HKEY hkey, PCHAR lpszSubKey, DWORD fdwType, 
		      PCHAR lpszData, DWORD cbData ) {
  LONG      retval;
  CHAR      fullname[256], data[512], process[16];
  
  GetFullName( hkey, lpszSubKey, NULL, fullname );
  retval = RealRegSetValue( hkey, lpszSubKey, fdwType, lpszData, cbData );
  data[0] = 0;
  if( lpszData ) {
    strcpy( data,"\"");
    strncat(data, lpszData, 256 );
    strcat( data, "\"");
  }
  UpdateStore( Sequence++, "%s\tSetValue\t%s\t%s\t%s", 
	       GetProcess(process), fullname, data,
	       ErrorString( retval ));  
  return retval;
}

LONG HookRegSetValueEx( HKEY hkey, PCHAR lpszValueName, 
			 DWORD lpdwReserved, DWORD fdwType, 
			 PBYTE lpbData, DWORD lpcbData ) {
  LONG      retval;
  int       i, len;
  CHAR      fullname[256], data[256], tmp[16], process[16];
  
  GetFullName( hkey, NULL, lpszValueName, fullname );
  retval = RealRegSetValueEx( hkey, lpszValueName, lpdwReserved, 
			      fdwType, lpbData, lpcbData );
  data[0] = 0;
  if( fdwType == REG_SZ ) {
    strcpy( data, "\"");
    strcat( data+1, lpbData );
    strcat( data, "\"");
  } else if( fdwType == REG_BINARY ) {
    if( lpcbData > 8 ) len = 8;
    else len = lpcbData;
    for( i = 0; i < len; i++ ) {
      sprintf( tmp, "%X ", lpbData[i]);
      strcat( data, tmp );
    }
    if( lpcbData > 8) strcat( data, "...");
  } else if( fdwType == REG_DWORD ) 
    sprintf( data, "0x%X", *(PDWORD) lpbData );
  UpdateStore( Sequence++, "%s\tSetValueEx\t%s\t%s\t%s", 
	       GetProcess(process), fullname, data,
	       ErrorString( retval ));  
  return retval;
}

LONG HookRegRemapPreDefKey( HKEY hkey, HKEY hRootKey ) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hkey, NULL, NULL, fullname );
  retval = RealRegRemapPreDefKey( hkey, hRootKey );
  UpdateStore( Sequence++, "%s\tRemapPreKey\t%s\tRoot: %s\t%s", 
	       GetProcess(process), fullname, 
	       hRootKey == HKEY_CURRENT_USER ? "CURRENT_USER" : 
	       "CURRENT_CONFIG", ErrorString( retval ));  
  return retval;
}

LONG HookRegQueryMultipleValues( HKEY hKey, PVALENT pValent, 
				DWORD dwNumVals, PCHAR pValueBuf, 
				PDWORD pTotSize ) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  GetFullName( hKey, NULL, NULL, fullname );
  retval = RealRegQueryMultipleValues( hKey, pValent,
				       dwNumVals, pValueBuf, pTotSize );
  UpdateStore( Sequence++, "%s\tQueryMultVal\t%s\t\t%s", 
	       GetProcess(process), fullname, 
	       ErrorString( retval ));  
  return retval;
}

LONG HookRegCreateDynKey( PCHAR szName, PVOID KeyContext, 
			  PVOID pInfo, PVOID pValList, 
			  DWORD dwNumVals, PVMMHKEY pKeyHandle ) {
  LONG      retval;
  CHAR      fullname[256], process[16];

  sprintf(fullname, "DYNDAT\\%s", szName );
  retval = RealRegCreateDynKey( szName, KeyContext, pInfo, pValList,
				dwNumVals, pKeyHandle );
  UpdateStore( Sequence++, "%s\tCreateDynKey\t%s\thKey: 0x%X\t%s", 
	       GetProcess(process), fullname,
	       *pKeyHandle, ErrorString( retval ));  
  if( retval == ERROR_SUCCESS )
    RegmonStoreHash( *pKeyHandle, fullname );
  return retval;
}

//----------------------------------------------------------------------
//
// OnSysDynamicDeviceInit
//
// Dynamic init. Hook all registry related VxD APIs.
//
//----------------------------------------------------------------------
BOOL OnSysDynamicDeviceInit()
{
  int i;

  // initialize mutex
  MUTEX_INIT( StoreMutex );
  MUTEX_INIT( HashMutex );

  // zero hash table
  for(i = 0; i < NUMHASH; i++ ) HashTable[i] = NULL;

  // allocate two buffers
  PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
	       (PMEMHANDLE) &Store, (PVOID) &Store );
  PageAllocate(STORESIZE, PG_SYS, 0, 0, 0, 0, NULL, PAGELOCKED, 
	       (PMEMHANDLE) &StoreExtra, (PVOID) &StoreExtra );
  Store->Len = StoreExtra->Len   = 0;
  Store->Next = StoreExtra->Next = NULL;
  NumStore = 2;

  // hook all registry routines
  RealRegOpenKey = Hook_Device_Service_C(___RegOpenKey, HookRegOpenKey,
					 &ROKThunk );
  RealRegCloseKey = Hook_Device_Service_C(___RegCloseKey, HookRegCloseKey,
					  &RCKThunk );
  RealRegCreateKey = Hook_Device_Service_C(___RegCreateKey, HookRegCreateKey,
					  &RCRKThunk );
  RealRegDeleteKey = Hook_Device_Service_C(___RegDeleteKey, HookRegDeleteKey,
					  &RDKThunk );
  RealRegDeleteValue = Hook_Device_Service_C(___RegDeleteValue, 
					     HookRegDeleteValue,
					     &RDVThunk );
  RealRegEnumKey   = Hook_Device_Service_C(___RegEnumKey,
					     HookRegEnumKey,
					     &REKThunk );
  RealRegEnumValue = Hook_Device_Service_C(___RegEnumValue, 
					     HookRegEnumValue,
					     &REVThunk );
  RealRegFlushKey  = Hook_Device_Service_C(___RegFlushKey,
					     HookRegFlushKey,
					     &RFKThunk );
  RealRegQueryInfoKey  = Hook_Device_Service_C(___RegQueryInfoKey,
					     HookRegQueryInfoKey,
					     &RQIKThunk );
  RealRegQueryValue = Hook_Device_Service_C(___RegQueryValue,
					     HookRegQueryValue,
					     &RQVThunk );
  RealRegQueryValueEx = Hook_Device_Service_C(___RegQueryValueEx,
					     HookRegQueryValueEx,
					     &RQVEThunk );
  RealRegSetValue = Hook_Device_Service_C(___RegSetValue,
					     HookRegSetValue,
					     &RSVThunk );
  RealRegSetValueEx = Hook_Device_Service_C(___RegSetValueEx,
					     HookRegSetValueEx,
					     &RSVEThunk );
  RealRegRemapPreDefKey = Hook_Device_Service_C(___RegRemapPreDefKey,
					     HookRegRemapPreDefKey,
					     &RRPDKThunk );
  RealRegQueryMultipleValues = Hook_Device_Service_C(___RegQueryMultipleValues,
					     HookRegQueryMultipleValues,
					     &RQMVThunk );
  RealRegCreateDynKey = Hook_Device_Service_C(___RegCreateDynKey,
					     HookRegCreateDynKey,
					     &RCDKThunk );
  return TRUE;
}

//----------------------------------------------------------------------
//
// OnSysDynamicDeviceExit
//
// Dynamic exit. Unhook everything.
//
//----------------------------------------------------------------------
BOOL OnSysDynamicDeviceExit()
{
  Unhook_Device_Service_C(___RegOpenKey, &ROKThunk );
  Unhook_Device_Service_C(___RegCloseKey, &RCKThunk );
  Unhook_Device_Service_C(___RegCreateKey, &RCRKThunk );
  Unhook_Device_Service_C(___RegDeleteKey, &RDKThunk );
  Unhook_Device_Service_C(___RegDeleteValue, &RDVThunk );
  Unhook_Device_Service_C(___RegEnumKey, &REKThunk );
  Unhook_Device_Service_C(___RegEnumValue, &REVThunk );
  Unhook_Device_Service_C(___RegFlushKey, &RFKThunk );
  Unhook_Device_Service_C(___RegQueryInfoKey, &RQIKThunk );
  Unhook_Device_Service_C(___RegQueryValue, &RQVThunk );
  Unhook_Device_Service_C(___RegQueryValueEx, &RQVEThunk );
  Unhook_Device_Service_C(___RegSetValue, &RSVThunk );
  Unhook_Device_Service_C(___RegSetValueEx, &RSVEThunk );
  Unhook_Device_Service_C(___RegRemapPreDefKey, &RRPDKThunk );
  Unhook_Device_Service_C(___RegQueryMultipleValues, &RQMVThunk );
  Unhook_Device_Service_C(___RegCreateDynKey, &RCDKThunk );

  // free all memory
  RegmonHashCleanup();
  RegmonFreeStore();
  return TRUE;
}

//----------------------------------------------------------------------
//
// OnW32Deviceiocontrol
//
// Interface with the GUI.
//
//----------------------------------------------------------------------
DWORD OnW32Deviceiocontrol(PIOCTLPARAMS p)
{
  PSTORE_BUF      old;

  switch( p->dioc_IOCtlCode ) {
  case 0:
    return 0;
  case REGMON_zerostats:
    MUTEX_P( StoreMutex );
    while ( Store->Next )  {
      // release next
      old = Store->Next;
      Store->Next = old->Next;
      MUTEX_P( StoreMutex );
      PageFree( (MEMHANDLE) old, 0 );
      MUTEX_V( StoreMutex );
      NumStore--;
    }
    Store->Len = 0;
    MUTEX_V( StoreMutex );
    return 0;

  case REGMON_getstats:
    // Copy buffer into user space.
    MUTEX_P( StoreMutex );
    if ( MAX_STORE > p->dioc_cbOutBuf ) {
      // Don't do this; its not worth implementing the fix.
      MUTEX_V( StoreMutex );
      return 1;
    } else if ( Store->Len  ||  Store->Next ) {

      // Switch to a new store
      RegmonNewStore();

      // Fetch the oldest to give to user
      old = RegmonOldestStore();
      MUTEX_V( StoreMutex );

      // Copy it
      memcpy( p->dioc_OutBuf, old->Data, old->Len );

      // Return length of copied info
      *p->dioc_bytesret = old->Len;

      // Deallocate buffer
      PageFree( (MEMHANDLE) old, 0 );
      NumStore--;
    } else {
      // no unread data
      MUTEX_V( StoreMutex );
      *p->dioc_bytesret = 0;
    }
    return 0;

  default:
    return 1;
  }
}
