/******************************************************************************
*             Transparent Label Control  11/29/94 by Mark Gamber              *
*                                                                             *
*  This example demonstrates a "transparent" control in the form of moveable  *
*  transparent 3-D label controls displayed over a bitmap on a dialog box.    *
*  The label control is self-contained and displays a dialog box with an edit *
*  control when double-clicked to allow the control's text to be changed.     *
*                                                                             *
*  By using the program or any included part, the user assumes responsibility *
*  for it's use and may not hold the author liable for any loss or damage.    *
*  If unable to accept this restriction, the program and all included parts   *
*  must be destroyed immediately. This program and all included parts is      *
*  public domain and may be freely distributed.                               *
*                                                                             *
******************************************************************************/

#include "windows.h"



#define CLASSNAME "TRANSLBL"         //  Change this here name for your own use

// === Function Prototypes ====================================================

BOOL WINAPI MainDlgProc( HWND, UINT, WPARAM, LPARAM );
BOOL WINAPI InputDlgProc( HWND, UINT, WPARAM, LPARAM );
LRESULT WINAPI LabelWndProc( HWND, UINT, WPARAM, LPARAM );
BOOL PaintDialog( HWND );

// === Global Variables =======================================================

HINSTANCE hInst;

// === Application Entry Point ================================================

int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd,
                      int nShow )
{
   WNDCLASS wc;

   wc.style = CS_DBLCLKS;             //  Register a window class for the label
   wc.lpfnWndProc = LabelWndProc;
   wc.hInstance = hInstance;
   wc.lpszClassName = CLASSNAME;
   wc.lpszMenuName = (LPSTR)NULL;
   wc.hIcon = (HICON)NULL;
   wc.hCursor = LoadCursor( NULL, IDC_ARROW );
   wc.hbrBackground = (HBRUSH)NULL;      //  This must be NULL for transparency
   wc.cbClsExtra = 0;
   wc.cbWndExtra = sizeof(HANDLE);                        //  Font handle space

   if( ! RegisterClass( &wc ) )           //  Base continuation on registration
      return( FALSE );

   hInst = hInstance;               //  If it worked, start the main dialog box
   DialogBox( hInstance, MAKEINTRESOURCE( 10000 ), NULL, MainDlgProc );
   return( FALSE );                                    //  And exit when closed
}                                                          //  End of WinMain()

// === Main Dialog Box Procedure ==============================================

BOOL WINAPI MainDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:                     //  When the dialog box starts...
      {
         char szStr[ 20 ];
         int i;

         for( i = 0; i < 3; i++ )             //  Create three labels with text
         {
            wsprintf( szStr, "Label &%d:", i + 1 );
            CreateWindow( CLASSNAME, szStr, 
                          WS_CHILD | WS_VISIBLE,
                          4 + ( i * 80 ), 4, 65, 18,
                          hDlg, (HMENU)100, hInst, NULL );
         }
                                              //  Size dialog box to the bitmap
         SetWindowPos( hDlg, NULL, 0, 0, 244, 174, 
                       SWP_NOZORDER | SWP_NOMOVE );
         return( TRUE );
      }                                                //  End of WM_INITDIALOG


      case WM_COMMAND:
         if( wParam == IDCANCEL )               //  Close dialog when cancelled
         {
            EndDialog( hDlg, TRUE );
            return( TRUE );
         }
         break;

                                //  Handling this message prevents flicker when
      case WM_ERASEBKGND:     //  redrawing, but the dialog isn't painted where
         return( TRUE );       //  there's no bitmap. Comment this out if blank
                              //  spaces are expected on the dialog box surface

      case WM_PAINT:
         return( PaintDialog( hDlg ) );             //  Paint bitmap background
   }
   return( FALSE );                                    //  End of MainDlgProc()
}

// === Called by WM_PAINT of Main Dialog Box ==================================

BOOL PaintDialog( HWND hDlg )
{
   HDC hDC, mDC;
   PAINTSTRUCT ps;
   HBITMAP hBmp;

   hDC = BeginPaint( hDlg, &ps );                         //  Usual paint stuff
                                                    //  Load the texture bitmap
   hBmp = LoadBitmap( hInst, MAKEINTRESOURCE( 100 ) );
   if( hBmp )
   {
      mDC = CreateCompatibleDC( hDC );                   //  Create a memory DC
      if( mDC )
      {
         SelectObject( mDC, hBmp );       //  Apply the bitmap to the memory DC

         if( ps.rcPaint.right > 239 )    //  Stay within the size of the bitmap
            ps.rcPaint.right = 239;

         if( ps.rcPaint.bottom > 149 )
            ps.rcPaint.bottom = 149;
                               //  Paint bitmap section as specified by rcPaint
         BitBlt( hDC, ps.rcPaint.left, ps.rcPaint.top, 
                 ps.rcPaint.right - ps.rcPaint.left, 
                 ps.rcPaint.bottom - ps.rcPaint.top,
                 mDC, ps.rcPaint.left, ps.rcPaint.top, SRCCOPY );

         DeleteDC( mDC );                           //  Clean up on the way out
      }
      DeleteObject( hBmp );
   }

   EndPaint( hDlg, &ps );                                    // And that's that
   return( TRUE );
}                                                      //  End of PaintDialog()

// === Transparent Label Window Procedure =====================================

LRESULT WINAPI LabelWndProc( HWND hWnd, UINT msg, WPARAM wParam,
                             LPARAM lParam )
{
   switch( msg )
   {
      case WM_CREATE:                 //  Get parent's font and store for later
         SetWindowLong( hWnd, 0, 
                        (DWORD)SendMessage( GetParent( hWnd ), 
                                            WM_GETFONT, 0, 0 ) );
         break;


      case WM_MOVE:                          //  If the label has been moved...
      {
         RECT Rect;
                              //  Get the new position using screen coordinates
         GetWindowRect( hWnd, &Rect );
         ScreenToClient( GetParent( hWnd ),        //  Convert to parent window
                         (LPPOINT)&Rect );                      //  coordinates
                             //  Force the parent to repaint under this control
                                    //  and erase the background in the process
         InvalidateRect( GetParent( hWnd ), &Rect, TRUE );
         InvalidateRect( hWnd, NULL, FALSE );     //  Then force a repaint here
                         //  without erasing the background (tada! transparent)
         break;
      }

      
      case WM_PAINT:                             //  Time to paint this control
      {
         PAINTSTRUCT ps;
         HDC hDC;
         char szText[ 128 ];
         RECT Rect;
         HPEN hPen;


         hDC = BeginPaint( hWnd, &ps );              //  More usual paint stuff
         GetClientRect( hWnd, &Rect );                //  Get position and size
                                                       //  Draw highlight lines
         hPen = CreatePen( PS_SOLID, 1, 
                           GetSysColor( COLOR_BTNHIGHLIGHT ) );
         SelectObject( hDC, hPen );

         MoveToEx( hDC, 0, Rect.bottom, NULL );          //  Left and top sides
         LineTo( hDC, 0, 0 );                                   //  double-wide
         LineTo( hDC, Rect.right, 0 );
         MoveToEx( hDC, 1, Rect.bottom - 1, NULL );
         LineTo( hDC, 1, 1 );
         LineTo( hDC, Rect.right - 1, 1 );
                                                //  Now draw the shadowed edges
         hPen = CreatePen( PS_SOLID, 1, 
                           GetSysColor( COLOR_BTNSHADOW ) );
         DeleteObject( SelectObject( hDC, hPen ) );

         MoveToEx( hDC, 1, Rect.bottom - 1, NULL );        //  Right and bottom
         LineTo( hDC, Rect.right - 1, Rect.bottom - 1 );        //  double-wide
         LineTo( hDC, Rect.right - 1, 0 );
         MoveToEx( hDC, 2, Rect.bottom - 2, NULL );
         LineTo( hDC, Rect.right - 2, Rect.bottom - 2 );
         LineTo( hDC, Rect.right - 2, 1 );

         if( GetWindowText( hWnd, szText, 128 ) )   //  If there's text to show
         {
            SetBkMode( hDC, TRANSPARENT );               //  No text background
                                     //  Use current font and button text color
            SelectObject( hDC, (HFONT)GetWindowLong( hWnd, 0 ) );
            SetTextColor( hDC, GetSysColor( COLOR_BTNTEXT ) );
            Rect.left += 4;                       //  Move text in a little bit
                                          //  Draw it to display any underscore
            DrawText( hDC, szText, lstrlen( szText ), &Rect, 
                      DT_LEFT | DT_VCENTER | DT_SINGLELINE );
         }
         EndPaint( hWnd, &ps );                             //  Clean up and...
         DeleteObject( hPen );
         return( TRUE );
      }                                                  //  ...end of WM_PAINT

      
      case WM_NCHITTEST:           //  This just allows the control to be moved
         return( HTCAPTION );      //  by telling Windows it's a window caption


      case WM_SETFONT:                     //  Save any new fonts sent this way
         SetWindowLong( hWnd, 0, (DWORD)wParam );
         if( lParam )                //  if lParam is non-zero, force a repaint
            SendMessage( hWnd, WM_MOVE, 0, 0 );
         return( TRUE );


      case WM_GETFONT:                       //  Someone wants this font handle
         return( GetWindowLong( hWnd, 0 ) );


      case WM_NCLBUTTONDBLCLK:                  //  Double-click to change text
         if( DialogBoxParam( hInst, MAKEINTRESOURCE( 10001 ), 
                             hWnd, InputDlgProc,     //  Start input dialog and
                             (LPARAM)hWnd ) )  //    pass it this window handle
            PostMessage( hWnd, WM_MOVE, 0, 0 );     //  If OKed, repaint window
         break;

      
      default:
         return( DefWindowProc( hWnd, msg, wParam, lParam ) );
   }
   return( FALSE );
}                                                     //  End of LabelWndProc()

// === Dialog box to allow Label Text Edit ====================================

BOOL WINAPI InputDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:
      {
         RECT Rect;
         char szStr[ 32 ];

                                       //  Get screen position of calling label
         GetWindowRect( (HWND)lParam, &Rect );      //  Set this dialog over it
         SetWindowPos( hDlg, NULL, Rect.left, Rect.top,     //  and size to fit
                       Rect.right - Rect.left, Rect.bottom - Rect.top,
                       SWP_NOZORDER );           //  Position edit control, too
         SetWindowPos( GetDlgItem( hDlg, 100 ), NULL, 0, 0,
                       Rect.right - Rect.left, Rect.bottom - Rect.top,
                       SWP_NOMOVE | SWP_NOZORDER );
         ShowWindow( GetDlgItem( hDlg, 1 ), SW_HIDE );
                                                  //  Save calling label handle
         SetProp( hDlg, MAKEINTATOM( 10000 ), (HANDLE)lParam );
         GetWindowText( (HWND)lParam, szStr, 32 );  //  Put label's text in the
         SetWindowText( GetDlgItem( hDlg, 100 ), szStr ); //  edit control here
         SendDlgItemMessage( hDlg, 100, EM_LIMITTEXT, 32, 0 );
         return( TRUE );
      }                                                //  End of WM_INITDIALOG


      case WM_COMMAND:
         if( wParam == IDCANCEL )     //  If cancelled, close with FALSE status
         {                             //  to prevent the label from repainting
            RemoveProp( hDlg, MAKEINTATOM( 10000 ) );               //  in vain
            EndDialog( hDlg, FALSE );
            return( TRUE );                             //  In vain, I tell you
         }
         if( wParam == IDOK )   //  But if it's OKed...a different story ensues
         {
            char szStr[ 32 ];
                                                //  Get the edit control's text
            GetDlgItemText( hDlg, 100, szStr, 32 );   //  and stuff it into the
            SetWindowText( GetProp( hDlg,           //  calling label's display
                           MAKEINTATOM( 10000 ) ),
                           szStr );
            RemoveProp( hDlg, MAKEINTATOM( 10000 ) );
            EndDialog( hDlg, TRUE ); //  Exit with TRUE status to repaint label
            return( TRUE );
         }
         break;
   }
   return( FALSE );
}                                                     //  End of InputDlgProc()

// === End of Application =====================================================
