/*
 *  This file forms part of "TKERN" - "Troy's Kernel for Windows".
 *
 *  Copyright (C) 1994  Troy Rollo <troy@cbme.unsw.EDU.AU>
 *
 *  This file is explicitly placed in the public domain. You may use it
 *  it for any purpose you see fit, including, but not limited to,
 *  incorperating it into a private, commercial, public domain,
 *  shareware, freeware or free software work.
 */


#include <windows.h>
#include <stdarg.h>
#include <dos.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/tkexec.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdlib.h>

CATCHBUF	__intr_catch_buf;

int	_creattype = O_TEXT;

extern	int	tkern_valid_file(int);
extern	int	tkern_errno(void);
extern	void	_init_all_files(void);
extern	void	_close_all_files(void);
extern	void	tkern_register_program(	int	*argc,
					char	***argv,
					char	***envp);
extern	int	tkern_close(int);
extern	int	tkern_open(char const *, int, ...);
extern	int	tkern_dup(int);
extern	int	tkern_dup2(int, int);
extern	int	tkern_read(int, char *, int);
extern	int	tkern_write(int, char const *, int);
extern	long	tkern_seek(int, long, int);
extern	int	tkern_isatty(int);
extern	int	tkern_ioctl(int, long, void *);
extern	void	tkern_register_sighandler(void far pascal (*_proc)(int));
extern	int	tkern_pipe(int *pfd);

/* _exitbuf is a global which stores the location of the file flushing
 * routine called by exit().
 */
extern	void	(*_exitbuf)(void);

/*
 * The char * const * below should really be 
 * char const * const *, but BC doesn't seem
 * to like it.
 */
extern	int	tkern_exec(	char const *,
				void *,
				char * const *,
				char const *);
extern short	tkern_wait(	union wait *wstatus);
extern short	tkern_wait3(	union wait *wstatus,
				int	nFlags,
				struct rusage *pZero);
extern short	tkern_waitpid(	int	pid,
				union wait *wstatus,
				int	nFlags);
extern int	tkern_kill(	int	pid,
				int	nSignal);

extern	int	main(int, char **, char **);

CATCHBUF	fork_buf;
extern	int	errno;

int	open(char const *pchPath, int nMode, int nAccess)
{
	int	rval;

	rval = tkern_open(pchPath, nMode, nAccess);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int	creat(char const *pchPath, int nAccess)
{
	return open(pchPath, O_WRONLY | O_CREAT | O_TRUNC | _creattype, nAccess);
}

int	close(int fd)
{
	int	rval;

	rval = tkern_close(fd);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}


int	read(int fd, char *pchBuffer, int nBytes)
{
	int	rval;

	rval = tkern_read(fd, pchBuffer, nBytes);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int	write(int fd, char *pchBuffer, int nBytes)
{
	int	rval;

	rval = tkern_write(fd, pchBuffer, nBytes);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int	dup(int fd)
{
	int	rval;

	rval = tkern_dup(fd);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int	dup2(int fd1, int fd2)
{
	int	rval;

	rval = tkern_dup2(fd1, fd2);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int	pipe(int *pfd)
{
	int	rval;

	rval = tkern_pipe(pfd);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

long	lseek(int fd, long nPosition, int nStart)
{
	long	rval;

	rval = tkern_seek(fd, nPosition, nStart);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int	isatty(int fd)
{
	int	rval;

	rval = tkern_isatty(fd);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int
ioctl(int fd, long nIOCtl, void *pvBuffer)
{
	int	rval;

	rval = tkern_ioctl(fd, nIOCtl, pvBuffer);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

static	void
doexec(	char	const *pchPath,
	void	const *pvArgs,
	char	* const *ppchEnv,
	char	const *pchCmdLine)
{
	int	iTask;

	iTask = tkern_exec(pchPath, pvArgs, ppchEnv, pchCmdLine);
	if (iTask == -1)
		errno = tkern_errno();
	Throw(fork_buf, iTask);
}

void
execl(	char const *pchPath, ...)
{
	va_list	args;

	va_start(args, pchPath);
	doexec(pchPath, args, environ, 0);
}

void
execl_ext(	char const *pchCmdLine,
		char const *pchPath,
		...)
{
	va_list	args;

	va_start(args, pchPath);
	doexec(pchPath, args, environ, pchCmdLine);
}

void
execv(	char	const	*pchPath,
	void	const	*pvArgs)
{

	doexec(pchPath, pvArgs, environ, 0);
}

void
execv_ext(	char	const	*pchCmdLine,
		char	const	*pchPath,
		void	const	*pvArgs)
{
	doexec(pchPath, pvArgs, environ, pchCmdLine);
}

short
wait(union wait *wstatus)
{
	short	rval;

	rval = tkern_wait(wstatus);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

short
wait3(	union wait	*wstatus,
	int		nFlags,
	struct rusage	*pZero)
{
	short	rval;

	rval = tkern_wait3(wstatus, nFlags, pZero);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

short
waitpid(int	pid,
	long	*wstatus,
	int	nFlags)
{
	short	rval;

	rval = tkern_waitpid(pid, (union wait *) wstatus, nFlags);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

int
kill(	int	pid,
	int	nSignal)
{
	int	rval;

	rval = tkern_kill(pid, nSignal);
	if (rval == -1)
		errno = tkern_errno();
	return rval;
}

void	far	pascal
InterruptMe(	int	nSignal)
{
	typedef	void (*sigfunc)(int);
	sigfunc cp;

	_DS = _SS;
	switch(nSignal)
	{
	case SIGINT:
		cp = signal(SIGINT, SIG_IGN);
		if (cp == SIG_IGN)
			return;
		if (cp == SIG_DFL)
			exit(-1);
		(*cp)(2);
		break;

	default:
		raise(nSignal);
	}
}

#pragma argsused
int	far		pascal
WinMain(HINSTANCE	hInstance,
	HINSTANCE	hPrec,
	LPSTR		lpCmdLine,
	int		nShow)
{
	int	argc;
	char	**argv;
	char	**envp;
	int	rval;

	tkern_register_program(&argc, &argv, &envp);
	tkern_register_sighandler(InterruptMe);
	if (!tkern_valid_file(0) ||
	    !tkern_valid_file(1) ||
	    !tkern_valid_file(2))
	{
		if (tkern_valid_file(0))
			tkern_close(0);
		if (tkern_valid_file(1))
			tkern_close(1);
		if (tkern_valid_file(2))
			tkern_close(2);
		tkern_open("/dev/window/console", O_RDWR);
		tkern_dup(0);
		tkern_dup(0);
	}
	_init_all_files();
	fdopen(0, "r");
	fdopen(1, "w");
	fdopen(2, "w");
	_exitbuf = _close_all_files;
	switch(Catch(__intr_catch_buf))
	{
	case 0:
		rval = main(argc, argv, envp);
		break;
	}
	exit(rval);
	return -1;
}
