/******************************************************************************
*            Vertical Tab Control example 11/25/94 by Mark Gamber             *
*                                                                             *
*     This example demonstrates how to create and maintain a vertical "tab    *
*  control" using the ever popular ownerdraw listbox. This example displays   *
*  only text, but adding a bitmap or structure is simple as the listbox item  *
*  data is not used in this example.                                          *
*                                                                             *
*     This program and all included parts are public domain and may be freely *
*  distributed. In using the program or any included parts, the user assumes  *
*  full responsibility for it's use and may not hold the author liable for    *
*  any loss or damage. If unable to accept this restriction, the program and  *
*  all included parts must be destroyed immediately.                          *
******************************************************************************/

#include "windows.h"

// === Function Prototypes ====================================================

BOOL WINAPI MainDlgProc( HWND, UINT, WPARAM, LPARAM );
BOOL PaintVTabCtrl( HWND, LPDRAWITEMSTRUCT );
BOOL PaintTabEdge( HWND );
BOOL PaintTabBody( HWND, BOOL );

// === Global Variables =======================================================

HINSTANCE hInst;

// === Application Entry Point ================================================

int APIENTRY WinMain( HINSTANCE hInstance, HINSTANCE hPrev, LPSTR lpCmd,
                      int nShow )
{
   hInst = hInstance;                     //  Save the instance in a global and
   DialogBox( hInstance, MAKEINTRESOURCE( 10000 ),   //  Fire up the dialog box
              NULL, MainDlgProc );
   return( FALSE );                      //  Exit when the dialog box is closed
}

// === Main Window Procedure (Dialog box) =====================================

BOOL WINAPI MainDlgProc( HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:                        //  When the dialog box starts
      {
         int i;
         char szStr[ 32 ];
                                     //  Create a brush to color the background
         SetProp( hDlg, MAKEINTATOM( 10000 ),    //  the listbox and dialog box
                  CreateSolidBrush( GetSysColor( COLOR_BTNFACE ) ) );
                                          //  Save as a prop for later deleting

         for( i = 0; i < 7; i++ )              //  Add something to the listbox
         {
            wsprintf( szStr, "Item #%d", i + 1 );
            SendDlgItemMessage( hDlg, 100, LB_ADDSTRING, 0, (LPARAM)szStr );
         }
                                                      //  Select the first item
         SendDlgItemMessage( hDlg, 100, LB_SETCURSEL, 0, 0 );
         InvalidateRect( hDlg, NULL, FALSE );
         return( TRUE );                                   //  And that is that
      }                                                //  End of WM_INITDIALOG


      case WM_MEASUREITEM:      //  Windows needs to know the list element size
      {
         LPMEASUREITEMSTRUCT lpMs = (LPMEASUREITEMSTRUCT)lParam;
         RECT Rect;
                                                //  We'll use the current width
         GetClientRect( GetDlgItem( hDlg, 100 ), &Rect );
         lpMs->itemWidth = Rect.right;
         lpMs->itemHeight = 32;        //  And some arbitrary number for height
         return( TRUE );
      }                                               //  End of WM_MEASUREITEM


      case WM_DRAWITEM:                     //  Ownerdraw listbox needs painted
         return( PaintVTabCtrl( hDlg, (LPDRAWITEMSTRUCT)lParam ) );

   
      case WM_CTLCOLORDLG:          //  Set background color for dialog box and
      case WM_CTLCOLORLISTBOX:     //  listbox to the current button face color
      case WM_CTLCOLORSTATIC:              //  Don't forget the static control!
         SetBkColor( (HDC)wParam, GetSysColor( COLOR_BTNFACE ) );
         return( (BOOL)GetProp( hDlg, MAKEINTATOM( 10000 ) ) );
               

      case WM_PAINT:
         return( PaintTabBody( hDlg, TRUE ) );


      case WM_COMMAND:
      {                                             //  If OK or Cancel pressed
         if( wParam == IDOK || wParam == IDCANCEL )
         {                              //  Delete the brush from WM_INITDIALOG
            DeleteObject( GetProp( hDlg, MAKEINTATOM( 10000 ) ) );
            RemoveProp( hDlg, MAKEINTATOM( 10000 ) );         //  Kill the prop

            EndDialog( hDlg, TRUE );      //  And kill the dialog (and the app)
            return( TRUE );
         }

         if( HIWORD( wParam ) == LBN_SELCHANGE )
         {
            char szStr[ 32 ];
            
            PaintTabEdge( hDlg );               //  Reflect listbox item change
            
            wsprintf( szStr, "Item %d selected",   //  Display latest selection
                      SendDlgItemMessage( hDlg, 100, LB_GETCURSEL, 0, 0 ) );
            SetDlgItemText( hDlg, 101, szStr );
            break;
         }
         break;
      }                                                   //  End of WM_COMMAND
   }
   return( FALSE );
}                                                      //  End of MainDlgProc()

// === Draw Listbox in response to WM_DRAWITEM ================================

BOOL PaintVTabCtrl( HWND hWnd, LPDRAWITEMSTRUCT lpDs )
{
   HPEN hPen;
   HFONT hFont;
   HBRUSH hBrush;
   BOOL bSelected;
   char szStr[ 32 ];
   POINT pt[ 6 ];

                                      //  Don't bother with non-items and focus
   if( lpDs->itemID < 0 || lpDs->itemAction & ODA_FOCUS )
      return( TRUE );

   if( lpDs->itemState & ODS_SELECTED )        //  Flag if the item is selected
      bSelected = 1;
   else
      bSelected = 0;
                                //  Create brush using the current button color
   hBrush = CreateSolidBrush( GetSysColor( COLOR_BTNFACE ) );
                                             //  Add one line to rect for erase
   ++lpDs->rcItem.bottom;         //  to cover a black border when non-selected
   FillRect( lpDs->hDC, &lpDs->rcItem, hBrush );           //  Erase background
   --lpDs->rcItem.bottom;                    //  Take line off for line drawing

   DeleteObject( hBrush );                              //  Done with the brush

   hPen = CreatePen( PS_SOLID, 1, RGB( 0, 0, 0 ) );
   hPen = SelectObject( lpDs->hDC, hPen );               //  Create a black pen
                                     //  Initialize POINT structures for border
   pt[ 0 ].x = lpDs->rcItem.right;        pt[ 0 ].y = lpDs->rcItem.bottom;
   pt[ 1 ].x = lpDs->rcItem.left + 3;     pt[ 1 ].y = lpDs->rcItem.bottom;
   pt[ 2 ].x = lpDs->rcItem.left;         pt[ 2 ].y = lpDs->rcItem.bottom - 3;
   pt[ 3 ].x = lpDs->rcItem.left;         pt[ 3 ].y = lpDs->rcItem.top + 3;
   pt[ 4 ].x = lpDs->rcItem.left + 3;     pt[ 4 ].y = lpDs->rcItem.top;
   pt[ 5 ].x = lpDs->rcItem.right;        pt[ 5 ].y = lpDs->rcItem.top;

   if( ! bSelected )               //  If not selected, push it in a few pixels
   {
      pt[ 1 ].x += 8;
      pt[ 2 ].x += 8;
      pt[ 3 ].x += 8;
      pt[ 4 ].x += 8;
   }
   Polyline( lpDs->hDC, pt, 6 );        //  Draw the border of the listbox item
                               //  Delete last pen while creating highlight pen
   DeleteObject( SelectObject( lpDs->hDC, CreatePen( PS_SOLID, 1,
                 GetSysColor( COLOR_BTNHIGHLIGHT ) ) ) );
                                        //  Initialize hilight POINT structures
   pt[ 0 ].x = lpDs->rcItem.left + 1;     pt[ 0 ].y = lpDs->rcItem.bottom - 3;
   pt[ 1 ].x = lpDs->rcItem.left + 1;     pt[ 1 ].y = lpDs->rcItem.top + 3;
   pt[ 2 ].x = lpDs->rcItem.left + 3;     pt[ 2 ].y = lpDs->rcItem.top + 1;
   pt[ 3 ].x = lpDs->rcItem.right;        pt[ 3 ].y = lpDs->rcItem.top + 1;

   if( ! bSelected )               //  If not selected, push it in a few pixels
   {
      pt[ 0 ].x += 8;
      pt[ 1 ].x += 8;
      pt[ 2 ].x += 8;
   }
   Polyline( lpDs->hDC, pt, 4 );                         //  Draw the highlight

   if( bSelected )            //  If the item is selected, draw highlight again
   {                                            //  so it stands out a bit more
      pt[ 0 ].x = lpDs->rcItem.left + 2;  pt[ 0 ].y = lpDs->rcItem.bottom - 2;
      pt[ 1 ].x = lpDs->rcItem.left + 2;  pt[ 1 ].y = lpDs->rcItem.top + 3;
      pt[ 2 ].x = lpDs->rcItem.left + 3;  pt[ 2 ].y = lpDs->rcItem.top + 2;
      pt[ 3 ].x = lpDs->rcItem.right;     pt[ 3 ].y = lpDs->rcItem.top + 2;

      Polyline( lpDs->hDC, pt, 4 );
   }
                               //  Delete hilight pen while creating shadow pen
   DeleteObject( SelectObject( lpDs->hDC, CreatePen( PS_SOLID, 1,
                 GetSysColor( COLOR_BTNSHADOW ) ) ) );
                                   //  Initialize POINT structures for a shadow
   pt[ 0 ].x = lpDs->rcItem.left + 3;     pt[ 0 ].y = lpDs->rcItem.bottom - 1;
   pt[ 1 ].x = lpDs->rcItem.right;        pt[ 1 ].y = lpDs->rcItem.bottom - 1;
   
   if( ! bSelected )
      pt[ 0 ].x += 8;          //  Move it in if not selected and draw the line

   Polyline( lpDs->hDC, pt, 2 );                  //  (yup, just one line here)

   if( bSelected )            //  If selected, draw it again to match highlight
   {
      pt[ 0 ].x = lpDs->rcItem.left + 3;  pt[ 0 ].y = lpDs->rcItem.bottom - 2;
      pt[ 1 ].x = lpDs->rcItem.right;     pt[ 1 ].y = lpDs->rcItem.bottom - 2;

      Polyline( lpDs->hDC, pt, 2 );                  //  (again, just one line)
   }

   if( bSelected )                              //  Use a bold font if selected
      hFont = CreateFont( -8, 0, 0, 0, FW_BOLD, 0, 0, 0, 0, 0, 0, 0, 0,
                          "MS Sans Serif" );
   else                                     //  Or a normal one if not selected
      hFont = CreateFont( -8, 0, 0, 0, FW_NORMAL, 0, 0, 0, 0, 0, 0, 0, 0,
                          "MS Sans Serif" );

   hFont = SelectObject( lpDs->hDC, hFont ); 
   SetBkMode( lpDs->hDC, TRANSPARENT );          //  Don't fill text background
   SetTextColor( lpDs->hDC, GetSysColor( COLOR_BTNTEXT ) );
                                                  //  Get the listbox item text
   SendMessage( lpDs->hwndItem, LB_GETTEXT, lpDs->itemID, (LPARAM)szStr );
                         //  And display it so it's roughly centered vertically
   TextOut( lpDs->hDC, lpDs->rcItem.left + 14, lpDs->rcItem.top + 10,
            szStr, lstrlen( szStr ) );
                                                    //  Clean up before leaving
   DeleteObject( SelectObject( lpDs->hDC, hPen ) );
   DeleteObject( SelectObject( lpDs->hDC, hFont ) );
   return( TRUE );
}                                                      //  End of PaintVTabCtrl

// === Draw "Tab Edge" on Dialog Box surface ==================================

BOOL PaintTabEdge( HWND hDlg )         //  Draw in response to list item change
{
   HDC hDC;
   HPEN hPen;
   RECT Rect;
   POINT pt, ppt[ 6 ];
   int iSelection;
 
                                       //  First, get current listbox selection
   iSelection = SendDlgItemMessage( hDlg, 100, LB_GETCURSEL, 0, 0 );
   if( iSelection == LB_ERR )
      return( FALSE );                      //  Quit if nothing is selected yet

   hDC = GetDC( hDlg );                          //  Get a DC to the dialog box
                                   //  Get the listbox WINDOW size and position
   GetWindowRect( GetDlgItem( hDlg, 100 ), &Rect );
                             //  Getting the window size and pos means only one
   pt.x = Rect.left;          //  conversion is needed plus it accounts for any
   pt.y = Rect.top;                                     //  border style in use
   ScreenToClient( hDlg, &pt );    //  Convert to dialog box client coordinates
   pt.x = pt.x + ( Rect.right - Rect.left ); //  Start where listbox leaves off

   hPen = CreatePen( PS_SOLID, 1, GetSysColor( COLOR_BTNFACE ) );
   SelectObject( hDC, hPen );                    //  First, cover any old lines

   ++pt.y;                                       //  Don't cover anything drawn
   Rect.bottom -= 4;                                //  by the WM_PAINT message

   ppt[ 0 ].x = pt.x;        ppt[ 0 ].y = pt.y;
   ppt[ 1 ].x = pt.x;        ppt[ 1 ].y = pt.y + ( Rect.bottom - Rect.top );
   ppt[ 2 ].x = pt.x + 1;    ppt[ 2 ].y = ppt[ 1 ].y;         //  Fill in POINT
   ppt[ 3 ].x = pt.x + 1;    ppt[ 3 ].y = pt.y;        //  array to cover three
   ppt[ 4 ].x = pt.x + 2;    ppt[ 4 ].y = pt.y;              //  vertical lines
   ppt[ 5 ].x = pt.x + 2;    ppt[ 5 ].y = ppt[ 1 ].y;
   Polyline( hDC, ppt, 6 );                            //  Draw lines (5 total)
   
   Rect.bottom += 4;                                  //  Put these values back
   --pt.y;                                                   // where they were

   hPen = CreatePen( PS_SOLID, 1, RGB( 0, 0, 0 ) );
   DeleteObject( SelectObject( hDC, hPen ) );             //  Draw black border

   ppt[ 0 ].x = pt.x;     ppt[ 0 ].y = pt.y;          //  Draw top to selection
   ppt[ 1 ].x = pt.x;     ppt[ 1 ].y = pt.y + ( iSelection * 32 ) + 1;
   Polyline( hDC, ppt, 2 );

   ppt[ 0 ].y = ppt[ 1 ].y + 31;                //  X position is same as above
   ppt[ 1 ].y = pt.y + ( Rect.bottom - Rect.top ) - 1;
   Polyline( hDC, ppt, 2 );    //  Then draw selection bottom to listbox bottom
                                                 //  Use button highlight color
   hPen = CreatePen( PS_SOLID, 1, GetSysColor( COLOR_BTNHIGHLIGHT ) );
   DeleteObject( SelectObject( hDC, hPen ) );
                                         //  Draw hilight from top to selection
   ppt[ 0 ].x = pt.x + 1;     ppt[ 0 ].y = pt.y + 1;
   ppt[ 1 ].x = pt.x + 1;     ppt[ 1 ].y = pt.y + ( iSelection * 32 ) + 1;
   ppt[ 2 ].x = pt.x - 1;     ppt[ 2 ].y = ppt[ 1 ].y;
   Polyline( hDC, ppt, 3 );     //  With a little indent to meet listbox border
                                               //  Do it again for double-width
   ++ppt[ 0 ].x;              ++ppt[ 0 ].y;       //  Just expand the positions
   ++ppt[ 1 ].x;              ++ppt[ 1 ].y;              //  from the last call
                              ++ppt[ 2 ].y;
   Polyline( hDC, ppt, 3 );
                                        //  Now draw bottom half of the hilight
   ppt[ 0 ].x = pt.x + 1;  ppt[ 0 ].y = pt.y + ( iSelection * 32 ) + 32;
   ppt[ 1 ].x = pt.x + 1;  ppt[ 1 ].y = pt.y + ( Rect.bottom - Rect.top ) - 2;
   Polyline( hDC, ppt, 2 );         //  From selection bottom to listbox bottom
                                           //  And do it again for double-width
   ++ppt[ 0 ].x;              --ppt[ 0 ].y;
   ++ppt[ 1 ].x;              --ppt[ 1 ].y;
   Polyline( hDC, ppt, 2 );               //  By expanding or contracting a bit

   hPen = CreatePen( PS_SOLID, 1, GetSysColor( COLOR_BTNSHADOW ) );
   DeleteObject( SelectObject( hDC, hPen ) );
                                            //  Now draw a small shadow section
   ppt[ 0 ].x = pt.x;         ppt[ 0 ].y = pt.y + ( iSelection * 32 ) + 31;
   ppt[ 1 ].x = pt.x + 2;     ppt[ 1 ].y = pt.y + ( iSelection * 32 ) + 31;
   Polyline( hDC, ppt, 2 );              //  to link listbox item to dialog box

   ++ppt[ 1 ].x;                      //  Draw that again for double-width, too
   --ppt[ 0 ].y;
   --ppt[ 1 ].y;
   Polyline( hDC, ppt, 2 );                        //  Pretty complicated, huh?

   ReleaseDC( hDlg, hDC );
   DeleteObject( hPen );                //  Be sure to delete the last pen used
   return( TRUE );
}                                                     //  End of PaintTabEdge()

// === Called in response to dialog box's WM_PAINT ============================

BOOL PaintTabBody( HWND hDlg, BOOL bInPaint )
{
   HDC hDC;
   PAINTSTRUCT ps;
   RECT Rect;
   POINT pt, ppt[ 4 ];
   HPEN hPen;
   int iSize;

   if( ! bInPaint )                //  If not called from WM_PAINT, create a DC
      hDC = GetDC( hDlg );             //  Note: This is unused in this example
   else
      hDC = BeginPaint( hDlg, &ps );      //  If called from WM_PAINT, use this
                                                 //  Get window rect of listbox
   GetWindowRect( GetDlgItem( hDlg, 100 ), &Rect );
   pt.x = Rect.left;
   pt.y = Rect.top;
   ScreenToClient( hDlg, &pt );          //  Convert to dialog rect coordinates
   pt.x = pt.x + ( Rect.right - Rect.left );
   iSize = Rect.bottom - Rect.top;  //  Save this cuz it's about to be trounced

   GetClientRect( hDlg, &Rect );                    //  Get the dialog box rect
                                                  //  Start with a black border
   hPen = CreatePen( PS_SOLID, 1, RGB( 0, 0, 0 ) );
   SelectObject( hDC, hPen );
                                              //  Draw edge of tab control body
   ppt[ 0 ].x = pt.x;               ppt[ 0 ].y = pt.y;
   ppt[ 1 ].x = Rect.right - 8;     ppt[ 1 ].y = pt.y;
   ppt[ 2 ].x = ppt[ 1 ].x;         ppt[ 2 ].y = iSize;
   ppt[ 3 ].x = pt.x;               ppt[ 3 ].y = iSize;
   Polyline( hDC, ppt, 4 );                    //  So it meets the listbox edge
                                                //  Now draw the highlight area
   hPen = CreatePen( PS_SOLID, 1, GetSysColor( COLOR_BTNHIGHLIGHT ) );
   DeleteObject( SelectObject( hDC, hPen ) );
                                                        //  Draw top edge twice
   ppt[ 0 ].x = pt.x;               ppt[ 0 ].y = pt.y + 1;
   ppt[ 1 ].x = Rect.right - 9;     ppt[ 1 ].y = ppt[ 0 ].y;
   ppt[ 2 ].x = pt.x;               ppt[ 2 ].y = pt.y + 2;
   ppt[ 3 ].x = ppt[ 1 ].x;         ppt[ 3 ].y = ppt[ 2 ].y;
   Polyline( hDC, ppt, 4 );

   hPen = CreatePen( PS_SOLID, 1, GetSysColor( COLOR_BTNSHADOW ) );
   DeleteObject( SelectObject( hDC, hPen ) );
                                               //  Draw tab control body shadow
   ppt[ 0 ].x = Rect.right - 9;    ppt[ 0 ].y = pt.y + 2;
   ppt[ 1 ].x = ppt[ 0 ].x;        ppt[ 1 ].y = iSize - 1;
   ppt[ 2 ].x = pt.x;              ppt[ 2 ].y = ppt[ 1 ].y;
   Polyline( hDC, ppt, 3 );                     //  Paint left and bottom twice
   ppt[ 0 ].x = Rect.right - 10;   ppt[ 0 ].y = pt.y + 3;
   ppt[ 1 ].x = ppt[ 0 ].x;        ppt[ 1 ].y = iSize - 2;
   ppt[ 2 ].x = pt.x;              ppt[ 2 ].y = ppt[ 1 ].y;
   Polyline( hDC, ppt, 3 );

   if( ! bInPaint )        //  Release the DC properly and kill the last object
      ReleaseDC( hDlg, hDC );
   else
      EndPaint( hDlg, &ps );

   DeleteObject( hPen );
   
   PaintTabEdge( hDlg );      //  Now paint the tab control edge and we're done
   return( TRUE );
}

// === End of Application =====================================================




