/****************************************************************/
/*                                                              */
/*      MODUL:  se.c                                            */
/*                                                              */
/*      FUNKTIONEN:                                             */
/*              - ask_replace (fragen, ob ersetzen)             */
/*              - upcase (Buchstabe in Grossbuchstabe wandeln)  */
/*              - instr (String in Text?)                       */
/*              - find (Begriff im Text suchen)                 */
/*              - str_to_num (String in Zahl wandeln)           */
/*              - o_to_vals (Optionen in Werte wandeln)         */
/*              - do_repfr (Latztes Suchen/Ersetzen wiederh.)   */
/*              - do_find (Suche ausfuehren)                    */
/*              - trag_e_ein (Ersetzungsbegriff eintragen)      */
/*              - flash_word (Wort zum Blinken bringen)         */
/*              - unflash_word (Wort "entblinken")              */
/*              - do_replace (Ersetzen ausfuehren)              */
/*                                                              */
/****************************************************************/

#include "defs.h"
#include <ctype.h>

extern char *fastzeichen();
extern char space;
extern WINDOW *status;
extern int ersetzaddr;

void do_find(), do_replace();

/* *** globale Daten und Initialisierung *** */
static char sbegriff[256],    /* Suchbegriff                                 */
	    ebegriff[256],    /* Ersetzungsbegriff                           */
	    optionen[256];    /* String fuer Optioneneingabe                 */
int         repeatflag=FALSE, /* Ist aktueller Vorgang eine Wiederholung ?   */
	    last_func=F_NONE; /* Art des letzten Vorgangs (Suchen, Ersetzen) */

unsigned char upcase();

/*****************************************************************************
*
*  Funktion       Fragen, ob ersetzt werden soll (ask_replace)
*  --------
*
*  Ergebnis     :
*                   Typ          : int char
*                   Wertebereich : (char) 0 - (char) 255
*                   Bedeutung    : Eingelesenes Zeichen (upcase)
*
*  Beschreibung : Es wird ein Prompt im Statusfenster ausgegeben, das
*                 der User durch Eingabe eines Zeichens beantworten kann.
*                 Es sind nur die Zeichen j,J,n,N,a,A zugelassen. Der
*                 Rueckgabewert ist das Zeichen, jedoch auf jeden Fall
*                 als Grossbuchstabe.
*
*****************************************************************************/

int ask_replace()
{
  /* *** interne Daten *** */
  int antw; /* zum Einlesen eines Zeichens */

  print_stat(PROMPT_REPLYN);
  do
    antw = newwgetch(status); /* Zeichen vom Tastatur / Macro lesen */
  while (!strchr ("jnJNyYAa",antw));
  clear_stat();
  if (strchr ("yY", antw)) /* Beantwortung mit Y(es) zulassen */
    antw = 'J';
  return(upcase(antw));
}

/*****************************************************************************
*
*  Funktion       Buchstabe in Grossbuchstabe wandeln (upcase)
*  --------
*
*  Parameter    : c         :
*                   Typ          : unsigned char
*                   Wertebereich : (char) 0 - (char) 255
*                   Bedeutung    : Umzuwandelndes Zeichen
*
*  Ergebnis     :
*                   Typ          : unsigned char
*                   Wertebereich : (char) 0 - (char) 255
*                   Bedeutung    : umgewandeltes Zeichen
*
*  Beschreibung : Falls das Zeichen ein Buchstabe ist, wird es in einen
*                 Grossbuchstaben umgewandelt.
*
*****************************************************************************/

unsigned char upcase (c)        /* eee klappt nicht bei Umlauten */
register unsigned char c;
{
  return (isalpha(c)?toupper(c):c);
}

/*****************************************************************************
*
*  Funktion       Text in String ? (instr)
*  --------
*
*  Parameter    : begriff   :
*                   Typ          : char*
*                   Wertebereich : Pointer auf ASCII-Zeichenkette
*                   Bedeutung    : String, nach dem gesucht wird
*
*                 zeile     :
*                   Typ          : char*
*                   Wertebereich : Pointer auf ASCII-Zeichenkette
*                   Bedeutung    : String, in dem gesucht wird.
*
*                 ignore    :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : TRUE : Gross-/Kleinschreibung ignorieren
*
*                 word      :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : TRUE : Nur ganze Woerter finden
*
*                 richtung  :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : TRUE : Vorwaerts suchen
*                                  FALSE: Rueckwaerts suchen
*
*  Ergebnis     :
*                   Typ          : char*
*                   Wertebereich : Pointer auf ASCII-Zeichen
*                   Bedeutung    : NULL = nicht gefunden
*                                  Sonst : Position des Suchstrings in
*                                          aktueller Zeile
*
*  Beschreibung : Die uebergebene Zeile wird auf den uebergebenen Suchstring
*                 begriff untersucht. Dabei wird, falls das ignore-Flag TRUE
*                 ist, die Gross-/Kleinschreibung ignoriert.
*                 Ist das Word-Flag gesetzt, so werden nur Suchbegriffe gefun-
*                 den, die rechts und links entweder Zeilenanfang bzw. Zeilen-
*                 ende oder einen Worttrenner haben.
*
*****************************************************************************/

char *instr(begriff,zeile,ignore,word,richtung)
register char *begriff;
char *zeile;
int  ignore,word,richtung;
{
  /* *** interne Daten und Initialisierung *** */
  register int  beg_sc,               /* Zwischenspeicher Cursorspalte  */
		len=strlen(begriff),  /* Laenge des Suchbegriffs        */
		index=0,              /* Index in aktuellen Suchbegriff */
		fr_ret;               /* Rueckgabewert von fastright    */
  register char *beg_start;           /* Position des Suchbegriffs in   */
				      /* der Zeile (nur bei word)       */

  if (!word)    /* Es muss kein abgeschlossenes Wort gefunden werden */
  {
    do
    {
      if (ignore? /* Wird Gross-/Kleinschreibung ignoriert, dann upcase */
      (upcase(*zeile) != upcase(begriff[index++])):(*zeile != begriff[index++]))
      {
	if (!fastleft(&zeile,index-richtung)) /* Nicht gefunden, ein Zeichen  */
	{                                     /* weiter rechts/links mit der  */
	  index = 0;                          /* Suche starten. Zeile zuende, */
	  break;                              /* dann do-while abbrechen      */
	}
	index = 0;      /* Im Suchbegriff wieder vorne beginnen */
	if (!richtung)  /* Falls rueckwaerts gesucht wird, nicht mehr nach */
	  continue;     /* rechts, da Position 0 getestet werden muss.     */
      }
      if(fr_ret = !fastright(&zeile,1)) /* dadurch steht man auch wenn gefunden */
	break;/* wurde hinter dem Suchbegriff, ausser Suchbegriff am Zeilenende */
    }while (index < len); /* Solange suchen, bis Suchbegriff gefunden wurde */
    if (index == len)     /* Wenn Suchbegriff gefunden wurde: */
    {                               /* Zum Anfang des Suchbegriffs gehen */
      fastleft(&zeile,len-fr_ret);  /* Falls Suchbegriff am Zeilenende, klappte */
      return (zeile);               /* fastright nicht, also 1 weniger nach links */
    }
    else                  /* Wurde Suchbegriff nicht gefunden, NULL zurueck */
      return (NULL);
  }
  else          /* Es soll ein abgeschlossenes Wort gefunden werden */
  {
    while (1)
    {
      /* Durch rekursiven Aufruf Begriff zunaechst normal in der Zeile */
      /* suchen. Wird er nicht gefunden, return(NULL), sonst testen,   */
      /* ob Wortgrenzen vorhanden                                      */
      if (beg_start = zeile = instr(begriff,zeile,ignore,FALSE,richtung))
      {
	beg_sc = akt_winp->screencol;
	if ((!fastleft(&zeile,1) || wortende(*zeile))) /* Wortende oder */
	{                                         /* Zeilenende links ? */
	  zeile = beg_start;
	  akt_winp->screencol = beg_sc;
	  fastright(&zeile,len - 1);   /* Zum Ende des Suchbegriffs gehen */
	  if (!fastright(&zeile,1) || wortende(*zeile)) /* Wenn hinter Such-   */
	  {                                             /* begriff Zeilenende, */
	    akt_winp->screencol = beg_sc;
	    return (beg_start);         /* dann auch Wortende, also gefunden ! */
	  }
	}
	akt_winp->screencol = beg_sc;   /* Cursorspalte restaurieren    */
	zeile = beg_start;              /* Zeiger in Zeile restaurieren */
	/* Suche ein Z. weiter rechts/links fortsetzen */
	if(richtung ? !fastright(&zeile,1)
		    : !fastleft (&zeile,1))
	  return(NULL);          /* Zeilenende, nicht gefunden */
      }
      else
	return (NULL); /* Begriff in aktueller Zeile nicht gefunden  */
    } /* Falls kein Wortende gefunden, nach erneutem Vorkommen suchen */
  }
}

/*****************************************************************************
*
*  Funktion       Begriff im Text suchen (find)
*  --------
*
*  Parameter    : begriff   :
*                   Typ          : char*
*                   Wertebereich : Pointer auf ASCII-Zeichenkette
*                   Bedeutung    : Suchbegriff
*
*                 richtung  :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : TRUE : vorwaerts , FALSE : rueckwaerts
*
*                 ignore    :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : Gross-/Kleinschreibung ignorieren
*
*                 word      :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : Nur ganze Woerter finden
*
*                 start_line:
*                   Typ          : int
*                   Wertebereich : 0-MAX_ANZ_LINES
*                   Bedeutung    : Nummer der Zeile, in der die Suche
*                                  gestartet wurde.
*
*                 start_col :
*                   Typ          : int
*                   Wertebereich : 0-MAXLENGTH
*                   Bedeutung    : Nummer der Spalte, in der die Suche
*                                  gestartet wurde.
*
*                 count     :
*                   Typ          : int*
*                   Wertebereich : Pointer auf Integer
*                   Bedeutung    : Wenn die Zeile, in der die Suche begann,
*                                  schon durchsucht wurde, ist *count 0,
*                                  sonst 1.
*
*  Ergebnis     :
*                   Typ          : int
*                   Wertebereich : TRUE, FALSE
*                   Bedeutung    : TRUE : begriff gefunden
*                                  FALSE: begriff nicht gefunden
*
*  Beschreibung : Es wird ab der aktuellen Position im Text nach dem ueber-
*                 gebenen Begriff gesucht. Dabei werden die Parameter richtung,
*                 ignore und word beruecksichtigt. Kommt man in der Zeile hinter
*                 der aktuellen an und count ist 0, so wird die Suche beendet.
*                 Findet man den Suchbegriff in der Zeile, in der die Suche
*                 gestartet wurde, hinter der Startspalte, so wird FALSE
*                 zurueckgegeben.
*                 Die Suche innerhalb einer Zeile wird durch die Funktion instr
*                 durchgefuehrt. Dadurch ist die Suche innerhalb einer Zeile
*                 unabhaengig vom Parameter richtung.
*
******************************************************************************/

int find(begriff,richtung,ignore,word,start_line,start_col,count)
register char *begriff;
int  richtung,ignore,word,start_line,start_col,*count;
{
  /* *** interne Daten und Initialisierung *** */
  register char *zeile;                       /* Zeiger in akt. Zeile    */
  register int  blen = strlen(begriff),       /* Laenge des Suchbegriffs */
		old_sc = akt_winp->screencol, /* Alte Cursorspalte       */
		old_line=akt_winp->textline,  /* Alte Cursorzeile        */
		nsl=start_line+2*richtung-1;  /* erste Zeile, die nicht  */
					      /* zweimal durchsucht wer- */
					      /* den darf.               */
  int           fl;      /* Laenge der aktuellen Zeile im screencol-Mass */

  if (!start_line && !richtung)
    nsl=akt_winp->maxline+1;
  else
    if (start_line == akt_winp->maxline && richtung)
      nsl = -1;
  print_stat(PROMPT_SEARCHING);

  /* Falls Cursor auf oder hinter dem Zeilenende steht und rueckwaerts */
  /* gesucht werden soll, Cursor um die Laenge des Suchbegriffs vor    */
  /* das Zeilenende setzen.                                            */
  if(akt_winp->screencol >= (fl = fastll(akt_winp->alinep->text)) && !richtung)
    akt_winp->screencol = fl - blen;  /* screencol kann zwar < 0 werden,    */
				      /* da aber fastzeichen dann hinterher */
  else                                /* &space liefert, ist das egal.      */
    if(akt_winp->screencol < 0)
    {
      if(!up())
	gotox(akt_winp->maxline);
      akt_winp->screencol =  fl - blen;
    }
  do
  {
    /* bei vorwaerts suchen werden leere Zeilen uebersprungen,
       bei rueckwaerts suchen zu kurze Zeilen */
    if((zeile = fastzeichen(akt_winp->screencol)) != &space)
    {
      if(*zeile == '_' && zeile[1] == '' && zeile[2]) /* Falls aktuelles     */
	zeile+=2; /* Zeichen unterstrichen ist, zeile auf Zeichen statt auf _ */
      if (instr(begriff,zeile,ignore,word,richtung))     /* Zeile durchsuchen */
	/* Wenn Teil der Zeile schonmal durchsucht, testen, ob gefundener Teil*/
	/* im schon durchsuchten Teil liegt.                                  */
	if(akt_winp->textline == start_line &&
	(richtung ? akt_winp->screencol >= start_col
		  : akt_winp->screencol <= start_col) && !*count)
	  break; /* Wenn ja, ist Ergebnis ungueltig, abbrechen */
	else
	{
	  clear_stat();
	  return (TRUE);
	}
    }
    /* Zur naechsten zu durchsuchenden Zeile gehen */
    akt_winp->alinep = richtung?akt_winp->alinep->next:akt_winp->alinep->prev;
    if((akt_winp->textline += 2*richtung-1) < 0)
      akt_winp->textline = akt_winp->maxline + 1; /* alinep steht auf dummyp */
    else
      if(akt_winp->textline > akt_winp->maxline)
	akt_winp->textline = -1;        /* alinep steht auf dummyp */

    /* Bei vorwaertssuchen ab erster Spalte beginnen, bei Rueckwaertssuchen */
    /* blen Spalten vor Zeilenende, da Suchbegriff nicht spaeter anfangen   */
    /* kann.                                                                */
    akt_winp->screencol = richtung?0:fastll(akt_winp->alinep->text)-blen;
  }while(akt_winp->textline != nsl || (*count)--); /* ganzen Text durchlaufen */
  gotox (old_line);             /* falls nicht gefunden, an alte Position */
  akt_winp->screencol = old_sc;
  clear_stat();
  return (FALSE);
}

/*****************************************************************************
*
*  Funktion       In einem String enthaltene Zahl berechnen (str_to_num)
*  --------
*
*  Parameter    : string    :
*                   Typ          : char*
*                   Wertebereich : Pointer auf ASCII-Zeichenkette
*                   Bedeutung    : String, in dem die Zahl enthalten ist
*
*  Ergebnis     :
*                   Typ          : int
*                   Wertebereich : 0-MAXINT
*                   Bedeutung    : Im String enthaltene Zahl als int
*
*  Beschreibung : Zuerst wird die erste im String vorkommende Ziffer gesucht.
*                 Dann wird die Ziffer in einen Integer umgewandelt. Bei
*                 jeder weiteren gefundenen Ziffer wird der bisherige Wert
*                 mit 10 multipliziert und der Wert der gefundenen Ziffer
*                 addiert. Sobald das Stringende oder keine Ziffer gefuden
*                 wird, bricht die Funktion ab.
*
*****************************************************************************/

int str_to_num(string)
register char *string;
{
  /* *** interne Daten *** */
  register int hilf=0; /* vorlaeufiges Ergebnis */

  while (*string && (*string < '0' || *string > '9'))
    string++;   /* erste Ziffer suchen */
  if (!*string) /* Keine Ziffer gefunden, -1 zurueck */
    return (-1);
  while (isdigit(*string))
    hilf = 10*hilf + *string++ - '0';
  return (hilf);
}

/*****************************************************************************
*
*  Funktion       Optionen in Werte wandeln (o_to_vals)
*  --------
*
*  Parameter    : optionen  :
*                   Typ          : char*
*                   Wertebereich : Pointer auf ASCII-Zeichenkette
*                   Bedeutung    : Auszuwertender String
*
*               : richtung  :
*                   Typ          : int*
*                   Wertebereich : Pointer auf Integer
*                   Bedeutung    : TRUE:vorwaerts, FALSE:rueckwaerts
*
*               : anzahl    :
*                   Typ          : int*
*                   Wertebereich : Pointer auf integer
*                   Bedeutung    : Anzahl der Wiederholungen
*
*               : ignore    :
*                   Typ          : int*
*                   Wertebereich : Pointer auf integer
*                   Bedeutung    : Gross-/Kleinschreibung ignorieren
*
*               : word      :
*                   Typ          : int*
*                   Wertebereich : Pointer auf integer
*                   Bedeutung    : Nur ganze Worte finden
*
*               : anf_ende  :
*                   Typ          : int*
*                   Wertebereich : Pointer auf Integer
*                   Bedeutung    : Soll vom Anfang oder vom Ende gesucht/
*                                  ersetzt werden ?
*
*  Beschreibung : Anhand des Optionenstrings werden die Parameter gesetzt.
*                 anzahl wird ueber str_to_num ermittelt, alle anderen
*                 ueber die Funktion strchr.
*
*****************************************************************************/

void o_to_vals(optionen,richtung,anzahl,ignore,word,anf_ende)
char *optionen;
int  *richtung,*anzahl,*ignore,*word,*anf_ende;
{
  /* Wenn der aktuelle Vorgang eine Wiederholung ist */
  /* oder der User keine Anzahl angegeben hat, wird  */
  /* die Anzahl auf 1 gesetzt.                       */
  if (repeatflag || (*anzahl = str_to_num(optionen))==-1)
    *anzahl = 1;

  *richtung = !strchr(optionen,'r') && !strchr(optionen,'R');
  *ignore   =  strchr(optionen,'i') ||  strchr(optionen,'I');
  *word     =  strchr(optionen,'w') ||  strchr(optionen,'W');
  *anf_ende =  FALSE;         /* default: nicht von Anfang oder Ende */
  if (strchr(optionen,'b') || strchr(optionen,'B'))
  {
    if(!repeatflag) /* bei repeat nicht mehr vom anfang an */
    {
      *anf_ende = TRUE;
      gotox(0);                /* Zeile 0, Spalte 0 aufsuchen */
      akt_winp->screencol = 0;
    }
    *richtung = 1; /* Bei Suchen vom anfang immer vorawerts */
  }
  else
    if (strchr(optionen,'e') || strchr(optionen,'E'))
    {
      if(!repeatflag)
      {
	*anf_ende = TRUE;         /* letzte Zeile, letzte Spalte aufsuchen */
	gotox(akt_winp->maxline);
	akt_winp->screencol = fastll(akt_winp->alinep->text)-1;
      }
      *richtung = 0;      /* Wenn vom Ende aus gesucht werden soll, dann  */
    }                     /* auf jeden Fall rueckwaerts.                  */
}

/*****************************************************************************
*
*  Funktion       Letztes Suchen/Ersetzen wiederholen (do_repfr)
*  --------
*
*  Beschreibung : Falls schon eine Suche bzw. Ersetzung stattgefunden hat,
*                 wird diese mit den gleichen Optionen, jedoch anzahl=1
*                 wiederholt.
*
*****************************************************************************/

void do_repfr()
{
  if (last_func != F_NONE)
  {
    repeatflag = TRUE;
    if (last_func == F_FIND)
      do_find();
    else
      do_replace();
    repeatflag = FALSE;
  }
}

/*****************************************************************************
*
*  Funktion       Suchen ausfuehren (do_find)
*  --------
*
*  Beschreibung : Falls die Funktion nicht durch do_repfr() aufgerunfen wurde,
*                 werden Suchbegriff und Optionen eingelesen. Anschliessend
*                 wird mit der Funktion o_to_vals der Optionenstring in die
*                 Parameter zerteilt und anzahl mal find() aufgerufen.
*
*****************************************************************************/

void do_find()
{
  /* *** interne Daten und Initialisierung *** */
  int          richtung,   /* Flag, ob vorawerts gesucht wird                */
	       anzahl,     /* Anzahl der Suchwiederholungen                  */
	       alt_anz,    /* Zwischenspeicher fuer Wiederholungszahl        */
	       ignore,     /* Flag, ob Gross-/Kleinschreibung ignoriert wird */
	       word,       /* Flag, ob nur ganze Woerter gefunden werden     */
	       anf_ende,   /* Flag, ob vom Anfang oder vom Ende gesucht wird */
	       count=1;    /* Zeigt an, ob die Zeile hinter der Anfangszeile */
			   /* schon einmal durchsucht wurde.                 */
  register int slen,       /* Laenge des Suchbegriffs                        */
	       start_line, /* Zeile, in der die Suche begann                 */
	       start_col;  /* Spalte, in der die Suche begann                */
  char         dummy[256]; /* String fuer zusammengeflickte Fehlermeldung    */

  if(akt_winp->maxline >= 0) /* Wenn Text leer, dann auch nicht suchen */
  {
    last_func = F_FIND;      /* Fuer do_repfr Funktionstyp merken */
    if (!repeatflag)         /* Handelt es sich um eine Wiederholung, dann   */
    {                        /* wird der Suchbegriff nicht erneut eingelesen */
      print_stat(PROMPT_SEARCH);
      read_stat(sbegriff,255,GS_ANY);
      clear_stat();
    }
    if(slen = strlen(sbegriff)) /* Bei leerem Suchbegriff nichts unternehmen */
    {
      if (!repeatflag) /* Optionen nur einlesen, wenn es keine Wiederholung ist */
      {
	print_stat(PROMPT_FOPTIONS);
	read_stat(optionen,255,GS_ANY);
	clear_stat();
      }
      akt_winp->lastline = akt_winp->textline; /* Aktuelle Position als letzte */
      akt_winp->lastcol = akt_winp->screencol; /* Position merken              */

      /* Optionen auswerten */
      o_to_vals(optionen,&richtung,&anzahl,&ignore,&word,&anf_ende);

      start_line = akt_winp->textline;
      if (!anf_ende) /* Wenn nicht vom Anfang oder vom Ende gesucht werden soll,*/
	akt_winp->screencol += 2*richtung-1;      /* Dann von einer Pos. weiter */
      start_col=akt_winp->screencol;              /* rechts/links suchen        */
      check_buff();       /* Evtl. Puffer zurueck, da auch alinep->text wichtig */

      /* Wenn der Cursor beim Vorwaertssuchen am Zeilenende steht oder beim */
      /* Rueckwaertssuchen am Zeilenanfang, dann wird die Nummer der Start- */
      /* zeile und der Startspalte angepasst. */
      if (richtung ? fastzeichen(start_col) == &space : akt_winp->screencol == -1)
      {
	if((start_line += 2*richtung-1) < 0)
	  start_line = akt_winp->maxline;
	else
	  if(start_line > akt_winp->maxline)
	    start_line = 0;
	start_col=richtung?0:MAXLENGTH - slen; /* Fall rueckwaerts suchen, dann */
      }                                  /* Zeilenende als Suchbeginn markieren */
      if ((alt_anz = anzahl) > 0)
      {
	/* Suchbegriff suchen bis nicht mehr zu finden oder gewuenschte Anzahl */
	while (find(sbegriff,richtung,ignore,word,start_line,start_col,&count)
	      && --anzahl)
	  akt_winp->screencol += richtung ? slen : -1;    /* begriff skippen */
	adapt_screen(sbegriff); /* anschliessend Bildschirm anpassen */
	if (anzahl) /* Wurde nicht die geforderte Anzahl gefunden ? */
	{
	  if (alt_anz == anzahl) /* Keinmal ? */
	  {
	    if (!anf_ende)
	      akt_winp->screencol--;
	    strcpy(dummy,PROMPT_NOTFOUND);
	  }
	  else
	  {
	    akt_winp->screencol -= richtung ? slen : -1; /* Skippen rueckgaengig machen */
	    sprintf (dummy,PROMPT_FOUNDN,alt_anz-anzahl);
	  }
	  pe_or(dummy);
	}
      }
    }
    setz_cursor(W_AKT);
  }
  else
    print_err(PROMPT_FEMPTY);
}

/*****************************************************************************
*
*  Funktion       Ersetzungsbegriff eintragen (trag_e_ein)
*  --------
*
*  Parameter    : ebegriff  :
*                   Typ          : char *
*                   Wertebereich : Pointer auf ASCII-Zeichenkette
*                   Bedeutung    : Einzutragender Begriff
*
*                 elen      :
*                   Typ          : int
*                   Wertebereich : 0 - MAX_INT
*                   Bedeutung    : Laenge des Ersetzungsbegriffs
*
*  Beschreibung : Der uebergebene Ersetzungsbegriff wird mit der Funktion
*                 enter_char() ab der aktuellen Position in den Text ein-
*                 getragen. Dabei wird der Insert-Mode abgeschaltet.
*
*****************************************************************************/

void trag_e_ein(ebegriff,elen)
register char *ebegriff;
register int elen;
{
  /* *** interne Daten und Initialisierung *** */
  register int i,                        /* Schleifenzaehler              */
	       oldi = akt_winp->insflag; /* Zwischenspeicher fuer insflag */

  akt_winp->insflag = FALSE;    /* damit enter_char Zeichen PUTtet */
  for (i=0;i<elen;i++)
    enter_char(*ebegriff++);
  akt_winp->insflag = oldi;     /* Insert-Mode restaurieren */
}

/*****************************************************************************
*
*  Funktion       Wort zum Blinken bringen (flash_word)
*  --------
*
*  Parameter    : laenge    :
*                   Typ          : int
*                   Wertebereich : 0-MAXLENGTH
*                   Bedeutung    : Laenge des zum Bilnek zu bringenden Worts
*
*  Beschreibung : Ab der aktuellen Cursorposition werden laenge Zeichen zum
*                 blinken gebracht, d.h. mit dem Ersetzungsattribut.
*                 versehen.
*
*****************************************************************************/

void flash_word(laenge)
register int laenge;
{
  while (laenge--)
    waddch(akt_winp->winp,winch(akt_winp->winp) | 256*ersetzaddr);
}

/*****************************************************************************
*
*  Funktion       Wort "entblinken" (unflash_word)
*  --------
*
*  Parameter    : laenge    :
*                   Typ          : int
*                   Wertebereich : 0-MAXLENGTH
*                   Bedeutung    : Laenge des zu "entblinkenden" Wortes
*
*  Beschreibung : Ab der aktuellen Cursorposition wird bei laenge Zeichen
*                 das Attribut fuer Blinken (ersatzaddr) geloescht.
*
*****************************************************************************/

void unflash_word(laenge)
register int laenge;
{
  while (laenge--)
    waddch(akt_winp->winp,winch(akt_winp->winp) & ~(256*ersetzaddr));
}

/*****************************************************************************
*
*  Funktion       Ersetzen ausfuehren (do_replace)
*  --------
*
*  Beschreibung : Falls die Funktion nicht von do_repfr aufgerufen wurde,
*                 kann der Benutzer Such-, Ersetzungsbegriff und Optionen
*                 eingeben. Dann wird ab der aktuellen Position aus der Such-
*                 begriff gesucht und je nach Option mit oder ohne Abfrage
*                 ersetzt. Die Suche kann vorwaerts und rueckwaerts, nur
*                 nach ganzen worten oder unter Ignorierung der Gross-/Klein-
*                 schreibnug durchgefuehrt werden. Zur Suche wird die Funktion
*                 find() benutzt, um den Begriff zu ersetzen die Funktion
*                 trag_e_ein().
*
*****************************************************************************/

void do_replace()
{
  /* *** interne Daten und Initialisierung *** */
  int          richtung,    /* Flag, ob vorawerts gesucht wird                */
	       anzahl,      /* Anzahl der Ersetzungswiederholungen            */
	       ignore,      /* Flag, ob Gross-/Kleinschreibung ignoriert wird */
	       word,        /* Flag, ob nur ganze Woerter gefunden werden     */
	       alt_anz,     /* Zwischenspeicher fuer Wiederholungszahl        */
	       diff,
	       dl,
	       globalflag;
  register int oldu = akt_winp->underflag, /* Zwischenspeicher */
	       slen,        /* Laenge des Suchbegriffs                        */
	       elen,        /* Laenge des Ersetzungsbegriffs                  */
	       ulflag,      /* Flag, ob Ersetzungsbegriff nach aktuellem      */
			    /* Modus oder nach dem ersten Zeichen des gefun-  */
			    /* denen Suchbegriffs unterstrichen werden soll   */
	       allflag;     /* Flag, ob alle vorkommenden Suchstrings ersetzt */
			    /* werden sollen.                                 */
  int          ret,         /* Rueckgabewert von ask_replace                  */
	       start_line=akt_winp->textline, /* Suchstart Zeile              */
	       start_col=akt_winp->screencol, /* Suchstart Spalte             */
	       count=1;     /* Zeigt an, ob die Zeile hinter der Anfangszeile */
			    /* schon einmal durchsucht wurde.                 */
  char         dummy[256];

  if(akt_winp->maxline >= 0) /* Wenn Text leer, nichts suchen */
  {
    last_func = F_REPLACE; /* Funktionstyp fuer evtl. Wiederholung merken */
    check_buff(); /* Evtl. Puffer zurueckschreiben, da alinep->text gebraucht */
    if (!repeatflag)       /* wird. */
    {                      /* Bei Wiederholung keinen neuen Suchbegriff,  */
      print_stat(PROMPT_SEARCH); /* keinen neuen Ersetzungsbegriff und */
      read_stat(sbegriff,255,GS_ANY); /* keine neuen Optionen einlesen    */
      clear_stat();
      if(!sbegriff[0])    /* falls leer, raus */
      {
	setz_cursor(W_AKT);
	return;
      }
      print_stat(PROMPT_REPLACE);
      read_stat(ebegriff,255,GS_ANY);
      clear_stat();
      print_stat(PROMPT_ROPTIONS);
      read_stat(optionen,255,GS_ANY);
      clear_stat();
    }
    /* Laengendifferenz zwischen Such- und Ersetzungsbegriff berechnen */
    diff = (elen = strlen(ebegriff))-(slen = strlen(sbegriff));
    akt_winp->lastline = akt_winp->textline; /* aktuelle Position als  */
    akt_winp->lastcol = akt_winp->screencol; /* letzte Position merken */

    /* Optionenstring auswerten */
    o_to_vals(optionen,&richtung,&anzahl,&ignore,&word,&dl); /* dl ist DUMMY */
    alt_anz = anzahl;
    globalflag = strchr(optionen,'g') || strchr(optionen,'G');
    allflag = strchr(optionen,'a') || strchr(optionen,'A'); /* alle? */
    ulflag = strchr(optionen,'u') || strchr(optionen,'U'); /* unterstr. nach modus?*/

    /* Falls Cursor hinter Zeilenende oder vor Zeilenanfang steht, */
    /* naechste Suchposition als Startposition eintragen.          */
    if (richtung && fastzeichen(start_col) == &space)
    {
      if((start_line += 2*richtung-1) < 0) /* Bei Textanfang oder */
	start_line = akt_winp->maxline;    /* Textende wraparound */
      else
	if(start_line > akt_winp->maxline)
	  start_line = 0;
      start_col=richtung?0:MAXLENGTH - slen; /* Fall rueckwaerts suchen, dann Zeilenende als Suchbeginn markieren */
    }

    /* Suchbegriff entweder anzahl mal oder bei gesetztem allflag so oft */
    /* es geht suchen und ersetzen.                                      */
    while ((allflag || anzahl--) && find(sbegriff,richtung,ignore,word,start_line,start_col,&count))
    {
      if (!globalflag)  /* muss Abfrage vorgenommen werden ? */
      {
	adapt_screen(sbegriff); /* Ja, dann Bildschirm anpassen */
	flash_word(slen);       /* Suchbegriff highlighten */
	setz_cursor(W_AKT);          /* und refreshen */
	unflash_word(slen);     /* Highlight aus, aber noch kein refresh */
	if((ret = ask_replace()) != 'J') /* soll nicht ersetzt werden ? */
	{
	  if(ret == 'A')    /* Abbruch? */
	  {
	    anzahl = -1;    /* Meldung am Ende unterdruecken */
	    break;
	  }
	  akt_winp->screencol+=slen; /* Suchbegriff skippen */
	  wrefresh(akt_winp->winp);  /* Fenster refreshen   */
	  continue;
	}
      }
      if(fastll(akt_winp->alinep->text) + diff > MAXLENGTH)
	print_err(PROMPT_ERREPLACE);
      else
      {
	/* falls erstes Zeichen des Wortes unterstrichen, wird gesamtes Wort
	   unterstrichen, wenn ulflag = FALSE, sonst wird je nach underline-
	   modus unterstrichen */
	if(!ulflag)
	  if(ul_char())
	    akt_winp->underflag = TRUE;
	  else
	    akt_winp->underflag = FALSE;
	mdelete(slen);
	insert(elen); /* Platz fuer neuen (weil evtl. underlined) */
	trag_e_ein(ebegriff,elen); /* Ersetzungsbegriff eintragen */
	check_buff(); /* Puffer zurueckschreiben, da alinep->text gebraucht */
      }
      if(!globalflag) /* Wenn Abgefragt wurde, dann Bildschirm */
      {               /* korrigieren */
	lineout(akt_winp->textline - akt_winp->ws_line);
	wrefresh(akt_winp->winp); /* veraenderte Zeile restaurieren */
      }
    } /* of while */

    /* Wenn nicht alle vorkommenden Suchbegriffe gefunden werden sollten */
    /* und nicht abgebrochen wurde, testen ob alle gewuenschten Er-      */
    /* setzungen durchgefuehrt werden konnten.                           */
    if (!allflag && anzahl != -1)
    {
      if (alt_anz-1 == anzahl)
	strcpy(dummy,PROMPT_NOTFOUND);
      else
	sprintf (dummy,PROMPT_FOUNDN,alt_anz-anzahl-1);
      print_err(dummy);
    }
    if (globalflag) /* Wenn keine Abfrage stattfand, dann jetzt Bildschirm */
      adapt_screen(sbegriff); /* anpassen */
    show_win(W_AKT);         /* eee evtl. show_win zuviel, wenn adapt_screen */
  }                     /* schon show_win() gemacht hatte               */
  else
    print_err(PROMPT_REMPTY);
  akt_winp->underflag = oldu; /* underflag restaurieren */
}
