procedure HighTraffic;
{ compute those high traffic areas }
var
  weights : array [1..MaxSector] of integer;
  counts  : array [1..MaxSector] of real;
  s1, s2  : sector;
  log, d  : boolean;
  f       : text;
  i       : integer;

procedure UniformProbs;
{ Assign all sectors to weight 1 }
var
  s : sector;
begin
  for s := 1 to MaxSector do
    weights[s] := 1;
end; {Uniform probabilities}

procedure PortProbs;
{ assign probabilities based upon sector type.  Empty sectors get nothing.
Space dock gets 50, terra 20, the other ports based upon port type. }
var
  s : sector;
begin
  for s := 1 to maxSector do
    case space.sectors[s].portType of
      NotAPort   : weights[s] := 0;
      0, 7       : weights[s] := 1;
      1, 6       : weights[s] := 2;
      2, 3, 4, 5 : weights[s] := 4;
      Class0     : weights[s] :=10;
    end; {case}
  weights[1] := 20;
  if space.dock <> 0 then
    weights[ space.dock ] := 50;
end; {port probs}

procedure AssignWeights;
var
  ch : char;
begin
  write('Port-heavy probabilities, (1) or uniform probabilities (2)?');
  readln( ch );
  case ch of
    '1' : PortProbs;
  else
          UniformProbs;
  end; {case}
end; {Assign the "weights" table}

procedure InitCounts;
var
  s : sector;
begin
  for s := 1 to maxSector do
    counts[s] := 0.0;
end; {Initialize all counts to zero}

procedure DisplayCounts( short, disk : boolean; var diskfile : text);
const
  header = ' sctr prob sctr prob sctr prob sctr prob';

var
  highestsector, linecount : integer;
  s : sector;
  highestval : real;
  quit : boolean;
begin
  linecount := 0;
  quit := false;
  if short then
    writeln( header, header );
  if short and disk then
    writeln(diskfile, header, header );
  repeat
    highestval := 0;
    for s := 1 to maxSector do
      if counts[s] > highestval then
        begin
          highestval := counts[s];
          highestsector := s;
        end; {if}
    if highestval > 0 then
      if short then
        begin
          write(highestsector:4,  ' ', highestval :5:0);
          if disk then
            write(diskfile, highestsector : 4,' ', highestval : 5:0);
        end {if}
      else
        DisplaySector( highestsector, ' prb:', round( highestval/100),
                       disk, diskfile);
    counts[ highestsector ] := 0;
    linecount := linecount + 1;
    if linecount mod 8 = 0 then
      if disk then
        writeln( diskfile );
    if (linecount mod 160 = 0) and not disk then
      begin
        writeln;
        quit := not prompt('more?');
      end; {if}
  until (highestval = 0) or quit;
  writeln;
  if disk then
    writeln( diskfile );
end;

procedure BackTrack( fromSector, toSector : sector );
begin
  counts[ fromSector ] := counts[ fromSector ] + weights[ fromSector ]
                                               + weights[ toSector ];
  if fromSector <> toSector then
    BackTrack( distances[ fromSector ].s , toSector );
end;

begin {HighTraffic}
  writeln('Warning: this computation will take a fairly long time.');
  if prompt('Try some other time? ') then
    exit;
  AssignWeights;
  InitCounts;
  log := prompt( 'Log to disk? ');
  d   := prompt('Short report? ');
  if log then
    begin
      assign( f, GetNewFileName('File name for report?  ', 'traffic.txt') );
      rewrite( f );
    end;
  for s1 := 1 to maxSector do
    begin
      if space.sectors[s1].number <> unexplored then
        begin
{ Do a shortest path search, finding spots for all sectors.  Mark parent.
  Now for each explored sector, just trace back toward s1. }
          write( s1, ' ');
          TwoWayDistances( s1, distances, false, true );
          for s2 := 1 to MaxSector do
            if (space.sectors[s2].number <> unexplored)
                and (distances[s2].d <> maxint) then
              BackTrack( s2, s1);
        end; {explored}
    end; {for s1}
  DisplayCounts( d, log, f);
  if log then
    close( f );
end;
