/*
        GEODUMP.C

        by Marcus Grber 1991-94

        Strukturierte Ausgabe von PC/GEOS-Dateien (soweit Format bekannt)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "geos.h"
#include "geos2.h"
#include "geostool.inc"


// #define NODISASM        /* if defined, disassembler module is excluded */
#define MAX_PASSES 10      /* max. pass number before disassembly is aborted */

#ifndef NODISASM
void disassemble(unsigned char *c,unsigned len,unsigned ofs,int all_silent);
#endif


#define GetStruct(str)      fread(&(str),sizeof(str),1,f)
#define GetStructP(str,ptr) fseek(f,ptr,SEEK_SET);fread(&(str),sizeof(str),1,f)

unsigned dodump;
unsigned ver;
unsigned base;

struct {                        /*** Segment-Beschreibung */
  GEOSseglen len;               // Lnge
  GEOSsegpos pos;               // Position in der Datei
  GEOSsegfix fixlen;            // Lnge der Fixup-Tabelle
  GEOSsegflags flags;           // Flags
} *segd;
unsigned numexp,numseg;         // Anzahl Export-Routinen, Segmente
GEOSexplist *expt;              // Export-Routinen
GEOSliblist *lib;               // Library-Namen

#ifndef NODISASM
unsigned dolist,disasm_silent;

unsigned this_seg;              // Segment number that is currently processed

/******************************************************************************/
#define MAX_GLOBAL_JMP 16000
#define MAX_DATA_MAP   16000

struct {
  unsigned seg,ofs;
} *global_jmp_map;
unsigned n_global_jmp;                  // number of global jumps
unsigned global_inf_changed;

void init_global_jmp(void)
{
        global_jmp_map=calloc(MAX_GLOBAL_JMP,sizeof(global_jmp_map[0]));
        n_global_jmp=0;                 // no global jump
}

void add_global_jmp(unsigned seg,unsigned ofs)
{
        unsigned i;

        for(i=0;i<n_global_jmp;i++)     // all existing global jumps
          if(global_jmp_map[i].seg==seg && global_jmp_map[i].ofs==ofs)
            return;                     // already in there - no change
        if(i==MAX_GLOBAL_JMP) return;   // global jump table full - abort
        global_jmp_map[i].seg=seg;      // add global jmp target
        global_jmp_map[i].ofs=ofs;      // add global jmp target
        n_global_jmp++;                 // one more global jmp target
        global_inf_changed=1;           // in case anyone wants to know...
}

int test_global_jmp(unsigned ofs)
{
        unsigned i;

        for(i=0;i<n_global_jmp;i++)
          if(global_jmp_map[i].seg == this_seg && global_jmp_map[i].ofs == ofs)
             return 1;
        return 0;
}


struct {
  unsigned seg,ofs;                     // location of jump list
  unsigned len;                         // size of list
  unsigned char size;                   // size of each entry
} jmplist[256];
unsigned n_jmplist;                     // number of jumplists

void init_jmplist(void)
{
        n_jmplist=0;                    // no jumplist yet
}

unsigned test_jmplist(unsigned ofs,unsigned *flag,int *addinfo)
{
        unsigned i;
        int firsthalf,ptrfirst,type;

        *flag=0;                        // default: no jumplist
        *addinfo=0xFFFF;                // additional info: just jumplist...
        for(i=0;i<n_jmplist;i++)        // all jumplists
          if(jmplist[i].seg == this_seg &&
             ofs >= jmplist[i].ofs &&
             ofs < jmplist[i].ofs+jmplist[i].len) {
            if(ofs == jmplist[i].ofs ||
               (jmplist[i].size>15 && ofs == (jmplist[i].ofs+jmplist[i].len/2)))
              *flag=1;                  // set flag if jumplist just starts...
            if(jmplist[i].size<16)      // location in normal jumplist?
              return jmplist[i].size;   // just return size

            /* the following piece of trickery creates a special return
               value for split pointer tables indicating whether this is
               the segment or the offset half and where the other half can
               be found (distance must be less than 2048 bytes). */

            firsthalf = (ofs<jmplist[i].ofs+jmplist[i].len/2);
                                        // indicates if in 1st half of jmplist
            ptrfirst = (jmplist[i].size==17);
                                        // indicates if segment or pointer first
            if(firsthalf^ptrfirst)      // is this segment half? [^^ is xor!]
              type=0x2800;              // fixup for split ptr segment
            else
              type=0x1800;              // fixup for split ptr offset
            *addinfo=type+(int)(jmplist[i].len/2)*(int)(firsthalf?1:-1);
                                        // offset to other half of pointer
            return 2;                   // this will create a "dw" instruction
          }
        *addinfo=0;                     // no jumplist
        return 0;                       // no jumplist applies
}

void add_jmplist(unsigned seg,unsigned ofs,unsigned len,unsigned size)
{
        jmplist[n_jmplist].seg=seg;     // add new jump list
        jmplist[n_jmplist].ofs=ofs;
        jmplist[n_jmplist].len=len;
        jmplist[n_jmplist].size=size;
        n_jmplist++;                    // count entries
}


struct {
  unsigned seg,ofs;
} *data_map;
unsigned n_data_map;                    // number of global jumps

void init_data_map(void)
{
        data_map=calloc(MAX_DATA_MAP,sizeof(data_map[0]));
        n_data_map=0;                   // no data map yet
}

void add_data_map(unsigned seg,unsigned ofs)
{
        unsigned i;

        for(i=0;i<n_data_map;i++)       // all existing data entry points
          if(data_map[i].seg==seg && data_map[i].ofs==ofs)
            return;                     // already in there - no change
        if(i==MAX_DATA_MAP) return;     // global data table full - abort
        data_map[n_data_map].seg=seg;   // add global jmp target
        data_map[n_data_map].ofs=ofs;   // add global jmp target
        n_data_map++;                   // one more global jmp target
        global_inf_changed=1;           // in case anyone wants to know...
}

int test_data_map(unsigned ofs)
{
        unsigned i;

        for(i=0;i<n_data_map;i++)
          if(data_map[i].seg == this_seg && data_map[i].ofs == ofs)
             return 1;
        return 0;
}

/******************************************************************************/
char *create_label(unsigned seg,unsigned ofs,char type)
{
        static char buffer[32];

        if(seg==0xFFFF) seg=this_seg;   // segment 0xFFFF means: current segment
        sprintf(buffer,"%c%d_%04X",type,seg,ofs);
        return buffer;
}

/******************************************************************************/
static unsigned code_fixlen;
static GEOSfixup *code_fixup;
static FILE *code_file;

/*
  callback routine from disasm module to test if an adress is hit by a
  fixup. If yes, a string describing the symbolic value of the fixup
  is returned. Returns NULL if no fixup.

  addinfo give more information on the nature of the position being fixed up:
    0000h    a general data item that can only be identified by a fixup
    FFFFh    context indicates an absolute jump adress, no matter if fixed
             up or not
    other    passed from test_jmplist to handle more complex jumplists
 */
char *test_fixup(unsigned ofs,int type,char *buf,int addinfo)
{
  static char buffer[32],buf2[GEOS_FNAME+1];
  unsigned i,j;
  int dist;

  *buffer=0;
  for(i=0; i<code_fixlen && code_fixup[i].ofs!=ofs; i++)
    ;
/*
   handle cases where no fixup has been found but context indicates presence
   of a code pointer (e.g. in a jump table) - this is mainly for correctly
   identifying pointers to code inside the program in which the offset is
   not fixed up but stored as a constant.
*/
  if(i==code_fixlen) {                  // no fixup found
    if((addinfo & 0xF000)==0x1000 || (type==4 && addinfo==0xFFFF)) {
                                        // offset part of "split" 32 bit pointer
      if(addinfo==0xFFFF)               // no jumplist - plain pointer?
        dist = 2;                       // offset is 2 bytes after segment
      else
        dist = (addinfo & 0x0FFF)-0x800;// extract distance
      for(j=0;
          j<code_fixlen                 // scan for suitable segment
          && (code_fixup[j].ofs!=ofs+dist || (code_fixup[j].type & 0xFE)!=0x22);
          j++)
          ;
      if(j==code_fixlen && dist>=0) return NULL;
                                        // none found - quit (for offset-after-
                                        // segment split jumplists, the segment
                                        // has already been removed, so ignore
                                        // missing fixup)
      sprintf(buffer,(type==4)?"%s":"offset %s",
         create_label(*(unsigned *)(buf+dist),*(unsigned *)buf,'H'));
                                        // successive segment/offset is as
                                        // good as a seg/ofs single fixup
      add_global_jmp(*(unsigned *)(buf+dist),*(unsigned *)buf);
      if((dist<0 || type==4) && j<code_fixlen) code_fixup[j].ofs=0xFFFF;
                                        // segment fixup has been handled
    }
    else if(type==2 && addinfo==0xFFFF) {
      strcpy(buffer, create_label(0xFFFF,*(unsigned *)buf,'H') );
      add_global_jmp(this_seg,*(unsigned *)buf);
    }
    else
      return NULL;
  }

/*
   handle segment component of a split pointer jumplist consisting of separate
   tables containing segment & offset.
*/
  else if((addinfo & 0xF000)==0x2000) { // found fixup and is split ptr segment
    if((code_fixup[i].type & 0xFE)==0x22) {
      dist = (addinfo & 0x0FFF)-0x800;  // extract distance
      sprintf(buffer,"segment %s",
         create_label(*(unsigned *)buf,*(unsigned *)(buf+dist),'H'));
      add_global_jmp(*(unsigned *)buf,*(unsigned *)(buf+dist));
    }
  }

/*
   identify library name if library relative fixup
*/
  if(*buffer==0 && (code_fixup[i].type & 0xF0) == 0x10) {
    strcpy(buf2,lib[code_fixup[i].type>>8].name);
                                        // get library name
    for(j=GEOS_FNAME-1;j>0 && buf2[j-1]==' ';j--)
      ;
    buf2[j]=0;                          // truncate trailing spaces
  }

/*
  handle different fixup types
*/
  if(*buffer==0)                        // not identified anything yet
   switch(type) {

    case 2:
      switch(code_fixup[i].type & 0xFF) {
      case 0x11:                        // offset of library entry
        sprintf(buffer,"offset %s_%u",buf2,*(unsigned *)buf);
        break;
      case 0x12:                        // segment of library entry
        sprintf(buffer,"segment %s_%u",buf2,*(unsigned *)buf);
        break;
      case 0x22:                        // fixups to program segments
      case 0x23:
        sprintf(buffer,"SEG%u",*(unsigned *)buf);
        break;
      default: return NULL;             
      }
      break;

    case 4:
      switch(code_fixup[i].type & 0xFF) {
      case 0x00:
        sprintf(buffer,"geos_%u",*(unsigned *)buf);
        break;
      case 0x11:                        // offset of library entry
                                        // segment might follow...
        for(j=0;
            j<code_fixlen               // scan for suitable segment
            && (code_fixup[j].ofs!=ofs+2 || (code_fixup[j].type & 0xFF)!=0x12);
            j++)
          ;
        if(j==code_fixlen) return NULL;
        sprintf(buffer,"%s_%u",buf2,*(unsigned *)buf);
                                        // successive segment/offset is as
                                        // good as a seg/ofs single fixup
        code_fixup[j].ofs=0xFFFF;       // segment fixup has been handled
        break;
      case 0x14:
        sprintf(buffer,"%s_%u",buf2,*(unsigned *)buf);
        break;
      case 0x24:
        strcpy(buffer,create_label(*(unsigned *)buf,*(unsigned *)(buf+2),'H'));
        add_global_jmp(*(unsigned *)buf,*(unsigned *)(buf+2));
        break;
      default: return NULL;
      }
      break;

    default: return NULL;               // can only fixup words and dwords
   }
  code_fixup[i].ofs=0xFFFF;             // fixup has been handled
  return buffer;
}
#else
unsigned dolist,global_inf_changed,disasm_silent;
#endif

/******************************************************************************/
char *DispFix(char *buf,FILE *f,long pos,GEOSfixup *fix)
{
        long old;
        unsigned w1,w2;
        char *p;

        old=ftell(f);                   // Position merken
        fseek(f,pos+fix->ofs,SEEK_SET); // Zur Fixup-Position
        sprintf(buf,"[%02x] ",fix->type & 0xFF);
        p=buf+strlen(buf);
        switch(fix->type & 0xF0) {
          case 0x00: strcpy(p, "Kernel       "); break;
          case 0x10: sprintf(p,"Library #%02x  ",fix->type>>8); break;
          case 0x20: strcpy(p,"Program      "); break;
            default: strcpy(p,"???          "); break;
        }
        p=buf+strlen(buf);
        switch(fix->type & 0x0F) {
          case 0x0: case 0x4:
            fread(&w1,sizeof(w1),1,f);
            fread(&w2,sizeof(w2),1,f);
            sprintf(p,((fix->type & 0xF0)==0x20)?"Seg:0x%04x Ofs:0x%04x":
                                                 "Ptr #0x%04x",w1,w2);
            break;
          case 0x1:
            fread(&w1,sizeof(w1),1,f);
            sprintf(p,"Off #0x%04x",w1);
            break;
          case 0x2: case 0x3:
            fread(&w1,sizeof(w1),1,f);
            sprintf(p,((fix->type & 0xF0)==0x20)?"Seg:0x%04x":
                                                 "Seg #0x%04x",w1);
            break;
        }
        fseek(f,old,SEEK_SET);          // Zurck zur gemerkten Position
        return buf;                     // Zeiger auf Puffer zurck
}

void prepost_hook(unsigned ofs,unsigned len,int silent_run,int type)
{
        unsigned i;
        char buf[64];

        switch(type) {                  // check various locations in code
          case 1:                       // before instruction
            break;

          /* "post" hook: Display fixups in the range covered by the current
             instruction that could not be integrated. Usually, these are
             pointers in data areas being mistaken for code etc. */
          case 2:                       // after instruction
            for(i=0;i<code_fixlen; i++) // check all fixups
              if(code_fixup[i].ofs>=ofs && code_fixup[i].ofs<ofs+len) {
                                        // falls into instruction range
                if(silent_run==0)       // visible run: display fixup
                  printf("; Bad reloc @ 0x%04x %s\n",
                    code_fixup[i].ofs,
                    DispFix(buf,
                      code_file,segd[this_seg].pos+base,&code_fixup[i]));
                if(silent_run>=0) {     // Final run for each segment
                  if(silent_run==1)     // Make segment "data" in final
                                        // run for segment (globally invisible)
                    add_data_map(this_seg,ofs);
                  code_fixup[i].ofs=0xFFFF;
                                        // fixup has been dealt with
                }
              }
            break;
        }
}

/******************************************************************************/

char *strncpy2(char *d,char *s,int n)
{
        strncpy(d,s,n);
        d[n]='\0';
        return d;
}

void strtrunc(char *s,int len)
{
        char *p;

        if(p=strchr(s,'\r')) *p='\0';   // Max. 1. Zeile anzeigen
        if(strlen(s)>(size_t)len)       // String trotzdem zu lang?
          strcpy(s+len-3,"...");        // Ja: String mit "..." abschneiden
}

char *DispTok(char *buf,GEOStoken *tok)
{
        char t[GEOS_TOKENLEN+1];

        strncpy(t,tok->str,GEOS_TOKENLEN);
                                        // Token in Puffer
        t[GEOS_TOKENLEN]='\0';          // Endnull anfgen
        if(*t)                          // Token vorhanden?
          sprintf(buf,"%4s,%u",t,tok->num);
                                        // Token in Buffer formatieren
        else
          strcpy(buf,"-");              // Sonst nur Ersatz
        return buf;                     // Zeiger auf Puffer zurck
}

char *DispRel(char *buf,GEOSrelease *rel)
{
        sprintf(buf,"%u.%u  %u-%u",     // Release-Nummer formatieren
                    rel->versmaj,rel->versmin,rel->revmaj,rel->revmin);
        return buf;                     // Zeiger auf Puffer zurck
}

char *DispPro(char *buf,GEOSprotocol *rel)
{
        sprintf(buf,"%u.%03u",rel->vers,rel->rev);
                                        // Protokoll-Nummer formatieren
        return buf;                     // Zeiger auf Puffer zurck
}

#define Hexdump(buf) DispHex(#buf":",(unsigned char *)&buf,-(int)sizeof(buf))
void DispHex(char *s,unsigned char *p,int n)
{
        int i,j;

        if(n<0) {                       // n negativ: Leerbereiche unterdrcken
          n=-n;
          for(i=0;i<n && !p[i];i++)     // Prfen, ob alles Nullen
            ;                           
          if(i==n) return;              // Nur Nullen: Nicht zeigen
        }
        for(i=0;i<n;i++) {              // Ganzen Buffer durchgehen
          if(i%16==0)                   // Neue Zeile?
            printf("%14s ",i?"":s);     // Einrckung ausgeben
          printf("%02x ",p[i]);         // Ein Byte ausgeben
          if(i%16==15 || i==n-1) {      // Zeilenende?
            for(j=i%16;j<15;j++)        // Evtl. bis zum rechten Rand fuellen
              printf("   ");
            for(j=i-(i%16);j<=i;j++)    // Bytes der letzten Zeile durgehen
              if(p[j]>=' ' && p[j]<='z')// ASCII-Zeichen?
                putchar(p[j]);          // Ausgeben
              else
                putchar('.');           // Sonst durch "." ersetzen
            putchar('\n');
          }
        }
}

void DispFile(FILE *f,long pos,unsigned len,unsigned ofs,long hdl)
{
        char buf[80],adr[16];
        unsigned x,bs;

        if(pos!=-1)                     // Position angebgen?
          fseek(f,pos,SEEK_SET);        // Ja: zum Anfang des Blocks
        for(x=0;x<len;x+=bs) {          // Segment lesen
          bs=len-x;                     // default: Ganzen Rest lesen
          if(bs>16)                     // Zu gro fr Puffer?
            bs=16;                      // Nur einen Puffer lesen
          if(hdl>-1)                    // Handle bergeben?
            sprintf(adr,"[%04X] %04X:",(unsigned)hdl,x+ofs);
                                        // Handle/Offset als Hexzahlen
          else
            sprintf(adr,(hdl==-2)?"* Data: %04X:":"%04X:",x+ofs);
                                        // Offset als Hexzahl
          hdl=-1;                       // Handles etc. nur in erster Zeile
          fread(buf,bs,1,f);            // Daten in Puffer lesen
          DispHex(adr,buf,bs);          // Pufferinhalt ausgeben
        }
}

void DisplayHeap(FILE *f,long pos,unsigned size)
{
        GEOSlocalheap lh;               // Kopf des lokalen Heaps
        GEOSlocallist *hdl;             // Maximal 256 Handles pro Heap
        unsigned i;
        unsigned blksize;
        struct { unsigned blksize; unsigned nextofs; } freehead;
        unsigned ofs;

        GetStructP(lh,pos);             // Kopf des lokalen Heaps holen
        printf("\n"
               " * LMem: Size: %5d Bytes   Handle list: @ 0x%04x   Entries: %d\n"
               "         Free: %5d Bytes\n"
               "        Flags: %04x\n"
               "         Type: %04x\n",
               lh.blocksize,lh.hdllistofs,lh.hdllistnum,lh.freesize,
               lh.LMBH_flags,lh.LMBH_lmemType);
                                        // Heap anzeigen
        hdl=calloc(lh.hdllistnum,sizeof(hdl[0]));
                                        // Platz fr Handle-Liste reservieren
        fseek(f,pos+lh.hdllistofs,SEEK_SET);
                                        // Zum Anfang der Handle-Liste im File
        fread(hdl,lh.hdllistnum,sizeof(hdl[0]),f);
                                        // Handle-Liste einlesen
        for(i=0;i<lh.hdllistnum;i++) {  // Liste durchgehen
          if(hdl[i] && hdl[i]!=0xFFFF) {// Unbenutzte Handles nicht zeigen
            GetStructP(blksize,pos+hdl[i]-2);
                                        // Angeforderte Lnge des Blocks
                                        // (tatschlich belegte Lnge ist immer
                                        // auf Vielfache von 4 aufgerundet)
            if(blksize>2)               // Nur wenn Block wirklich gefllt
              DispFile(f,-1,blksize-2,
                       hdl[i],lh.hdllistofs+i*sizeof(GEOSlocallist));
                                        // Daten des Handles anzeigen
          }
        }
        ofs=lh.freeofs;                 // Offset des ersten freien Blocks
        while(ofs) {                    // Solange freien Blcke da sind
          GetStructP(freehead,pos+ofs-2);
                                        // Kopf des freien Blocks holen
          printf("         %04X: *** free: %5d bytes\n",ofs,freehead.blksize);
                                        // Freien Block anzeigen
          ofs=freehead.nextofs;         // Zeiger auf nchsten Block
        }
        free(hdl);                      // Handle-Liste nicht mehr ntig
}

#ifndef NODISASM
void DispCode(FILE *f,long pos,unsigned size,
              GEOSfixup *fix,unsigned fixlen,
              unsigned seg,unsigned ofs)
{
        unsigned char *p=malloc(size);

        fseek(f,pos,SEEK_SET);
        fread(p,size,1,f);
        code_fixup=fix;
        code_fixlen=fixlen;
        code_file=f;
        this_seg=seg;                   // make segment publicly available
        disassemble(p,size,ofs,disasm_silent);
        free(p);
}
#endif

void DisplaySegment(FILE *f,unsigned n,unsigned i)
{
        GEOSfixup *fix;                 // Fixup-Eintrag
        GEOSlocalheap lh;               // Kopf des lokalen Heaps
        unsigned x;
        char buf[80];
        unsigned data_ofs,data_len;

        if(!disasm_silent) {
#ifndef NODISASM
          if(dolist)
            printf("SEG%-3d  SEGMENT  ;",i);
          else
#endif
            printf("Resource %d:\n",i);
          printf("  File offset: @ 0x0%-8lx Size: 0x0%04x     Relocs: %d\n"
                 "%c       Flags: %04x\n",
                 segd[i].pos+base,segd[i].len,segd[i].fixlen/sizeof(*fix),
                 dolist?';':' ',segd[i].flags);
        }
        if(dodump) {
          fix=malloc(segd[i].fixlen);   // Platz fr Fixups
          fseek(f,segd[i].pos+base+((segd[i].len+0xF)&0xFFF0),SEEK_SET);
                                        // Zu Fixups springen
          fread(fix,1,segd[i].fixlen,f);// Fixups lesen

          data_ofs = 0;
          data_len = segd[i].len;

          if(segd[i].flags & 0x8) {
            GetStructP(lh,segd[i].pos+base);
                                        // Kopf des lokalen Heaps holen
            data_ofs = sizeof(lh);
            data_len = lh.hdllistofs - data_ofs;
#ifndef NODISASM
            if(dolist)
              if (fix[0].ofs==0)        // Fixup in Heap-Kopf immer vorhanden
                fix[0].ofs = 0xFFFF;
#endif
          }

          if(data_len)                  // Daten im Segment dumpen?
#ifndef NODISASM
            if(dolist && segd[i].len>0)
              DispCode(f,segd[i].pos+base+data_ofs,data_len,fix,segd[i].fixlen/sizeof(*fix),i,data_ofs);
            else
#endif
            {
              putchar('\n');
              DispFile(f,segd[i].pos+base+data_ofs,data_len,data_ofs,-2);
            }

          if(!disasm_silent && (segd[i].flags & 0x8))
            DisplayHeap(f,segd[i].pos+base,segd[i].len);

          if(!disasm_silent) {
            for(x=0; x < segd[i].fixlen/sizeof(*fix); x++)
                                        // Alle Fixups durchgehen
              if(fix[x].ofs != 0xFFFF)  // Nur, wenn nicht im Listing
                printf(
#ifndef NODISASM
                       dolist?"; %12s @ 0x%04x  %s\n":
#endif
                       "%14s @ 0x%04x  %s\n",
                       (x?"":"Relocs:"),
                       fix[x].ofs,DispFix(buf,f,segd[i].pos+base,&fix[x]));
            free(fix);
#ifndef NODISASM
            if(dolist)
              printf("SEG%-3d\tENDS\n",i);
#endif
            printf("\n");
          }
        }
}

void DisplayApplication(FILE *f)
{
        GEOSappheader ah;               // Zusatz-Dateikopf fr Programme
        char buf[80],buf2[80];
        unsigned i;
        long segbase;
        int pass=1;

        GetStructP(ah,(ver==1)?0xC0:(base-8));
                                        // Dateikopf holen
        strcpy(buf,ah.name);            // Namen/Extension zusammensetzen
        strcpy(buf+GEOS_FNAME,ah.ext);
        buf[GEOS_FNAME+GEOS_FEXT]=0;
        printf("\n   Perm. name: %s\n",buf);
        printf(  "         Type: %02x %-33s\n"
                 "    Attribute: %04x\n",
                 ah.type,
                 (ah.type==1)?"[Application]":
                 (ah.type==2)?"[Library]":
                 (ah.type==3)?"[Driver]":
                              "???",
                 ah.attr);

        printf(  " Kernel prot.: %s\n",DispPro(buf,&ah.kernalprot));
        printf(  "         CRC?: 0x0%04x\n",ah.CRC);
        printf(  " Stack/Uninit: %u bytes\n",ah.stacksize);

        Hexdump(ah._x1); Hexdump(ah._x21);
        Hexdump(ah._x3); Hexdump(ah._x33); Hexdump(ah._x4);
        Hexdump(ah._x5);
            
        lib=malloc(ah.numlib*sizeof(*lib));
        segd=malloc(ah.numseg*sizeof(*segd));

        fread(lib,ah.numlib,sizeof(*lib),f);
                                        // Get libraries
        for(i=0;i<ah.numlib;i++) {      // Alle Libraries durchgehen
          strcpy(buf,lib[i].name);      // Library-Namen holen
          buf[GEOS_FNAME]=0;
          printf("%14s  %2d: %s %s (Protocol %s)\n",
                 i?"":"Libraries:",i,buf,
                 (lib[i].type==0x2000)?"Driver":
		 (lib[i].type==0x4000)?"Library":
                                    "???",
                 DispPro(buf2,&lib[i].protocol));
        }

        if(ah.attr & 0x8000) {
          printf("          ???: Res:0x%04x Off:0x%04x\n",ah.x2_seg,ah.x2_ofs);
          printf(" Process obj?: Res:0x%04x Chunk:0x%04x\n",ah.tokenres_seg,ah.tokenres_item);
#ifndef NODISASM
          if(dolist) add_data_map(ah.x2_seg,ah.x2_ofs);
#endif
                                        // I didn't find any code here...
        }
        if(ah.attr & 0x4000) {
          printf("         Init: Res:0x%04x Off:0x%04x\n",ah.initseg,ah.initofs);
#ifndef NODISASM
          if(dolist) add_global_jmp(ah.initseg,ah.initofs);
#endif
        }
        if(ah.attr & 0x2000) {
          printf("     Strategy: Vector located @ Res:0x%04x Off:0x%04x\n",
                 ah.startseg,ah.startofs);
#ifndef NODISASM
          if(dolist) add_jmplist(ah.startseg,ah.startofs,4,4);
#endif
                                        // Add entry point as "jmplist"
        }

        numexp=ah.numexp;               // Anzahl Exportfunktionen global machen
        expt=malloc(numexp*sizeof(*expt));
        fread(expt,ah.numexp,sizeof(*expt),f);
                                        // Exportfunktionen lesen
        if(dodump) {                    // full listing?
          for(i=0;i<ah.numexp;i++) {    // Alle Exportfunktionen durchgehen
            if(i%4==0) printf("%14s ",i?"":"Exports:");
            printf("%5d=%04x:%04x ",i,expt[i].seg,expt[i].ofs);
            if((i%4==3) || (i==ah.numexp-1)) putchar('\n');
#ifndef NODISASM
            add_global_jmp(expt[i].seg,expt[i].ofs);
#endif
          }
        }
        else
          if(ah.numexp)
            printf("%14s %d locations\n","Exports:",ah.numexp);

        segbase=ftell(f);               // Start der Segmenttabelle
        numseg=ah.numseg;
        for(i=0;i<ah.numseg;i++) {
          fseek(f,segbase+sizeof(GEOSseglen)*i,SEEK_SET);
          fread(&segd[i].len,sizeof(GEOSseglen),1,f);
          fseek(f,segbase+sizeof(GEOSseglen)*ah.numseg+sizeof(GEOSsegpos)*i,SEEK_SET);
          fread(&segd[i].pos,sizeof(GEOSsegpos),1,f);
          fseek(f,segbase+(sizeof(GEOSseglen)+sizeof(GEOSsegpos))*ah.numseg
                     +sizeof(GEOSsegfix)*i,SEEK_SET);
          fread(&segd[i].fixlen,sizeof(GEOSsegfix),1,f);
          fseek(f,segbase+(sizeof(GEOSseglen)+sizeof(GEOSsegpos)+sizeof(GEOSsegfix))*ah.numseg
                     +sizeof(GEOSsegflags)*i,SEEK_SET);
          fread(&segd[i].flags,sizeof(GEOSsegflags),1,f);
                                        // Aus jeder Tabelle 1 Eintrag holen
        }

        if(dolist) puts("%");
        puts("");

        disasm_silent=1;                // first runs all silent
#ifndef NODISASM
        global_inf_changed=dolist;      // at least 1 additional run for listing
#else
        global_inf_changed=0;
#endif
        do {
          if(!global_inf_changed || pass>MAX_PASSES) disasm_silent=0;
                                        // converged? Then final run is visible
          if(disasm_silent) fprintf(stderr,"Pass %d...\n",pass);
#ifndef NODISASM
          if(!disasm_silent && dolist) {
            if(n_global_jmp==MAX_GLOBAL_JMP || n_data_map==MAX_DATA_MAP)
              printf("; Aborted prematurely - jump table limit exceeded.\n\n");
            printf("; Disassembly Passes required: %d\n\n",pass);
          }
#endif
          global_inf_changed=0;         // notice all changes in jmp list
          for(i=0;i<ah.numseg;i++) {    // check all segments
            if(disasm_silent) fprintf(stderr,"%4d",i);
            DisplaySegment(f,ah.numseg,i);
                                        // display segment
          }
          if(disasm_silent) fputc('\n',stderr);
          pass++;                       // count passes
        } while(global_inf_changed || disasm_silent);

        free(expt);
        free(segd);
        free(lib);
}

void DisplayDbHdr(FILE *f,long pos,unsigned size)
{
        GEOSdbheader dh;                // Header of dbmanager file

        GetStructP(dh,pos);             // Get header block
        printf("      Mem Hdl: [%04x]    PrimGroup: %04x PrimItem: %04x  _x: %04x\n",
               dh.seg,dh.prim.group,dh.prim.item,dh._x);
}

void DisplayIdx(FILE *f,long pos,unsigned size)
{
        char *bl;
        GEOSdbidx *lh;                  // header of group index
        unsigned min_block,max_block;
        unsigned i,j;

        bl=malloc(size);                // Allocate space for block data
        fseek(f,pos,SEEK_SET);          // Seek to block image
        fread(bl,size,1,f);             // Read block into memory
        lh=(GEOSdbidx *)bl;             // Pointer to block header
        printf("      Mem Hdl: [%04x]    Flags: %04x\n",lh->seg,lh->flags);
        if(!lh->maxitemlist) {          // no items: abort
          printf("        (empty index)\n");
          return;
        }

        min_block=0xFFFF; max_block=0;
        for(i=sizeof(GEOSdbidx),j=0;i<lh->curitemlist;
            i+=sizeof(GEOSdbitemlist)) {
                                        // Dump items
          printf("       <%04x>: block %04x, local hdl [%04x]\n",
                 i,((GEOSdbitemlist *)(bl+i))->block,
                   ((GEOSdbitemlist *)(bl+i))->hdl);
          min_block=min(((GEOSdbitemlist *)(bl+i))->block,min_block);
          max_block=max(((GEOSdbitemlist *)(bl+i))->block,max_block);
          if(++j==64) {
            j=0;
            i+=30;
          }
        }

        for(i=min_block;i<=max_block;i+=sizeof(GEOSdbblocklist))
                                        // Dump blocks
          printf("   block %04x: VM block [%04x],%3d items  _x: %04x\n",
                 i,((GEOSdbblocklist *)(bl+i))->hdl,
                   ((GEOSdbblocklist *)(bl+i))->num,
                   ((GEOSdbblocklist *)(bl+i))->_x);

        free(bl);                       // Release memory for lists
}

void DisplayVMFile(FILE *f)
{
        GEOSvmfheader vh;               // VM-Dateikopf
        GEOSvmfdirheader dh;            // Kopf des VM-DIR-Blocks
        GEOSvmfdirrec dr;               // Einzelner Eintrag im VM-DIR-Block
        int i;
        unsigned idx,hdl;
        long pos;
        long totalUsed,totalFree;

        GetStructP(vh,(ver==1)?0xC0:(base-8));
                                        // Dateikopf holen
        if(vh.IDVM!=GEOS_IDVM) {
          printf("Invalid VM file.\n");
          return;
        }
        printf("\n VM Directory: @ 0x%06lx   Length: %d bytes\n",
               vh.dirptr+base,vh.dirsize);

        GetStructP(dh,vh.dirptr+base);  // Kopf des DIR-Blocks holen
        printf("\n  Blocks free: %-25d"
                 " Handles free: %d"
               "\n  Blocks used: %-23d"
               "Blocks attached: %d"
               "\n   Total size: %ld Bytes"
               "\n        Flags: %04x\n",
               dh.nblocks_free,dh.nhdls_free,dh.nblocks_used,dh.nblocks_loaded,
               dh.totalsize,dh.flags);
        Hexdump(dh._x2); Hexdump(dh._x2b);

        printf("\n Map Block Hdl: [%04x]  DBMap Block Hdl: [%04x]\n"
                 "1st free block: [%04x]  last free block: [%04x]"
               "  1st free Hdl: [%04x]\n",
               dh.hdl_first,dh.hdl_dbmap,
               dh.hdl_1stfree,dh.hdl_lastfree,dh.hdl_1stunused);
        if(dolist) puts("%");
        puts("");

        idx=0;                          // Laufender Zhler fr Handle-Rechnung
        totalUsed=totalFree=0;
        for(i=(dh.dirsize-sizeof(dh))/sizeof(dr);i;i--) {
                                        // Alle DIR-Eintraege durchgehen
          GetStruct(dr);                // Einen Block-Eintrag holen
          printf("[%04x]: ",GeosIdx2Hdl(idx));
          if(dr.blocksize) {            // Belegter Block
            totalUsed += dr.blocksize;  // Blockgre aufsummieren
            printf("     @ 0x%06lx %5d bytes  MemHdl: %04x  UserID: %04x\n",
                   dr.blockptr+base,dr.blocksize,dr.used.hdl,dr.used.ID);
            if(dr.used.flags!=0x00FF) { // Ungewhnliche Flags?
              printf("             Flags: %04x",dr.used.flags);
                                        // Flags ausgeben und aufschlsseln
              if(dr.used.flags & 0x100) printf(", LMem heap");
              if(dr.used.flags & 0x200) printf(", unSAVEed");
              if(!(dr.used.flags & 4))  printf(", Changes in [%04x]",
                                               dr.used.ID);
              printf("\n");
            }
            if(dodump) {
              if(idx==0)                // Directory nicht mehr dumpen
                printf("        (Handle directory)\n");
              else {
                if(GeosIdx2Hdl(idx)==dh.hdl_first)
                  printf("        (Map block)\n");
                if(GeosIdx2Hdl(idx)==dh.hdl_dbmap)
                  printf("        (DBMap block)\n");
                                        // Map-Blcke markieren

                pos=ftell(f);           // Position merken
                GetStructP(hdl,dr.blockptr+base);
                                        // erste Bytes des Blocks lesen
                fseek(f,pos,SEEK_SET);  // Zurck zur Position in Tabelle
                if(dr.used.flags&0x100) // Block mit lokalem Heap?
                  DisplayHeap(f,dr.blockptr+base,dr.blocksize);
                                        // Ja: Im Heap-Format ausdumpen
#if 0
                else if(dr.used.ID==0x1111)
                  DisplayDbHdr(f,dr.blockptr+base,dr.blocksize);
                else if(dr.used.ID==0x2222)
                  DisplayIdx(f,dr.blockptr+base,dr.blocksize);
#endif
                else
                  DispFile(f,dr.blockptr+base,dr.blocksize,0,-2);
                                        // Als Hexdatei ausdumpen
                fseek(f,pos,SEEK_SET);  // Zurck zur Position in Tabelle
              }
              printf("\n");
            }
          }
          else if(dr.blockptr) {        // Freier Block
            totalFree += dr.free.size;
            printf("free @ 0x%06lx %5d bytes   "
                   "Previous: [%04x]  Next: [%04x]\n",
                   dr.blockptr+base,dr.free.size,dr.free.prev,dr.free.next);
          }
          else                          // Freier Eintrag
            printf("**** (unused)\n");
          idx++;
        }
        printf("\nTotal in blocks: %ld Bytes (%ld used, %ld free)\n",
               totalUsed+totalFree,totalUsed,totalFree);
}

/*****************************************************************************
 *      Routines for displaying the common GEOS file header
 *****************************************************************************/
unsigned DisplayHeader(FILE *f)
{
        union {
          GEOSheader h1;
          GEOS2header h2;
        } hd;                         // Standard-Dateikopf
        char buf1[80],buf2[80];

        fread(&hd,sizeof(hd),1,f);    // Dateikopf einlesen

        if(hd.h1.ID==GEOS_ID) {       // G1-Identifikation stimmt?

/*** Version 1 header ***/
          GeosToIBM(hd.h1.name,1);    // Zeichensatz konvertieren
          GeosToIBM(hd.h1.info,1);
          GeosToIBM(hd.h1._copyright,1);
          printf("         Name: %-36s       Token: %s\n",hd.h1.name,DispTok(buf1,&hd.h1.token));
          printf("GEOS filetype: %-36s Application: %s\n",(hd.h1.class==0)?"Executable (1.x)":"VM file (1.x)",DispTok(buf1,&hd.h1.appl));
          printf("      Release: %-36s    Procotol: %s\n",DispRel(buf1,&hd.h1.release),DispPro(buf2,&hd.h1.protocol));
          printf("        Flags: %04x\n",hd.h1.flags);
          if(*hd.h1.info) {           // Evtl. Info ausgeben
            strtrunc(hd.h1.info,60);  // Vor Ausgabe auf 60 Zeichen krzen
            printf("    User info: %s\n",hd.h1.info);
          }
          if(*hd.h1._copyright)  {    // Evtl. Copyright ausgeben
            strncpy2(buf1,hd.h1._copyright,sizeof(hd.h1._copyright));
            printf("    Copyright: %s\n",buf1);
          }

          ver=1;
          base=0;
          return hd.h1.class;         // Dateityp zurckgeben
        }
        else if(hd.h2.ID==GEOS2_ID) { // G2-Identifikation stimmt?

/*** Version 2 header ***/
          GeosToIBM(hd.h2.name,1);    // Zeichensatz konvertieren
          GeosToIBM(hd.h2.info,1);
          GeosToIBM(hd.h2._copyright,1);
          printf("         Name: %-36s       Token: %s\n",hd.h2.name,DispTok(buf1,&hd.h2.token));
          printf("GEOS filetype: %-36s Application: %s\n",
            (hd.h2.class==1)?"Executable (2.x)":
            (hd.h2.class==2)?"VM file (2.x)":
            (hd.h2.class==3)?"Byte level file":
            (hd.h2.class==4)?"Directory info file":
            (hd.h2.class==5)?"Link":
                             "???",
            DispTok(buf1,&hd.h2.appl));
          printf("      Release: %-36s    Procotol: %s\n",DispRel(buf1,&hd.h2.release),DispPro(buf2,&hd.h2.protocol));
          printf("      Created: %02d/%02d/%04d  %02d:%02d.%02d",
            hd.h2.create_date.m,hd.h2.create_date.d,1980+hd.h2.create_date.y,
            hd.h2.create_time.h,hd.h2.create_time.m,2*hd.h2.create_time.s_2);
          if(*hd.h2.password) {       // Evtl. Password ausgeben
            strncpy2(buf1,hd.h2.password,sizeof(hd.h2.password));
            printf("                    Password: %s",buf1);
          }
          putchar('\n');
          printf("        Flags: %04x  %s%s%s%s\n",hd.h2.flags,
            (hd.h2.flags & 0x8000)?"[Template]":"",
            (hd.h2.flags & 0x4000)?"[Public Multiple]":"",
            (hd.h2.flags & 0x2000)?"[Public Single]":"",
            (hd.h2.flags & 0x0800)?"[Hidden]":"");
          if(*hd.h2.info) {           // Evtl. Info ausgeben
            strtrunc(hd.h2.info,60);  // Vor Ausgabe auf 60 Zeichen krzen
            printf("    User info: %s\n",hd.h2.info);
          }
          if(*hd.h2._copyright) {     // Evtl. Copyright ausgeben
            strncpy2(buf1,hd.h2._copyright,sizeof(hd.h2._copyright));
            printf("    Copyright: %s\n",buf1);
          }

          ver=2;                      // GW-Version
          base=sizeof(GEOS2header);
          return hd.h2.class;         // Dateityp zurckgeben
        }

        puts("Not a PC/GEOS file.");
        return 0xFFFF;                // Fehlermeldung - Abbruch
}

#ifndef NODISASM
void disasm_info(char *fname)
{
        FILE *f;
        char buf[256];
        int line=0,n;
        unsigned seg,ofs,len;
        char type;

        if(f=fopen(fname,"rt")) {
          while(fgets(buf,sizeof(buf),f)) {
                                        // Bis zum Dateiende
            line++;                     // Zeilen zhlen
            switch(toupper(buf[0])) {
              case ' ':                 // Kommentarzeilen werden bersprungen
              case '\t':
              case ';':
              case '\n':
                n=0;
                break;
              case 'C':                 // "C seg ofs" adds code entry
                n = sscanf(buf+1," %04x %04x",&seg,&ofs) - 2;
                add_global_jmp(seg,ofs);
                break;
              case 'D':                 // "D seg ofs" adds data entry
                n = sscanf(buf+1," %04x %04x",&seg,&ofs) - 2;
                add_data_map(seg,ofs);
                break;
              case 'J':                 // "J seg ofs len type" adds jumplist
                n = sscanf(buf+1," %04x %04x %04x %c",&seg,&ofs,&len,&type) - 4;
                type=toupper(type);     // Gro/klein egal...
                add_jmplist(seg,ofs,len,
                  (type=='2')?2:        // near jumplist
                  (type=='4')?4:        // far jumplist
                  (type=='S')?16:       // split, segment list comes first
                  (type=='O')?17:       // split, offset list comes first
                              (n=2));   // unknown, force abort...
                break;
              default:
                n=1;
            }
            if(n) {
              printf("Bad disasm info in line %d.\n",line);
              exit(1);
            }
          }
          fclose(f);
        }
        else {
          puts("Cannot load disasm info file.");
          exit(1);
        }
}
#endif

void main(int argc,char *argv[])
{
        FILE *f;
        char path[_MAX_PATH],drive[_MAX_DRIVE],dir[_MAX_DIR],
             name[_MAX_FNAME],ext[_MAX_EXT];
        unsigned class;
        int i;

        dodump=0;                       // default: kein Dump
#ifndef NODISASM
        init_global_jmp();              // no global jmp targets yet
        init_data_map();                // no global jmp targets yet
        init_jmplist();                 // no jumplist yet
#endif

        i=1;
        if(argc>i && (argv[i][0]=='/' || argv[i][0]=='-')) {
          switch(toupper(argv[i][1])) {
#ifndef NODISASM
          case 'L':
            dolist=1;                   // '/L'-Switch: Listing
            if(argv[i][2])              // Filename angegeben?
              disasm_info(argv[i]+2);   // Infos fr Disassembler laden
#endif
          case 'D':
            dodump=1;                   // '/D'-Switch: Dumpen
            break;
          }
          i++;                          // Parameter bergehen
        }
        if(argc<=i) {                   // Zu wenig Parameter?
          puts("\nGeoDump 0.4 -=- by Marcus Grber, "__DATE__"\n"
                 "Disassembly engine based on a module by Robin Hilliard\n"
                 "Analysis of PC/Geos file formats\n");

          puts("Syntax: GEODUMP [/D|/L[file]] filename\n"
               "          /D      Dump contents of blocks/resources"
#ifndef NODISASM
             "\n          /L      List code resources (includes /D)"
             "\n          /Lfile  List, use specified file for additional info"
#endif
          );
          exit(1);
        }

        _splitpath(argv[i],drive,dir,name,ext);
        strupr(name); strupr(ext);      // Dateinamen in Grobuchstaben
        if(*ext=='\0')                  // Extension fehlt: GEO annehmen
          strcpy(ext,".GEO");
        _makepath(path,drive,dir,name,ext);

        if(!(f=fopen(path,"rb"))) {     // Datei zum Lesen ffnen
          puts("File not found.");
          exit(1);                      // Datei fehlt? Fehlermeldung
        }

        if(dolist)                      // Header wird auskommentiert
          puts("comment %");
        printf("\n\t\t\t    Dump of %s%s\n\n",name,ext);
                                        // berschrift
        class=DisplayHeader(f);         // Informationen des allgemeinen Kopfes
        if((class==0 && ver==1) || (class==1 && ver!=1))
                                        // Dateityp "Applikation"
          DisplayApplication(f);        // Applikationsspezifische Informationen
        else if((class==1 && ver==1) || (class==2 && ver!=1))
                                        // Dateityp "VM-Datei"
          DisplayVMFile(f);             // Dateispezifische Informationen
        fclose(f);                      // Datei schlieen

#ifndef NODISASM
        free(global_jmp_map);           // play it safe...
        free(data_map);
#endif
}
