/* Copyright (C) 1993 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gsprops.c */
/* Default device properties for Ghostscript library */
#include "memory_.h"			/* for memcpy */
#include "gx.h"
#include "gserrors.h"
#include "gsparam.h"
#include "gxdevice.h"
#include "gxfixed.h"

/* Get the device properties. */
int
gs_getdeviceprops(gx_device *dev, gs_prop_list *plist)
{	gx_device_set_procs(dev);
	fill_dev_proc(dev, get_props, gx_default_get_props);
	return (*dev_proc(dev, get_props))(dev, plist);
}

/* Get standard properties. */
int
gx_default_get_props(gx_device *dev, gs_prop_list *plist)
{	int colors = dev->color_info.num_components;
	int depth = dev->color_info.depth;
	int GrayValues = dev->color_info.max_gray + 1;
	int code;
	float HWResolution[2];
	  gs_prop_float_array hwra;
	int HWSize[2];
	  gs_prop_int_array hwsa;
	gs_matrix mat;
	float InitialMatrix[6];
	  gs_prop_float_array ima;
	gs_prop_string dns;

	/* Fill in values common to all devices. */

	HWResolution[0] = dev->x_pixels_per_inch;
	HWResolution[1] = dev->y_pixels_per_inch;
	hwra.data = HWResolution, hwra.size = 2, hwra.persistent = false;
	HWSize[0] = dev->width;
	HWSize[1] = dev->height;
	hwsa.data = HWSize, hwsa.size = 2, hwsa.persistent = false;
	fill_dev_proc(dev, get_initial_matrix, gx_default_get_initial_matrix);
	(*dev_proc(dev, get_initial_matrix))(dev, &mat);
	InitialMatrix[0] = mat.xx;
	InitialMatrix[1] = mat.xy;
	InitialMatrix[2] = mat.yx;
	InitialMatrix[3] = mat.yy;
	InitialMatrix[4] = mat.tx;
	InitialMatrix[5] = mat.ty;
	ima.data = InitialMatrix, ima.size = 6, ima.persistent = false;
	prop_string_from_string(dns, dev->dname);

	if ( (code = prop_float_array_param(plist, "HWResolution", &hwra)) < 0 ||
	     (code = prop_int_array_param(plist, "HWSize", &hwsa)) < 0 ||
	     (code = prop_float_array_param(plist, "InitialMatrix", &ima)) < 0 ||
	     (code = prop_string_param(plist, "Name", &dns)) < 0 ||
	     (code = prop_int_param(plist, "Colors", &colors)) < 0 ||
	     (code = prop_int_param(plist, "HWBitsPerPixel", &depth)) < 0 ||
	     (code = prop_int_param(plist, "GrayValues", &GrayValues)) < 0 ||
	     (code = prop_int_param(plist, "PageCount", &dev->page_count)) < 0
	   )
		return code;

	/* Fill in color information. */

	if ( colors > 1 )
	{	int RGBValues = dev->color_info.max_rgb + 1;
		long ColorValues = 1L << depth;
		if ( (code = prop_int_param(plist, "RedValues", &RGBValues)) < 0 ||
		     (code = prop_int_param(plist, "GreenValues", &RGBValues)) < 0 ||
		     (code = prop_int_param(plist, "BlueValues", &RGBValues)) < 0 ||
		     (code = prop_long_param(plist, "ColorValues", &ColorValues)) < 0
		   )
			return code;
	}

	if ( depth <= 8 && colors <= 3 )
	{	byte palette[3 << 8];
		byte *p = palette;
		gs_prop_string hwcms;
		gx_color_value rgb[3];
		gx_color_index i;
		if ( palette == 0 )
			return_error(gs_error_VMerror);
		for ( i = 0; (i >> depth) == 0; i++ )
		{	int j;
			(*dev_proc(dev, map_color_rgb))(dev, i, rgb);
			for ( j = 0; j < colors; j++ )
				*p++ = gx_color_value_to_byte(rgb[j]);
		}
		hwcms.data = palette, hwcms.size = colors << depth, hwcms.persistent = false;
		if ( (code = prop_string_param(plist, "HWColorMap", &hwcms)) < 0 )
			return code;
	}

	return 0;
}

/* Set the device properties. */
/* If the device was open and the put_props procedure closed it, */
/* return 1; otherwise, return 0 or an error code as usual. */
int
gs_putdeviceprops(gx_device *dev, gs_prop_list *plist)
{	bool was_open = dev->is_open;
	int code;
	fill_dev_proc(dev, put_props, gx_default_put_props);
	code = (*dev_proc(dev, put_props))(dev, plist);
	return (code < 0 ? code : was_open && !dev->is_open ? 1 : code);
}

/* Set standard properties. */
/* Note that setting the size or resolution closes the device. */
/* We break out a procedure that only sets the properties, */
/* without closing or reopening the device, for the benefit of */
/* window devices that need to do the latter themselves. */
int
gx_default_put_props_only(gx_device *dev, gs_prop_list *plist)
{	int code = 0;
	gs_prop_int_array hwsa;
	gs_prop_float_array hwra;

	switch ( code = prop_int_array_param(plist, "HWSize", &hwsa) )
	{
	default:
		return code;
	case 1:
		break;
	case 0:
		if ( hwsa.size != 2 )
			return_error(gs_error_typecheck);
#define ap hwsa.data
		if ( ap[0] <= 0 || ap[1] <= 0 )
			return_error(gs_error_rangecheck);
#define max_coord (max_fixed / fixed_1)
#if max_coord < max_int
		if ( ap[0] > max_coord || ap[1] > max_coord )
			return_error(gs_error_limitcheck);
#endif
#undef max_coord
		dev->width = ap[0];
		dev->height = ap[1];
#undef ap
		code = 1;
	}

	switch ( code = prop_float_array_param(plist, "HWResolution", &hwra) )
	{
	default:
		return code;
	case 1:
		break;
	case 0:
		if ( hwra.size != 2 )
			return_error(gs_error_typecheck);
#define ap hwra.data
		if ( ap[0] <= 0 || ap[1] <= 0 )
			return_error(gs_error_rangecheck);
		dev->x_pixels_per_inch = ap[0];
		dev->y_pixels_per_inch = ap[1];
#undef ap
		code = 1;
	}

	return code;
}
int
gx_default_put_props(gx_device *dev, gs_prop_list *plist)
{	gx_device temp_dev;
	int code;
	temp_dev = *dev;
	code = gx_default_put_props_only(&temp_dev, plist);
	if ( code < 0 )
		return code;
	/* Close the device; gs_putdeviceprops will reopen it. */
	if ( dev->is_open && code )
	{	int ccode = gs_closedevice(dev);
		if ( ccode < 0 ) return ccode;
	}
	dev->x_pixels_per_inch = temp_dev.x_pixels_per_inch;
	dev->y_pixels_per_inch = temp_dev.y_pixels_per_inch;
	dev->width = temp_dev.width;
	dev->height = temp_dev.height;
	return code;
}
