/* This file is LOADPRG.C
** contains :
**
**              - loader for a.out programs (emx-format)
**
** Copyright (c) Rainer Schnitker 92,93
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include "DPMI.H"
#include "DPMIDOS.H"
#include "PROCESS.H"
#include "GNUAOUT.H"
#include "LOADPRG.H"
#include "COPY32.H"
#include "START32.H"
#include "CDOSX32.H"
#include "RSX.H"

#define DEFAULT_STACK 0x10000L

#ifdef _EMX_

/*
** arguments and environment at process start:
**
**	environment and argument strings
**	argv[argc]
**	envp[envc]
**	<- ESP
*/
int argvenv(int argc, char **argv, int envc, char **env, NEWPROCESS * proc)
{
    int i;
    DWORD len, stkp;
    DWORD *vectors;
    WORD count = 0;

    vectors = (DWORD *) malloc((argc + envc + 2) * sizeof(DWORD));
    if (vectors == NULL)
	return E2BIG;

    stkp = proc->stackp32;

    /* store env strings in user stack, built vectors */
    for (i = 0; i < envc; i++) {
	len = (DWORD) (WORD) strlen(env[i]) + 1;
	stkp -= len;
	stkp &= ~3L;		/* align4 */
	cpy16_32(proc->data32sel, stkp, env[i], len);
	vectors[count++] = stkp;
    }
    vectors[count++] = 0;	/* last is a NULL pointer */

    /* store arg strings in user stack, built vectors */
    for (i = 0; i < argc; i++) {
	len = (DWORD) (WORD) strlen(argv[i]) + 1;
	stkp -= len;
	stkp &= ~3L;		/* align4 */
	cpy16_32(proc->data32sel, stkp, argv[i], len);
	vectors[count] = stkp;
	count++;
    }
    vectors[count++] = 0;	/* last is a NULL pointer */

    len = (DWORD) (count * sizeof(long));
    stkp -= len;
    cpy16_32(proc->data32sel, stkp, vectors, len);
    free(vectors);

    proc->stackp32 = stkp;
    return 0;
}

int skip_exe_hdr(int filehandle, DWORD * headoff)
{
    struct exe_hdr exehdr;
    struct emx_hdr emxhdr;

    read(filehandle, &exehdr, sizeof(struct exe_hdr));

    if (exehdr.signatur == 0x5a4d) {	/* falls exe-kopf */
	*headoff = ((DWORD) exehdr.hdr_para) * 16;
	if (lseek(filehandle, *headoff, SEEK_SET) == -1) {
	    goto fail;
	}
	read(filehandle, &emxhdr, sizeof(struct emx_hdr));
	if (memcmp(emxhdr.sig, "emx", 3) != 0) {
	    goto fail;
	}
	*headoff = emxhdr.next_hdr;
    }
    if (lseek(filehandle, *headoff, SEEK_SET) == -1)
	goto fail;
    return 0;

  fail:
    *headoff = 0;
    lseek(filehandle, 0, SEEK_SET);
    return -1;
}

#else

/*
** arguments and environment at process start
**
**	environment and argument strings
**	argv[]
**	envp[]
**	pointer to env[0]
**	pointer to argv[0]
**	int argc
**	<- ESP
*/
int argvenv(int argc, char **argv, int envc, char **env, NEWPROCESS * proc)
{
    int i;
    DWORD len, stkp;
    DWORD *vectors;
    WORD count = 3;		/* 0=argc 1=argv 2=env */

    vectors = (DWORD *) malloc((argc + envc + 2 + 3) * sizeof(DWORD));
    if (vectors == NULL)
	return E2BIG;

    stkp = proc->stackp32;

    /* store env strings in user stack, built vectors */
    for (i = 0; i < envc; i++) {
	len = (DWORD) (WORD) strlen(env[i]) + 1;
	stkp -= len;
	stkp &= ~3L;		/* align4 */
	cpy16_32(proc->data32sel, stkp, env[i], len);
	vectors[count++] = stkp;
    }
    vectors[count++] = 0;	/* last is a NULL pointer */

    /* store arg strings in user stack, built vectors */
    for (i = 0; i < argc; i++) {
	len = (DWORD) (WORD) strlen(argv[i]) + 1;
	stkp -= len;
	stkp &= ~3L;		/* align4 */
	cpy16_32(proc->data32sel, stkp, argv[i], len);
	vectors[count] = stkp;
	count++;
    }
    vectors[count++] = 0;	/* last is a NULL pointer */

    len = (DWORD) (count * sizeof(long));
    stkp -= len;
    vectors[0] = argc;
    vectors[1] = stkp + (4L + envc) * sizeof(long);	/* & vectors[3+nenvp+1] */
    vectors[2] = stkp + 3 * sizeof(long);	/* & vectors[3] */
    cpy16_32(proc->data32sel, stkp, vectors, len);
    free(vectors);

    proc->stackp32 = stkp;
    return 0;
}

int skip_exe_hdr(int filehandle, DWORD * headoff)
{
    struct exe_hdr exehdr;

    exehdr.signatur = 0;
    read(filehandle, &exehdr, sizeof(WORD) * 3);

    if (exehdr.signatur == 0x5a4d) {	/* falls exe-kopf */
	*headoff += (DWORD) exehdr.high * 512L;
	if (exehdr.low)
	    *headoff += (DWORD) exehdr.low - 512L;
    }
    if (lseek(filehandle, *headoff, SEEK_SET) != -1)
	return 0;
    else {
	*headoff = 0;
	lseek(filehandle, 0, SEEK_SET);
	return -1;
    }
}

#endif

static int readin_file(short r_bx, short r_ds, long r_edx, long r_ecx)
{
    long count;

    count = r_ecx;
    (WORD) tr.ebx = r_bx;
    (WORD) tr.edx = (WORD) iobuf;
    while (count > 0) {		/* bytes left */
	(WORD) tr.eax = 0x3f00;
	tr.ecx = (count <= IOBUF_SIZE) ? count : (DWORD) IOBUF_SIZE;

	if (realdos())
	    return -1;

	cpy16_32(r_ds, r_edx, iobuf, tr.eax);
	count -= tr.eax;
	if ((WORD) tr.ecx != (WORD) tr.eax)
	    break;
	r_edx += tr.eax;
    }

    return 0;
}

/*
**  rsx/gcc program layout
**
**		    .....................................
**		    |					| no limit
**		    |	       heap			|
**		    |					|
**	 page aligned-----------------------------------|
**		    |					|
**		    |	    stack (default 64KB)	|
**		    |					|
**	 page aligned-----------------------------------|
**		    |	       BSS			|
**		    |					|
**		    |	       DATA			|
**	 t * 0x10000|-----------------------------------|
**		    |					|
**		    |	       TEXT			|
**		    |					|
**  memory   0x10000|----------text-entry---------------|> DPMI memory begin
**  base addr	    |					|
**		    |	       empty			|> no memory
**		    |					|> access->page fault
**		    0-----------------------------------|
*/

int load_protected_program(char *filename, NEWPROCESS * proc)
{
    GNUOUT aout_hdr;
    DWORD sizetext, sizedata, sizestack;
    DWORD headoff;
    int fhandle;

    fhandle = open(filename, O_RDONLY | O_BINARY);
    if (fhandle == -1) {
	return ENOENT;
    }
    /* skip exe-header, return correct offset in file */
    headoff = 0;
    skip_exe_hdr(fhandle, &headoff);

    /* read gnu aout header */
    read(fhandle, &aout_hdr, sizeof(aout_hdr));

    /* test header */
    if ((WORD) aout_hdr.a_info != 0x10b || aout_hdr.a_entry != 0x10000) {
	close(fhandle);
	return ENOEXEC;
    }
    sizetext = (aout_hdr.a_text + SEGMENT_SIZE - 1L) & ~(SEGMENT_SIZE - 1L);
    sizedata = aout_hdr.a_data + ((aout_hdr.a_bss + 4095L) & ~4095L);
    if (opt_stackval)
	sizestack = (DWORD) opt_stackval *1024;
    else
	sizestack = DEFAULT_STACK;
    if (proc == &RSX_PROCESS)	    /* save stackspace for emu */
	sizestack = 0;
    proc->membytes = sizedata + sizetext + sizestack;
    proc->codesize = aout_hdr.a_text;
    proc->datasize = aout_hdr.a_data + aout_hdr.a_bss;
    proc->entry = aout_hdr.a_entry;
    proc->brk_value = N_DATADDR(aout_hdr) + sizedata + sizestack;
    proc->init_brk = proc->brk_value;
    proc->pagefree = 0;
    proc->stackp32 = proc->init_brk - 4L;
    proc->stacksize = sizestack ;
    if (proc == &RSX_PROCESS) {
	proc->stacksize = sizetext - proc->codesize ;
	proc->stackp32 = 0x1FFF0L ;
    }

    /* MEMORY per DPMI besorgen */
    if (AllocMem(proc->membytes, &(proc->memhandle), &(proc->memaddress)))
	return ENOMEM;

    /* descriptor base address ( first segment without real memory ) */
    proc->dsbase = proc->memaddress - N_TXTADDR(aout_hdr);

    /* alloc 32bit cs,ds ldt selector */
    AllocLDT(2, &(proc->code32sel));
    proc->data32sel = proc->code32sel + 8;

    /* fill descriptors */
    SetBaseAddress(proc->code32sel, proc->dsbase);
    SetBaseAddress(proc->data32sel, proc->dsbase);
    SetAccess(proc->code32sel, APP_CODE_SEL, DEFAULT_BIT | GRANULAR_BIT);
    SetAccess(proc->data32sel, APP_DATA_SEL, DEFAULT_BIT | GRANULAR_BIT);

    /* allow execute data & stack  (GDB??) */
    if (opt_allow)
	SetLimit(proc->code32sel,
		 N_TXTADDR(aout_hdr) + proc->membytes - 1L);
    else
	SetLimit(proc->code32sel,
		 aout_hdr.a_text + N_TXTADDR(aout_hdr) - 1L);

    /* allow mem access (sorry,destroy protection) */
    if (opt_memaccess)
	SetLimit(proc->data32sel, 0xFFFFFFFF);
    else
	SetLimit(proc->data32sel,
		 N_TXTADDR(aout_hdr) + proc->membytes - 1L);

    /* read in code */
    lseek(fhandle, headoff + N_TXTOFF(aout_hdr), SEEK_SET);
    readin_file(fhandle, proc->data32sel, N_TXTADDR(aout_hdr), aout_hdr.a_text);

    /* read in data,bss */
    lseek(fhandle, headoff + N_DATOFF(aout_hdr), SEEK_SET);
    readin_file(fhandle, proc->data32sel, N_DATADDR(aout_hdr),
		aout_hdr.a_data + aout_hdr.a_bss);

    close(fhandle);

    /* zero bss segment */
    bzero32(proc->data32sel, N_BSSADDR(aout_hdr), aout_hdr.a_bss);

    return 0;
}
