/* This file is GDT.C
**
** this program shows the GDT, IDT and LDT tables
*/

#include <stdio.h>
#include <stdlib.h>

typedef unsigned char BYTE;
typedef unsigned short WORD;
typedef unsigned long DWORD;

typedef struct {
    WORD lim_lo;
    WORD base_lo;
    BYTE base_mi, access;
    BYTE lim_hi, base_hi;
} DESCRIPTOR;

typedef struct {
    WORD limit, lo, hi;
} GDTR;

#define DESC_BASE(d)  (((((DWORD)d.base_hi<<8)|(DWORD)d.base_mi)<<16)|(DWORD)d.base_lo)
#define DESC_LIMIT(d) ((((DWORD)(d.lim_hi&15))<<16)|(DWORD)d.lim_lo)

/* descriptor fields */
#define ACCESS_BIT              1
#define READ_BIT                2
#define WRITE_BIT               2
#define CONFIRMING_BIT          4
#define EXPAND_BIT              4
#define CODE_BIT                8
#define SEGMENT_BIT             16
#define PRESENT_BIT             128
#define AVL_BIT                 16
#define DEFAULT_BIT             64
#define BIG_BIT                 64
#define GRANULAR_BIT            128
#define DPL_MASK                96
#define TYPE_MASK               15
#define LIMIT_HI_MASK           15

#define APP_CODE_SEL            250	/* dpl=3 , read */
#define APP_DATA_SEL            242	/* dpl=3 ,write */

#define TYPE_CODE               10
#define TYPE_CONFIRM_CODE       14
#define TYPE_DATA               2
#define TYPE_EXPAND_DATA        6
#define TYPE_286TSS             1
#define TYPE_LDT                2
#define TYPE_BUSY286TSS         3
#define TYPE_286CALLGATE        4
#define TYPE_TASKGATE           5
#define TYPE_286INTGATE         6
#define TYPE_286TRAPGATE        7
#define TYPE_386TSS             9
#define TYPE_BUSY386TSS         11
#define TYPE_386CALLGATE        12
#define TYPE_386INTGATE         14
#define TYPE_386TRAPGATE        15


/* LDT Descriptor management services  DPMI 0.9 */
int AllocLDT(WORD, WORD *);
int FreeLDT(WORD);
int SegToSel(WORD, WORD *);
WORD SelInc(void);		/* can't fail */
int LockSel(WORD);
int UnlockSel(WORD);
int GetBaseAddress(WORD, DWORD *);
int SetBaseAddress(WORD, DWORD);
int SetLimit(WORD, DWORD);
int SetAccess(WORD, BYTE, BYTE);
int CreatAlias(WORD, WORD *);
int GetDescriptor(WORD, DESCRIPTOR *);
int SetDescriptor(WORD, DESCRIPTOR *);
int AllocSpecialLDT(WORD);
int GetDescriptor32(WORD, DESCRIPTOR *);
int SetDescriptor32(WORD, DESCRIPTOR *);

void sgdt(GDTR *);
void sidt(GDTR *);
WORD sldt(void);

void get_segment(WORD, void *, void *, unsigned int);


void PrintDescriptor(DESCRIPTOR * d)
{
    BYTE flag;

    if (d->access & PRESENT_BIT == 0) {
	printf("Not Present\n");
	return;
    }
    if (d->access & SEGMENT_BIT)/* code or data selector */
	printf("Base=%02X%02X%04X Limit=%01X%04X %s DPL=%u %s %s %s %s\n",
	       d->base_hi, d->base_mi, d->base_lo,
	       d->lim_hi & LIMIT_HI_MASK, d->lim_lo,
	       (d->lim_hi & GRANULAR_BIT) ? "Pages" : "Bytes",
	       (d->access & DPL_MASK) >> 5,
	       (d->lim_hi & DEFAULT_BIT) ? "32bit" : "16bit",
	       (d->access & CODE_BIT) ?
	       (d->access & CONFIRMING_BIT) ? "CONF" : "CODE" :
	       (d->access & EXPAND_BIT) ? "EXPD" : "DATA",
	       (d->access & WRITE_BIT) ?
	       (d->access & CODE_BIT) ? "R" : "W" : "-",
	       (d->access & ACCESS_BIT) ? "ACC" : " ");

    else {			/* system selector */
	flag = d->access & (BYTE) 7;
	switch (flag) {
	case 0:
	    break;
	case 1:
	    printf("TSS%s ", (d->access & 8) ? "386" : "286");
	    break;
	case 2:
	    printf("LDT ");
	    break;
	case 3:
	    printf("BUSYTSS%s ", (d->access & 8) ? "386" : "286");
	    break;
	case 4:
	    printf("CALLGATE%s ", (d->access & 8) ? "386" : "286");
	    break;
	case 5:
	    printf("TASKGATE ");
	    break;
	case 6:
	    printf("INT_GATE%s ", (d->access & 8) ? "386" : "286");
	    break;
	case 7:
	    printf("TRAPGATE%s ", (d->access & 8) ? "386" : "286");
	    break;
	}


	if (flag <= 3) {
	    printf("Base=%02X%02X%04X Limit=%01X%04X DPL=%u\n",
		   d->base_hi, d->base_mi, d->base_lo,
		   d->lim_hi & LIMIT_HI_MASK, d->lim_lo,
		   (d->access & DPL_MASK) >> 5);
	} else {
	    printf("Sel=%04X Offset=%02X%02X%04X DPL=%u\n",
		   d->base_lo,
		   d->base_hi, d->lim_hi, d->lim_lo,
		   (d->access & DPL_MASK) >> 5);
	}
    }				/* else system descriptor */
}


void gdt()
{
    GDTR gdtr;			/* struct for sgdt,sidt call ,6 Byte Buffer */
    WORD sel, ldtsel, idtsel;
    DWORD gdtbase, ldtbase, idtbase, gdtlimit, ldtlimit, idtlimit;
    unsigned i;
    DESCRIPTOR desc;
    DESCRIPTOR *gdt;		/* pointer to gdt of DPMI server */
    DESCRIPTOR *idt;		/* pointer to idt of DPMI server */

    sgdt(&gdtr);
    printf("\nGDT : limit %04X address %04X%04X \n",
	   gdtr.limit, gdtr.hi, gdtr.lo);
    printf("LDT selector= %X\n", (ldtsel = sldt()));

    /* now we must alloced one selector and set the address
    **    to the gdt address and limit */

    gdtbase = (DWORD) gdtr.hi << 16 | gdtr.lo;
    gdtlimit = gdtr.limit;

    if (AllocLDT(1, &sel)) {
	printf("error allocldt\n");
	return;
    }
    if (SetBaseAddress(sel, gdtbase)) {
	printf("err setbase\n");
	return;
    }
    if (SetLimit(sel, gdtlimit)) {
	printf("err setlimit\n");
	return;
    }
    /* the same procedure for the idt table */
    sidt(&gdtr);
    printf("IDT : limit %04X address %04X%04X \n", gdtr.limit, gdtr.hi, gdtr.lo);
    idtbase = ((DWORD) gdtr.hi << 16) | gdtr.lo;
    idtlimit = gdtr.limit;

    printf("GLOBALE DESCRIPTOR TABELLE:\n");

    gdt = (DESCRIPTOR *) 0;
    gdtlimit++;
    gdtlimit /= 8;		/* one descriptor = 8 bytes */

    for (i = 0; i < gdtlimit; i++) {
	get_segment(sel, gdt + i, &desc, sizeof(DESCRIPTOR));
	if (desc.access != 0) {	/* exclude null selectors */
	    printf("g=%04X ", i << 3);
	    PrintDescriptor(&desc);
	}
	if (ldtsel == i * 8) {	/* to get the address of the ldt table */
	    ldtlimit = DESC_LIMIT(desc);
	    ldtbase = DESC_BASE(desc);
	}
    }

    printf("ldtbase ldt=%lX  limit %lX\n", ldtbase, ldtlimit);
    printf("gebrauchter selektor %X\n", sel);
    if ((SetBaseAddress(sel, ldtbase)) == -1) {
	printf("err setbase\n");
	return;
    };
    if ((SetLimit(sel, ldtlimit)) == -1) {
	printf("err setlimit\n");
	return;
    };

    printf("LOKALE DESCRIPTOR TABELLE:\n");
    ldtlimit++;
    ldtlimit /= 8;
    for (i = 0; i < ldtlimit; i++) {
	get_segment(sel, gdt + i, &desc, sizeof(DESCRIPTOR));
	if (desc.access != 0) {	/* exclude null selectors */
	    printf("l=%04X ", (i << 3) + 7);
	    PrintDescriptor(&desc);
	}
    }

    printf("INTERRUPT DESCRIPTOR TABELLE:\n");
    if ((SetBaseAddress(sel, idtbase)) == -1) {
	printf("err setbase\n");
	return;
    }
    if ((SetLimit(sel, idtlimit)) == -1) {
	printf("err setlimit\n");
	return;
    }
    idt = (DESCRIPTOR *) 0;
    idtlimit++;
    idtlimit /= 8;

    for (i = 0; i < idtlimit; i++) {
	printf("int %3X ", i);
	get_segment(sel, idt + i, &desc, sizeof(DESCRIPTOR));
	PrintDescriptor(&desc);
    }

    FreeLDT(sel);
}

void main()
{
    gdt();
}
