/******************************************************************************/
/*                                                                            */
/*      Modul zur Ansteuerung der seriellen Schnittstelle ber INT 14H        */
/*                                                                            */
/* (C) 1992 by                                                                */
/*     Ullrich von Bassewitz                                                  */
/*     Zwehrenbhlstrae 33                                                   */
/*     D-7400 Tbingen                                                        */
/*                                                                            */
/******************************************************************************/




#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include "i14.h"



static const unsigned char IBaudrate [8] = {
    0x00, 0x20, 0x40, 0x60, 0x80, 0xA0, 0xC0, 0xE0
};
static const unsigned char IParity [3] = {
    0x00, 0x08, 0x18
};
static const unsigned char IStopBits [2] = {
    0x00, 0x04
};
static const unsigned char IDataBits [4] = {
    0x00, 0x01, 0x02, 0x03
};


static const char Xon = 0x11;           // ^Q
static const char Xoff = 0x13;          // ^S


// Anzahl Wiederholungen bei Timeout
static const unsigned InitTimeoutRetries = 20;




/*********************************************************************/
/*                                                                   */
/*                        class SerialPort                           */
/*                                                                   */
/*********************************************************************/



// Konstruktor
SerialPort::SerialPort (unsigned Port,
                        TBaudRate Baudrate,
                        TParity Parity,
                        unsigned StopBits,
                        unsigned DataBits) :
        PortNr (Port-1),
        MaxTries (InitTimeoutRetries)

{
    // Serielle Schnittstelle initialisieren
    unsigned char InitVal = IBaudrate [Baudrate] | IParity [Parity] |
                            IStopBits [StopBits-1] | IDataBits [DataBits-5];

    asm les     bx, [this]
    asm mov     ah, 0
    asm mov     al, [InitVal]
    asm mov     dx, es:[bx]. (SerialPort) PortNr
    asm int     14h

}





void SerialPort::OutByte (char B)
{
    unsigned PortStatus;


    // Warten bis ausgegeben werden darf
    while (!HandShakeOk ()) ;

    // Zeichen ausgeben
    unsigned Tries = MaxTries;
    do {
        asm les     bx, [this]
        asm mov     dx, es:[bx]. (SerialPort) PortNr
        asm mov     ah, 1
        asm mov     al, [B]
        asm int     14h
        asm mov     al, 0               // Unteres Byte ist ungltig
        PortStatus = _AX;
    } while ((--Tries) && ((PortStatus & 0x8000) != 0));

    // Portstatus prfen
    CheckPortStatus (PortStatus);
}






void SerialPort::OutString (char far *S)
{
    if (S != NULL) {
        while (*S) {
            OutByte (*S++);
        }
    }
}





char SerialPort::InByte ()
// Liest ein Zeichen vom seriellen Port ein
{
    char B;

    asm les     bx, [this]
    asm mov     dx, es:[bx]. (SerialPort) PortNr
    asm mov     ah, 02h
    asm int     14h
    asm mov     [B], al         // Byte merken
    asm mov     al, 0           // Status nur in ah

    CheckPortStatus (_AX);

    return B;
}




int SerialPort::ByteAvail ()
// Ergibt 1 wenn ein Eingabe-Byte verfgbar ist, 0 wenn keines
{
    asm les     bx, [this]
    asm mov     dx, es:[bx]. (SerialPort) PortNr
    asm mov     ah, 03h
    asm mov     al, ah          // Ergebnis ist in Bit 8 von ax
    asm and     ax, 0001h
    return _AX;
}





void SerialPort::Error (unsigned Errno)
// Bricht das Programm mit einer Fehlermeldung ab. Mu fr andere Reaktionen
// berladen werden.
{
    const unsigned MaxNo = 2;
    static char * Messages [MaxNo] = {
        "Timeout",
        "bertragungsfehler",
    };
    char *Msg;

    Msg = (Errno < MaxNo) ? Messages [Errno-1] : "Unbekannter Fehler";

    fprintf (stderr, "SerialPort: Port #%u: %s\n", PortNr + 1, Msg);
    abort ();

}




void SerialPort::CheckPortStatus (unsigned PortStatus)
// Prft den bergebenen Status und ruft Error auf bei Fehlern
{
    if (PortStatus & 0x8000) {
        // Timeout
        Error (1);
    } else if (PortStatus & 0x0E00) {
        // -Fehler
        Error (2);
    }
}





/*********************************************************************/
/*                                                                   */
/*                       class HardWiredPort                         */
/*                                                                   */
/*********************************************************************/




int HardWiredPort::HandShakeOk ()
{
    asm les     bx, [this]
    asm mov     dx, es:[bx]. (SerialPort) PortNr
    asm mov     ah, 03h
    asm int     14h
    asm xor     bx, bx          // Ergebnis = FALSE annehmen
    asm and     al, 30h         // DSR & CTS prfen
    asm cmp     al, 30h
    asm jne     L1              // Springe wenn nicht ok
    asm inc     bx              // Ergebnis = TRUE
L1: asm xchg    ax, bx          // Ergebnis nach ax
    return _AX;
}







/*********************************************************************/
/*                                                                   */
/*                        class XonXoffPort                          */
/*                                                                   */
/*********************************************************************/




int XonXoffPort::HandShakeOk ()
{
    while (ByteAvail ()) {

        switch (InByte ()) {

            case Xon:   Stopped = 0;
                        break;
            case Xoff:  Stopped = 1;
                        break;

        }
    }

    // Aktuellen Stop-Zustand zurckgeben
    return Stopped;
}




/*****************************************************************************/
/*                                                                           */
/* Die Funktion gibt die Datei auf die serielle Schnittstelle aus. Benutzt   */
/* wird dazu das ROM-BIOS, also der Interrupt 14h. Die bergebenen Parameter */
/* haben folgende Bedeutung:                                                 */
/*                                                                           */
/*   FileName        ist der Name der auszugebenden Datei.                   */
/*   PreString       ist ein String, der vor dieser Datei noch auf die       */
/*                   Schnittstelle ausgegeben wird. Dies kann z.B. ein       */
/*                   Steuerstring sein, der dem Plotter das Protokoll        */
/*                   mitteilt.                                               */
/*   Port            ist ein bereits initialisiertes Schnittstellenobjekt.   */
/*                                                                           */
/* Das Ergebnis ist 0 wenn die Datei erfolgreich ausgegeben werden konnte    */
/* und entspricht bei Fehlern errno. Bei Fehlern bei der Ausgabe auf die     */
/* Schnittstelle erfolgt ein Programmabbruch, es sei denn, es wird ein       */
/* Objekt bergeben, bei dem die Error-Funktion berdefiniert worden ist.    */
/*                                                                           */
/* ------------------------------------------------------------------------- */



int far Plot (char far *FileName, char far *PreString, SerialPort far &Port)
{
    static const unsigned BufSize = 512;

    char far *Buf = NULL;
    FILE *F;
    unsigned Count, I;


    // Puffer belegen
    if ((Buf = new char [BufSize]) == NULL) {
        // Nicht genug Speicher
        return ENOMEM;
    }

    // Datei ffnen
    if ((F = fopen (FileName, "rb")) == NULL) {
        // Speicher freigeben
        delete Buf;

        // Fehlercode zurckgeben
        return errno;
    }

    // Vorstring ausgeben
    Port.OutString (PreString);

    // Dateigre bestimmen
    (void) fseek (F, 0L, SEEK_END);
    long Size = ftell (F);
    (void) fseek (F, 0L, SEEK_SET);

    while (Size) {
        Count = (Size > BufSize) ? BufSize : (unsigned) Size;

        // Puffer aus der Datei lesen
        (void) fread (Buf, Count, 1, F);

        // Puffer ausgeben
        for (I = 0; I < Count; I++) {
            Port.OutByte (Buf [I]);
        }

        // Restzahl verminden
        Size -= Count;
    }


    // Fertig. Datei schlieen, Puffer freigeben, Ergebnis ist 0
    (void) fclose (F);
    delete Buf;

    return 0;

}
